/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.streaming;

import java.io.IOException;
import java.util.Collection;
import java.util.function.UnaryOperator;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.lifecycle.LifecycleNewTracker;
import org.apache.cassandra.db.streaming.CassandraStreamHeader;
import org.apache.cassandra.db.streaming.CassandraStreamReceiver;
import org.apache.cassandra.db.streaming.ComponentManifest;
import org.apache.cassandra.db.streaming.IStreamReader;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableMultiWriter;
import org.apache.cassandra.io.sstable.format.SSTableFormat;
import org.apache.cassandra.io.sstable.format.big.BigTableZeroCopyWriter;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.streaming.ProgressInfo;
import org.apache.cassandra.streaming.StreamReceiver;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.messages.StreamMessageHeader;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CassandraEntireSSTableStreamReader
implements IStreamReader {
    private static final Logger logger = LoggerFactory.getLogger(CassandraEntireSSTableStreamReader.class);
    private final TableId tableId;
    private final StreamSession session;
    private final StreamMessageHeader messageHeader;
    private final CassandraStreamHeader header;
    private final int fileSequenceNumber;

    public CassandraEntireSSTableStreamReader(StreamMessageHeader messageHeader, CassandraStreamHeader streamHeader, StreamSession session) {
        if (streamHeader.format != SSTableFormat.Type.BIG) {
            throw new AssertionError((Object)("Unsupported SSTable format " + (Object)((Object)streamHeader.format)));
        }
        if (session.getPendingRepair() != null && !session.getPendingRepair().equals(messageHeader.pendingRepair)) {
            throw new IllegalStateException(String.format("Stream Session & SSTable (%s) pendingRepair UUID mismatch.", messageHeader.tableId));
        }
        this.header = streamHeader;
        this.session = session;
        this.messageHeader = messageHeader;
        this.tableId = messageHeader.tableId;
        this.fileSequenceNumber = messageHeader.sequenceNumber;
    }

    @Override
    public SSTableMultiWriter read(DataInputPlus in) throws Throwable {
        ColumnFamilyStore cfs = ColumnFamilyStore.getIfExists(this.tableId);
        if (cfs == null) {
            throw new IOException("Table " + this.tableId + " was dropped during streaming");
        }
        ComponentManifest manifest = this.header.componentManifest;
        long totalSize = manifest.totalSize();
        logger.debug("[Stream #{}] Started receiving sstable #{} from {}, size = {}, table = {}", new Object[]{this.session.planId(), this.fileSequenceNumber, this.session.peer, FBUtilities.prettyPrintMemory(totalSize), cfs.metadata()});
        BigTableZeroCopyWriter writer = null;
        try {
            writer = this.createWriter(cfs, totalSize, manifest.components());
            long bytesRead = 0L;
            for (Component component : manifest.components()) {
                long length = manifest.sizeOf(component);
                logger.debug("[Stream #{}] Started receiving {} component from {}, componentSize = {}, readBytes = {}, totalSize = {}", new Object[]{this.session.planId(), component, this.session.peer, FBUtilities.prettyPrintMemory(length), FBUtilities.prettyPrintMemory(bytesRead), FBUtilities.prettyPrintMemory(totalSize)});
                writer.writeComponent(component.type, in, length);
                this.session.progress(writer.descriptor.filenameFor(component), ProgressInfo.Direction.IN, length, length);
                logger.debug("[Stream #{}] Finished receiving {} component from {}, componentSize = {}, readBytes = {}, totalSize = {}", new Object[]{this.session.planId(), component, this.session.peer, FBUtilities.prettyPrintMemory(length), FBUtilities.prettyPrintMemory(bytesRead += length), FBUtilities.prettyPrintMemory(totalSize)});
            }
            UnaryOperator transform = stats -> stats.mutateLevel(this.header.sstableLevel).mutateRepairedMetadata(this.messageHeader.repairedAt, this.messageHeader.pendingRepair, false);
            String description = String.format("level %s and repairedAt time %s and pendingRepair %s", this.header.sstableLevel, this.messageHeader.repairedAt, this.messageHeader.pendingRepair);
            writer.descriptor.getMetadataSerializer().mutate(writer.descriptor, description, transform);
            return writer;
        }
        catch (Throwable e) {
            logger.error("[Stream {}] Error while reading sstable from stream for table = {}", new Object[]{this.session.planId(), cfs.metadata(), e});
            if (writer != null) {
                e = writer.abort(e);
            }
            throw e;
        }
    }

    private File getDataDir(ColumnFamilyStore cfs, long totalSize) throws IOException {
        Directories.DataDirectory localDir = cfs.getDirectories().getWriteableLocation(totalSize);
        if (localDir == null) {
            throw new IOException(String.format("Insufficient disk space to store %s", FBUtilities.prettyPrintMemory(totalSize)));
        }
        File dir = cfs.getDirectories().getLocationForDisk(cfs.getDiskBoundaries().getCorrectDiskForKey(this.header.firstKey));
        if (dir == null) {
            return cfs.getDirectories().getDirectoryForNewSSTables();
        }
        return dir;
    }

    protected BigTableZeroCopyWriter createWriter(ColumnFamilyStore cfs, long totalSize, Collection<Component> components) throws IOException {
        File dataDir = this.getDataDir(cfs, totalSize);
        StreamReceiver streamReceiver = this.session.getAggregator(this.tableId);
        assert (streamReceiver instanceof CassandraStreamReceiver);
        LifecycleNewTracker lifecycleNewTracker = CassandraStreamReceiver.fromReceiver(this.session.getAggregator(this.tableId)).createLifecycleNewTracker();
        Descriptor desc = cfs.newSSTableDescriptor(dataDir, this.header.version, this.header.format);
        logger.debug("[Table #{}] {} Components to write: {}", new Object[]{cfs.metadata(), desc.filenameFor(Component.DATA), components});
        return new BigTableZeroCopyWriter(desc, cfs.metadata, lifecycleNewTracker, components);
    }
}

