/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.snapshot;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import java.time.Instant;
import java.util.Collection;
import java.util.Comparator;
import java.util.PriorityQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.cassandra.concurrent.ExecutorFactory;
import org.apache.cassandra.concurrent.ScheduledExecutorPlus;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.service.snapshot.SnapshotLoader;
import org.apache.cassandra.service.snapshot.TableSnapshot;
import org.apache.cassandra.utils.ExecutorUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotManager {
    private static final ScheduledExecutorPlus executor = ExecutorFactory.Global.executorFactory().scheduled(false, "SnapshotCleanup");
    private static final Logger logger = LoggerFactory.getLogger(SnapshotManager.class);
    private final long initialDelaySeconds;
    private final long cleanupPeriodSeconds;
    @VisibleForTesting
    protected volatile ScheduledFuture cleanupTaskFuture;
    private final PriorityQueue<TableSnapshot> expiringSnapshots = new PriorityQueue<TableSnapshot>(Comparator.comparing(x -> x.getExpiresAt()));

    public SnapshotManager() {
        this(CassandraRelevantProperties.SNAPSHOT_CLEANUP_INITIAL_DELAY_SECONDS.getInt(), CassandraRelevantProperties.SNAPSHOT_CLEANUP_PERIOD_SECONDS.getInt());
    }

    @VisibleForTesting
    protected SnapshotManager(long initialDelaySeconds, long cleanupPeriodSeconds) {
        this.initialDelaySeconds = initialDelaySeconds;
        this.cleanupPeriodSeconds = cleanupPeriodSeconds;
    }

    public Collection<TableSnapshot> getExpiringSnapshots() {
        return this.expiringSnapshots;
    }

    public synchronized void start() {
        this.loadSnapshots();
        this.resumeSnapshotCleanup();
    }

    public synchronized void stop() throws InterruptedException, TimeoutException {
        this.expiringSnapshots.clear();
        if (this.cleanupTaskFuture != null) {
            this.cleanupTaskFuture.cancel(false);
            this.cleanupTaskFuture = null;
        }
    }

    public synchronized void addSnapshot(TableSnapshot snapshot) {
        if (snapshot.isExpiring()) {
            logger.debug("Adding expiring snapshot {}", (Object)snapshot);
            this.expiringSnapshots.add(snapshot);
        }
    }

    @VisibleForTesting
    protected synchronized void loadSnapshots() {
        SnapshotLoader loader = new SnapshotLoader(DatabaseDescriptor.getAllDataFileLocations());
        this.addSnapshots(loader.loadSnapshots());
    }

    @VisibleForTesting
    protected synchronized void addSnapshots(Collection<TableSnapshot> snapshots) {
        logger.debug("Adding snapshots: {}.", (Object)Joiner.on((String)", ").join((Iterable)snapshots.stream().map(s -> s.getId()).collect(Collectors.toList())));
        snapshots.forEach(this::addSnapshot);
    }

    private synchronized void resumeSnapshotCleanup() {
        if (this.cleanupTaskFuture == null) {
            logger.info("Scheduling expired snapshot cleanup with initialDelaySeconds={} and cleanupPeriodSeconds={}", (Object)this.initialDelaySeconds, (Object)this.cleanupPeriodSeconds);
            this.cleanupTaskFuture = executor.scheduleWithFixedDelay(this::clearExpiredSnapshots, this.initialDelaySeconds, this.cleanupPeriodSeconds, TimeUnit.SECONDS);
        }
    }

    @VisibleForTesting
    protected synchronized void clearExpiredSnapshots() {
        Instant now = FBUtilities.now();
        while (!this.expiringSnapshots.isEmpty() && this.expiringSnapshots.peek().isExpired(now)) {
            TableSnapshot expiredSnapshot = this.expiringSnapshots.peek();
            if (expiredSnapshot == null) continue;
            logger.debug("Removing expired snapshot {}.", (Object)expiredSnapshot);
            this.clearSnapshot(expiredSnapshot);
        }
    }

    protected void clearSnapshot(TableSnapshot snapshot) {
        for (File snapshotDir : snapshot.getDirectories()) {
            Directories.removeSnapshotDirectory(DatabaseDescriptor.getSnapshotRateLimiter(), snapshotDir);
        }
        this.expiringSnapshots.remove(snapshot);
    }

    @VisibleForTesting
    public static void shutdownAndWait(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        ExecutorUtils.shutdownNowAndWait(timeout, unit, executor);
    }
}

