/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.lang.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.flink.hadoop.shaded.com.google.common.base.Supplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CacheFlag;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemTestHelper;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.BlockReaderTestUtil;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.LogVerificationAppender;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveEntry;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveInfo;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveIterator;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveStats;
import org.apache.hadoop.hdfs.protocol.CachePoolEntry;
import org.apache.hadoop.hdfs.protocol.CachePoolInfo;
import org.apache.hadoop.hdfs.protocol.CachePoolStats;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.namenode.CacheManager;
import org.apache.hadoop.hdfs.server.namenode.CachedBlock;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.SecondaryNameNode;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocols;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.GSet;
import org.apache.log4j.Appender;
import org.apache.log4j.Logger;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestCacheDirectives {
    static final Log LOG = LogFactory.getLog(TestCacheDirectives.class);
    private static final UserGroupInformation unprivilegedUser = UserGroupInformation.createRemoteUser("unprivilegedUser");
    private static Configuration conf;
    private static MiniDFSCluster cluster;
    private static DistributedFileSystem dfs;
    private static NamenodeProtocols proto;
    private static NameNode namenode;
    private static NativeIO.POSIX.CacheManipulator prevCacheManipulator;
    private static final long BLOCK_SIZE = 4096L;
    private static final int NUM_DATANODES = 4;
    private static final long CACHE_CAPACITY = 16384L;

    private static HdfsConfiguration createCachingConf() {
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setLong("dfs.blocksize", 4096L);
        conf.setLong("dfs.datanode.max.locked.memory", 16384L);
        conf.setLong("dfs.heartbeat.interval", 1L);
        conf.setLong("dfs.cachereport.intervalMsec", 1000L);
        conf.setLong("dfs.namenode.path.based.cache.refresh.interval.ms", 1000L);
        conf.setInt("dfs.namenode.list.cache.pools.num.responses", 2);
        conf.setInt("dfs.namenode.list.cache.directives.num.responses", 2);
        return conf;
    }

    @Before
    public void setup() throws Exception {
        conf = TestCacheDirectives.createCachingConf();
        cluster = new MiniDFSCluster.Builder(conf).numDataNodes(4).build();
        cluster.waitActive();
        dfs = cluster.getFileSystem();
        proto = cluster.getNameNodeRpc();
        namenode = cluster.getNameNode();
        prevCacheManipulator = NativeIO.POSIX.getCacheManipulator();
        NativeIO.POSIX.setCacheManipulator(new NativeIO.POSIX.NoMlockCacheManipulator());
        BlockReaderTestUtil.enableHdfsCachingTracing();
    }

    @After
    public void teardown() throws Exception {
        RemoteIterator<CacheDirectiveEntry> iter = dfs.listCacheDirectives(null);
        while (iter.hasNext()) {
            dfs.removeCacheDirective(iter.next().getInfo().getId());
        }
        TestCacheDirectives.waitForCachedBlocks(namenode, 0, 0, "teardown");
        if (cluster != null) {
            cluster.shutdown();
        }
        NativeIO.POSIX.setCacheManipulator(prevCacheManipulator);
    }

    @Test(timeout=60000L)
    public void testBasicPoolOperations() throws Exception {
        String poolName = "pool1";
        CachePoolInfo info = new CachePoolInfo("pool1").setOwnerName("bob").setGroupName("bobgroup").setMode(new FsPermission(493)).setLimit(150L);
        dfs.addCachePool(info);
        try {
            dfs.addCachePool(info);
            Assert.fail((String)"added the pool with the same name twice");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("pool1 already exists", ioe);
        }
        try {
            dfs.addCachePool(new CachePoolInfo(""));
            Assert.fail((String)"added empty pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            dfs.addCachePool(null);
            Assert.fail((String)"added null pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("CachePoolInfo is null", ioe);
        }
        try {
            proto.addCachePool(new CachePoolInfo(""));
            Assert.fail((String)"added empty pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            proto.addCachePool(null);
            Assert.fail((String)"added null pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("CachePoolInfo is null", ioe);
        }
        info.setOwnerName("jane").setGroupName("janegroup").setMode(new FsPermission(448)).setLimit(314L);
        dfs.modifyCachePool(info);
        try {
            dfs.modifyCachePool(new CachePoolInfo("fool"));
            Assert.fail((String)"modified non-existent cache pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("fool does not exist", ioe);
        }
        try {
            dfs.modifyCachePool(new CachePoolInfo(""));
            Assert.fail((String)"modified empty pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            dfs.modifyCachePool(null);
            Assert.fail((String)"modified null pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("CachePoolInfo is null", ioe);
        }
        try {
            proto.modifyCachePool(new CachePoolInfo(""));
            Assert.fail((String)"modified empty pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            proto.modifyCachePool(null);
            Assert.fail((String)"modified null pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("CachePoolInfo is null", ioe);
        }
        dfs.removeCachePool("pool1");
        try {
            dfs.removeCachePool("pool99");
            Assert.fail((String)"expected to get an exception when removing a non-existent pool.");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("Cannot remove non-existent cache pool", ioe);
        }
        try {
            dfs.removeCachePool("pool1");
            Assert.fail((String)"expected to get an exception when removing a non-existent pool.");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("Cannot remove non-existent cache pool", ioe);
        }
        try {
            dfs.removeCachePool("");
            Assert.fail((String)"removed empty pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            dfs.removeCachePool(null);
            Assert.fail((String)"removed null pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            proto.removeCachePool("");
            Assert.fail((String)"removed empty pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        try {
            proto.removeCachePool(null);
            Assert.fail((String)"removed null pool");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("invalid empty cache pool name", ioe);
        }
        info = new CachePoolInfo("pool2");
        dfs.addCachePool(info);
    }

    @Test(timeout=60000L)
    public void testCreateAndModifyPools() throws Exception {
        String poolName = "pool1";
        String ownerName = "abc";
        String groupName = "123";
        FsPermission mode = new FsPermission(493);
        long limit = 150L;
        dfs.addCachePool(new CachePoolInfo(poolName).setOwnerName(ownerName).setGroupName(groupName).setMode(mode).setLimit(limit));
        RemoteIterator<CachePoolEntry> iter = dfs.listCachePools();
        CachePoolInfo info = iter.next().getInfo();
        Assert.assertEquals((Object)poolName, (Object)info.getPoolName());
        Assert.assertEquals((Object)ownerName, (Object)info.getOwnerName());
        Assert.assertEquals((Object)groupName, (Object)info.getGroupName());
        ownerName = "def";
        groupName = "456";
        mode = new FsPermission(448);
        limit = 151L;
        dfs.modifyCachePool(new CachePoolInfo(poolName).setOwnerName(ownerName).setGroupName(groupName).setMode(mode).setLimit(limit));
        iter = dfs.listCachePools();
        info = iter.next().getInfo();
        Assert.assertEquals((Object)poolName, (Object)info.getPoolName());
        Assert.assertEquals((Object)ownerName, (Object)info.getOwnerName());
        Assert.assertEquals((Object)groupName, (Object)info.getGroupName());
        Assert.assertEquals((Object)mode, (Object)info.getMode());
        Assert.assertEquals((long)limit, (long)info.getLimit());
        dfs.removeCachePool(poolName);
        iter = dfs.listCachePools();
        Assert.assertFalse((String)"expected no cache pools after deleting pool", (boolean)iter.hasNext());
        proto.listCachePools(null);
        try {
            proto.removeCachePool("pool99");
            Assert.fail((String)"expected to get an exception when removing a non-existent pool.");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("Cannot remove non-existent", ioe);
        }
        try {
            proto.removeCachePool(poolName);
            Assert.fail((String)"expected to get an exception when removing a non-existent pool.");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains("Cannot remove non-existent", ioe);
        }
        iter = dfs.listCachePools();
        Assert.assertFalse((String)"expected no cache pools after deleting pool", (boolean)iter.hasNext());
    }

    private static void validateListAll(RemoteIterator<CacheDirectiveEntry> iter, Long ... ids) throws Exception {
        for (Long id : ids) {
            Assert.assertTrue((String)"Unexpectedly few elements", (boolean)iter.hasNext());
            Assert.assertEquals((String)"Unexpected directive ID", (Object)id, (Object)iter.next().getInfo().getId());
        }
        Assert.assertFalse((String)"Unexpectedly many list elements", (boolean)iter.hasNext());
    }

    private static long addAsUnprivileged(final CacheDirectiveInfo directive) throws Exception {
        return unprivilegedUser.doAs(new PrivilegedExceptionAction<Long>(){

            @Override
            public Long run() throws IOException {
                DistributedFileSystem myDfs = (DistributedFileSystem)FileSystem.get(conf);
                return myDfs.addCacheDirective(directive);
            }
        });
    }

    @Test(timeout=60000L)
    public void testAddRemoveDirectives() throws Exception {
        proto.addCachePool(new CachePoolInfo("pool1").setMode(new FsPermission(511)));
        proto.addCachePool(new CachePoolInfo("pool2").setMode(new FsPermission(511)));
        proto.addCachePool(new CachePoolInfo("pool3").setMode(new FsPermission(511)));
        proto.addCachePool(new CachePoolInfo("pool4").setMode(new FsPermission(0)));
        CacheDirectiveInfo alpha = new CacheDirectiveInfo.Builder().setPath(new Path("/alpha")).setPool("pool1").build();
        CacheDirectiveInfo beta = new CacheDirectiveInfo.Builder().setPath(new Path("/beta")).setPool("pool2").build();
        CacheDirectiveInfo delta = new CacheDirectiveInfo.Builder().setPath(new Path("/delta")).setPool("pool1").build();
        long alphaId = TestCacheDirectives.addAsUnprivileged(alpha);
        long alphaId2 = TestCacheDirectives.addAsUnprivileged(alpha);
        Assert.assertFalse((String)"Expected to get unique directives when re-adding an existing CacheDirectiveInfo", (alphaId == alphaId2 ? 1 : 0) != 0);
        long betaId = TestCacheDirectives.addAsUnprivileged(beta);
        try {
            TestCacheDirectives.addAsUnprivileged(new CacheDirectiveInfo.Builder().setPath(new Path("/unicorn")).setPool("no_such_pool").build());
            Assert.fail((String)"expected an error when adding to a non-existent pool.");
        }
        catch (InvalidRequestException ioe) {
            GenericTestUtils.assertExceptionContains("Unknown pool", ioe);
        }
        try {
            TestCacheDirectives.addAsUnprivileged(new CacheDirectiveInfo.Builder().setPath(new Path("/blackhole")).setPool("pool4").build());
            Assert.fail((String)"expected an error when adding to a pool with mode 0 (no permissions for anyone).");
        }
        catch (AccessControlException e) {
            GenericTestUtils.assertExceptionContains("Permission denied while accessing pool", e);
        }
        try {
            TestCacheDirectives.addAsUnprivileged(new CacheDirectiveInfo.Builder().setPath(new Path("/illegal:path/")).setPool("pool1").build());
            Assert.fail((String)"expected an error when adding a malformed path to the cache directives.");
        }
        catch (IllegalArgumentException e) {
            GenericTestUtils.assertExceptionContains("is not a valid DFS filename", e);
        }
        try {
            TestCacheDirectives.addAsUnprivileged(new CacheDirectiveInfo.Builder().setPath(new Path("/emptypoolname")).setReplication((short)1).setPool("").build());
            Assert.fail((String)"expected an error when adding a cache directive with an empty pool name.");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("Invalid empty pool name", e);
        }
        long deltaId = TestCacheDirectives.addAsUnprivileged(delta);
        long relativeId = TestCacheDirectives.addAsUnprivileged(new CacheDirectiveInfo.Builder().setPath(new Path("relative")).setPool("pool1").build());
        RemoteIterator<CacheDirectiveEntry> iter = dfs.listCacheDirectives(null);
        TestCacheDirectives.validateListAll(iter, alphaId, alphaId2, betaId, deltaId, relativeId);
        iter = dfs.listCacheDirectives(new CacheDirectiveInfo.Builder().setPool("pool3").build());
        Assert.assertFalse((boolean)iter.hasNext());
        iter = dfs.listCacheDirectives(new CacheDirectiveInfo.Builder().setPool("pool1").build());
        TestCacheDirectives.validateListAll(iter, alphaId, alphaId2, deltaId, relativeId);
        iter = dfs.listCacheDirectives(new CacheDirectiveInfo.Builder().setPool("pool2").build());
        TestCacheDirectives.validateListAll(iter, betaId);
        dfs.removeCacheDirective(betaId);
        iter = dfs.listCacheDirectives(new CacheDirectiveInfo.Builder().setPool("pool2").build());
        Assert.assertFalse((boolean)iter.hasNext());
        try {
            dfs.removeCacheDirective(betaId);
            Assert.fail((String)"expected an error when removing a non-existent ID");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("No directive with ID", e);
        }
        try {
            proto.removeCacheDirective(-42L);
            Assert.fail((String)"expected an error when removing a negative ID");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("Invalid negative ID", e);
        }
        try {
            proto.removeCacheDirective(43L);
            Assert.fail((String)"expected an error when removing a non-existent ID");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("No directive with ID", e);
        }
        dfs.removeCacheDirective(alphaId);
        dfs.removeCacheDirective(alphaId2);
        dfs.removeCacheDirective(deltaId);
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(relativeId).setReplication((short)555).build());
        iter = dfs.listCacheDirectives(null);
        Assert.assertTrue((boolean)iter.hasNext());
        CacheDirectiveInfo modified = iter.next().getInfo();
        Assert.assertEquals((long)relativeId, (long)modified.getId());
        Assert.assertEquals((long)555L, (long)modified.getReplication().shortValue());
        dfs.removeCacheDirective(relativeId);
        iter = dfs.listCacheDirectives(null);
        Assert.assertFalse((boolean)iter.hasNext());
        CacheDirectiveInfo directive = new CacheDirectiveInfo.Builder().setPath(new Path(".")).setPool("pool1").build();
        long id = dfs.addCacheDirective(directive);
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(directive).setId(id).setReplication((short)2).build());
        dfs.removeCacheDirective(id);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=60000L)
    public void testCacheManagerRestart() throws Exception {
        SecondaryNameNode secondary = null;
        try {
            conf.set("dfs.namenode.secondary.http-address", "0.0.0.0:0");
            secondary = new SecondaryNameNode(conf);
            String pool = "poolparty";
            String groupName = "partygroup";
            FsPermission mode = new FsPermission(511);
            long limit = 747L;
            dfs.addCachePool(new CachePoolInfo("poolparty").setGroupName(groupName).setMode(mode).setLimit(limit));
            RemoteIterator<CachePoolEntry> pit = dfs.listCachePools();
            Assert.assertTrue((String)"No cache pools found", (boolean)pit.hasNext());
            CachePoolInfo info = pit.next().getInfo();
            Assert.assertEquals((Object)"poolparty", (Object)info.getPoolName());
            Assert.assertEquals((Object)groupName, (Object)info.getGroupName());
            Assert.assertEquals((Object)mode, (Object)info.getMode());
            Assert.assertEquals((long)limit, (long)info.getLimit());
            Assert.assertFalse((String)"Unexpected # of cache pools found", (boolean)pit.hasNext());
            int numEntries = 10;
            String entryPrefix = "/party-";
            long prevId = -1L;
            Date expiry = new Date();
            for (int i = 0; i < numEntries; ++i) {
                prevId = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(new Path(entryPrefix + i)).setPool("poolparty").setExpiration(CacheDirectiveInfo.Expiration.newAbsolute(expiry.getTime())).build());
            }
            RemoteIterator<CacheDirectiveEntry> dit = dfs.listCacheDirectives(null);
            for (int i = 0; i < numEntries; ++i) {
                Assert.assertTrue((String)("Unexpected # of cache entries: " + i), (boolean)dit.hasNext());
                CacheDirectiveInfo cd = dit.next().getInfo();
                Assert.assertEquals((long)(i + 1), (long)cd.getId());
                Assert.assertEquals((Object)(entryPrefix + i), (Object)cd.getPath().toUri().getPath());
                Assert.assertEquals((Object)"poolparty", (Object)cd.getPool());
            }
            Assert.assertFalse((String)"Unexpected # of cache directives found", (boolean)dit.hasNext());
            secondary.doCheckpoint();
            String imagePool = "imagePool";
            dfs.addCachePool(new CachePoolInfo("imagePool"));
            prevId = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(new Path("/image")).setPool("imagePool").build());
            dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
            dfs.saveNamespace();
            dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
            boolean fetchImage = secondary.doCheckpoint();
            Assert.assertTrue((String)"Secondary should have fetched a new fsimage from NameNode", (boolean)fetchImage);
            dfs.removeCachePool("imagePool");
            cluster.restartNameNode(new String[0]);
            pit = dfs.listCachePools();
            Assert.assertTrue((String)"No cache pools found", (boolean)pit.hasNext());
            info = pit.next().getInfo();
            Assert.assertEquals((Object)"poolparty", (Object)info.getPoolName());
            Assert.assertEquals((Object)"poolparty", (Object)info.getPoolName());
            Assert.assertEquals((Object)groupName, (Object)info.getGroupName());
            Assert.assertEquals((Object)mode, (Object)info.getMode());
            Assert.assertEquals((long)limit, (long)info.getLimit());
            Assert.assertFalse((String)"Unexpected # of cache pools found", (boolean)pit.hasNext());
            dit = dfs.listCacheDirectives(null);
            for (int i = 0; i < numEntries; ++i) {
                Assert.assertTrue((String)("Unexpected # of cache entries: " + i), (boolean)dit.hasNext());
                CacheDirectiveInfo cd = dit.next().getInfo();
                Assert.assertEquals((long)(i + 1), (long)cd.getId());
                Assert.assertEquals((Object)(entryPrefix + i), (Object)cd.getPath().toUri().getPath());
                Assert.assertEquals((Object)"poolparty", (Object)cd.getPool());
                Assert.assertEquals((long)expiry.getTime(), (long)cd.getExpiration().getMillis());
            }
            Assert.assertFalse((String)"Unexpected # of cache directives found", (boolean)dit.hasNext());
            long nextId = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(new Path("/foobar")).setPool("poolparty").build());
            Assert.assertEquals((long)(prevId + 1L), (long)nextId);
        }
        finally {
            if (secondary != null) {
                secondary.shutdown();
            }
        }
    }

    private static void waitForCachedBlocks(NameNode nn, final int expectedCachedBlocks, final int expectedCachedReplicas, final String logString) throws Exception {
        final FSNamesystem namesystem = nn.getNamesystem();
        final CacheManager cacheManager = namesystem.getCacheManager();
        LOG.info((Object)("Waiting for " + expectedCachedBlocks + " blocks with " + expectedCachedReplicas + " replicas."));
        GenericTestUtils.waitFor(new Supplier<Boolean>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public Boolean get() {
                int numCachedBlocks = 0;
                int numCachedReplicas = 0;
                namesystem.readLock();
                try {
                    GSet<CachedBlock, CachedBlock> cachedBlocks = cacheManager.getCachedBlocks();
                    if (cachedBlocks != null) {
                        for (CachedBlock cachedBlock : cachedBlocks) {
                            ++numCachedBlocks;
                            numCachedReplicas += cachedBlock.getDatanodes(DatanodeDescriptor.CachedBlocksList.Type.CACHED).size();
                        }
                    }
                }
                finally {
                    namesystem.readUnlock();
                }
                if (!(expectedCachedBlocks != -1 && numCachedBlocks != expectedCachedBlocks || expectedCachedReplicas != -1 && numCachedReplicas != expectedCachedReplicas)) {
                    return true;
                }
                LOG.info((Object)(logString + " cached blocks: have " + numCachedBlocks + " / " + expectedCachedBlocks + ".  " + "cached replicas: have " + numCachedReplicas + " / " + expectedCachedReplicas));
                return false;
            }
        }, 500, 60000);
    }

    private static void waitForCacheDirectiveStats(final DistributedFileSystem dfs, final long targetBytesNeeded, final long targetBytesCached, final long targetFilesNeeded, final long targetFilesCached, final CacheDirectiveInfo filter, final String infoString) throws Exception {
        LOG.info((Object)("Polling listCacheDirectives " + (filter == null ? "ALL" : filter.toString()) + " for " + targetBytesNeeded + " targetBytesNeeded, " + targetBytesCached + " targetBytesCached, " + targetFilesNeeded + " targetFilesNeeded, " + targetFilesCached + " targetFilesCached"));
        GenericTestUtils.waitFor(new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                RemoteIterator<CacheDirectiveEntry> iter = null;
                CacheDirectiveEntry entry = null;
                try {
                    iter = dfs.listCacheDirectives(filter);
                    entry = iter.next();
                }
                catch (IOException e) {
                    Assert.fail((String)("got IOException while calling listCacheDirectives: " + e.getMessage()));
                }
                Assert.assertNotNull((Object)entry);
                CacheDirectiveStats stats = entry.getStats();
                if (targetBytesNeeded == stats.getBytesNeeded() && targetBytesCached == stats.getBytesCached() && targetFilesNeeded == stats.getFilesNeeded() && targetFilesCached == stats.getFilesCached()) {
                    return true;
                }
                LOG.info((Object)(infoString + ": " + "filesNeeded: " + stats.getFilesNeeded() + "/" + targetFilesNeeded + ", filesCached: " + stats.getFilesCached() + "/" + targetFilesCached + ", bytesNeeded: " + stats.getBytesNeeded() + "/" + targetBytesNeeded + ", bytesCached: " + stats.getBytesCached() + "/" + targetBytesCached));
                return false;
            }
        }, 500, 60000);
    }

    private static void waitForCachePoolStats(final DistributedFileSystem dfs, final long targetBytesNeeded, final long targetBytesCached, final long targetFilesNeeded, final long targetFilesCached, final CachePoolInfo pool, final String infoString) throws Exception {
        LOG.info((Object)("Polling listCachePools " + pool.toString() + " for " + targetBytesNeeded + " targetBytesNeeded, " + targetBytesCached + " targetBytesCached, " + targetFilesNeeded + " targetFilesNeeded, " + targetFilesCached + " targetFilesCached"));
        GenericTestUtils.waitFor(new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                block6: {
                    CachePoolEntry entry;
                    RemoteIterator<CachePoolEntry> iter = null;
                    try {
                        iter = dfs.listCachePools();
                    }
                    catch (IOException e) {
                        Assert.fail((String)("got IOException while calling listCachePools: " + e.getMessage()));
                    }
                    do {
                        entry = null;
                        try {
                            if (!iter.hasNext()) break block6;
                            entry = iter.next();
                        }
                        catch (IOException e) {
                            Assert.fail((String)("got IOException while iterating through listCachePools: " + e.getMessage()));
                        }
                        if (entry == null) break block6;
                    } while (!entry.getInfo().getPoolName().equals(pool.getPoolName()));
                    CachePoolStats stats = entry.getStats();
                    if (targetBytesNeeded == stats.getBytesNeeded() && targetBytesCached == stats.getBytesCached() && targetFilesNeeded == stats.getFilesNeeded() && targetFilesCached == stats.getFilesCached()) {
                        return true;
                    }
                    LOG.info((Object)(infoString + ": " + "filesNeeded: " + stats.getFilesNeeded() + "/" + targetFilesNeeded + ", filesCached: " + stats.getFilesCached() + "/" + targetFilesCached + ", bytesNeeded: " + stats.getBytesNeeded() + "/" + targetBytesNeeded + ", bytesCached: " + stats.getBytesCached() + "/" + targetBytesCached));
                    return false;
                }
                return false;
            }
        }, 500, 60000);
    }

    private static void checkNumCachedReplicas(DistributedFileSystem dfs, List<Path> paths, int expectedBlocks, int expectedReplicas) throws Exception {
        int numCachedBlocks = 0;
        int numCachedReplicas = 0;
        for (Path p : paths) {
            FileStatus f = dfs.getFileStatus(p);
            long len = f.getLen();
            long blockSize = f.getBlockSize();
            long numBlocks = (len + blockSize - 1L) / blockSize;
            BlockLocation[] locs = dfs.getFileBlockLocations(p, 0L, len);
            Assert.assertEquals((String)("Unexpected number of block locations for path " + p), (long)numBlocks, (long)locs.length);
            for (BlockLocation l : locs) {
                if (l.getCachedHosts().length > 0) {
                    ++numCachedBlocks;
                }
                numCachedReplicas += l.getCachedHosts().length;
            }
        }
        LOG.info((Object)("Found " + numCachedBlocks + " of " + expectedBlocks + " blocks"));
        LOG.info((Object)("Found " + numCachedReplicas + " of " + expectedReplicas + " replicas"));
        Assert.assertEquals((String)"Unexpected number of cached blocks", (long)expectedBlocks, (long)numCachedBlocks);
        Assert.assertEquals((String)"Unexpected number of cached replicas", (long)expectedReplicas, (long)numCachedReplicas);
    }

    @Test(timeout=120000L)
    public void testWaitForCachedReplicas() throws Exception {
        FileSystemTestHelper helper = new FileSystemTestHelper();
        GenericTestUtils.waitFor(new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                return namenode.getNamesystem().getCacheCapacity() == 65536L && namenode.getNamesystem().getCacheUsed() == 0L;
            }
        }, 500, 60000);
        NamenodeProtocols nnRpc = namenode.getRpcServer();
        DataNode dn0 = cluster.getDataNodes().get(0);
        String bpid = cluster.getNamesystem().getBlockPoolId();
        LinkedList<Long> bogusBlockIds = new LinkedList<Long>();
        bogusBlockIds.add(999999L);
        nnRpc.cacheReport(dn0.getDNRegistrationForBP(bpid), bpid, bogusBlockIds);
        Path rootDir = helper.getDefaultWorkingDirectory(dfs);
        String pool = "friendlyPool";
        nnRpc.addCachePool(new CachePoolInfo("friendlyPool"));
        int numFiles = 2;
        int numBlocksPerFile = 2;
        ArrayList<String> paths = new ArrayList<String>(2);
        for (int i = 0; i < 2; ++i) {
            Path p = new Path(rootDir, "testCachePaths-" + i);
            FileSystemTestHelper.createFile(dfs, p, 2, 4096);
            paths.add(p.toUri().getPath());
        }
        TestCacheDirectives.waitForCachedBlocks(namenode, 0, 0, "testWaitForCachedReplicas:0");
        int expected = 0;
        for (int i = 0; i < 2; ++i) {
            CacheDirectiveInfo directive = new CacheDirectiveInfo.Builder().setPath(new Path((String)paths.get(i))).setPool("friendlyPool").build();
            nnRpc.addCacheDirective(directive, EnumSet.noneOf(CacheFlag.class));
            TestCacheDirectives.waitForCachedBlocks(namenode, expected += 2, expected, "testWaitForCachedReplicas:1");
        }
        DatanodeInfo[] live = dfs.getDataNodeStats(HdfsConstants.DatanodeReportType.LIVE);
        Assert.assertEquals((String)"Unexpected number of live nodes", (long)4L, (long)live.length);
        long totalUsed = 0L;
        for (DatanodeInfo dn : live) {
            long cacheCapacity = dn.getCacheCapacity();
            long cacheUsed = dn.getCacheUsed();
            long cacheRemaining = dn.getCacheRemaining();
            Assert.assertEquals((String)"Unexpected cache capacity", (long)16384L, (long)cacheCapacity);
            Assert.assertEquals((String)"Capacity not equal to used + remaining", (long)cacheCapacity, (long)(cacheUsed + cacheRemaining));
            Assert.assertEquals((String)"Remaining not equal to capacity - used", (long)(cacheCapacity - cacheUsed), (long)cacheRemaining);
            totalUsed += cacheUsed;
        }
        Assert.assertEquals((long)((long)expected * 4096L), (long)totalUsed);
        CacheDirectiveIterator entries = new CacheDirectiveIterator(nnRpc, null);
        for (int i = 0; i < 2; ++i) {
            CacheDirectiveEntry entry = (CacheDirectiveEntry)entries.next();
            nnRpc.removeCacheDirective(entry.getInfo().getId());
            TestCacheDirectives.waitForCachedBlocks(namenode, expected -= 2, expected, "testWaitForCachedReplicas:2");
        }
    }

    @Test(timeout=120000L)
    public void testWaitForCachedReplicasInDirectory() throws Exception {
        String pool = "friendlyPool";
        CachePoolInfo poolInfo = new CachePoolInfo("friendlyPool");
        dfs.addCachePool(poolInfo);
        LinkedList<Path> paths = new LinkedList<Path>();
        paths.add(new Path("/foo/bar"));
        paths.add(new Path("/foo/baz"));
        paths.add(new Path("/foo2/bar2"));
        paths.add(new Path("/foo2/baz2"));
        dfs.mkdir(new Path("/foo"), FsPermission.getDirDefault());
        dfs.mkdir(new Path("/foo2"), FsPermission.getDirDefault());
        int numBlocksPerFile = 2;
        for (Path path : paths) {
            FileSystemTestHelper.createFile(dfs, path, 2, 4096, (short)3, false);
        }
        TestCacheDirectives.waitForCachedBlocks(namenode, 0, 0, "testWaitForCachedReplicasInDirectory:0");
        long id = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(new Path("/foo")).setReplication((short)2).setPool("friendlyPool").build());
        TestCacheDirectives.waitForCachedBlocks(namenode, 4, 8, "testWaitForCachedReplicasInDirectory:1:blocks");
        TestCacheDirectives.waitForCacheDirectiveStats(dfs, 32768L, 32768L, 2L, 2L, new CacheDirectiveInfo.Builder().setPath(new Path("/foo")).build(), "testWaitForCachedReplicasInDirectory:1:directive");
        TestCacheDirectives.waitForCachePoolStats(dfs, 32768L, 32768L, 2L, 2L, poolInfo, "testWaitForCachedReplicasInDirectory:1:pool");
        long id2 = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(new Path("/foo/bar")).setReplication((short)4).setPool("friendlyPool").build());
        TestCacheDirectives.waitForCachedBlocks(namenode, 4, 10, "testWaitForCachedReplicasInDirectory:2:blocks");
        TestCacheDirectives.waitForCacheDirectiveStats(dfs, 32768L, 32768L, 2L, 2L, new CacheDirectiveInfo.Builder().setPath(new Path("/foo")).build(), "testWaitForCachedReplicasInDirectory:2:directive-1");
        TestCacheDirectives.waitForCacheDirectiveStats(dfs, 32768L, 24576L, 1L, 0L, new CacheDirectiveInfo.Builder().setPath(new Path("/foo/bar")).build(), "testWaitForCachedReplicasInDirectory:2:directive-2");
        TestCacheDirectives.waitForCachePoolStats(dfs, 65536L, 57344L, 3L, 2L, poolInfo, "testWaitForCachedReplicasInDirectory:2:pool");
        dfs.removeCacheDirective(id);
        dfs.removeCacheDirective(id2);
        TestCacheDirectives.waitForCachedBlocks(namenode, 0, 0, "testWaitForCachedReplicasInDirectory:3:blocks");
        TestCacheDirectives.waitForCachePoolStats(dfs, 0L, 0L, 0L, 0L, poolInfo, "testWaitForCachedReplicasInDirectory:3:pool");
    }

    @Test(timeout=120000L)
    public void testReplicationFactor() throws Exception {
        int i;
        String pool = "friendlyPool";
        dfs.addCachePool(new CachePoolInfo("friendlyPool"));
        LinkedList<Path> paths = new LinkedList<Path>();
        paths.add(new Path("/foo/bar"));
        paths.add(new Path("/foo/baz"));
        paths.add(new Path("/foo2/bar2"));
        paths.add(new Path("/foo2/baz2"));
        dfs.mkdir(new Path("/foo"), FsPermission.getDirDefault());
        dfs.mkdir(new Path("/foo2"), FsPermission.getDirDefault());
        int numBlocksPerFile = 2;
        for (Path path : paths) {
            FileSystemTestHelper.createFile(dfs, path, 2, 4096, (short)3, false);
        }
        TestCacheDirectives.waitForCachedBlocks(namenode, 0, 0, "testReplicationFactor:0");
        TestCacheDirectives.checkNumCachedReplicas(dfs, paths, 0, 0);
        long id = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(new Path("/foo")).setReplication((short)1).setPool("friendlyPool").build());
        TestCacheDirectives.waitForCachedBlocks(namenode, 4, 4, "testReplicationFactor:1");
        TestCacheDirectives.checkNumCachedReplicas(dfs, paths, 4, 4);
        for (i = 2; i <= 3; ++i) {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(id).setReplication((short)i).build());
            TestCacheDirectives.waitForCachedBlocks(namenode, 4, 4 * i, "testReplicationFactor:2");
            TestCacheDirectives.checkNumCachedReplicas(dfs, paths, 4, 4 * i);
        }
        for (i = 2; i >= 1; --i) {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(id).setReplication((short)i).build());
            TestCacheDirectives.waitForCachedBlocks(namenode, 4, 4 * i, "testReplicationFactor:3");
            TestCacheDirectives.checkNumCachedReplicas(dfs, paths, 4, 4 * i);
        }
        dfs.removeCacheDirective(id);
        TestCacheDirectives.waitForCachedBlocks(namenode, 0, 0, "testReplicationFactor:4");
        TestCacheDirectives.checkNumCachedReplicas(dfs, paths, 0, 0);
    }

    @Test(timeout=60000L)
    public void testListCachePoolPermissions() throws Exception {
        UserGroupInformation myUser = UserGroupInformation.createRemoteUser("myuser");
        DistributedFileSystem myDfs = (DistributedFileSystem)DFSTestUtil.getFileSystemAs(myUser, conf);
        String poolName = "poolparty";
        dfs.addCachePool(new CachePoolInfo("poolparty").setMode(new FsPermission(448)));
        RemoteIterator<CachePoolEntry> it = myDfs.listCachePools();
        CachePoolInfo info = it.next().getInfo();
        Assert.assertFalse((boolean)it.hasNext());
        Assert.assertEquals((String)"Expected pool name", (Object)"poolparty", (Object)info.getPoolName());
        Assert.assertNull((String)"Unexpected owner name", (Object)info.getOwnerName());
        Assert.assertNull((String)"Unexpected group name", (Object)info.getGroupName());
        Assert.assertNull((String)"Unexpected mode", (Object)info.getMode());
        Assert.assertNull((String)"Unexpected limit", (Object)info.getLimit());
        long limit = 99L;
        dfs.modifyCachePool(new CachePoolInfo("poolparty").setOwnerName(myUser.getShortUserName()).setLimit(99L));
        it = myDfs.listCachePools();
        info = it.next().getInfo();
        Assert.assertFalse((boolean)it.hasNext());
        Assert.assertEquals((String)"Expected pool name", (Object)"poolparty", (Object)info.getPoolName());
        Assert.assertEquals((String)"Mismatched owner name", (Object)myUser.getShortUserName(), (Object)info.getOwnerName());
        Assert.assertNotNull((String)"Expected group name", (Object)info.getGroupName());
        Assert.assertEquals((String)"Mismatched mode", (long)448L, (long)info.getMode().toShort());
        Assert.assertEquals((String)"Mismatched limit", (long)99L, (long)info.getLimit());
    }

    @Test(timeout=120000L)
    public void testExpiry() throws Exception {
        String pool = "pool1";
        dfs.addCachePool(new CachePoolInfo(pool));
        Path p = new Path("/mypath");
        DFSTestUtil.createFile(dfs, p, 8192L, (short)2, 2457L);
        Date start = new Date();
        Date expiry = DateUtils.addSeconds((Date)start, (int)120);
        long id = dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPath(p).setPool(pool).setExpiration(CacheDirectiveInfo.Expiration.newAbsolute(expiry)).setReplication((short)2).build());
        TestCacheDirectives.waitForCachedBlocks(cluster.getNameNode(), 2, 4, "testExpiry:1");
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(id).setExpiration(CacheDirectiveInfo.Expiration.newRelative(0L)).build());
        TestCacheDirectives.waitForCachedBlocks(cluster.getNameNode(), 0, 0, "testExpiry:2");
        RemoteIterator<CacheDirectiveEntry> it = dfs.listCacheDirectives(null);
        CacheDirectiveEntry ent = it.next();
        Assert.assertFalse((boolean)it.hasNext());
        Date entryExpiry = new Date(ent.getInfo().getExpiration().getMillis());
        Assert.assertTrue((String)"Directive should have expired", (boolean)entryExpiry.before(new Date()));
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(id).setExpiration(CacheDirectiveInfo.Expiration.newRelative(120000L)).build());
        TestCacheDirectives.waitForCachedBlocks(cluster.getNameNode(), 2, 4, "testExpiry:3");
        it = dfs.listCacheDirectives(null);
        ent = it.next();
        Assert.assertFalse((boolean)it.hasNext());
        entryExpiry = new Date(ent.getInfo().getExpiration().getMillis());
        Assert.assertTrue((String)"Directive should not have expired", (boolean)entryExpiry.after(new Date()));
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(id).setExpiration(CacheDirectiveInfo.Expiration.newRelative(-1L)).build());
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("Cannot set a negative expiration", e);
        }
    }

    @Test(timeout=120000L)
    public void testLimit() throws Exception {
        try {
            dfs.addCachePool(new CachePoolInfo("poolofnegativity").setLimit(-99L));
            Assert.fail((String)"Should not be able to set a negative limit");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("negative", e);
        }
        String destiny = "poolofdestiny";
        Path path1 = new Path("/destiny");
        DFSTestUtil.createFile(dfs, path1, 8192L, (short)1, 38036L);
        CachePoolInfo poolInfo = new CachePoolInfo("poolofdestiny").setLimit(8191L);
        dfs.addCachePool(poolInfo);
        CacheDirectiveInfo info1 = new CacheDirectiveInfo.Builder().setPool("poolofdestiny").setPath(path1).build();
        try {
            dfs.addCacheDirective(info1);
            Assert.fail((String)"Should not be able to cache when there is no more limit");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("remaining capacity", e);
        }
        poolInfo.setLimit(8192L);
        dfs.modifyCachePool(poolInfo);
        long id1 = dfs.addCacheDirective(info1);
        TestCacheDirectives.waitForCachePoolStats(dfs, 8192L, 8192L, 1L, 1L, poolInfo, "testLimit:1");
        Path path2 = new Path("/failure");
        DFSTestUtil.createFile(dfs, path2, 4096L, (short)1, 38037L);
        try {
            dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPool("poolofdestiny").setPath(path2).build(), EnumSet.noneOf(CacheFlag.class));
            Assert.fail((String)"Should not be able to add another cached file");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("remaining capacity", e);
        }
        poolInfo.setLimit(4096L);
        dfs.modifyCachePool(poolInfo);
        TestCacheDirectives.waitForCachePoolStats(dfs, 8192L, 0L, 1L, 0L, poolInfo, "testLimit:2");
        RemoteIterator<CachePoolEntry> it = dfs.listCachePools();
        Assert.assertTrue((String)"Expected a cache pool", (boolean)it.hasNext());
        CachePoolStats stats = it.next().getStats();
        Assert.assertEquals((String)"Overlimit bytes should be difference of needed and limit", (long)4096L, (long)stats.getBytesOverlimit());
        CachePoolInfo inadequate = new CachePoolInfo("poolofinadequacy").setLimit(4096L);
        dfs.addCachePool(inadequate);
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(info1).setId(id1).setPool(inadequate.getPoolName()).build(), EnumSet.noneOf(CacheFlag.class));
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("remaining capacity", e);
        }
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(info1).setId(id1).setPool(inadequate.getPoolName()).build(), EnumSet.of(CacheFlag.FORCE));
        dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPool(inadequate.getPoolName()).setPath(path1).build(), EnumSet.of(CacheFlag.FORCE));
    }

    @Test(timeout=30000L)
    public void testMaxRelativeExpiry() throws Exception {
        try {
            dfs.addCachePool(new CachePoolInfo("failpool").setMaxRelativeExpiryMs(-1L));
            Assert.fail((String)"Added a pool with a negative max expiry.");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("negative", e);
        }
        try {
            dfs.addCachePool(new CachePoolInfo("failpool").setMaxRelativeExpiryMs(0x7FFFFFFFFFFFFFFEL));
            Assert.fail((String)"Added a pool with too big of a max expiry.");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("too big", e);
        }
        CachePoolInfo coolPool = new CachePoolInfo("coolPool");
        long poolExpiration = 600000L;
        dfs.addCachePool(coolPool.setMaxRelativeExpiryMs(600000L));
        RemoteIterator<CachePoolEntry> poolIt = dfs.listCachePools();
        CachePoolInfo listPool = poolIt.next().getInfo();
        Assert.assertFalse((String)"Should only be one pool", (boolean)poolIt.hasNext());
        Assert.assertEquals((String)"Expected max relative expiry to match set value", (long)600000L, (long)listPool.getMaxRelativeExpiryMs());
        try {
            dfs.addCachePool(coolPool.setMaxRelativeExpiryMs(-1L));
            Assert.fail((String)"Added a pool with a negative max expiry.");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("negative", e);
        }
        try {
            dfs.modifyCachePool(coolPool.setMaxRelativeExpiryMs(0x2000000000000000L));
            Assert.fail((String)"Added a pool with too big of a max expiry.");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("too big", e);
        }
        CacheDirectiveInfo defaultExpiry = new CacheDirectiveInfo.Builder().setPath(new Path("/blah")).setPool(coolPool.getPoolName()).build();
        dfs.addCacheDirective(defaultExpiry);
        RemoteIterator<CacheDirectiveEntry> dirIt = dfs.listCacheDirectives(defaultExpiry);
        CacheDirectiveInfo listInfo = dirIt.next().getInfo();
        Assert.assertFalse((String)"Should only have one entry in listing", (boolean)dirIt.hasNext());
        long listExpiration = listInfo.getExpiration().getAbsoluteMillis() - new Date().getTime();
        Assert.assertTrue((String)"Directive expiry should be approximately the pool's max expiry", (Math.abs(listExpiration - 600000L) < 10000L ? 1 : 0) != 0);
        CacheDirectiveInfo.Builder builder = new CacheDirectiveInfo.Builder().setPath(new Path("/lolcat")).setPool(coolPool.getPoolName());
        try {
            dfs.addCacheDirective(builder.setExpiration(CacheDirectiveInfo.Expiration.newRelative(600001L)).build());
            Assert.fail((String)"Added a directive that exceeds pool's max relative expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("exceeds the max relative expiration", e);
        }
        try {
            dfs.addCacheDirective(builder.setExpiration(CacheDirectiveInfo.Expiration.newAbsolute(new Date().getTime() + 600000L + 10000L)).build());
            Assert.fail((String)"Added a directive that exceeds pool's max relative expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("exceeds the max relative expiration", e);
        }
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(defaultExpiry).setId(listInfo.getId()).setExpiration(CacheDirectiveInfo.Expiration.newRelative(600001L)).build());
            Assert.fail((String)"Modified a directive to exceed pool's max relative expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("exceeds the max relative expiration", e);
        }
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(defaultExpiry).setId(listInfo.getId()).setExpiration(CacheDirectiveInfo.Expiration.newAbsolute(new Date().getTime() + 600000L + 10000L)).build());
            Assert.fail((String)"Modified a directive to exceed pool's max relative expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("exceeds the max relative expiration", e);
        }
        try {
            dfs.addCacheDirective(builder.setExpiration(CacheDirectiveInfo.Expiration.newRelative(Long.MAX_VALUE)).build());
            Assert.fail((String)"Added a directive with a gigantic max value");
        }
        catch (IllegalArgumentException e) {
            GenericTestUtils.assertExceptionContains("is too far in the future", e);
        }
        try {
            dfs.addCacheDirective(builder.setExpiration(CacheDirectiveInfo.Expiration.newAbsolute(Long.MAX_VALUE)).build());
            Assert.fail((String)"Added a directive with a gigantic max value");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("is too far in the future", e);
        }
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(defaultExpiry).setId(listInfo.getId()).setExpiration(CacheDirectiveInfo.Expiration.NEVER).build());
            Assert.fail((String)"Modified a directive to exceed pool's max relative expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("exceeds the max relative expiration", e);
        }
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(defaultExpiry).setId(listInfo.getId()).setExpiration(CacheDirectiveInfo.Expiration.newAbsolute(Long.MAX_VALUE)).build());
            Assert.fail((String)"Modified a directive to exceed pool's max relative expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("is too far in the future", e);
        }
        CachePoolInfo destPool = new CachePoolInfo("destPool");
        dfs.addCachePool(destPool.setMaxRelativeExpiryMs(300000L));
        try {
            dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(defaultExpiry).setId(listInfo.getId()).setPool(destPool.getPoolName()).build());
            Assert.fail((String)"Modified a directive to a pool with a lower max expiration");
        }
        catch (InvalidRequestException e) {
            GenericTestUtils.assertExceptionContains("exceeds the max relative expiration", e);
        }
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder(defaultExpiry).setId(listInfo.getId()).setPool(destPool.getPoolName()).setExpiration(CacheDirectiveInfo.Expiration.newRelative(300000L)).build());
        dirIt = dfs.listCacheDirectives(new CacheDirectiveInfo.Builder().setPool(destPool.getPoolName()).build());
        listInfo = dirIt.next().getInfo();
        listExpiration = listInfo.getExpiration().getAbsoluteMillis() - new Date().getTime();
        Assert.assertTrue((String)("Unexpected relative expiry " + listExpiration + " expected approximately " + 300000L), (Math.abs(300000L - listExpiration) < 10000L ? 1 : 0) != 0);
        dfs.modifyCachePool(destPool.setMaxRelativeExpiryMs(0x1FFFFFFFFFFFFFFFL));
        poolIt = dfs.listCachePools();
        listPool = poolIt.next().getInfo();
        while (!listPool.getPoolName().equals(destPool.getPoolName())) {
            listPool = poolIt.next().getInfo();
        }
        Assert.assertEquals((String)"Expected max relative expiry to match set value", (long)0x1FFFFFFFFFFFFFFFL, (long)listPool.getMaxRelativeExpiryMs());
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(listInfo.getId()).setExpiration(CacheDirectiveInfo.Expiration.newRelative(0x1FFFFFFFFFFFFFFFL)).build());
        dfs.modifyCacheDirective(new CacheDirectiveInfo.Builder().setId(listInfo.getId()).setExpiration(CacheDirectiveInfo.Expiration.newRelative(0x1FFFFFFFFFFFFFFEL)).build());
    }

    @Test(timeout=60000L)
    public void testExceedsCapacity() throws Exception {
        Path fileName = new Path("/exceeds");
        long fileLen = 131072L;
        int numCachedReplicas = 16;
        DFSTestUtil.createFile(dfs, fileName, 131072L, (short)4, 1027565L);
        LogVerificationAppender appender = new LogVerificationAppender();
        Logger logger = Logger.getRootLogger();
        logger.addAppender((Appender)appender);
        dfs.addCachePool(new CachePoolInfo("pool"));
        dfs.addCacheDirective(new CacheDirectiveInfo.Builder().setPool("pool").setPath(fileName).setReplication((short)1).build());
        TestCacheDirectives.waitForCachedBlocks(namenode, -1, numCachedReplicas, "testExceeds:1");
        int lines = appender.countLinesWithMessage("more bytes in the cache: dfs.datanode.max.locked.memory");
        Assert.assertEquals((String)"Namenode should not send extra CACHE commands", (long)0L, (long)lines);
        dfs.delete(fileName, false);
        DFSTestUtil.createFile(dfs, fileName, 4096, 131072L, 32768L, (short)1, 1027565L);
        Thread.sleep(4000L);
        lines = appender.countLinesWithMessage("more bytes in the cache: dfs.datanode.max.locked.memory");
        Assert.assertEquals((String)"Namenode should not send extra CACHE commands", (long)0L, (long)lines);
    }

    static {
        NativeIO.POSIX.setCacheManipulator(new NativeIO.POSIX.NoMlockCacheManipulator());
        EditLogFileOutputStream.setShouldSkipFsyncForTesting(false);
    }
}

