/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.PrivilegedExceptionAction;
import java.util.concurrent.Semaphore;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class TestFileSystemCaching {
    @Test
    public void testCacheEnabled() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        FileSystem fs1 = FileSystem.get(new URI("cachedfile://a"), conf);
        FileSystem fs2 = FileSystem.get(new URI("cachedfile://a"), conf);
        junit.framework.Assert.assertSame((Object)fs1, (Object)fs2);
    }

    @Test
    public void testDefaultFsUris() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.defaultfs.impl", DefaultFs.class.getName());
        URI defaultUri = URI.create("defaultfs://host");
        FileSystem.setDefaultUri(conf, defaultUri);
        FileSystem fs = null;
        FileSystem defaultFs = FileSystem.get(conf);
        Assert.assertEquals((Object)defaultUri, (Object)defaultFs.getUri());
        fs = FileSystem.get(URI.create("defaultfs:/"), conf);
        junit.framework.Assert.assertSame((Object)defaultFs, (Object)fs);
        fs = FileSystem.get(URI.create("defaultfs:///"), conf);
        junit.framework.Assert.assertSame((Object)defaultFs, (Object)fs);
        fs = FileSystem.get(URI.create("defaultfs://host"), conf);
        junit.framework.Assert.assertSame((Object)defaultFs, (Object)fs);
        fs = FileSystem.get(URI.create("defaultfs://host2"), conf);
        junit.framework.Assert.assertNotSame((Object)defaultFs, (Object)fs);
        fs = FileSystem.get(URI.create("/"), conf);
        junit.framework.Assert.assertSame((Object)defaultFs, (Object)fs);
        try {
            fs = FileSystem.get(URI.create("//host"), conf);
            Assert.fail((String)"got fs with auth but no scheme");
        }
        catch (Exception e) {
            Assert.assertEquals((Object)"No FileSystem for scheme: null", (Object)e.getMessage());
        }
        try {
            fs = FileSystem.get(URI.create("//host2"), conf);
            Assert.fail((String)"got fs with auth but no scheme");
        }
        catch (Exception e) {
            Assert.assertEquals((Object)"No FileSystem for scheme: null", (Object)e.getMessage());
        }
    }

    @Test
    public void testCacheEnabledWithInitializeForeverFS() throws Exception {
        final Configuration conf = new Configuration();
        Thread t = new Thread(){

            @Override
            public void run() {
                conf.set("fs.localfs1.impl", "org.apache.hadoop.fs.TestFileSystemCaching$InitializeForeverFileSystem");
                try {
                    FileSystem.get(new URI("localfs1://a"), conf);
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
                catch (URISyntaxException e) {
                    e.printStackTrace();
                }
            }
        };
        t.start();
        InitializeForeverFileSystem.sem.acquire();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        FileSystem.get(new URI("cachedfile://a"), conf);
        t.interrupt();
        t.join();
    }

    @Test
    public void testCacheDisabled() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.uncachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        conf.setBoolean("fs.uncachedfile.impl.disable.cache", true);
        FileSystem fs1 = FileSystem.get(new URI("uncachedfile://a"), conf);
        FileSystem fs2 = FileSystem.get(new URI("uncachedfile://a"), conf);
        junit.framework.Assert.assertNotSame((Object)fs1, (Object)fs2);
    }

    @Test
    public <T extends TokenIdentifier> void testCacheForUgi() throws Exception {
        final Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser("foo");
        UserGroupInformation ugiB = UserGroupInformation.createRemoteUser("bar");
        FileSystem fsA = ugiA.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        FileSystem fsA1 = ugiA.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        junit.framework.Assert.assertSame((Object)fsA, (Object)fsA1);
        FileSystem fsB = ugiB.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        junit.framework.Assert.assertNotSame((Object)fsA, (Object)fsB);
        Token t1 = (Token)Mockito.mock(Token.class);
        UserGroupInformation ugiA2 = UserGroupInformation.createRemoteUser("foo");
        fsA = ugiA2.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        junit.framework.Assert.assertNotSame((Object)fsA, (Object)fsA1);
        ugiA.addToken(t1);
        fsA = ugiA.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        junit.framework.Assert.assertSame((Object)fsA, (Object)fsA1);
    }

    @Test
    public void testUserFS() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        FileSystem fsU1 = FileSystem.get(new URI("cachedfile://a"), conf, "bar");
        FileSystem fsU2 = FileSystem.get(new URI("cachedfile://a"), conf, "foo");
        junit.framework.Assert.assertNotSame((Object)fsU1, (Object)fsU2);
    }

    @Test
    public void testFsUniqueness() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        FileSystem fs1 = FileSystem.get(conf);
        FileSystem fs2 = FileSystem.get(conf);
        Assert.assertTrue((fs1 == fs2 ? 1 : 0) != 0);
        fs1 = FileSystem.newInstance(new URI("cachedfile://a"), conf, "bar");
        fs2 = FileSystem.newInstance(new URI("cachedfile://a"), conf, "bar");
        Assert.assertTrue((fs1 != fs2 && !fs1.equals(fs2) ? 1 : 0) != 0);
        fs1.close();
        fs2.close();
    }

    @Test
    public void testCloseAllForUGI() throws Exception {
        final Configuration conf = new Configuration();
        conf.set("fs.cachedfile.impl", FileSystem.getFileSystemClass("file", null).getName());
        UserGroupInformation ugiA = UserGroupInformation.createRemoteUser("foo");
        FileSystem fsA = ugiA.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        FileSystem fsA1 = ugiA.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        junit.framework.Assert.assertSame((Object)fsA, (Object)fsA1);
        FileSystem.closeAllForUGI(ugiA);
        fsA1 = ugiA.doAs(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get(new URI("cachedfile://a"), conf);
            }
        });
        junit.framework.Assert.assertNotSame((Object)fsA, (Object)fsA1);
    }

    @Test
    public void testDelete() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        ((FileSystem)fs).delete(path, false);
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)false));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        ((FileSystem)fs).delete(path, true);
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)true));
    }

    @Test
    public void testDeleteOnExit() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
        Assert.assertTrue((boolean)fs.deleteOnExit(path));
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
        ((FileSystem)fs).close();
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs)).delete((Path)Mockito.eq((Object)path), Mockito.eq((boolean)true));
    }

    @Test
    public void testDeleteOnExitFNF() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        Assert.assertFalse((boolean)fs.deleteOnExit(path));
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        ((FileSystem)fs).close();
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)Mockito.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testDeleteOnExitRemoved() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
        Assert.assertTrue((boolean)fs.deleteOnExit(path));
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        ((FileSystem)fs).close();
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)Mockito.any(Path.class), Mockito.anyBoolean());
    }

    @Test
    public void testCancelDeleteOnExit() throws IOException {
        FileSystem mockFs = (FileSystem)Mockito.mock(FileSystem.class);
        FilterFileSystem fs = new FilterFileSystem(mockFs);
        Path path = new Path("/a");
        Mockito.when((Object)mockFs.getFileStatus((Path)Mockito.eq((Object)path))).thenReturn((Object)new FileStatus());
        Assert.assertTrue((boolean)fs.deleteOnExit(path));
        ((FileSystem)Mockito.verify((Object)mockFs)).getFileStatus((Path)Mockito.eq((Object)path));
        Assert.assertTrue((boolean)fs.cancelDeleteOnExit(path));
        Assert.assertFalse((boolean)fs.cancelDeleteOnExit(path));
        Mockito.reset((Object[])new FileSystem[]{mockFs});
        ((FileSystem)fs).close();
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).getFileStatus((Path)Mockito.any(Path.class));
        ((FileSystem)Mockito.verify((Object)mockFs, (VerificationMode)Mockito.never())).delete((Path)Mockito.any(Path.class), Mockito.anyBoolean());
    }

    public static class InitializeForeverFileSystem
    extends LocalFileSystem {
        static final Semaphore sem = new Semaphore(0);

        @Override
        public void initialize(URI uri, Configuration conf) throws IOException {
            sem.release();
            try {
                while (true) {
                    Thread.sleep(1000L);
                }
            }
            catch (InterruptedException e) {
                return;
            }
        }
    }

    static class DefaultFs
    extends LocalFileSystem {
        URI uri;

        DefaultFs() {
        }

        @Override
        public void initialize(URI uri, Configuration conf) {
            this.uri = uri;
        }

        @Override
        public URI getUri() {
            return this.uri;
        }
    }
}

