/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.protocol.datatransfer;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.TextInputCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import javax.security.sasl.RealmCallback;
import javax.security.sasl.RealmChoiceCallback;
import javax.security.sasl.Sasl;
import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslException;
import javax.security.sasl.SaslServer;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.flink.hadoop.shaded.com.google.common.base.Charsets;
import org.apache.flink.hadoop.shaded.com.google.common.collect.Maps;
import org.apache.flink.hadoop.shaded.com.google.protobuf.ByteString;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.InvalidEncryptionKeyException;
import org.apache.hadoop.hdfs.protocol.proto.DataTransferProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelper;
import org.apache.hadoop.hdfs.security.token.block.BlockPoolTokenSecretManager;
import org.apache.hadoop.hdfs.security.token.block.DataEncryptionKey;
import org.apache.hadoop.security.SaslInputStream;
import org.apache.hadoop.security.SaslOutputStream;

@InterfaceAudience.Private
public class DataTransferEncryptor {
    public static final Log LOG = LogFactory.getLog(DataTransferEncryptor.class);
    private static final int ENCRYPTED_TRANSFER_MAGIC_NUMBER = -559038737;
    private static final String NAME_DELIMITER = " ";
    private static final String SERVER_NAME = "0";
    private static final String PROTOCOL = "hdfs";
    private static final String MECHANISM = "DIGEST-MD5";
    private static final Map<String, String> SASL_PROPS = new TreeMap<String, String>();

    public static IOStreamPair getEncryptedStreams(OutputStream underlyingOut, InputStream underlyingIn, BlockPoolTokenSecretManager blockPoolTokenSecretManager, String encryptionAlgorithm) throws IOException {
        DataInputStream in = new DataInputStream(underlyingIn);
        DataOutputStream out = new DataOutputStream(underlyingOut);
        HashMap<String, String> saslProps = Maps.newHashMap(SASL_PROPS);
        saslProps.put("com.sun.security.sasl.digest.cipher", encryptionAlgorithm);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Server using encryption algorithm " + encryptionAlgorithm));
        }
        SaslParticipant sasl = new SaslParticipant(Sasl.createSaslServer(MECHANISM, PROTOCOL, SERVER_NAME, saslProps, new SaslServerCallbackHandler(blockPoolTokenSecretManager)));
        int magicNumber = in.readInt();
        if (magicNumber != -559038737) {
            throw new InvalidMagicNumberException(magicNumber);
        }
        try {
            DataTransferEncryptor.performSaslStep1(out, in, sasl);
            byte[] remoteResponse = DataTransferEncryptor.readSaslMessage(in);
            byte[] localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            DataTransferEncryptor.sendSaslMessage(out, localResponse);
            DataTransferEncryptor.checkSaslComplete(sasl);
            return sasl.createEncryptedStreamPair(out, in);
        }
        catch (IOException ioe) {
            if (ioe instanceof SaslException && ioe.getCause() != null && ioe.getCause() instanceof InvalidEncryptionKeyException) {
                DataTransferEncryptor.sendInvalidKeySaslErrorMessage(out, ioe.getCause().getMessage());
            } else {
                DataTransferEncryptor.sendGenericSaslErrorMessage(out, ioe.getMessage());
            }
            throw ioe;
        }
    }

    public static IOStreamPair getEncryptedStreams(OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKey encryptionKey) throws IOException {
        HashMap<String, String> saslProps = Maps.newHashMap(SASL_PROPS);
        saslProps.put("com.sun.security.sasl.digest.cipher", encryptionKey.encryptionAlgorithm);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Client using encryption algorithm " + encryptionKey.encryptionAlgorithm));
        }
        DataOutputStream out = new DataOutputStream(underlyingOut);
        DataInputStream in = new DataInputStream(underlyingIn);
        String userName = DataTransferEncryptor.getUserNameFromEncryptionKey(encryptionKey);
        SaslParticipant sasl = new SaslParticipant(Sasl.createSaslClient(new String[]{MECHANISM}, userName, PROTOCOL, SERVER_NAME, saslProps, new SaslClientCallbackHandler(encryptionKey.encryptionKey, userName)));
        out.writeInt(-559038737);
        out.flush();
        try {
            DataTransferEncryptor.sendSaslMessage(out, new byte[0]);
            DataTransferEncryptor.performSaslStep1(out, in, sasl);
            byte[] remoteResponse = DataTransferEncryptor.readSaslMessage(in);
            byte[] localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            assert (localResponse == null);
            DataTransferEncryptor.checkSaslComplete(sasl);
            return sasl.createEncryptedStreamPair(out, in);
        }
        catch (IOException ioe) {
            DataTransferEncryptor.sendGenericSaslErrorMessage(out, ioe.getMessage());
            throw ioe;
        }
    }

    private static void performSaslStep1(DataOutputStream out, DataInputStream in, SaslParticipant sasl) throws IOException {
        byte[] remoteResponse = DataTransferEncryptor.readSaslMessage(in);
        byte[] localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
        DataTransferEncryptor.sendSaslMessage(out, localResponse);
    }

    private static void checkSaslComplete(SaslParticipant sasl) throws IOException {
        if (!sasl.isComplete()) {
            throw new IOException("Failed to complete SASL handshake");
        }
        if (!sasl.supportsConfidentiality()) {
            throw new IOException("SASL handshake completed, but channel does not support encryption");
        }
    }

    private static void sendSaslMessage(DataOutputStream out, byte[] payload) throws IOException {
        DataTransferEncryptor.sendSaslMessage(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.SUCCESS, payload, null);
    }

    private static void sendInvalidKeySaslErrorMessage(DataOutputStream out, String message) throws IOException {
        DataTransferEncryptor.sendSaslMessage(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY, null, message);
    }

    private static void sendGenericSaslErrorMessage(DataOutputStream out, String message) throws IOException {
        DataTransferEncryptor.sendSaslMessage(out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR, null, message);
    }

    private static void sendSaslMessage(OutputStream out, DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus status, byte[] payload, String message) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto.Builder builder = DataTransferProtos.DataTransferEncryptorMessageProto.newBuilder();
        builder.setStatus(status);
        if (payload != null) {
            builder.setPayload(ByteString.copyFrom(payload));
        }
        if (message != null) {
            builder.setMessage(message);
        }
        DataTransferProtos.DataTransferEncryptorMessageProto proto = builder.build();
        proto.writeDelimitedTo(out);
        out.flush();
    }

    private static byte[] readSaslMessage(DataInputStream in) throws IOException {
        DataTransferProtos.DataTransferEncryptorMessageProto proto = DataTransferProtos.DataTransferEncryptorMessageProto.parseFrom(PBHelper.vintPrefixed(in));
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR_UNKNOWN_KEY) {
            throw new InvalidEncryptionKeyException(proto.getMessage());
        }
        if (proto.getStatus() == DataTransferProtos.DataTransferEncryptorMessageProto.DataTransferEncryptorStatus.ERROR) {
            throw new IOException(proto.getMessage());
        }
        return proto.getPayload().toByteArray();
    }

    private static String getUserNameFromEncryptionKey(DataEncryptionKey encryptionKey) {
        return encryptionKey.keyId + NAME_DELIMITER + encryptionKey.blockPoolId + NAME_DELIMITER + new String(Base64.encodeBase64((byte[])encryptionKey.nonce, (boolean)false), Charsets.UTF_8);
    }

    private static byte[] getEncryptionKeyFromUserName(BlockPoolTokenSecretManager blockPoolTokenSecretManager, String userName) throws IOException {
        String[] nameComponents = userName.split(NAME_DELIMITER);
        if (nameComponents.length != 3) {
            throw new IOException("Provided name '" + userName + "' has " + nameComponents.length + " components instead of the expected 3.");
        }
        int keyId = Integer.parseInt(nameComponents[0]);
        String blockPoolId = nameComponents[1];
        byte[] nonce = Base64.decodeBase64((String)nameComponents[2]);
        return blockPoolTokenSecretManager.retrieveDataEncryptionKey(keyId, blockPoolId, nonce);
    }

    private static char[] encryptionKeyToPassword(byte[] encryptionKey) {
        return new String(Base64.encodeBase64((byte[])encryptionKey, (boolean)false), Charsets.UTF_8).toCharArray();
    }

    static {
        SASL_PROPS.put("javax.security.sasl.qop", "auth-conf");
        SASL_PROPS.put("javax.security.sasl.server.authentication", "true");
    }

    @InterfaceAudience.Private
    public static class InvalidMagicNumberException
    extends IOException {
        private static final long serialVersionUID = 1L;

        public InvalidMagicNumberException(int magicNumber) {
            super(String.format("Received %x instead of %x from client.", magicNumber, -559038737));
        }
    }

    private static class SaslParticipant {
        public SaslServer saslServer;
        public SaslClient saslClient;

        public SaslParticipant(SaslServer saslServer) {
            this.saslServer = saslServer;
        }

        public SaslParticipant(SaslClient saslClient) {
            this.saslClient = saslClient;
        }

        public byte[] evaluateChallengeOrResponse(byte[] challengeOrResponse) throws SaslException {
            if (this.saslClient != null) {
                return this.saslClient.evaluateChallenge(challengeOrResponse);
            }
            return this.saslServer.evaluateResponse(challengeOrResponse);
        }

        public boolean isComplete() {
            if (this.saslClient != null) {
                return this.saslClient.isComplete();
            }
            return this.saslServer.isComplete();
        }

        public boolean supportsConfidentiality() {
            String qop = null;
            qop = this.saslClient != null ? (String)this.saslClient.getNegotiatedProperty("javax.security.sasl.qop") : (String)this.saslServer.getNegotiatedProperty("javax.security.sasl.qop");
            return qop != null && qop.equals("auth-conf");
        }

        private IOStreamPair createEncryptedStreamPair(DataOutputStream out, DataInputStream in) {
            if (this.saslClient != null) {
                return new IOStreamPair(new SaslInputStream((InputStream)in, this.saslClient), new SaslOutputStream((OutputStream)out, this.saslClient));
            }
            return new IOStreamPair(new SaslInputStream((InputStream)in, this.saslServer), new SaslOutputStream((OutputStream)out, this.saslServer));
        }
    }

    private static class SaslClientCallbackHandler
    implements CallbackHandler {
        private final byte[] encryptionKey;
        private final String userName;

        public SaslClientCallbackHandler(byte[] encryptionKey, String userName) {
            this.encryptionKey = encryptionKey;
            this.userName = userName;
        }

        @Override
        public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
            NameCallback nc = null;
            PasswordCallback pc = null;
            TextInputCallback rc = null;
            for (Callback callback : callbacks) {
                if (callback instanceof RealmChoiceCallback) continue;
                if (callback instanceof NameCallback) {
                    nc = (NameCallback)callback;
                    continue;
                }
                if (callback instanceof PasswordCallback) {
                    pc = (PasswordCallback)callback;
                    continue;
                }
                if (callback instanceof RealmCallback) {
                    rc = (RealmCallback)callback;
                    continue;
                }
                throw new UnsupportedCallbackException(callback, "Unrecognized SASL client callback");
            }
            if (nc != null) {
                nc.setName(this.userName);
            }
            if (pc != null) {
                pc.setPassword(DataTransferEncryptor.encryptionKeyToPassword(this.encryptionKey));
            }
            if (rc != null) {
                rc.setText(rc.getDefaultText());
            }
        }
    }

    private static class SaslServerCallbackHandler
    implements CallbackHandler {
        private final BlockPoolTokenSecretManager blockPoolTokenSecretManager;

        public SaslServerCallbackHandler(BlockPoolTokenSecretManager blockPoolTokenSecretManager) {
            this.blockPoolTokenSecretManager = blockPoolTokenSecretManager;
        }

        @Override
        public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
            NameCallback nc = null;
            PasswordCallback pc = null;
            AuthorizeCallback ac = null;
            for (Callback callback : callbacks) {
                if (callback instanceof AuthorizeCallback) {
                    ac = (AuthorizeCallback)callback;
                    continue;
                }
                if (callback instanceof PasswordCallback) {
                    pc = (PasswordCallback)callback;
                    continue;
                }
                if (callback instanceof NameCallback) {
                    nc = (NameCallback)callback;
                    continue;
                }
                if (callback instanceof RealmCallback) continue;
                throw new UnsupportedCallbackException(callback, "Unrecognized SASL DIGEST-MD5 Callback: " + callback);
            }
            if (pc != null) {
                byte[] encryptionKey = DataTransferEncryptor.getEncryptionKeyFromUserName(this.blockPoolTokenSecretManager, nc.getDefaultName());
                pc.setPassword(DataTransferEncryptor.encryptionKeyToPassword(encryptionKey));
            }
            if (ac != null) {
                ac.setAuthorized(true);
                ac.setAuthorizedID(ac.getAuthorizationID());
            }
        }
    }
}

