/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ShortNode;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.raft.ElectionState;
import org.apache.kafka.raft.QuorumStateStore;
import org.apache.kafka.raft.generated.QuorumStateData;
import org.apache.kafka.raft.generated.QuorumStateDataJsonConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileQuorumStateStore
implements QuorumStateStore {
    private static final Logger log = LoggerFactory.getLogger(FileQuorumStateStore.class);
    private static final String DATA_VERSION = "data_version";
    static final short LOWEST_SUPPORTED_VERSION = 0;
    static final short HIGHEST_SUPPORTED_VERSION = 1;
    public static final String DEFAULT_FILE_NAME = "quorum-state";
    private final File stateFile;

    public FileQuorumStateStore(File stateFile) {
        this.stateFile = stateFile;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private QuorumStateData readStateFromFile(File file) {
        try (BufferedReader reader = Files.newBufferedReader(file.toPath(), StandardCharsets.UTF_8);){
            String line = reader.readLine();
            if (line == null) {
                throw new EOFException("File ended prematurely.");
            }
            ObjectMapper objectMapper = new ObjectMapper();
            JsonNode readNode = objectMapper.readTree(line);
            if (!(readNode instanceof ObjectNode)) {
                throw new IOException("Deserialized node " + readNode + " is not an object node");
            }
            ObjectNode dataObject = (ObjectNode)readNode;
            JsonNode dataVersionNode = dataObject.get(DATA_VERSION);
            if (dataVersionNode == null) {
                throw new IOException("Deserialized node " + readNode + " does not have " + DATA_VERSION + " field");
            }
            short dataVersion = dataVersionNode.shortValue();
            if (dataVersion < 0 || dataVersion > 1) {
                throw new IllegalStateException(String.format("data_version (%d) is not within the min (%d) and max ($d) supported version", dataVersion, (short)0, (short)1));
            }
            QuorumStateData quorumStateData = QuorumStateDataJsonConverter.read((JsonNode)dataObject, dataVersion);
            return quorumStateData;
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Error while reading the Quorum status from the file %s", file), e);
        }
    }

    @Override
    public Optional<ElectionState> readElectionState() {
        if (!this.stateFile.exists()) {
            return Optional.empty();
        }
        return Optional.of(ElectionState.fromQuorumStateData(this.readStateFromFile(this.stateFile)));
    }

    @Override
    public void writeElectionState(ElectionState latest, short kraftVersion) {
        short quorumStateVersion = this.quorumStateVersionFromKRaftVersion(kraftVersion);
        this.writeElectionStateToFile(this.stateFile, latest.toQuorumStateData(quorumStateVersion), quorumStateVersion);
    }

    @Override
    public Path path() {
        return this.stateFile.toPath();
    }

    private short quorumStateVersionFromKRaftVersion(short kraftVersion) {
        if (kraftVersion == 0) {
            return 0;
        }
        if (kraftVersion == 1) {
            return 1;
        }
        throw new IllegalArgumentException(String.format("Unknown kraft.version %d", kraftVersion));
    }

    private void writeElectionStateToFile(File stateFile, QuorumStateData state, short version) {
        if (version > 1) {
            throw new IllegalArgumentException(String.format("Quorum state data version (%d) is greater than the supported version (%d)", version, (short)1));
        }
        File temp = new File(stateFile.getAbsolutePath() + ".tmp");
        this.deleteFileIfExists(temp);
        log.trace("Writing tmp quorum state {}", (Object)temp.getAbsolutePath());
        try {
            try (FileOutputStream fileOutputStream = new FileOutputStream(temp);
                 BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)fileOutputStream, StandardCharsets.UTF_8));){
                ObjectNode jsonState = (ObjectNode)QuorumStateDataJsonConverter.write(state, version);
                jsonState.set(DATA_VERSION, (JsonNode)new ShortNode(version));
                writer.write(jsonState.toString());
                writer.flush();
                fileOutputStream.getFD().sync();
            }
            Utils.atomicMoveWithFallback((Path)temp.toPath(), (Path)stateFile.toPath());
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Error while writing the Quorum status from the file %s", stateFile.getAbsolutePath()), e);
        }
        finally {
            this.deleteFileIfExists(temp);
        }
    }

    @Override
    public void clear() {
        this.deleteFileIfExists(this.stateFile);
        this.deleteFileIfExists(new File(this.stateFile.getAbsolutePath() + ".tmp"));
    }

    public String toString() {
        return "Quorum state filepath: " + this.stateFile.getAbsolutePath();
    }

    private void deleteFileIfExists(File file) {
        try {
            Files.deleteIfExists(file.toPath());
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Error while deleting file %s", file.getAbsoluteFile()), e);
        }
    }
}

