/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.cluster.manager.impl;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.filter.ClientFilter;
import com.sun.jersey.api.client.filter.GZIPContentEncodingFilter;
import com.sun.jersey.core.util.MultivaluedMapImpl;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import javax.ws.rs.core.MultivaluedMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.cluster.manager.HttpRequestReplicator;
import org.apache.nifi.cluster.manager.NodeResponse;
import org.apache.nifi.cluster.manager.exception.UriConstructionException;
import org.apache.nifi.cluster.manager.impl.WebClusterManager;
import org.apache.nifi.cluster.protocol.NodeIdentifier;
import org.apache.nifi.logging.NiFiLog;
import org.apache.nifi.util.FormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpRequestReplicatorImpl
implements HttpRequestReplicator {
    private static final int DEFAULT_SHUTDOWN_REPLICATOR_SECONDS = 30;
    private static final Logger logger = new NiFiLog(LoggerFactory.getLogger(HttpRequestReplicatorImpl.class));
    private final Client client;
    private final int numThreads;
    private final int connectionTimeoutMs;
    private final int readTimeoutMs;
    private ExecutorService executorService;
    private int shutdownReplicatorSeconds = 30;
    private String nodeProtocolScheme = null;

    public HttpRequestReplicatorImpl(int numThreads, Client client) {
        this(numThreads, client, "0 sec", "0 sec");
    }

    public HttpRequestReplicatorImpl(int numThreads, Client client, String connectionTimeout, String readTimeout) {
        if (numThreads <= 0) {
            throw new IllegalArgumentException("The number of threads must be greater than zero.");
        }
        if (client == null) {
            throw new IllegalArgumentException("Client may not be null.");
        }
        this.numThreads = numThreads;
        this.client = client;
        this.connectionTimeoutMs = (int)FormatUtils.getTimeDuration((String)connectionTimeout, (TimeUnit)TimeUnit.MILLISECONDS);
        this.readTimeoutMs = (int)FormatUtils.getTimeDuration((String)readTimeout, (TimeUnit)TimeUnit.MILLISECONDS);
        client.getProperties().put("com.sun.jersey.client.property.connectTimeout", this.connectionTimeoutMs);
        client.getProperties().put("com.sun.jersey.client.property.readTimeout", this.readTimeoutMs);
        client.getProperties().put("com.sun.jersey.client.property.followRedirects", Boolean.TRUE);
    }

    @Override
    public void start() {
        if (this.isRunning()) {
            throw new IllegalStateException("Instance is already started.");
        }
        this.executorService = Executors.newFixedThreadPool(this.numThreads);
    }

    @Override
    public boolean isRunning() {
        return this.executorService != null && !this.executorService.isShutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void stop() {
        if (!this.isRunning()) {
            throw new IllegalStateException("Instance is already stopped.");
        }
        try {
            if (this.getShutdownReplicatorSeconds() <= 0) {
                this.executorService.shutdownNow();
            } else {
                this.executorService.shutdown();
            }
            this.executorService.awaitTermination(this.getShutdownReplicatorSeconds(), TimeUnit.SECONDS);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
        finally {
            if (this.executorService.isTerminated()) {
                logger.info("HTTP Request Replicator has been terminated successfully.");
            } else {
                logger.warn("HTTP Request Replicator has not terminated properly.  There exists an uninterruptable thread that will take an indeterminate amount of time to stop.");
            }
        }
    }

    public synchronized void setNodeProtocolScheme(String nodeProtocolScheme) {
        if (StringUtils.isNotBlank((CharSequence)nodeProtocolScheme) && !"http".equalsIgnoreCase(nodeProtocolScheme) && !"https".equalsIgnoreCase(nodeProtocolScheme)) {
            throw new IllegalArgumentException("Node Protocol Scheme must be either HTTP or HTTPS");
        }
        this.nodeProtocolScheme = nodeProtocolScheme;
    }

    public synchronized String getNodeProtocolScheme() {
        return this.nodeProtocolScheme;
    }

    private synchronized String getNodeProtocolScheme(URI uri) {
        if (StringUtils.isBlank((CharSequence)this.nodeProtocolScheme)) {
            return uri.getScheme();
        }
        return this.nodeProtocolScheme;
    }

    public int getConnectionTimeoutMs() {
        return this.connectionTimeoutMs;
    }

    public int getReadTimeoutMs() {
        return this.readTimeoutMs;
    }

    public int getShutdownReplicatorSeconds() {
        return this.shutdownReplicatorSeconds;
    }

    public void setShutdownReplicatorSeconds(int shutdownReplicatorSeconds) {
        this.shutdownReplicatorSeconds = shutdownReplicatorSeconds;
    }

    @Override
    public Set<NodeResponse> replicate(Set<NodeIdentifier> nodeIds, String method, URI uri, Map<String, List<String>> parameters, Map<String, String> headers) throws UriConstructionException {
        if (nodeIds == null) {
            throw new IllegalArgumentException("Node IDs may not be null.");
        }
        if (method == null) {
            throw new IllegalArgumentException("HTTP method may not be null.");
        }
        if (uri == null) {
            throw new IllegalArgumentException("URI may not be null.");
        }
        if (parameters == null) {
            throw new IllegalArgumentException("Parameters may not be null.");
        }
        if (headers == null) {
            throw new IllegalArgumentException("HTTP headers map may not be null.");
        }
        return this.replicateHelper(nodeIds, method, this.getNodeProtocolScheme(uri), uri.getPath(), parameters, null, headers);
    }

    @Override
    public Set<NodeResponse> replicate(Set<NodeIdentifier> nodeIds, String method, URI uri, Object entity, Map<String, String> headers) throws UriConstructionException {
        if (nodeIds == null) {
            throw new IllegalArgumentException("Node IDs may not be null.");
        }
        if (method == null) {
            throw new IllegalArgumentException("HTTP method may not be null.");
        }
        if (method.equalsIgnoreCase("DELETE") || method.equalsIgnoreCase("GET") || method.equalsIgnoreCase("HEAD") || method.equalsIgnoreCase("OPTIONS")) {
            throw new IllegalArgumentException("HTTP (DELETE | GET | HEAD | OPTIONS) requests cannot have a body containing an entity.");
        }
        if (uri == null) {
            throw new IllegalArgumentException("URI may not be null.");
        }
        if (entity == null) {
            throw new IllegalArgumentException("Entity may not be null.");
        }
        if (headers == null) {
            throw new IllegalArgumentException("HTTP headers map may not be null.");
        }
        return this.replicateHelper(nodeIds, method, this.getNodeProtocolScheme(uri), uri.getPath(), null, entity, headers);
    }

    private Set<NodeResponse> replicateHelper(Set<NodeIdentifier> nodeIds, String method, String scheme, String path, Map<String, List<String>> parameters, Object entity, Map<String, String> headers) throws UriConstructionException {
        if (nodeIds.isEmpty()) {
            return new HashSet<NodeResponse>();
        }
        ExecutorCompletionService<NodeResponse> completionService = new ExecutorCompletionService<NodeResponse>(this.executorService);
        ArrayList<NodeHttpRequestFutureWrapper> futureNodeHttpRequests = new ArrayList<NodeHttpRequestFutureWrapper>();
        HashMap<NodeIdentifier, URI> uriMap = new HashMap<NodeIdentifier, URI>();
        try {
            for (NodeIdentifier nodeId : nodeIds) {
                URI nodeUri = new URI(scheme, null, nodeId.getApiAddress(), nodeId.getApiPort(), path, null, null);
                uriMap.put(nodeId, nodeUri);
            }
        }
        catch (URISyntaxException use) {
            throw new UriConstructionException(use);
        }
        String requestId = UUID.randomUUID().toString();
        headers.put("X-RequestID", requestId);
        for (Map.Entry entry : uriMap.entrySet()) {
            NodeIdentifier nodeId = (NodeIdentifier)entry.getKey();
            URI nodeUri = (URI)entry.getValue();
            NodeHttpRequestCallable callable = entity == null ? new NodeHttpRequestCallable(nodeId, method, nodeUri, parameters, headers) : new NodeHttpRequestCallable(nodeId, method, nodeUri, entity, headers);
            futureNodeHttpRequests.add(new NodeHttpRequestFutureWrapper(nodeId, method, nodeUri, completionService.submit(callable)));
        }
        HashSet<NodeResponse> result = new HashSet<NodeResponse>();
        for (int i = 0; i < nodeIds.size(); ++i) {
            NodeResponse nodeResponse;
            NodeHttpRequestFutureWrapper futureNodeHttpRequest = null;
            try {
                Future futureNodeResourceResponse = completionService.take();
                for (NodeHttpRequestFutureWrapper futureNodeHttpRequestElem : futureNodeHttpRequests) {
                    if (futureNodeHttpRequestElem.getFuture() != futureNodeResourceResponse) continue;
                    futureNodeHttpRequest = futureNodeHttpRequestElem;
                }
                nodeResponse = (NodeResponse)futureNodeResourceResponse.get();
                result.add(nodeResponse);
                continue;
            }
            catch (InterruptedException | ExecutionException ex) {
                logger.warn("Node request for " + futureNodeHttpRequest.getNodeId() + " encountered exception: " + ex, (Throwable)ex);
                nodeResponse = new NodeResponse(futureNodeHttpRequest.getNodeId(), futureNodeHttpRequest.getHttpMethod(), futureNodeHttpRequest.getRequestUri(), ex);
                result.add(nodeResponse);
            }
        }
        if (logger.isDebugEnabled()) {
            NodeResponse min = null;
            NodeResponse max = null;
            long nanosSum = 0L;
            int nanosAdded = 0;
            for (NodeResponse response : result) {
                long maxNanos;
                long requestNanos = response.getRequestDuration(TimeUnit.NANOSECONDS);
                long minNanos = min == null ? -1L : min.getRequestDuration(TimeUnit.NANOSECONDS);
                long l = maxNanos = max == null ? -1L : max.getRequestDuration(TimeUnit.NANOSECONDS);
                if (requestNanos < minNanos || minNanos < 0L) {
                    min = response;
                }
                if (requestNanos > maxNanos || maxNanos < 0L) {
                    max = response;
                }
                if (requestNanos < 0L) continue;
                nanosSum += requestNanos;
                ++nanosAdded;
            }
            StringBuilder sb = new StringBuilder();
            sb.append("Node Responses for ").append(method).append(" ").append(path).append(" (Request ID ").append(requestId).append("):\n");
            for (NodeResponse response : result) {
                sb.append(response).append("\n");
            }
            long averageNanos = nanosAdded == 0 ? -1L : nanosSum / (long)nanosAdded;
            long averageMillis = averageNanos < 0L ? averageNanos : TimeUnit.MILLISECONDS.convert(averageNanos, TimeUnit.NANOSECONDS);
            logger.debug("For {} {} (Request ID {}), minimum response time = {}, max = {}, average = {} ms", new Object[]{method, path, requestId, min, max, averageMillis});
            logger.debug(sb.toString());
        }
        return result;
    }

    private class NodeHttpRequestCallable
    implements Callable<NodeResponse> {
        private final NodeIdentifier nodeId;
        private final String method;
        private final URI uri;
        private final Object entity;
        private final Map<String, List<String>> parameters = new HashMap<String, List<String>>();
        private final Map<String, String> headers = new HashMap<String, String>();

        private NodeHttpRequestCallable(NodeIdentifier nodeId, String method, URI uri, Object entity, Map<String, String> headers) {
            this.nodeId = nodeId;
            this.method = method;
            this.uri = uri;
            this.entity = entity;
            this.headers.putAll(headers);
        }

        private NodeHttpRequestCallable(NodeIdentifier nodeId, String method, URI uri, Map<String, List<String>> parameters, Map<String, String> headers) {
            this.nodeId = nodeId;
            this.method = method;
            this.uri = uri;
            this.entity = null;
            this.parameters.putAll(parameters);
            this.headers.putAll(headers);
        }

        @Override
        public NodeResponse call() {
            try {
                ClientResponse clientResponse;
                WebResource.Builder resourceBuilder = this.getResourceBuilder();
                String requestId = this.headers.get("x-nifi-request-id");
                long startNanos = System.nanoTime();
                if ("DELETE".equalsIgnoreCase(this.method)) {
                    clientResponse = (ClientResponse)resourceBuilder.delete(ClientResponse.class);
                } else if ("GET".equalsIgnoreCase(this.method)) {
                    clientResponse = (ClientResponse)resourceBuilder.get(ClientResponse.class);
                } else if ("HEAD".equalsIgnoreCase(this.method)) {
                    clientResponse = resourceBuilder.head();
                } else if ("OPTIONS".equalsIgnoreCase(this.method)) {
                    clientResponse = (ClientResponse)resourceBuilder.options(ClientResponse.class);
                } else if ("POST".equalsIgnoreCase(this.method)) {
                    clientResponse = (ClientResponse)resourceBuilder.post(ClientResponse.class);
                } else if ("PUT".equalsIgnoreCase(this.method)) {
                    clientResponse = (ClientResponse)resourceBuilder.put(ClientResponse.class);
                } else {
                    throw new IllegalArgumentException("HTTP Method '" + this.method + "' not supported for request replication.");
                }
                return new NodeResponse(this.nodeId, this.method, this.uri, clientResponse, System.nanoTime() - startNanos, requestId);
            }
            catch (UniformInterfaceException | IllegalArgumentException t) {
                return new NodeResponse(this.nodeId, this.method, this.uri, t);
            }
        }

        private WebResource.Builder getResourceBuilder() {
            WebResource.Builder builder;
            MultivaluedMapImpl map = new MultivaluedMapImpl();
            map.putAll(this.parameters);
            WebResource resource = HttpRequestReplicatorImpl.this.client.resource(this.uri);
            if (WebClusterManager.isResponseInterpreted(this.uri, this.method)) {
                resource.addFilter((ClientFilter)new GZIPContentEncodingFilter(false));
            }
            if ("DELETE".equalsIgnoreCase(this.method) || "HEAD".equalsIgnoreCase(this.method) || "GET".equalsIgnoreCase(this.method) || "OPTIONS".equalsIgnoreCase(this.method)) {
                resource = resource.queryParams((MultivaluedMap)map);
                builder = resource.getRequestBuilder();
            } else {
                builder = this.entity == null ? resource.entity((Object)map) : resource.entity(this.entity);
            }
            boolean foundContentType = false;
            for (Map.Entry<String, String> entry : this.headers.entrySet()) {
                builder.header(entry.getKey(), (Object)entry.getValue());
                if (!entry.getKey().equalsIgnoreCase("content-type")) continue;
                foundContentType = true;
            }
            if (!foundContentType) {
                builder.type("application/x-www-form-urlencoded");
            }
            return builder;
        }
    }

    private class NodeHttpRequestFutureWrapper {
        private final NodeIdentifier nodeId;
        private final String httpMethod;
        private final URI requestUri;
        private final Future<NodeResponse> future;

        public NodeHttpRequestFutureWrapper(NodeIdentifier nodeId, String httpMethod, URI requestUri, Future<NodeResponse> future) {
            if (nodeId == null) {
                throw new IllegalArgumentException("Node ID may not be null.");
            }
            if (StringUtils.isBlank((CharSequence)httpMethod)) {
                throw new IllegalArgumentException("Http method may not be null or empty.");
            }
            if (requestUri == null) {
                throw new IllegalArgumentException("Request URI may not be null.");
            }
            if (future == null) {
                throw new IllegalArgumentException("Future may not be null.");
            }
            this.nodeId = nodeId;
            this.httpMethod = httpMethod;
            this.requestUri = requestUri;
            this.future = future;
        }

        public NodeIdentifier getNodeId() {
            return this.nodeId;
        }

        public String getHttpMethod() {
            return this.httpMethod;
        }

        public URI getRequestUri() {
            return this.requestUri;
        }

        public Future<NodeResponse> getFuture() {
            return this.future;
        }
    }
}

