/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tez.http;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.IOUtils;
import org.apache.tez.common.security.JobTokenSecretManager;
import org.apache.tez.http.BaseHttpConnection;
import org.apache.tez.http.HttpConnectionParams;
import org.apache.tez.http.SSLFactory;
import org.apache.tez.runtime.library.common.security.SecureShuffleUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpConnection
extends BaseHttpConnection {
    private static final Logger LOG = LoggerFactory.getLogger(HttpConnection.class);
    private URL url;
    private final String logIdentifier;
    @VisibleForTesting
    protected volatile HttpURLConnection connection;
    private volatile DataInputStream input;
    private volatile boolean connectionSucceeed;
    private volatile boolean cleanup;
    private final JobTokenSecretManager jobTokenSecretMgr;
    private String encHash;
    private String msgToEncode;
    private final HttpConnectionParams httpConnParams;
    private final Stopwatch stopWatch;

    public HttpConnection(URL url, HttpConnectionParams connParams, String logIdentifier, JobTokenSecretManager jobTokenSecretManager) throws IOException {
        this.logIdentifier = logIdentifier;
        this.jobTokenSecretMgr = jobTokenSecretManager;
        this.httpConnParams = connParams;
        this.url = url;
        this.stopWatch = new Stopwatch();
        if (LOG.isDebugEnabled()) {
            LOG.debug("MapOutput URL :" + url.toString());
        }
    }

    @VisibleForTesting
    public void computeEncHash() throws IOException {
        this.msgToEncode = SecureShuffleUtils.buildMsgFrom(this.url);
        this.encHash = SecureShuffleUtils.hashFromString(this.msgToEncode, this.jobTokenSecretMgr);
    }

    private void setupConnection() throws IOException {
        this.connection = (HttpURLConnection)this.url.openConnection();
        if (this.httpConnParams.isSslShuffle()) {
            SSLFactory sslFactory = this.httpConnParams.getSslFactory();
            Preconditions.checkArgument((sslFactory != null ? 1 : 0) != 0, (Object)"SSLFactory can not be null");
            sslFactory.configure(this.connection);
        }
        this.computeEncHash();
        this.connection.addRequestProperty("UrlHash", this.encHash);
        this.connection.setReadTimeout(this.httpConnParams.getReadTimeout());
        this.connection.addRequestProperty("name", "mapreduce");
        this.connection.addRequestProperty("version", "1.0.0");
    }

    @Override
    public boolean connect() throws IOException {
        return this.connect(this.httpConnParams.getConnectionTimeout());
    }

    private boolean connect(int connectionTimeout) throws IOException {
        this.stopWatch.reset().start();
        if (this.connection == null) {
            this.setupConnection();
        }
        int unit = 0;
        if (connectionTimeout < 0) {
            throw new IOException("Invalid timeout [timeout = " + connectionTimeout + " ms]");
        }
        if (connectionTimeout > 0) {
            unit = Math.min(60000, connectionTimeout);
        }
        this.connection.setConnectTimeout(unit);
        int connectionFailures = 0;
        while (true) {
            long connectStartTime = System.currentTimeMillis();
            try {
                this.connection.connect();
                this.connectionSucceeed = true;
            }
            catch (IOException ioe) {
                ++connectionFailures;
                if (this.cleanup) {
                    LOG.info("Cleanup is set to true. Not attempting to connect again. Last exception was: [" + ioe.getClass().getName() + ", " + ioe.getMessage() + "]");
                    return false;
                }
                if ((connectionTimeout -= unit) <= 0) {
                    throw new IOException("Failed to connect to " + this.url + ", #connectionFailures=" + connectionFailures, ioe);
                }
                long elapsed = System.currentTimeMillis() - connectStartTime;
                if (elapsed < (long)unit) {
                    try {
                        long sleepTime = (long)unit - elapsed;
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Sleeping for " + sleepTime + " while establishing connection to " + this.url + ", since connectAttempt returned in " + elapsed + " ms");
                        }
                        Thread.sleep(sleepTime);
                    }
                    catch (InterruptedException e) {
                        throw new IOException("Connection establishment sleep interrupted, #connectionFailures=" + connectionFailures, e);
                    }
                }
                if (connectionTimeout >= unit) continue;
                unit = connectionTimeout;
                this.connection.setConnectTimeout(unit);
                continue;
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Time taken to connect to " + this.url.toString() + " " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms; connectionFailures=" + connectionFailures);
        }
        return true;
    }

    @Override
    public void validate() throws IOException {
        this.stopWatch.reset().start();
        int rc = this.connection.getResponseCode();
        if (rc != 200) {
            throw new IOException("Got invalid response code " + rc + " from " + this.url + ": " + this.connection.getResponseMessage());
        }
        if (!"mapreduce".equals(this.connection.getHeaderField("name")) || !"1.0.0".equals(this.connection.getHeaderField("version"))) {
            throw new IOException("Incompatible shuffle response version");
        }
        String replyHash = this.connection.getHeaderField("ReplyHash");
        if (replyHash == null) {
            throw new IOException("security validation of TT Map output failed");
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("url=" + this.msgToEncode + ";encHash=" + this.encHash + ";replyHash=" + replyHash);
        }
        SecureShuffleUtils.verifyReply(replyHash, this.encHash, this.jobTokenSecretMgr);
        LOG.info("for url=" + this.url + " sent hash and receievd reply " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms");
    }

    @Override
    public DataInputStream getInputStream() throws IOException {
        this.stopWatch.reset().start();
        if (this.connectionSucceeed) {
            this.input = new DataInputStream(new BufferedInputStream(this.connection.getInputStream(), this.httpConnParams.getBufferSize()));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Time taken to getInputStream (connect) " + this.url + " " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms");
        }
        return this.input;
    }

    @Override
    public void cleanup(boolean disconnect) throws IOException {
        this.cleanup = true;
        this.stopWatch.reset().start();
        try {
            if (this.input != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Closing input on " + this.logIdentifier);
                }
                this.input.close();
                this.input = null;
            }
            if (this.httpConnParams.isKeepAlive() && this.connectionSucceeed) {
                this.readErrorStream(this.connection.getErrorStream());
            }
            if (this.connection != null && (disconnect || !this.httpConnParams.isKeepAlive())) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Closing connection on " + this.logIdentifier + ", disconnectParam=" + disconnect);
                }
                this.connection.disconnect();
                this.connection = null;
            }
        }
        catch (IOException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Exception while shutting down fetcher " + this.logIdentifier, (Throwable)e);
            }
            LOG.info("Exception while shutting down fetcher " + this.logIdentifier + ": " + e.getMessage());
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Time taken to cleanup connection to " + this.url + " " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms");
        }
    }

    private void readErrorStream(InputStream errorStream) {
        if (errorStream == null) {
            return;
        }
        try {
            DataOutputBuffer errorBuffer = new DataOutputBuffer();
            IOUtils.copyBytes((InputStream)errorStream, (OutputStream)errorBuffer, (int)4096);
            IOUtils.closeStream((Closeable)errorBuffer);
            IOUtils.closeStream((Closeable)errorStream);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }
}

