/**
 *
 * Copyright 2005-2006 The Apache Software Foundation or its licensors, as applicable.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.xbean.kernel;

import java.util.Set;

/**
 * The StartStrategy interface is used to assist the kernel in determining how to handle problems that occur while
 * starting a service.
 *
 * @author Dain Sundstrom
 * @version $Id$
 * @since 2.0
 */
public interface StartStrategy {
    /**
     * Determines if the kernel should wait for the unsatified conditions to be satisfied.
     *
     * @param serviceName the name of the service that has the unsatisfied condtions
     * @param conditions the unsatisfied condtions
     * @return true if the kernel should wait for the conditions to be satisfied; false if the strategy would like
     *         silently leave the service in the starting state
     * @throws UnsatisfiedConditionsException the the strategy would like to leave the service in the starting state
     * and throw an exception the caller
     * @throws UnregisterServiceException if the strategy would like to ignore the unsatisfied conditions and continue to
     * destry the service
     */
    boolean waitForUnsatisfiedConditions(ServiceName serviceName, Set conditions) throws UnsatisfiedConditionsException, UnregisterServiceException;

    /**
     * Handle the start error.  The strategy can rethrow the exception, throw an {@link UnregisterServiceException}, or
     * return.  If this method rethrows the exception, the service will be destroyed and the exception will be thrown to
     * the caller.  If an UnregisterServiceException is thrown, the kernel will unregister the service and rethrow
     * {@link UnregisterServiceException#getCause()}.  If this method returns without throwing an exception, the kernel
     * will pass the exception to the service monitor for processing and leave the service in the starting state.
     *
     * @param serviceName the name of the service that has the error
     * @param startError the Exception or Error
     * @throws UnregisterServiceException if the strategy would like the service to be destroyed and unregistered
     * @throws Exception if the strategy would like to destroy the service and throw the exception to the caller
     */
    void startError(ServiceName serviceName, Throwable startError) throws UnregisterServiceException, Exception;
}
