/*
 * Decompiled with CFR 0.152.
 */
package org.ballerinalang.stdlib.crypto;

import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.ballerinalang.jvm.BallerinaErrors;
import org.ballerinalang.jvm.values.ArrayValue;
import org.ballerinalang.jvm.values.ErrorValue;

public class CryptoUtils {
    private static final Pattern varPattern = Pattern.compile("\\$\\{([^}]*)}");
    private static final int[] VALID_GCM_TAG_SIZES = new int[]{32, 63, 96, 104, 112, 120, 128};
    private static final int[] VALID_AES_KEY_SIZES = new int[]{16, 24, 32};

    private CryptoUtils() {
    }

    public static byte[] hmac(String algorithm, byte[] key, byte[] input) {
        try {
            SecretKeySpec secretKey = new SecretKeySpec(key, algorithm);
            Mac mac = Mac.getInstance(algorithm);
            mac.init(secretKey);
            return mac.doFinal(input);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw CryptoUtils.createError("Error occurred while calculating HMAC: " + e.getMessage());
        }
    }

    public static byte[] hash(String algorithm, byte[] input) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance(algorithm);
            messageDigest.update(input);
            return messageDigest.digest();
        }
        catch (NoSuchAlgorithmException e) {
            throw CryptoUtils.createError("Error occurred while calculating hash: " + e.getMessage());
        }
    }

    public static Object sign(String algorithm, PrivateKey privateKey, byte[] input) {
        try {
            Signature sig = Signature.getInstance(algorithm);
            sig.initSign(privateKey);
            sig.update(input);
            return new ArrayValue(sig.sign());
        }
        catch (InvalidKeyException e) {
            return CryptoUtils.createError("Uninitialized private key: " + e.getMessage());
        }
        catch (NoSuchAlgorithmException | SignatureException e) {
            throw CryptoUtils.createError("Error occurred while calculating signature: " + e.getMessage());
        }
    }

    public static Object verify(String algorithm, PublicKey publicKey, byte[] data, byte[] signature) {
        try {
            Signature sig = Signature.getInstance(algorithm);
            sig.initVerify(publicKey);
            sig.update(data);
            return sig.verify(signature);
        }
        catch (InvalidKeyException e) {
            return CryptoUtils.createError("Uninitialized public key: " + e.getMessage());
        }
        catch (NoSuchAlgorithmException | SignatureException e) {
            throw CryptoUtils.createError("Error occurred while calculating signature: " + e.getMessage());
        }
    }

    public static ErrorValue createError(String errMsg) {
        return BallerinaErrors.createError((String)"{ballerina/crypto}Error", (String)errMsg);
    }

    public static Object rsaEncryptDecrypt(CipherMode cipherMode, String algorithmMode, String algorithmPadding, Key key, byte[] input, byte[] iv, long tagSize) {
        try {
            String transformedAlgorithmMode = CryptoUtils.transformAlgorithmMode(algorithmMode);
            String transformedAlgorithmPadding = CryptoUtils.transformAlgorithmPadding(algorithmPadding);
            if (tagSize != -1L && Arrays.stream(VALID_GCM_TAG_SIZES).noneMatch(i -> tagSize == (long)i)) {
                return CryptoUtils.createError("Valid tag sizes are: " + Arrays.toString(VALID_GCM_TAG_SIZES));
            }
            AlgorithmParameterSpec paramSpec = CryptoUtils.buildParameterSpec(transformedAlgorithmMode, iv, (int)tagSize);
            Cipher cipher = Cipher.getInstance("RSA/" + transformedAlgorithmMode + "/" + transformedAlgorithmPadding);
            CryptoUtils.initCipher(cipher, cipherMode, key, paramSpec);
            return new ArrayValue(cipher.doFinal(input));
        }
        catch (NoSuchAlgorithmException e) {
            return CryptoUtils.createError("Unsupported algorithm: RSA " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (NoSuchPaddingException e) {
            return CryptoUtils.createError("Unsupported padding scheme defined in the algorithm: RSA " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException | ErrorValue e) {
            return CryptoUtils.createError("Error occurred while RSA encrypt/decrypt: " + e.getMessage());
        }
    }

    public static Object aesEncryptDecrypt(CipherMode cipherMode, String algorithmMode, String algorithmPadding, byte[] key, byte[] input, byte[] iv, long tagSize) {
        try {
            if (Arrays.stream(VALID_AES_KEY_SIZES).noneMatch(validSize -> validSize == key.length)) {
                return CryptoUtils.createError("Invalid key size. valid key sizes in bytes: " + Arrays.toString(VALID_AES_KEY_SIZES));
            }
            String transformedAlgorithmMode = CryptoUtils.transformAlgorithmMode(algorithmMode);
            String transformedAlgorithmPadding = CryptoUtils.transformAlgorithmPadding(algorithmPadding);
            SecretKeySpec keySpec = new SecretKeySpec(key, "AES");
            if (tagSize != -1L && Arrays.stream(VALID_GCM_TAG_SIZES).noneMatch(validSize -> (long)validSize == tagSize)) {
                return CryptoUtils.createError("Invalid tag size. valid tag sizes in bytes: " + Arrays.toString(VALID_GCM_TAG_SIZES));
            }
            AlgorithmParameterSpec paramSpec = CryptoUtils.buildParameterSpec(transformedAlgorithmMode, iv, (int)tagSize);
            Cipher cipher = Cipher.getInstance("AES/" + transformedAlgorithmMode + "/" + transformedAlgorithmPadding);
            CryptoUtils.initCipher(cipher, cipherMode, keySpec, paramSpec);
            return new ArrayValue(cipher.doFinal(input));
        }
        catch (NoSuchAlgorithmException e) {
            return CryptoUtils.createError("Unsupported algorithm: AES " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (NoSuchPaddingException e) {
            return CryptoUtils.createError("Unsupported padding scheme defined in  the algorithm: AES " + algorithmMode + " " + algorithmPadding + ": " + e.getMessage());
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException | ErrorValue e) {
            return CryptoUtils.createError("Error occurred while AES encrypt/decrypt: " + e.getMessage());
        }
    }

    private static void initCipher(Cipher cipher, CipherMode cipherMode, Key key, AlgorithmParameterSpec paramSpec) throws InvalidKeyException, InvalidAlgorithmParameterException {
        switch (cipherMode) {
            case ENCRYPT: {
                if (paramSpec == null) {
                    cipher.init(1, key);
                    break;
                }
                cipher.init(1, key, paramSpec);
                break;
            }
            case DECRYPT: {
                if (paramSpec == null) {
                    cipher.init(2, key);
                    break;
                }
                cipher.init(2, key, paramSpec);
            }
        }
    }

    private static AlgorithmParameterSpec buildParameterSpec(String algorithmMode, byte[] iv, int tagSize) {
        switch (algorithmMode) {
            case "GCM": {
                if (iv == null) {
                    throw BallerinaErrors.createError((String)"GCM mode requires 16 byte IV");
                }
                return new GCMParameterSpec(tagSize, iv);
            }
            case "CBC": {
                if (iv == null) {
                    throw BallerinaErrors.createError((String)"CBC mode requires 16 byte IV");
                }
                return new IvParameterSpec(iv);
            }
            case "ECB": {
                if (iv == null) break;
                throw BallerinaErrors.createError((String)"ECB mode cannot use IV");
            }
        }
        return null;
    }

    private static String transformAlgorithmMode(String algorithmMode) throws ErrorValue {
        if (!(algorithmMode.equals("CBC") || algorithmMode.equals("ECB") || algorithmMode.equals("GCM"))) {
            throw BallerinaErrors.createError((String)("Unsupported mode: " + algorithmMode));
        }
        return algorithmMode;
    }

    private static String transformAlgorithmPadding(String algorithmPadding) throws ErrorValue {
        switch (algorithmPadding) {
            case "PKCS1": {
                algorithmPadding = "PKCS1Padding";
                break;
            }
            case "PKCS5": {
                algorithmPadding = "PKCS5Padding";
                break;
            }
            case "OAEPwithMD5andMGF1": {
                algorithmPadding = "OAEPWithMD5AndMGF1Padding";
                break;
            }
            case "OAEPWithSHA1AndMGF1": {
                algorithmPadding = "OAEPWithSHA-1AndMGF1Padding";
                break;
            }
            case "OAEPWithSHA256AndMGF1": {
                algorithmPadding = "OAEPWithSHA-256AndMGF1Padding";
                break;
            }
            case "OAEPwithSHA384andMGF1": {
                algorithmPadding = "OAEPWithSHA-384AndMGF1Padding";
                break;
            }
            case "OAEPwithSHA512andMGF1": {
                algorithmPadding = "OAEPWithSHA-512AndMGF1Padding";
                break;
            }
            case "NONE": {
                algorithmPadding = "NoPadding";
                break;
            }
            default: {
                throw BallerinaErrors.createError((String)("Unsupported padding: " + algorithmPadding));
            }
        }
        return algorithmPadding;
    }

    public static String substituteVariables(String value) {
        Matcher matcher = varPattern.matcher(value);
        boolean found = matcher.find();
        if (!found) {
            return value;
        }
        StringBuffer sb = new StringBuffer();
        do {
            String sysPropKey;
            String sysPropValue;
            if ((sysPropValue = CryptoUtils.getSystemVariableValue(sysPropKey = matcher.group(1), null)) == null || sysPropValue.length() == 0) {
                throw new RuntimeException("System property " + sysPropKey + " is not specified");
            }
            sysPropValue = sysPropValue.replace("\\", "\\\\");
            matcher.appendReplacement(sb, sysPropValue);
        } while (matcher.find());
        matcher.appendTail(sb);
        return sb.toString();
    }

    private static String getSystemVariableValue(String variableName, String defaultValue) {
        if (System.getProperty(variableName) != null) {
            return System.getProperty(variableName);
        }
        if (System.getenv(variableName) != null) {
            return System.getenv(variableName);
        }
        return defaultValue;
    }

    public static enum CipherMode {
        ENCRYPT,
        DECRYPT;

    }
}

