/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.sdk.server.subscriptions;

import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.PeekingIterator;
import com.google.common.math.DoubleMath;
import com.google.common.primitives.Ints;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.milo.opcua.sdk.server.Session;
import org.eclipse.milo.opcua.sdk.server.items.BaseMonitoredItem;
import org.eclipse.milo.opcua.sdk.server.subscriptions.PublishQueue;
import org.eclipse.milo.opcua.sdk.server.subscriptions.SubscriptionManager;
import org.eclipse.milo.opcua.stack.core.application.services.ServiceRequest;
import org.eclipse.milo.opcua.stack.core.serialization.UaResponseMessage;
import org.eclipse.milo.opcua.stack.core.serialization.UaStructure;
import org.eclipse.milo.opcua.stack.core.types.builtin.DateTime;
import org.eclipse.milo.opcua.stack.core.types.builtin.DiagnosticInfo;
import org.eclipse.milo.opcua.stack.core.types.builtin.ExtensionObject;
import org.eclipse.milo.opcua.stack.core.types.builtin.StatusCode;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UInteger;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned;
import org.eclipse.milo.opcua.stack.core.types.structured.DataChangeNotification;
import org.eclipse.milo.opcua.stack.core.types.structured.EventFieldList;
import org.eclipse.milo.opcua.stack.core.types.structured.EventNotificationList;
import org.eclipse.milo.opcua.stack.core.types.structured.ModifySubscriptionRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.MonitoredItemNotification;
import org.eclipse.milo.opcua.stack.core.types.structured.NotificationMessage;
import org.eclipse.milo.opcua.stack.core.types.structured.PublishRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.PublishResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.ResponseHeader;
import org.eclipse.milo.opcua.stack.core.types.structured.SetPublishingModeRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.StatusChangeNotification;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Subscription {
    private static final double MIN_LIFETIME = 10000.0;
    private static final double MAX_LIFETIME = 3600000.0;
    private static final double MIN_PUBLISHING_INTERVAL = 1.0;
    private static final double MAX_PUBLISHING_INTERVAL = 60000.0;
    private static final int MAX_NOTIFICATIONS = 65535;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private volatile Iterator<BaseMonitoredItem<?>> lastIterator = Collections.emptyIterator();
    private final AtomicLong itemIds = new AtomicLong(1L);
    private final Map<UInteger, BaseMonitoredItem<?>> itemsById = Maps.newConcurrentMap();
    private final AtomicReference<State> state = new AtomicReference<State>(State.Normal);
    private final AtomicReference<StateListener> stateListener = new AtomicReference();
    private final AtomicLong sequenceNumber = new AtomicLong(1L);
    private final Map<UInteger, NotificationMessage> availableMessages = Maps.newConcurrentMap();
    private final PublishHandler publishHandler = new PublishHandler();
    private final TimerHandler timerHandler = new TimerHandler();
    private volatile boolean messageSent = false;
    private volatile boolean moreNotifications = false;
    private volatile long keepAliveCounter;
    private volatile long lifetimeCounter;
    private volatile double publishingInterval;
    private volatile long lifetimeCount;
    private volatile long maxKeepAliveCount;
    private volatile int maxNotificationsPerPublish;
    private volatile boolean publishingEnabled;
    private volatile int priority;
    private volatile SubscriptionManager subscriptionManager;
    private final UInteger subscriptionId;

    public Subscription(SubscriptionManager subscriptionManager, UInteger subscriptionId, double publishingInterval, long maxKeepAliveCount, long lifetimeCount, long maxNotificationsPerPublish, boolean publishingEnabled, int priority) {
        this.subscriptionManager = subscriptionManager;
        this.subscriptionId = subscriptionId;
        this.setPublishingInterval(publishingInterval);
        this.setMaxKeepAliveCount(maxKeepAliveCount);
        this.setLifetimeCount(lifetimeCount);
        this.setMaxNotificationsPerPublish(maxNotificationsPerPublish);
        this.publishingEnabled = publishingEnabled;
        this.priority = priority;
        this.resetKeepAliveCounter();
        this.resetLifetimeCounter();
        this.logger.debug("[id={}] subscription created, interval={}, keep-alive={}, lifetime={}", new Object[]{subscriptionId, publishingInterval, maxKeepAliveCount, lifetimeCount});
    }

    public synchronized void modifySubscription(ModifySubscriptionRequest request) {
        this.setPublishingInterval(request.getRequestedPublishingInterval());
        this.setMaxKeepAliveCount(request.getRequestedMaxKeepAliveCount().longValue());
        this.setLifetimeCount(request.getRequestedLifetimeCount().longValue());
        this.setMaxNotificationsPerPublish(request.getMaxNotificationsPerPublish().longValue());
        this.priority = request.getPriority().intValue();
        this.resetLifetimeCounter();
        this.logger.debug("[id={}] subscription modified, interval={}, keep-alive={}, lifetime={}", new Object[]{this.subscriptionId, this.publishingInterval, this.maxKeepAliveCount, this.lifetimeCount});
    }

    public synchronized List<BaseMonitoredItem<?>> deleteSubscription() {
        this.setState(State.Closed);
        this.logger.debug("[id={}] subscription deleted.", (Object)this.subscriptionId);
        return Lists.newArrayList(this.itemsById.values());
    }

    public synchronized void setPublishingMode(SetPublishingModeRequest request) {
        this.publishingEnabled = request.getPublishingEnabled();
        this.resetLifetimeCounter();
        this.logger.debug("[id={}] {}.", (Object)this.subscriptionId, (Object)(this.publishingEnabled ? "publishing enabled." : "publishing disabled."));
    }

    public synchronized void addMonitoredItems(List<BaseMonitoredItem<?>> createdItems) {
        for (BaseMonitoredItem<?> item : createdItems) {
            this.itemsById.put(item.getId(), item);
        }
        this.resetLifetimeCounter();
        this.logger.debug("[id={}] created {} MonitoredItems.", (Object)this.subscriptionId, (Object)createdItems.size());
    }

    public synchronized void removeMonitoredItems(List<BaseMonitoredItem<?>> deletedItems) {
        for (BaseMonitoredItem<?> item : deletedItems) {
            this.itemsById.remove(item.getId());
        }
        this.resetLifetimeCounter();
        this.logger.debug("[id={}] deleted {} MonitoredItems.", (Object)this.subscriptionId, (Object)deletedItems.size());
    }

    public synchronized Map<UInteger, BaseMonitoredItem<?>> getMonitoredItems() {
        return this.itemsById;
    }

    private void setPublishingInterval(double requestedPublishingInterval) {
        if (requestedPublishingInterval < 1.0 || Double.isNaN(requestedPublishingInterval) || Double.isInfinite(requestedPublishingInterval)) {
            requestedPublishingInterval = 1.0;
        }
        if (requestedPublishingInterval > 60000.0) {
            requestedPublishingInterval = 60000.0;
        }
        this.publishingInterval = requestedPublishingInterval;
    }

    private void setMaxKeepAliveCount(long maxKeepAliveCount) {
        double keepAliveInterval;
        if (maxKeepAliveCount == 0L) {
            maxKeepAliveCount = 3L;
        }
        if ((keepAliveInterval = (double)maxKeepAliveCount * this.publishingInterval) > 3600000.0) {
            maxKeepAliveCount = (long)(3600000.0 / this.publishingInterval);
            if (maxKeepAliveCount < 0xFFFFFFFFL && 3600000.0 % this.publishingInterval != 0.0) {
                ++maxKeepAliveCount;
            }
            keepAliveInterval = (double)maxKeepAliveCount * this.publishingInterval;
        }
        if (keepAliveInterval > 60000.0 && (maxKeepAliveCount = (long)(60000.0 / this.publishingInterval)) < 0xFFFFFFFFL && 60000.0 % this.publishingInterval != 0.0) {
            ++maxKeepAliveCount;
        }
        this.maxKeepAliveCount = maxKeepAliveCount;
    }

    private void setLifetimeCount(long lifetimeCount) {
        double lifetimeInterval = (double)lifetimeCount * this.publishingInterval;
        if (lifetimeInterval > 3600000.0 && (lifetimeCount = (long)(3600000.0 / this.publishingInterval)) < 0xFFFFFFFFL && 3600000.0 % this.publishingInterval != 0.0) {
            ++lifetimeCount;
        }
        if (this.maxKeepAliveCount < 0x55555555L) {
            if (this.maxKeepAliveCount * 3L > lifetimeCount) {
                lifetimeCount = this.maxKeepAliveCount * 3L;
            }
            lifetimeInterval = (double)lifetimeCount * this.publishingInterval;
        } else {
            lifetimeCount = 0xFFFFFFFFL;
            lifetimeInterval = Double.MAX_VALUE;
        }
        if (10000.0 > this.publishingInterval && 10000.0 > lifetimeInterval && (lifetimeCount = (long)(10000.0 / this.publishingInterval)) < 0xFFFFFFFFL && 10000.0 % this.publishingInterval != 0.0) {
            ++lifetimeCount;
        }
        this.lifetimeCount = lifetimeCount;
    }

    private void setMaxNotificationsPerPublish(long maxNotificationsPerPublish) {
        if (maxNotificationsPerPublish <= 0L || maxNotificationsPerPublish > 65535L) {
            maxNotificationsPerPublish = 65535L;
        }
        this.maxNotificationsPerPublish = Ints.saturatedCast((long)maxNotificationsPerPublish);
    }

    private synchronized PublishQueue publishQueue() {
        return this.subscriptionManager.getPublishQueue();
    }

    private long currentSequenceNumber() {
        return this.sequenceNumber.get();
    }

    private long nextSequenceNumber() {
        return this.sequenceNumber.getAndIncrement();
    }

    void resetLifetimeCounter() {
        this.lifetimeCounter = this.lifetimeCount;
        this.logger.debug("[id={}] lifetime counter reset to {}", (Object)this.subscriptionId, (Object)this.lifetimeCounter);
    }

    private void resetKeepAliveCounter() {
        this.keepAliveCounter = this.maxKeepAliveCount;
        this.logger.debug("[id={}] keep-alive counter reset to {}", (Object)this.subscriptionId, (Object)this.maxKeepAliveCount);
    }

    private void returnKeepAlive(ServiceRequest<PublishRequest, PublishResponse> service) {
        ResponseHeader header = service.createResponseHeader();
        UInteger sequenceNumber = Unsigned.uint((long)this.currentSequenceNumber());
        NotificationMessage notificationMessage = new NotificationMessage(sequenceNumber, DateTime.now(), new ExtensionObject[0]);
        UInteger[] available = this.getAvailableSequenceNumbers();
        UInteger requestHandle = ((PublishRequest)service.getRequest()).getRequestHeader().getRequestHandle();
        StatusCode[] acknowledgeResults = this.subscriptionManager.getAcknowledgeResults(requestHandle);
        PublishResponse response = new PublishResponse(header, this.subscriptionId, available, Boolean.valueOf(this.moreNotifications), notificationMessage, acknowledgeResults, new DiagnosticInfo[0]);
        service.setResponse((UaResponseMessage)response);
        this.logger.debug("[id={}] returned keep-alive NotificationMessage sequenceNumber={}.", (Object)this.subscriptionId, (Object)sequenceNumber);
    }

    void returnStatusChangeNotification(ServiceRequest<PublishRequest, PublishResponse> service) {
        StatusChangeNotification statusChange = new StatusChangeNotification(new StatusCode(0x800A0000L), null);
        UInteger sequenceNumber = Unsigned.uint((long)this.nextSequenceNumber());
        NotificationMessage notificationMessage = new NotificationMessage(sequenceNumber, new DateTime(), new ExtensionObject[]{ExtensionObject.encode((UaStructure)statusChange)});
        ResponseHeader header = service.createResponseHeader();
        PublishResponse response = new PublishResponse(header, this.subscriptionId, new UInteger[0], Boolean.valueOf(false), notificationMessage, new StatusCode[0], new DiagnosticInfo[0]);
        service.setResponse((UaResponseMessage)response);
        this.logger.debug("[id={}] returned StatusChangeNotification sequenceNumber={}.", (Object)this.subscriptionId, (Object)sequenceNumber);
    }

    private void returnNotifications(ServiceRequest<PublishRequest, PublishResponse> service) {
        LinkedHashSet items = new LinkedHashSet();
        this.lastIterator.forEachRemaining(items::add);
        this.itemsById.values().stream().filter(item -> item.hasNotifications() || item.isTriggered()).forEach(items::add);
        PeekingIterator iterator = Iterators.peekingIterator(items.iterator());
        this.gatherAndSend(iterator, Optional.of(service));
        this.lastIterator = iterator.hasNext() ? iterator : Collections.emptyIterator();
    }

    private void gatherAndSend(PeekingIterator<BaseMonitoredItem<?>> iterator, Optional<ServiceRequest<PublishRequest, PublishResponse>> service) {
        if (service.isPresent()) {
            ArrayList notifications = Lists.newArrayList();
            while (notifications.size() < this.maxNotificationsPerPublish && iterator.hasNext()) {
                BaseMonitoredItem item = (BaseMonitoredItem)iterator.peek();
                boolean gatheredAllForItem = this.gather(item, notifications, this.maxNotificationsPerPublish);
                if (!gatheredAllForItem || !iterator.hasNext()) continue;
                iterator.next();
            }
            this.moreNotifications = iterator.hasNext();
            this.sendNotifications(service.get(), notifications);
            if (this.moreNotifications) {
                this.gatherAndSend(iterator, Optional.ofNullable(this.publishQueue().poll()));
            }
        } else if (this.moreNotifications) {
            this.publishQueue().addSubscription(this);
        }
    }

    private boolean gather(BaseMonitoredItem<?> item, List<UaStructure> notifications, int maxNotifications) {
        int max = maxNotifications - notifications.size();
        return item.getNotifications(notifications, max);
    }

    private void sendNotifications(ServiceRequest<PublishRequest, PublishResponse> service, List<UaStructure> notifications) {
        ArrayList dataNotifications = Lists.newArrayList();
        ArrayList eventNotifications = Lists.newArrayList();
        notifications.forEach(notification -> {
            if (notification instanceof MonitoredItemNotification) {
                dataNotifications.add((MonitoredItemNotification)notification);
            } else if (notification instanceof EventFieldList) {
                eventNotifications.add((EventFieldList)notification);
            }
        });
        ArrayList notificationData = Lists.newArrayList();
        if (dataNotifications.size() > 0) {
            DataChangeNotification dataChange = new DataChangeNotification(dataNotifications.toArray(new MonitoredItemNotification[dataNotifications.size()]), new DiagnosticInfo[0]);
            notificationData.add(ExtensionObject.encode((UaStructure)dataChange));
        }
        if (eventNotifications.size() > 0) {
            EventNotificationList eventChange = new EventNotificationList(eventNotifications.toArray(new EventFieldList[eventNotifications.size()]));
            notificationData.add(ExtensionObject.encode((UaStructure)eventChange));
        }
        UInteger sequenceNumber = Unsigned.uint((long)this.nextSequenceNumber());
        NotificationMessage notificationMessage = new NotificationMessage(sequenceNumber, new DateTime(), notificationData.toArray(new ExtensionObject[notificationData.size()]));
        this.availableMessages.put(notificationMessage.getSequenceNumber(), notificationMessage);
        UInteger[] available = this.getAvailableSequenceNumbers();
        UInteger requestHandle = ((PublishRequest)service.getRequest()).getRequestHeader().getRequestHandle();
        StatusCode[] acknowledgeResults = this.subscriptionManager.getAcknowledgeResults(requestHandle);
        ResponseHeader header = service.createResponseHeader();
        PublishResponse response = new PublishResponse(header, this.subscriptionId, available, Boolean.valueOf(this.moreNotifications), notificationMessage, acknowledgeResults, new DiagnosticInfo[0]);
        service.setResponse((UaResponseMessage)response);
        this.logger.debug("[id={}] returning {} DataChangeNotification(s) and {} EventNotificationList(s) sequenceNumber={} moreNotifications={}.", new Object[]{this.subscriptionId, dataNotifications.size(), eventNotifications.size(), sequenceNumber, this.moreNotifications});
    }

    private boolean notificationsAvailable() {
        return this.itemsById.values().stream().anyMatch(item -> item.hasNotifications() || item.isTriggered());
    }

    private void setState(State state) {
        State previousState = this.state.getAndSet(state);
        this.logger.debug("[id={}] {} -> {}", new Object[]{this.subscriptionId, previousState, state});
        StateListener listener = this.stateListener.get();
        if (listener != null) {
            listener.onStateChange(this, previousState, state);
        }
    }

    public UInteger getId() {
        return this.subscriptionId;
    }

    public double getPublishingInterval() {
        return this.publishingInterval;
    }

    public long getMaxKeepAliveCount() {
        return this.maxKeepAliveCount;
    }

    public long getLifetimeCount() {
        return this.lifetimeCount;
    }

    public int getMaxNotificationsPerPublish() {
        return this.maxNotificationsPerPublish;
    }

    public boolean isPublishingEnabled() {
        return this.publishingEnabled;
    }

    public int getPriority() {
        return this.priority;
    }

    public synchronized UInteger[] getAvailableSequenceNumbers() {
        Set<UInteger> uIntegers = this.availableMessages.keySet();
        Object[] available = uIntegers.toArray(new UInteger[uIntegers.size()]);
        Arrays.sort(available);
        return available;
    }

    public synchronized SubscriptionManager getSubscriptionManager() {
        return this.subscriptionManager;
    }

    public synchronized void setSubscriptionManager(SubscriptionManager subscriptionManager) {
        this.subscriptionManager = subscriptionManager;
    }

    public Session getSession() {
        return this.subscriptionManager.getSession();
    }

    public long nextItemId() {
        return this.itemIds.getAndIncrement();
    }

    public void setStateListener(StateListener listener) {
        this.stateListener.set(listener);
    }

    synchronized void onPublish(ServiceRequest<PublishRequest, PublishResponse> service) {
        State state = this.state.get();
        this.logger.trace("[id={}] onPublish(), state={}, keep-alive={}, lifetime={}", new Object[]{this.subscriptionId, state, this.keepAliveCounter, this.lifetimeCounter});
        if (state == State.Normal) {
            this.publishHandler.whenNormal((ServiceRequest<PublishRequest, PublishResponse>)service);
        } else if (state == State.KeepAlive) {
            this.publishHandler.whenKeepAlive((ServiceRequest<PublishRequest, PublishResponse>)service);
        } else if (state == State.Late) {
            this.publishHandler.whenLate((ServiceRequest<PublishRequest, PublishResponse>)service);
        } else if (state == State.Closing) {
            this.publishHandler.whenClosing((ServiceRequest<PublishRequest, PublishResponse>)service);
        } else if (state == State.Closed) {
            this.publishHandler.whenClosed((ServiceRequest<PublishRequest, PublishResponse>)service);
        } else {
            throw new RuntimeException("Unhandled subscription state: " + (Object)((Object)state));
        }
    }

    synchronized void onPublishingTimer() {
        State state = this.state.get();
        this.logger.trace("[id={}] onPublishingTimer(), state={}, keep-alive={}, lifetime={}", new Object[]{this.subscriptionId, state, this.keepAliveCounter, this.lifetimeCounter});
        long startNanos = System.nanoTime();
        if (state == State.Normal) {
            this.timerHandler.whenNormal();
        } else if (state == State.KeepAlive) {
            this.timerHandler.whenKeepAlive();
        } else if (state == State.Late) {
            this.timerHandler.whenLate();
        } else if (state == State.Closed) {
            this.logger.debug("[id={}] onPublish(), state={}", (Object)this.subscriptionId, (Object)state);
        } else {
            throw new RuntimeException("unhandled subscription state: " + (Object)((Object)state));
        }
        long elapsedNanos = System.nanoTime() - startNanos;
        long elapsedMillis = TimeUnit.MILLISECONDS.convert(elapsedNanos, TimeUnit.NANOSECONDS);
        long adjustedInterval = DoubleMath.roundToLong((double)(this.publishingInterval - (double)elapsedMillis), (RoundingMode)RoundingMode.UP);
        this.startPublishingTimer(adjustedInterval);
    }

    synchronized void startPublishingTimer() {
        long interval = DoubleMath.roundToLong((double)this.publishingInterval, (RoundingMode)RoundingMode.UP);
        this.startPublishingTimer(interval);
    }

    private synchronized void startPublishingTimer(long interval) {
        if (this.state.get() == State.Closed) {
            return;
        }
        --this.lifetimeCounter;
        if (this.lifetimeCounter < 1L) {
            this.logger.debug("[id={}] lifetime expired.", (Object)this.subscriptionId);
            this.setState(State.Closing);
        } else {
            this.subscriptionManager.getServer().getScheduledExecutorService().schedule(this::onPublishingTimer, interval, TimeUnit.MILLISECONDS);
        }
    }

    public synchronized StatusCode acknowledge(UInteger sequenceNumber) {
        if (this.availableMessages.remove(sequenceNumber) != null) {
            this.logger.debug("[id={}] sequence number acknowledged: {}", (Object)this.subscriptionId, (Object)sequenceNumber);
            return StatusCode.GOOD;
        }
        this.logger.debug("[id={}] sequence number unknown: {}", (Object)this.subscriptionId, (Object)sequenceNumber);
        return new StatusCode(2155479040L);
    }

    public synchronized NotificationMessage republish(UInteger sequenceNumber) {
        this.resetLifetimeCounter();
        return this.availableMessages.get(sequenceNumber);
    }

    public static interface StateListener {
        public void onStateChange(Subscription var1, State var2, State var3);
    }

    public static enum State {
        Closing,
        Closed,
        Normal,
        KeepAlive,
        Late;

    }

    private class TimerHandler {
        private TimerHandler() {
        }

        private void whenNormal() {
            boolean publishRequestQueued = Subscription.this.publishQueue().isNotEmpty();
            boolean publishingEnabled = Subscription.this.publishingEnabled;
            boolean notificationsAvailable = Subscription.this.notificationsAvailable();
            if (publishRequestQueued && publishingEnabled && notificationsAvailable) {
                Optional<ServiceRequest<PublishRequest, PublishResponse>> service = Optional.ofNullable(Subscription.this.publishQueue().poll());
                if (service.isPresent()) {
                    Subscription.this.resetLifetimeCounter();
                    Subscription.this.returnNotifications((ServiceRequest<PublishRequest, PublishResponse>)service.get());
                    Subscription.this.messageSent = true;
                } else {
                    this.whenNormal();
                }
            } else if (publishRequestQueued && !Subscription.this.messageSent && (!publishingEnabled || publishingEnabled && !notificationsAvailable)) {
                Optional<ServiceRequest<PublishRequest, PublishResponse>> service = Optional.ofNullable(Subscription.this.publishQueue().poll());
                if (service.isPresent()) {
                    Subscription.this.resetLifetimeCounter();
                    Subscription.this.returnKeepAlive((ServiceRequest<PublishRequest, PublishResponse>)service.get());
                    Subscription.this.messageSent = true;
                } else {
                    this.whenNormal();
                }
            } else if (!publishRequestQueued && (!Subscription.this.messageSent || publishingEnabled && notificationsAvailable)) {
                Subscription.this.setState(State.Late);
                Subscription.this.publishQueue().addSubscription(Subscription.this);
            } else if (Subscription.this.messageSent && (!publishingEnabled || publishingEnabled && !notificationsAvailable)) {
                Subscription.this.setState(State.KeepAlive);
                Subscription.this.resetKeepAliveCounter();
            } else {
                throw new IllegalStateException("unhandled subscription state");
            }
        }

        private void whenLate() {
        }

        private void whenKeepAlive() {
            boolean publishingEnabled = Subscription.this.publishingEnabled;
            boolean notificationsAvailable = Subscription.this.notificationsAvailable();
            boolean publishRequestQueued = Subscription.this.publishQueue().isNotEmpty();
            if (publishingEnabled && notificationsAvailable && publishRequestQueued) {
                Optional<ServiceRequest<PublishRequest, PublishResponse>> service = Optional.ofNullable(Subscription.this.publishQueue().poll());
                if (service.isPresent()) {
                    Subscription.this.setState(State.Normal);
                    Subscription.this.resetLifetimeCounter();
                    Subscription.this.returnNotifications((ServiceRequest<PublishRequest, PublishResponse>)service.get());
                    Subscription.this.messageSent = true;
                } else {
                    this.whenKeepAlive();
                }
            } else if (publishRequestQueued && Subscription.this.keepAliveCounter == 1L && (!publishingEnabled || publishingEnabled && !notificationsAvailable)) {
                Optional<ServiceRequest<PublishRequest, PublishResponse>> service = Optional.ofNullable(Subscription.this.publishQueue().poll());
                if (service.isPresent()) {
                    Subscription.this.returnKeepAlive((ServiceRequest<PublishRequest, PublishResponse>)service.get());
                    Subscription.this.resetLifetimeCounter();
                    Subscription.this.resetKeepAliveCounter();
                } else {
                    this.whenKeepAlive();
                }
            } else if (Subscription.this.keepAliveCounter > 1L && (!publishingEnabled || publishingEnabled && !notificationsAvailable)) {
                Subscription.this.keepAliveCounter--;
            } else if (!publishRequestQueued && (Subscription.this.keepAliveCounter == 1L || Subscription.this.keepAliveCounter > 1L && publishingEnabled && notificationsAvailable)) {
                Subscription.this.setState(State.Late);
                Subscription.this.publishQueue().addSubscription(Subscription.this);
            }
        }
    }

    private class PublishHandler {
        private PublishHandler() {
        }

        private void whenNormal(ServiceRequest<PublishRequest, PublishResponse> service) {
            boolean publishingEnabled = Subscription.this.publishingEnabled;
            if (!publishingEnabled || publishingEnabled && !Subscription.this.moreNotifications) {
                Subscription.this.publishQueue().addRequest(service);
            } else if (publishingEnabled && Subscription.this.moreNotifications) {
                Subscription.this.resetLifetimeCounter();
                Subscription.this.returnNotifications((ServiceRequest<PublishRequest, PublishResponse>)service);
                Subscription.this.messageSent = true;
            } else {
                throw new IllegalStateException("unhandled subscription state");
            }
        }

        private void whenLate(ServiceRequest<PublishRequest, PublishResponse> service) {
            boolean publishingEnabled = Subscription.this.publishingEnabled;
            boolean notificationsAvailable = Subscription.this.notificationsAvailable();
            if (publishingEnabled && (notificationsAvailable || Subscription.this.moreNotifications)) {
                Subscription.this.setState(State.Normal);
                Subscription.this.resetLifetimeCounter();
                Subscription.this.returnNotifications((ServiceRequest<PublishRequest, PublishResponse>)service);
                Subscription.this.messageSent = true;
            } else if (!publishingEnabled || publishingEnabled && !notificationsAvailable && !Subscription.this.moreNotifications) {
                Subscription.this.setState(State.KeepAlive);
                Subscription.this.resetLifetimeCounter();
                Subscription.this.returnKeepAlive((ServiceRequest<PublishRequest, PublishResponse>)service);
                Subscription.this.messageSent = true;
            } else {
                throw new IllegalStateException("unhandled subscription state");
            }
        }

        private void whenKeepAlive(ServiceRequest<PublishRequest, PublishResponse> service) {
            Subscription.this.publishQueue().addRequest(service);
        }

        private void whenClosing(ServiceRequest<PublishRequest, PublishResponse> service) {
            Subscription.this.returnStatusChangeNotification(service);
            Subscription.this.setState(State.Closed);
        }

        private void whenClosed(ServiceRequest<PublishRequest, PublishResponse> service) {
            Subscription.this.publishQueue().addRequest(service);
        }
    }
}

