/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.sdk.server;

import com.google.common.base.Objects;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.math.DoubleMath;
import com.google.common.primitives.Bytes;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.eclipse.milo.opcua.sdk.server.OpcUaServer;
import org.eclipse.milo.opcua.sdk.server.SecurityConfiguration;
import org.eclipse.milo.opcua.sdk.server.Session;
import org.eclipse.milo.opcua.sdk.server.SessionListener;
import org.eclipse.milo.opcua.sdk.server.diagnostics.ServerDiagnosticsSummary;
import org.eclipse.milo.opcua.sdk.server.identity.IdentityValidator;
import org.eclipse.milo.opcua.sdk.server.services.ServiceAttributes;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.UaRuntimeException;
import org.eclipse.milo.opcua.stack.core.security.SecurityAlgorithm;
import org.eclipse.milo.opcua.stack.core.security.SecurityPolicy;
import org.eclipse.milo.opcua.stack.core.serialization.UaResponseMessage;
import org.eclipse.milo.opcua.stack.core.types.builtin.ByteString;
import org.eclipse.milo.opcua.stack.core.types.builtin.DiagnosticInfo;
import org.eclipse.milo.opcua.stack.core.types.builtin.ExtensionObject;
import org.eclipse.milo.opcua.stack.core.types.builtin.NodeId;
import org.eclipse.milo.opcua.stack.core.types.builtin.StatusCode;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UInteger;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned;
import org.eclipse.milo.opcua.stack.core.types.enumerated.ApplicationType;
import org.eclipse.milo.opcua.stack.core.types.enumerated.MessageSecurityMode;
import org.eclipse.milo.opcua.stack.core.types.enumerated.UserTokenType;
import org.eclipse.milo.opcua.stack.core.types.structured.ActivateSessionRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.ActivateSessionResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.AnonymousIdentityToken;
import org.eclipse.milo.opcua.stack.core.types.structured.ApplicationDescription;
import org.eclipse.milo.opcua.stack.core.types.structured.CloseSessionRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.CloseSessionResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.CreateSessionRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.CreateSessionResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.EndpointDescription;
import org.eclipse.milo.opcua.stack.core.types.structured.SignatureData;
import org.eclipse.milo.opcua.stack.core.types.structured.SignedSoftwareCertificate;
import org.eclipse.milo.opcua.stack.core.types.structured.UserIdentityToken;
import org.eclipse.milo.opcua.stack.core.types.structured.UserTokenPolicy;
import org.eclipse.milo.opcua.stack.core.util.CertificateUtil;
import org.eclipse.milo.opcua.stack.core.util.ConversionUtil;
import org.eclipse.milo.opcua.stack.core.util.DigestUtil;
import org.eclipse.milo.opcua.stack.core.util.EndpointUtil;
import org.eclipse.milo.opcua.stack.core.util.NonceUtil;
import org.eclipse.milo.opcua.stack.core.util.SignatureUtil;
import org.eclipse.milo.opcua.stack.server.security.ServerCertificateValidator;
import org.eclipse.milo.opcua.stack.server.services.AttributeHistoryServiceSet;
import org.eclipse.milo.opcua.stack.server.services.AttributeServiceSet;
import org.eclipse.milo.opcua.stack.server.services.MethodServiceSet;
import org.eclipse.milo.opcua.stack.server.services.MonitoredItemServiceSet;
import org.eclipse.milo.opcua.stack.server.services.NodeManagementServiceSet;
import org.eclipse.milo.opcua.stack.server.services.QueryServiceSet;
import org.eclipse.milo.opcua.stack.server.services.ServiceRequest;
import org.eclipse.milo.opcua.stack.server.services.SessionServiceSet;
import org.eclipse.milo.opcua.stack.server.services.SubscriptionServiceSet;
import org.eclipse.milo.opcua.stack.server.services.ViewServiceSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager
implements AttributeServiceSet,
AttributeHistoryServiceSet,
MethodServiceSet,
MonitoredItemServiceSet,
NodeManagementServiceSet,
QueryServiceSet,
SessionServiceSet,
SubscriptionServiceSet,
ViewServiceSet {
    private static final int MAX_SESSION_TIMEOUT_MS = 120000;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Map<NodeId, Session> createdSessions = Maps.newConcurrentMap();
    private final Map<NodeId, Session> activeSessions = Maps.newConcurrentMap();
    private final List<SessionListener> sessionListeners = new CopyOnWriteArrayList<SessionListener>();
    private final List<ByteString> clientNonces = Lists.newCopyOnWriteArrayList();
    private final OpcUaServer server;

    SessionManager(OpcUaServer server) {
        this.server = server;
    }

    public void killSession(NodeId nodeId, boolean deleteSubscriptions) {
        this.activeSessions.values().stream().filter(s -> s.getSessionId().equals((Object)nodeId)).findFirst().ifPresent(s -> s.close(deleteSubscriptions));
    }

    public void addSessionListener(SessionListener listener) {
        this.sessionListeners.add(listener);
    }

    public void removeSessionListener(SessionListener listener) {
        this.sessionListeners.remove(listener);
    }

    public List<Session> getAllSessions() {
        ArrayList<Session> sessions = new ArrayList<Session>();
        sessions.addAll(this.createdSessions.values());
        sessions.addAll(this.activeSessions.values());
        return sessions;
    }

    public UInteger getCurrentSessionCount() {
        return Unsigned.uint((int)(this.createdSessions.size() + this.activeSessions.size()));
    }

    private Session session(ServiceRequest service) throws UaException {
        long secureChannelId = service.getSecureChannelId();
        NodeId authToken = service.getRequest().getRequestHeader().getAuthenticationToken();
        Session session = this.activeSessions.get(authToken);
        if (session == null) {
            session = this.createdSessions.get(authToken);
            if (session != null) {
                session.close(true);
                throw new UaException(2150039552L);
            }
            throw new UaException(2149908480L);
        }
        if (session.getSecureChannelId() != secureChannelId) {
            session.getSessionDiagnostics().getUnauthorizedRequestCount().increment();
            throw new UaException(0x80220000L);
        }
        session.updateLastActivity();
        service.attr(ServiceAttributes.SERVER_KEY).set((Object)this.server);
        service.attr(ServiceAttributes.SESSION_KEY).set((Object)session);
        return session;
    }

    public void onCreateSession(ServiceRequest serviceRequest) {
        ServerDiagnosticsSummary serverDiagnosticsSummary = this.server.getDiagnosticsSummary();
        try {
            CreateSessionResponse response = this.createSession(serviceRequest);
            serverDiagnosticsSummary.getCumulatedSessionCount().increment();
            serviceRequest.setResponse((UaResponseMessage)response);
        }
        catch (UaException e) {
            serverDiagnosticsSummary.getRejectedSessionCount().increment();
            if (e.getStatusCode().isSecurityError()) {
                serverDiagnosticsSummary.getSecurityRejectedSessionCount().increment();
            }
            serviceRequest.setServiceFault(e);
        }
    }

    private CreateSessionResponse createSession(ServiceRequest serviceRequest) throws UaException {
        CreateSessionRequest request = (CreateSessionRequest)serviceRequest.getRequest();
        long maxSessionCount = this.server.getConfig().getLimits().getMaxSessionCount().longValue();
        if ((long)(this.createdSessions.size() + this.activeSessions.size()) >= maxSessionCount) {
            throw new UaException(2153119744L);
        }
        ByteString serverNonce = NonceUtil.generateNonce((int)32);
        NodeId authenticationToken = new NodeId(0, NonceUtil.generateNonce((int)32));
        long maxRequestMessageSize = serviceRequest.getServer().getConfig().getMessageLimits().getMaxMessageSize();
        double revisedSessionTimeout = Math.max(5000.0, Math.min(120000.0, request.getRequestedSessionTimeout()));
        ApplicationDescription clientDescription = request.getClientDescription();
        long secureChannelId = serviceRequest.getSecureChannelId();
        EndpointDescription endpoint = serviceRequest.getEndpoint();
        SecurityPolicy securityPolicy = SecurityPolicy.fromUri((String)endpoint.getSecurityPolicyUri());
        EndpointDescription[] serverEndpoints = (EndpointDescription[])this.server.getEndpointDescriptions().stream().filter(ed -> !ed.getEndpointUrl().endsWith("/discovery")).filter(ed -> SessionManager.endpointMatchesUrl(ed, request.getEndpointUrl())).filter(ed -> Objects.equal((Object)endpoint.getTransportProfileUri(), (Object)ed.getTransportProfileUri())).map(SessionManager::stripNonEssentialFields).toArray(EndpointDescription[]::new);
        if (serverEndpoints.length == 0) {
            serverEndpoints = (EndpointDescription[])this.server.getEndpointDescriptions().stream().filter(ed -> !ed.getEndpointUrl().endsWith("/discovery")).filter(ed -> Objects.equal((Object)endpoint.getTransportProfileUri(), (Object)ed.getTransportProfileUri())).map(SessionManager::stripNonEssentialFields).toArray(EndpointDescription[]::new);
        }
        ByteString clientNonce = request.getClientNonce();
        if (securityPolicy != SecurityPolicy.None) {
            NonceUtil.validateNonce((ByteString)clientNonce);
            if (this.clientNonces.contains(clientNonce)) {
                throw new UaException(2149842944L);
            }
        }
        if (securityPolicy != SecurityPolicy.None && clientNonce.isNotNull()) {
            this.clientNonces.add(clientNonce);
            while (this.clientNonces.size() > 64) {
                this.clientNonces.remove(0);
            }
        }
        ByteString clientCertificateBytes = request.getClientCertificate();
        if (securityPolicy != SecurityPolicy.None && serviceRequest.getClientCertificateBytes() != null && !Objects.equal((Object)clientCertificateBytes, (Object)serviceRequest.getClientCertificateBytes())) {
            throw new UaException(2148728832L, "certificate used to open secure channel differs from certificate used to create session");
        }
        SecurityConfiguration securityConfiguration = this.createSecurityConfiguration(endpoint, clientCertificateBytes);
        if (securityPolicy != SecurityPolicy.None) {
            X509Certificate clientCertificate = securityConfiguration.getClientCertificate();
            List<X509Certificate> clientCertificateChain = securityConfiguration.getClientCertificateChain();
            if (clientCertificate == null || clientCertificateChain == null) {
                throw new UaException(2148728832L, "client certificate must be non-null");
            }
            ServerCertificateValidator certificateValidator = this.server.getConfig().getCertificateValidator();
            certificateValidator.validateCertificateChain(clientCertificateChain, clientDescription.getApplicationUri());
        }
        SignatureData serverSignature = this.getServerSignature(securityPolicy, securityConfiguration.getKeyPair(), clientNonce, securityConfiguration.getClientCertificateBytes());
        NodeId sessionId = new NodeId(1, "Session:" + UUID.randomUUID());
        String sessionName = request.getSessionName();
        Duration sessionTimeout = Duration.ofMillis(DoubleMath.roundToLong((double)revisedSessionTimeout, (RoundingMode)RoundingMode.UP));
        Session session = new Session(this.server, sessionId, sessionName, sessionTimeout, clientDescription, request.getServerUri(), request.getMaxResponseMessageSize(), endpoint, secureChannelId, securityConfiguration);
        session.setLastNonce(serverNonce);
        session.addLifecycleListener((s, remove) -> {
            this.createdSessions.remove(authenticationToken);
            this.activeSessions.remove(authenticationToken);
            this.sessionListeners.forEach(l -> l.onSessionClosed(s));
        });
        this.createdSessions.put(authenticationToken, session);
        this.sessionListeners.forEach(l -> l.onSessionCreated(session));
        return new CreateSessionResponse(serviceRequest.createResponseHeader(), sessionId, authenticationToken, Double.valueOf(revisedSessionTimeout), serverNonce, endpoint.getServerCertificate(), serverEndpoints, new SignedSoftwareCertificate[0], serverSignature, Unsigned.uint((long)maxRequestMessageSize));
    }

    private SecurityConfiguration createSecurityConfiguration(EndpointDescription endpoint, ByteString clientCertificateBytes) throws UaException {
        SecurityPolicy securityPolicy = SecurityPolicy.fromUri((String)endpoint.getSecurityPolicyUri());
        MessageSecurityMode securityMode = endpoint.getSecurityMode();
        X509Certificate clientCertificate = null;
        List clientCertificateChain = null;
        KeyPair keyPair = null;
        X509Certificate serverCertificate = null;
        List serverCertificateChain = null;
        if (securityPolicy != SecurityPolicy.None) {
            clientCertificate = CertificateUtil.decodeCertificate((byte[])clientCertificateBytes.bytes());
            clientCertificateChain = CertificateUtil.decodeCertificates((byte[])clientCertificateBytes.bytes());
            ByteString thumbprint = ByteString.of((byte[])DigestUtil.sha1((byte[])endpoint.getServerCertificate().bytesOrEmpty()));
            keyPair = (KeyPair)this.server.getConfig().getCertificateManager().getKeyPair(thumbprint).orElseThrow(() -> new UaException(0x80890000L));
            serverCertificate = (X509Certificate)this.server.getConfig().getCertificateManager().getCertificate(thumbprint).orElseThrow(() -> new UaException(0x80890000L));
            serverCertificateChain = this.server.getConfig().getCertificateManager().getCertificateChain(thumbprint).map(Lists::newArrayList).orElseThrow(() -> new UaException(0x80890000L));
        }
        return new SecurityConfiguration(securityPolicy, securityMode, keyPair, serverCertificate, serverCertificateChain, clientCertificate, clientCertificateChain);
    }

    private static boolean endpointMatchesUrl(EndpointDescription endpoint, String requestedEndpointUrl) {
        String endpointHost = EndpointUtil.getHost((String)Strings.nullToEmpty((String)endpoint.getEndpointUrl()));
        String requestedHost = EndpointUtil.getHost((String)Strings.nullToEmpty((String)requestedEndpointUrl));
        return Strings.nullToEmpty((String)endpointHost).equalsIgnoreCase(Strings.nullToEmpty((String)requestedHost));
    }

    private static EndpointDescription stripNonEssentialFields(EndpointDescription endpoint) {
        ApplicationDescription applicationDescription = endpoint.getServer();
        ApplicationDescription newApplicationDescription = new ApplicationDescription(applicationDescription.getApplicationUri(), null, null, ApplicationType.Server, null, null, null);
        return new EndpointDescription(endpoint.getEndpointUrl(), newApplicationDescription, ByteString.NULL_VALUE, endpoint.getSecurityMode(), endpoint.getSecurityPolicyUri(), endpoint.getUserIdentityTokens(), endpoint.getTransportProfileUri(), endpoint.getSecurityLevel());
    }

    public void onActivateSession(ServiceRequest serviceRequest) {
        try {
            ActivateSessionResponse response = this.activateSession(serviceRequest);
            serviceRequest.setResponse((UaResponseMessage)response);
        }
        catch (UaException e) {
            serviceRequest.setServiceFault(e);
        }
    }

    private ActivateSessionResponse activateSession(ServiceRequest serviceRequest) throws UaException {
        ActivateSessionRequest request = (ActivateSessionRequest)serviceRequest.getRequest();
        long secureChannelId = serviceRequest.getSecureChannelId();
        NodeId authToken = request.getRequestHeader().getAuthenticationToken();
        List clientSoftwareCertificates = ConversionUtil.l((Object[])request.getClientSoftwareCertificates());
        Session session = this.createdSessions.get(authToken);
        if (session == null) {
            session = this.activeSessions.get(authToken);
            if (session == null) {
                throw new UaException(2149908480L);
            }
            SessionManager.verifyClientSignature(session, request);
            SecurityConfiguration securityConfiguration = session.getSecurityConfiguration();
            if (session.getSecureChannelId() == secureChannelId) {
                UserIdentityToken identityToken = this.decodeIdentityToken(request.getUserIdentityToken(), session.getEndpoint().getUserIdentityTokens());
                Object identityObject = this.validateIdentityToken(session, identityToken, request.getUserTokenSignature());
                Object[] results = new StatusCode[clientSoftwareCertificates.size()];
                Arrays.fill(results, StatusCode.GOOD);
                ByteString serverNonce = NonceUtil.generateNonce((int)32);
                session.setClientAddress(serviceRequest.getClientAddress());
                session.setIdentityObject(identityObject, identityToken);
                session.setLastNonce(serverNonce);
                session.setLocaleIds(request.getLocaleIds());
                return new ActivateSessionResponse(serviceRequest.createResponseHeader(), serverNonce, (StatusCode[])results, new DiagnosticInfo[0]);
            }
            ByteString clientCertificateBytes = serviceRequest.getClientCertificateBytes();
            UserIdentityToken identityToken = this.decodeIdentityToken(request.getUserIdentityToken(), session.getEndpoint().getUserIdentityTokens());
            Object identityObject = this.validateIdentityToken(session, identityToken, request.getUserTokenSignature());
            boolean sameIdentity = Objects.equal((Object)identityObject, (Object)session.getIdentityObject());
            boolean sameCertificate = Objects.equal((Object)clientCertificateBytes, (Object)securityConfiguration.getClientCertificateBytes());
            if (sameIdentity && sameCertificate) {
                SecurityConfiguration newSecurityConfiguration = this.createSecurityConfiguration(serviceRequest.getEndpoint(), clientCertificateBytes);
                session.setEndpoint(serviceRequest.getEndpoint());
                session.setSecureChannelId(secureChannelId);
                session.setSecurityConfiguration(newSecurityConfiguration);
                this.logger.debug("Session id={} is now associated with secureChannelId={}", (Object)session.getSessionId(), (Object)secureChannelId);
                Object[] results = new StatusCode[clientSoftwareCertificates.size()];
                Arrays.fill(results, StatusCode.GOOD);
                ByteString serverNonce = NonceUtil.generateNonce((int)32);
                session.setClientAddress(serviceRequest.getClientAddress());
                session.setLastNonce(serverNonce);
                session.setLocaleIds(request.getLocaleIds());
                return new ActivateSessionResponse(serviceRequest.createResponseHeader(), serverNonce, (StatusCode[])results, new DiagnosticInfo[0]);
            }
            throw new UaException(2148728832L);
        }
        if (secureChannelId != session.getSecureChannelId()) {
            throw new UaException(2148728832L);
        }
        SessionManager.verifyClientSignature(session, request);
        UserIdentityToken identityToken = this.decodeIdentityToken(request.getUserIdentityToken(), session.getEndpoint().getUserIdentityTokens());
        Object identityObject = this.validateIdentityToken(session, identityToken, request.getUserTokenSignature());
        this.createdSessions.remove(authToken);
        this.activeSessions.put(authToken, session);
        Object[] results = new StatusCode[clientSoftwareCertificates.size()];
        Arrays.fill(results, StatusCode.GOOD);
        ByteString serverNonce = NonceUtil.generateNonce((int)32);
        session.setClientAddress(serviceRequest.getClientAddress());
        session.setIdentityObject(identityObject, identityToken);
        session.setLocaleIds(request.getLocaleIds());
        session.setLastNonce(serverNonce);
        return new ActivateSessionResponse(serviceRequest.createResponseHeader(), serverNonce, (StatusCode[])results, new DiagnosticInfo[0]);
    }

    private static void verifyClientSignature(Session session, ActivateSessionRequest request) throws UaException {
        SecurityConfiguration securityConfiguration = session.getSecurityConfiguration();
        if (securityConfiguration.getSecurityPolicy() != SecurityPolicy.None) {
            SignatureData clientSignature = request.getClientSignature();
            ByteString serverCertificateBs = securityConfiguration.getServerCertificateBytes();
            ByteString lastNonceBs = session.getLastNonce();
            try {
                byte[] dataBytes = Bytes.concat((byte[][])new byte[][]{serverCertificateBs.bytesOrEmpty(), lastNonceBs.bytesOrEmpty()});
                try {
                    SignatureUtil.verify((SecurityAlgorithm)SecurityAlgorithm.fromUri((String)clientSignature.getAlgorithm()), (X509Certificate)securityConfiguration.getClientCertificate(), (byte[])dataBytes, (byte[])clientSignature.getSignature().bytesOrEmpty());
                }
                catch (UaException e) {
                    throw new UaException(0x80580000L, (Throwable)e);
                }
            }
            catch (UaException e) {
                ByteString serverCertificateChainBs = securityConfiguration.getServerCertificateChainBytes();
                if (serverCertificateBs.equals((Object)serverCertificateChainBs)) {
                    throw e;
                }
                byte[] dataBytes = Bytes.concat((byte[][])new byte[][]{serverCertificateChainBs.bytesOrEmpty(), lastNonceBs.bytesOrEmpty()});
                try {
                    SignatureUtil.verify((SecurityAlgorithm)SecurityAlgorithm.fromUri((String)clientSignature.getAlgorithm()), (X509Certificate)securityConfiguration.getClientCertificate(), (byte[])dataBytes, (byte[])clientSignature.getSignature().bytesOrEmpty());
                }
                catch (UaException ex) {
                    throw new UaException(0x80580000L, (Throwable)e);
                }
            }
        }
    }

    @Nonnull
    private UserIdentityToken decodeIdentityToken(@Nullable ExtensionObject identityTokenXo, @Nullable UserTokenPolicy[] tokenPolicies) {
        Object identityToken;
        if (identityTokenXo != null && !identityTokenXo.isNull() && (identityToken = identityTokenXo.decodeOrNull(this.server.getSerializationContext())) instanceof UserIdentityToken) {
            return (UserIdentityToken)identityToken;
        }
        String policyId = ConversionUtil.l((Object[])tokenPolicies).stream().filter(p -> p.getTokenType() == UserTokenType.Anonymous).findFirst().map(UserTokenPolicy::getPolicyId).orElse(null);
        return new AnonymousIdentityToken(policyId);
    }

    private Object validateIdentityToken(Session session, Object tokenObject, SignatureData tokenSignature) throws UaException {
        IdentityValidator identityValidator = this.server.getConfig().getIdentityValidator();
        UserTokenPolicy tokenPolicy = this.validatePolicyId(session, tokenObject);
        if (tokenObject instanceof UserIdentityToken) {
            return identityValidator.validateIdentityToken(session, (UserIdentityToken)tokenObject, tokenPolicy, tokenSignature);
        }
        throw new UaException(0x80200000L);
    }

    private UserTokenPolicy validatePolicyId(Session session, Object tokenObject) throws UaException {
        if (tokenObject instanceof UserIdentityToken) {
            UserIdentityToken token = (UserIdentityToken)tokenObject;
            String policyId = token.getPolicyId();
            List userIdentityTokens = ConversionUtil.l((Object[])session.getEndpoint().getUserIdentityTokens());
            Optional<UserTokenPolicy> policy = userIdentityTokens.stream().filter(t -> Objects.equal((Object)policyId, (Object)t.getPolicyId())).findFirst();
            return policy.orElseThrow(() -> new UaException(0x80200000L, "policy not found: " + policyId));
        }
        throw new UaException(0x80200000L);
    }

    public void onCloseSession(ServiceRequest service) {
        try {
            CloseSessionResponse response = this.closeSession(service);
            service.setResponse((UaResponseMessage)response);
        }
        catch (UaException e) {
            service.setServiceFault(e);
        }
    }

    private CloseSessionResponse closeSession(ServiceRequest service) throws UaException {
        CloseSessionRequest request = (CloseSessionRequest)service.getRequest();
        long secureChannelId = service.getSecureChannelId();
        NodeId authToken = service.getRequest().getRequestHeader().getAuthenticationToken();
        Session session = this.activeSessions.get(authToken);
        if (session != null) {
            if (session.getSecureChannelId() != secureChannelId) {
                throw new UaException(0x80220000L);
            }
            this.activeSessions.remove(authToken);
            session.close(request.getDeleteSubscriptions());
            return new CloseSessionResponse(service.createResponseHeader());
        }
        session = this.createdSessions.get(authToken);
        if (session == null) {
            throw new UaException(2149908480L);
        }
        if (session.getSecureChannelId() != secureChannelId) {
            throw new UaException(0x80220000L);
        }
        this.createdSessions.remove(authToken);
        session.close(request.getDeleteSubscriptions());
        return new CloseSessionResponse(service.createResponseHeader());
    }

    public void onCancel(ServiceRequest service) throws UaException {
        this.session(service).onCancel(service);
    }

    private SignatureData getServerSignature(SecurityPolicy securityPolicy, KeyPair keyPair, ByteString clientNonce, ByteString clientCertificate) throws UaException {
        if (securityPolicy == SecurityPolicy.None) {
            return new SignatureData(null, null);
        }
        try {
            SecurityAlgorithm algorithm = securityPolicy.getAsymmetricSignatureAlgorithm();
            byte[] data = Bytes.concat((byte[][])new byte[][]{clientCertificate.bytes(), clientNonce.bytes()});
            byte[] signature = SignatureUtil.sign((SecurityAlgorithm)algorithm, (PrivateKey)keyPair.getPrivate(), (ByteBuffer[])new ByteBuffer[]{ByteBuffer.wrap(data)});
            return new SignatureData(algorithm.getUri(), ByteString.of((byte[])signature));
        }
        catch (UaRuntimeException e) {
            throw new UaException(2148728832L);
        }
    }

    public void onRead(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getReadCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getAttributeServiceSet().onRead(service);
    }

    public void onWrite(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getWriteCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getAttributeServiceSet().onWrite(service);
    }

    public void onHistoryRead(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getHistoryReadCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getAttributeHistoryServiceSet().onHistoryRead(service);
    }

    public void onHistoryUpdate(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getHistoryUpdateCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getAttributeHistoryServiceSet().onHistoryUpdate(service);
    }

    public void onBrowse(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getBrowseCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getViewServiceSet().onBrowse(service);
    }

    public void onBrowseNext(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getBrowseNextCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getViewServiceSet().onBrowseNext(service);
    }

    public void onTranslateBrowsePaths(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getTranslateBrowsePathsToNodeIdsCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getViewServiceSet().onTranslateBrowsePaths(service);
    }

    public void onRegisterNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getRegisterNodesCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getViewServiceSet().onRegisterNodes(service);
    }

    public void onUnregisterNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getUnregisterNodesCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getViewServiceSet().onUnregisterNodes(service);
    }

    public void onAddNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getAddNodesCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getNodeManagementServiceSet().onAddNodes(service);
    }

    public void onAddReferences(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getAddReferencesCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getNodeManagementServiceSet().onAddReferences(service);
    }

    public void onDeleteNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getDeleteNodesCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getNodeManagementServiceSet().onDeleteNodes(service);
    }

    public void onDeleteReferences(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getDeleteReferencesCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getNodeManagementServiceSet().onDeleteReferences(service);
    }

    public void onCreateSubscription(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getCreateSubscriptionCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onCreateSubscription(service);
    }

    public void onModifySubscription(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getModifySubscriptionCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onModifySubscription(service);
    }

    public void onSetPublishingMode(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getSetPublishingModeCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onSetPublishingMode(service);
    }

    public void onPublish(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getPublishCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onPublish(service);
    }

    public void onRepublish(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getRepublishCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onRepublish(service);
    }

    public void onTransferSubscriptions(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getTransferSubscriptionsCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onTransferSubscriptions(service);
    }

    public void onDeleteSubscriptions(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getDeleteSubscriptionsCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getSubscriptionServiceSet().onDeleteSubscriptions(service);
    }

    public void onCreateMonitoredItems(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getCreateMonitoredItemsCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getMonitoredItemServiceSet().onCreateMonitoredItems(service);
    }

    public void onModifyMonitoredItems(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getModifyMonitoredItemsCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getMonitoredItemServiceSet().onModifyMonitoredItems(service);
    }

    public void onSetMonitoringMode(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getSetMonitoringModeCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getMonitoredItemServiceSet().onSetMonitoringMode(service);
    }

    public void onSetTriggering(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getSetTriggeringCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getMonitoredItemServiceSet().onSetTriggering(service);
    }

    public void onDeleteMonitoredItems(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getDeleteMonitoredItemsCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getMonitoredItemServiceSet().onDeleteMonitoredItems(service);
    }

    public void onCall(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getCallCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getMethodServiceSet().onCall(service);
    }

    public void onQueryFirst(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getQueryFirstCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getQueryServiceSet().onQueryFirst(service);
    }

    public void onQueryNext(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSessionDiagnostics().getQueryNextCount().record(service);
        session.getSessionDiagnostics().getTotalRequestCount().record(service);
        session.getQueryServiceSet().onQueryNext(service);
    }
}

