/*
 * Copyright 2013 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codehaus.groovy.grails.web.binding

import groovy.transform.CompileStatic
import groovy.util.logging.Commons

import org.grails.databinding.events.DataBindingListenerAdapter
import org.springframework.beans.factory.InitializingBean
import org.springframework.beans.factory.annotation.Autowired

/**
 * An adapter which supports notifying BindEventListener instances
 * of binding events generated by the Grails 2.3 data binder.
 * 
 * Note that the BindEventListener interface defines a doBind method
 * which accepts some parameters which make sense in the Spring data
 * binder but do not make sense in the Grails 2.3 data binder.  This
 * adapter invokes the doBind method and passes null for the
 * MutablePropertyValues and TypeConverter arguments.  The only real
 * value passed to the doBind method is the object being bound to,
 * which is often all that is needed.  For more flexible binding event
 * notification applications should convert their BindEventListener
 * listeners to DataBindingListener listeners.
 * 
 * @author Jeff Brown
 * @since 2.3.2
 * @see org.grails.databinding.events.DataBindingListener
 * @see org.grails.databinding.events.DataBindingListenerAdapter
 * @see org.codehaus.groovy.grails.web.binding.BindEventListener
 *
 */
@CompileStatic
@Commons
class BindEventListenerAdapter extends DataBindingListenerAdapter implements InitializingBean {

    @Autowired(required=false)
    List<BindEventListener> listeners
    
    @Override
    Boolean beforeBinding(Object obj, Object errors) {
        if(listeners) {
            for(BindEventListener listener : listeners) {
                try {
                    listener.doBind obj, null, null
                } catch (Exception e) {
                    log.error "An error occurred notifying the ${listener.getClass().getName()} listener.", e
                }
            }
        }
        true
    }
    
    @Override
    public void afterPropertiesSet() throws Exception {
        if(listeners) {
            log.debug "${listeners.size()} BindEventListener beans are configured in the BindEventListenerAdapter bean."
        } else {
            log.warn 'No BindEventListener beans are configured in the application context.  There is no need to have grails.databinding.enableSpringEventAdapter set to true in Config.groovy.'
        }
    }
}
