/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.api;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * A collection of ManagedComponent and structural information
 * about a deployment.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 90104 $
 */
public interface ManagedDeployment
{
   /** The phase of deployment
    * @deprecated - no replacement, to be dropped
    */
   @Deprecated()
   public enum DeploymentPhase {
      /** A deployment loaded during the server bootstrap phase */
      BOOTSTRAP,
      /** An mc/service deployment for a Deployer to be loaded after the BOOTSTRAP phase */
      DEPLOYER,
      /** Any deployment content to be loaded after the DEPLOYER phase */
      APPLICATION,
      /** Any unversioned deployment content to be loaded after the APPLICATION phase */
      APPLICATION_TRANSIENT
   };

   /**
    * Get the full name of the associated DeploymentUnit
    * (e.g. vfsfile:/C:/opt/jboss-eap-5.0.0-SNAPSHOT/server/default/deploy/foo.ear/).
    * @return the full name of the DeploymentUnit
    *         (e.g. vfsfile:/C:/opt/jboss-eap-5.0.0-SNAPSHOT/server/default/deploy/foo.ear/)
    */
   public String getName();

   /**
    * Get the simple name (e.g. foo.ear) for the deployment
    * @return the simple name (e.g. foo.ear) of the deployment
    */
   public String getSimpleName();

   /**
    * Get the phase this deployment is associated with
    * @return the phase
    */
   public DeploymentPhase getDeploymentPhase();

   /**
    * Get the deployment state of the deployment. 
    * @return the deployment state
    */
   public DeploymentState getDeploymentState();

   /**
    * Add a deployment/module type
    * @param type
    * @return true if the type was added, false if it already exists
    */
   public boolean addType(String type);

   /**
    * Get the deployment/module types.
    * @return deployment types
    */
   public Set<String> getTypes();

   /**
    * Get the deployment/module types.
    * @param types
    */
   public void setTypes(Set<String> types);

   /**
    * Get the managed property names
    * 
    * @return the property names
    */
   public Set<String> getPropertyNames();

   /**
    * Get a property
    * 
    * @param name the name
    * @return the property
    */
   public ManagedProperty getProperty(String name);
   
   /**
    * Get the properties
    * 
    * @return the properties
    */
   public Map<String, ManagedProperty> getProperties();

   /**
    * Get the names of the child managed objects.
    *
    * <b>NOTE:</b> Clients should not use this method - it is intended for server-side use only.
    *
    * @return the names of the child managed objects
    */
   public Set<String> getManagedObjectNames();

   /**
    * Gets the child managed objects.
    *
    * <b>NOTE:</b> Clients should not use this method - it is intended for server-side use only.
    *
    * @return the child managed objects
    */
   public Map<String, ManagedObject> getManagedObjects();

   /**
    * Gets the child managed object with the specified name. Returns null if there is no child managed object with that
    * name.
    *
    * <b>NOTE:</b> Clients should not use this method - it is intended for server-side use only.
    *
    * @param name the managed object name
    * @return the child managed object with the specified name, or null if there is no child managed object with that
    *         name
    */
   public ManagedObject getManagedObject(String name);

   /**
    * 
    * @return the parent
    */
   public ManagedDeployment getParent();

   /**
    * Gets the nested ManagedComponent with the specified name.
    * @param name the child component's name
    * @return the child ManagedComponent with the specified name
    */
   public ManagedComponent getComponent(String name);

   /**
    * Get the nested ManagedComponents for the deployment module.
    * @return the nested ManagedComponents for the deployment module.
    */
   public Map<String, ManagedComponent> getComponents();

   /**
    * Get the nested deployment modules.
    * @return nested deployment modules.
    */
   public List<ManagedDeployment> getChildren();

   /**
    * Get the DeploymentTemplate names for components
    * that can be added to this deployment.
    * @return  the template names
    */
   public Set<String> getComponentTemplateNames();

   /**
    * Add a component to this deployment
    * @param name the name
    * @param comp the component
    */
   public void addComponent(String name, ManagedComponent comp);

   /**
    * 
    * @param name the name
    * @return true when it was removed
    */
   public boolean removeComponent(String name);

   /**
    * Get the DeploymentTemplate names for deployments
    * that can be added to this deployment.
    * @return the template names
    */
   public Set<String> getDeploymentTemplateNames();

   /**
    * Add a deployment
    * @param deploymentBaseName
    * @param info
    * @return the deployment
    */
   public ManagedDeployment addModule(String deploymentBaseName, DeploymentTemplateInfo info);

   /**
    * Get an attachment from the deployment.
    *
    * <b>NOTE:</b> Clients should not use this method - it is intended for server-side use only.
    * 
    * @see #setAttachment(String, Object)
    * 
    * @param name the name
    * @return the attachment
    */
   Object getAttachment(String name);
   
   /**
    * Get an attachment from the deployment,
    * uses the expected type as both the name
    * and to cast the resulting object.
    *
    * <b>NOTE:</b> Clients should not use this method - it is intended for server-side use only.
    * 
    * @param <T> the expected type
    * @param expectedType the expected type
    * @return the attachment
    * @throws ClassCastException when the object is not of the expected type
    */
   <T> T getAttachment(Class<T> expectedType);

   /**
    * Set an attachment against the deployment. Attachments added to a deployment
    * are serialized to external clients like admin tools, so the attachment
    * type should be restricted to jdk types and commonly provided management
    * related classes.
    *
    * <b>NOTE:</b> Clients should not use this method - it is intended for server-side use only.
    * 
    * @param name the name
    * @param attachment the attachment, pass null to remove an attachment
    * @throws IllegalArgumentException for a null name
    */
   void setAttachment(String name, Object attachment);
}
