/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.clientpolicy.executor;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collections;
import java.util.stream.Stream;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.keycloak.services.clientpolicy.ClientPolicyException;
import org.keycloak.services.clientpolicy.executor.SecureRedirectUrisEnforcerExecutor;
import org.keycloak.util.JsonSerialization;

public class SecureRedirectUrisEnforcerExecutorTest {
    private static SecureRedirectUrisEnforcerExecutor executor;
    private ObjectNode configuration;

    @BeforeClass
    public static void setupAll() {
        executor = new SecureRedirectUrisEnforcerExecutor(null);
    }

    @Before
    public void setup() {
        this.configuration = JsonSerialization.createObjectNode();
    }

    @Test
    public void defaultConfiguration() {
        this.setupConfiguration((JsonNode)this.configuration);
        SecureRedirectUrisEnforcerExecutor.Configuration configuration = executor.getConfiguration();
        Assert.assertFalse((boolean)configuration.isAllowIPv4LoopbackAddress());
        Assert.assertFalse((boolean)configuration.isAllowIPv6LoopbackAddress());
        Assert.assertFalse((boolean)configuration.isAllowPrivateUseUriScheme());
        Assert.assertFalse((boolean)configuration.isAllowHttpScheme());
        Assert.assertFalse((boolean)configuration.isAllowWildcardContextPath());
        Assert.assertFalse((boolean)configuration.isOAuth2_1Compliant());
        Assert.assertFalse((boolean)configuration.isAllowOpenRedirect());
        Assert.assertTrue((boolean)configuration.getAllowPermittedDomains().isEmpty());
    }

    @Test
    public void failUriSyntax() {
        this.checkFail("https://keycloak.org\n", false, "Invalid Redirect Uri: invalid uri");
        this.checkFail("Collins'&1=1;--", false, "Invalid Redirect Uri: invalid uri");
    }

    @Test
    public void failValidatePrivateUseUriScheme() {
        this.checkFail("myapp:/oauth.redirect", false, "Invalid Redirect Uri: invalid private use scheme");
        this.enable("allow-private-use-uri-scheme");
        this.checkFail("myapp.example.com:/*", false, "Invalid Redirect Uri: invalid private use scheme");
        this.enable("allow-wildcard-context-path");
        this.checkFail("myapp.example.com:/*/abc/*/efg", false, "Invalid Redirect Uri: invalid private use scheme");
        this.enable("oauth-2-1-compliant");
        Stream.of("myapp:/oauth.redirect#pinpoint", "myapp.example.com:/oauth.redirect/*", "myapp:/oauth.redirect").forEach(i -> this.checkFail((String)i, false, "Invalid Redirect Uri: invalid private use scheme"));
    }

    @Test
    public void successValidatePrivateUseUriScheme() {
        this.enable("allow-private-use-uri-scheme");
        Stream.of("com.example.app:/oauth2redirect/example-provider", "com.example.app:51004/oauth2redirect/example-provider").forEach(i -> this.checkSuccess((String)i, false));
        this.enable("allow-wildcard-context-path");
        this.checkSuccess("myapp.example.com:/*", false);
        this.enable("oauth-2-1-compliant");
        this.checkSuccess("com.example.app:/oauth2redirect/example-provider", false);
    }

    @Test
    public void failValidateIPv4LoopbackAddress() {
        this.checkFail("https://127.0.0.1/auth/admin", false, "Invalid Redirect Uri: invalid loopback address");
        this.enable("allow-ipv4-loopback-address");
        Stream.of("http://127.0.0.1:8080/auth/admin", "https://127.0.0.1:8080/*").forEach(i -> this.checkFail((String)i, false, "Invalid Redirect Uri: invalid loopback address"));
        this.enable("allow-wildcard-context-path");
        this.checkFail("https://127.0.0.1:8080/*/efg/*/abc", false, "Invalid Redirect Uri: invalid loopback address");
        this.enable("allow-http-scheme");
        this.enable("oauth-2-1-compliant");
        Stream.of("http://127.0.0.1:8080/auth/admin", "http://127.0.0.1/*", "http://127.0.0.1/auth/admin#fragment").forEach(i -> this.checkFail((String)i, false, "Invalid Redirect Uri: invalid loopback address"));
        Stream.of("http://127.0.0.1:123456/oauth2redirect/example-provider", "http://127.0.0.1/oauth2redirect/example-provider").forEach(i -> this.checkFail((String)i, true, "Invalid Redirect Uri: invalid loopback address"));
    }

    @Test
    public void successValiIPv4LoopbackAddress() {
        this.enable("allow-ipv4-loopback-address");
        Stream.of("https://127.0.0.1:8443", "https://localhost/auth/admin").forEach(i -> this.checkSuccess((String)i, false));
        this.enable("allow-wildcard-context-path");
        this.checkSuccess("https://localhost/*", false);
        this.enable("allow-http-scheme");
        this.checkSuccess("http://127.0.0.1:8080/oauth2redirect", false);
        this.enable("oauth-2-1-compliant");
        this.checkSuccess("http://127.0.0.1/oauth2redirect/example-provider", false);
        this.checkSuccess("http://127.0.0.1:43321/oauth2redirect/example-provider", true);
    }

    @Test
    public void failValidateIPv6LoopbackAddress() {
        this.checkFail("https://[::1]:9999/oauth2redirect/example-provider", false, "Invalid Redirect Uri: invalid loopback address");
        this.enable("allow-ipv6-loopback-address");
        this.checkFail("http://[::1]:9999/oauth2redirect/example-provider", false, "Invalid Redirect Uri: invalid loopback address");
        this.checkFail("https://[::1]:9999/*", false, "Invalid Redirect Uri: invalid loopback address");
        this.enable("allow-wildcard-context-path");
        this.checkFail("https://[::1]/*/efg/*/abc", false, "Invalid Redirect Uri: invalid loopback address");
        this.enable("allow-http-scheme");
        this.enable("oauth-2-1-compliant");
        Stream.of("http://[::1]:8080/auth/admin", "http://[::1]/*", "http://[::1]/auth/admin#fragment", "https://[0:0:0:0:0:0:0:1]:8080").forEach(i -> this.checkFail((String)i, false, "Invalid Redirect Uri: invalid loopback address"));
        Stream.of("http://[::1]:123456/oauth2redirect/example-provider", "http://[::1]/oauth2redirect/example-provider").forEach(i -> this.checkFail((String)i, true, "Invalid Redirect Uri: invalid loopback address"));
    }

    @Test
    public void successValiIPv6LoopbackAddress() {
        this.enable("allow-ipv6-loopback-address");
        for (String i : Arrays.asList("https://[::1]/oauth2redirect/example-provider", "https://[::1]:8443")) {
            this.checkSuccess(i, false);
        }
        this.enable("allow-wildcard-context-path");
        this.checkSuccess("https://[::1]/*", false);
        this.enable("allow-http-scheme");
        this.checkSuccess("http://[::1]:8080/oauth2redirect", false);
        this.enable("oauth-2-1-compliant");
        this.checkSuccess("http://[::1]/oauth2redirect/example-provider", false);
        this.checkSuccess("http://[::1]:43321/oauth2redirect/example-provider", true);
    }

    @Test
    public void failNormalUri() {
        this.checkFail("http://192.168.1.211:9088/oauth2redirect/example-provider/test", false, "Invalid Redirect Uri: invalid uri");
        this.enable("allow-wildcard-context-path");
        this.checkFail("https://192.168.1.211/*/efg/*/abc", false, "Invalid Redirect Uri: invalid uri");
        this.enable("allow-http-scheme");
        this.permittedDomains("oauth.redirect", "((dev|test)-)*example.org");
        this.checkFail("http://192.168.1.211/oauth2redirect/example-provider/test", false, "Invalid Redirect Uri: invalid uri");
        this.enable("oauth-2-1-compliant");
        Stream.of("http://dev-example.com/auth/callback", "https://test-example.com:8443/*", "https://oauth.redirect:8443/auth/callback#fragment").forEach(i -> this.checkFail((String)i, true, "Invalid Redirect Uri: invalid uri"));
    }

    @Test
    public void successNormalUri() {
        Stream.of("https://example.org/realms/master", "https://192.168.1.211:9088/oauth2redirect/example-provider/test", "https://192.168.1.211/").forEach(i -> this.checkSuccess((String)i, false));
        this.enable("allow-wildcard-context-path");
        this.checkSuccess("https://example.org/*", false);
        this.enable("allow-http-scheme");
        this.permittedDomains("oauth.redirect", "((dev|test)-)*example.org");
        this.checkSuccess("http://test-example.org:8080/*", false);
        this.enable("oauth-2-1-compliant");
        this.checkSuccess("https://dev-example.org:8080/redirect", false);
    }

    @Test
    public void successDefaultConfiguration() {
        Stream.of("https://example.org/realms/master", "https://192.168.1.211:9088/oauth2redirect/example-provider/test", "https://192.168.1.211/").forEach(i -> this.checkSuccess((String)i, false));
    }

    @Test
    public void successAllConfigurationEnabled() {
        this.enable("allow-wildcard-context-path", "allow-ipv4-loopback-address", "allow-ipv6-loopback-address", "allow-http-scheme", "allow-private-use-uri-scheme");
        this.permittedDomains("oauth.redirect", "((dev|test)-)*example.org");
        Stream.of("http://127.0.0.1/*/realms/master", "http://[::1]/*/realms/master", "myapp.example.com://oauth.redirect", "https://test-example.org/*/auth/admin").forEach(i -> this.checkSuccess((String)i, false));
    }

    @Test
    public void successAllConfigurationDisabled() {
        this.disable("allow-wildcard-context-path", "allow-ipv4-loopback-address", "allow-ipv6-loopback-address", "allow-http-scheme", "allow-private-use-uri-scheme", "allow-open-redirect");
        Stream.of("https://keycloak.org/sso/silent-callback.html", "https://example.org/auth/realms/master/broker/oidc/endpoint", "https://192.168.8.1:12345/auth/realms/master/broker/oidc/endpoint").forEach(i -> this.checkSuccess((String)i, false));
    }

    private void permittedDomains(String ... domains) {
        ArrayNode arrayNode = JsonSerialization.mapper.createArrayNode();
        Arrays.stream(domains).forEach(arg_0 -> ((ArrayNode)arrayNode).add(arg_0));
        this.configuration.set("allow-permitted-domains", (JsonNode)arrayNode);
    }

    private void disable(String ... config) {
        Arrays.stream(config).forEach(it -> this.configuration.set(it, (JsonNode)BooleanNode.getFalse()));
    }

    private void enable(String ... config) {
        Arrays.stream(config).forEach(it -> this.configuration.set(it, (JsonNode)BooleanNode.getTrue()));
    }

    private void setupConfiguration(JsonNode node) {
        SecureRedirectUrisEnforcerExecutor.Configuration configuration = (SecureRedirectUrisEnforcerExecutor.Configuration)JsonSerialization.mapper.convertValue((Object)node, executor.getExecutorConfigurationClass());
        executor.setupConfiguration(configuration);
    }

    private void checkFail(String redirectUri, boolean isRedirectUriParameter, String errorDetail) {
        try {
            this.doValidate(redirectUri, isRedirectUriParameter);
            Assert.fail();
        }
        catch (ClientPolicyException cpe) {
            Assert.assertEquals((Object)"invalid_request", (Object)cpe.getMessage());
            Assert.assertEquals((Object)errorDetail, (Object)cpe.getErrorDetail());
        }
    }

    private void checkSuccess(String redirectUri, boolean isRedirectUriParameter) {
        try {
            this.doValidate(redirectUri, isRedirectUriParameter);
        }
        catch (ClientPolicyException e) {
            Assert.assertNull((String)e.getErrorDetail(), (Object)((Object)e));
        }
    }

    private void doValidate(String redirectUri, boolean isRedirectUriParameter) throws ClientPolicyException {
        this.setupConfiguration((JsonNode)this.configuration);
        executor.verifyRedirectUri(redirectUri, isRedirectUriParameter);
    }

    @Test
    public void matchDomains() throws URISyntaxException {
        String[] expectNoneMatches;
        String[] expectMatches;
        SecureRedirectUrisEnforcerExecutor.Configuration config = new SecureRedirectUrisEnforcerExecutor.Configuration();
        SecureRedirectUrisEnforcerExecutor.UriValidation validation = new SecureRedirectUrisEnforcerExecutor.UriValidation("http://localhost:8080/auth/realms/master/account", false, config);
        boolean matches = validation.matchDomains(Collections.emptyList());
        Assert.assertFalse((boolean)matches);
        matches = validation.matchDomains(Collections.singletonList("local-\\w+"));
        Assert.assertFalse((boolean)matches);
        matches = validation.matchDomains(Collections.singletonList("localhost"));
        Assert.assertTrue((boolean)matches);
        matches = validation.matchDomains(Collections.singletonList("local\\w+"));
        Assert.assertTrue((boolean)matches);
        matches = validation.matchDomains(Arrays.asList("local-\\w+", "localhost2"));
        Assert.assertFalse((boolean)matches);
        matches = validation.matchDomains(Arrays.asList("local\\w+", "localhost"));
        Assert.assertTrue((boolean)matches);
        String givenPattern = "((dev|test)-)*example.org";
        for (String match : expectMatches = new String[]{"https://dev-example.org", "https://test-example.org", "https://example.org"}) {
            validation = new SecureRedirectUrisEnforcerExecutor.UriValidation(match, false, config);
            Assert.assertTrue((String)match, (boolean)validation.matchDomain(givenPattern));
        }
        for (String match : expectNoneMatches = new String[]{"https://prod-example.org", "https://testexample.org"}) {
            validation = new SecureRedirectUrisEnforcerExecutor.UriValidation(match, false, config);
            Assert.assertFalse((String)match, (boolean)validation.matchDomain(givenPattern));
        }
    }
}

