/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.referencing;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Set;
import org.geotools.factory.Hints;
import org.geotools.io.TableWriter;
import org.geotools.referencing.CRS;
import org.geotools.referencing.ReferencingFactoryFinder;
import org.geotools.referencing.datum.BursaWolfParameters;
import org.geotools.referencing.datum.DefaultGeodeticDatum;
import org.geotools.referencing.factory.AbstractAuthorityFactory;
import org.geotools.referencing.factory.FactoryDependencies;
import org.geotools.referencing.wkt.Parser;
import org.geotools.resources.Arguments;
import org.geotools.resources.CRSUtilities;
import org.geotools.resources.i18n.Vocabulary;
import org.opengis.metadata.Identifier;
import org.opengis.metadata.citation.Citation;
import org.opengis.referencing.AuthorityFactory;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.IdentifiedObject;
import org.opengis.referencing.NoSuchAuthorityCodeException;
import org.opengis.referencing.crs.CRSAuthorityFactory;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.operation.CoordinateOperation;
import org.opengis.referencing.operation.CoordinateOperationAuthorityFactory;
import org.opengis.referencing.operation.CoordinateOperationFactory;
import org.opengis.referencing.operation.OperationNotFoundException;
import org.opengis.util.InternationalString;

final class Command {
    private static final Hints HINTS = null;
    private final AuthorityFactory factory;
    private final Parser formatter;

    private Command(String authority) {
        this.factory = authority == null ? CRS.getAuthorityFactory(false) : ReferencingFactoryFinder.getCRSAuthorityFactory(authority, HINTS);
        this.formatter = new Parser();
    }

    private static char[] getSeparator() {
        char[] separator = new char[79];
        Arrays.fill(separator, '\u2500');
        return separator;
    }

    private static void help(PrintWriter out) {
        out.println("Display informations about CRS identified by authority codes.");
        out.println("Usage: java org.geotools.referencing.CRS [options] [codes]");
        out.println("Options:");
        out.println(" -authority=ARG : Uses the specified authority factory (default to all).");
        out.println(" -bursawolfs    : Lists Bursa-Wolf parameters for the specified CRS.");
        out.println(" -codes         : Lists all available CRS codes with their description.");
        out.println(" -colors        : Enable syntax coloring on ANSI X3.64 compatible terminal.");
        out.println(" -dependencies  : Lists authority factory dependencies as a tree.");
        out.println(" -factories     : Lists all availables CRS authority factories.");
        out.println(" -forcexy       : Force \"longitude first\" axis order.");
        out.println(" -help          : Prints this message.");
        out.println(" -locale=ARG    : Formats texts in the specified locale.");
        out.println(" -operations    : Prints all available coordinate operations between a pair of CRS.");
        out.println(" -transform     : Prints the preferred math transform between a pair of CRS.");
    }

    private void list(PrintWriter out, String[] args) throws FactoryException {
        char[] separator = null;
        for (int i = 0; i < args.length; ++i) {
            if (separator == null) {
                separator = Command.getSeparator();
            } else {
                out.println(separator);
            }
            out.println(this.formatter.format(this.factory.createObject(args[i])));
            String warning = this.formatter.getWarning();
            if (warning == null) continue;
            out.println();
            out.print(Vocabulary.format(242));
            out.print(": ");
            out.println(warning);
        }
    }

    private void codes(PrintWriter out) throws FactoryException {
        Set<String> codes = this.factory.getAuthorityCodes(CoordinateReferenceSystem.class);
        TableWriter table = new TableWriter(out);
        table.writeHorizontalSeparator();
        table.write(Vocabulary.format(21));
        table.nextColumn();
        table.write(Vocabulary.format(46));
        table.writeHorizontalSeparator();
        for (String code : codes) {
            table.write(code);
            table.nextColumn();
            try {
                InternationalString description = this.factory.getDescriptionText(code);
                if (description != null) {
                    table.write(((Object)description).toString());
                }
            }
            catch (NoSuchAuthorityCodeException e) {
                table.write(e.getLocalizedMessage());
            }
            table.nextLine();
        }
        table.writeHorizontalSeparator();
        try {
            table.flush();
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
    }

    private static void factories(PrintWriter out) {
        HashSet<Citation> done = new HashSet<Citation>();
        TableWriter table = new TableWriter((Writer)out, " \u2502 ");
        TableWriter notes = new TableWriter((Writer)out, " ");
        int noteCount = 0;
        notes.setMultiLinesCells(true);
        table.setMultiLinesCells(true);
        table.writeHorizontalSeparator();
        table.write(Vocabulary.format(6));
        table.nextColumn();
        table.write(Vocabulary.format(46));
        table.nextColumn();
        table.write(Vocabulary.format(151));
        table.writeHorizontalSeparator();
        for (CRSAuthorityFactory factory : ReferencingFactoryFinder.getCRSAuthorityFactories(HINTS)) {
            Citation authority = factory.getAuthority();
            Iterator<? extends Identifier> identifiers = authority.getIdentifiers().iterator();
            if (!identifiers.hasNext() || !done.add(authority)) continue;
            table.write(identifiers.next().getCode());
            table.nextColumn();
            table.write(((Object)authority.getTitle()).toString().trim());
            if (factory instanceof AbstractAuthorityFactory) {
                String description;
                try {
                    description = ((AbstractAuthorityFactory)((Object)factory)).getBackingStoreDescription();
                }
                catch (FactoryException e) {
                    description = e.getLocalizedMessage();
                }
                if (description != null) {
                    String n = String.valueOf(++noteCount);
                    table.nextColumn();
                    table.write(40);
                    table.write(n);
                    table.write(41);
                    notes.write(40);
                    notes.write(n);
                    notes.write(41);
                    notes.nextColumn();
                    notes.write(description.trim());
                    notes.nextLine();
                }
            }
            table.nextLine();
        }
        table.writeHorizontalSeparator();
        try {
            table.flush();
            notes.flush();
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
    }

    private void bursaWolfs(PrintWriter out, String[] args) throws FactoryException {
        int i;
        NumberFormat nf = NumberFormat.getNumberInstance();
        nf.setMinimumFractionDigits(3);
        nf.setMaximumFractionDigits(3);
        TableWriter table = new TableWriter(out);
        table.writeHorizontalSeparator();
        String[] titles = new String[]{Vocabulary.format(210), "dx", "dy", "dz", "ex", "ey", "ez", "ppm"};
        for (i = 0; i < titles.length; ++i) {
            table.write(titles[i]);
            table.nextColumn();
            table.setAlignment(1);
        }
        table.writeHorizontalSeparator();
        for (i = 0; i < args.length; ++i) {
            IdentifiedObject object = this.factory.createObject(args[i]);
            if (object instanceof CoordinateReferenceSystem) {
                object = CRSUtilities.getDatum((CoordinateReferenceSystem)object);
            }
            if (!(object instanceof DefaultGeodeticDatum)) continue;
            BursaWolfParameters[] params = ((DefaultGeodeticDatum)object).getBursaWolfParameters();
            for (int j = 0; j < params.length; ++j) {
                BursaWolfParameters p = params[j];
                table.setAlignment(0);
                table.write(p.targetDatum.getName().getCode());
                table.nextColumn();
                table.setAlignment(2);
                for (int k = 0; k < 7; ++k) {
                    double v;
                    switch (k) {
                        case 0: {
                            v = p.dx;
                            break;
                        }
                        case 1: {
                            v = p.dy;
                            break;
                        }
                        case 2: {
                            v = p.dz;
                            break;
                        }
                        case 3: {
                            v = p.ex;
                            break;
                        }
                        case 4: {
                            v = p.ey;
                            break;
                        }
                        case 5: {
                            v = p.ez;
                            break;
                        }
                        case 6: {
                            v = p.ppm;
                            break;
                        }
                        default: {
                            throw new AssertionError(k);
                        }
                    }
                    table.write(nf.format(v));
                    table.nextColumn();
                }
                table.nextLine();
            }
            table.writeHorizontalSeparator();
        }
        try {
            table.flush();
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
    }

    private void operations(PrintWriter out, String[] args) throws FactoryException {
        if (!(this.factory instanceof CoordinateOperationAuthorityFactory)) {
            return;
        }
        CoordinateOperationAuthorityFactory factory = (CoordinateOperationAuthorityFactory)this.factory;
        char[] separator = null;
        for (int i = 0; i < args.length; ++i) {
            for (int j = i + 1; j < args.length; ++j) {
                Set<CoordinateOperation> op = factory.createFromCoordinateReferenceSystemCodes(args[i], args[j]);
                for (CoordinateOperation operation : op) {
                    if (separator == null) {
                        separator = Command.getSeparator();
                    } else {
                        out.println(separator);
                    }
                    out.println(this.formatter.format(operation));
                }
            }
        }
    }

    private void transform(PrintWriter out, String[] args) throws FactoryException {
        if (!(this.factory instanceof CRSAuthorityFactory)) {
            return;
        }
        CRSAuthorityFactory factory = (CRSAuthorityFactory)this.factory;
        CoordinateOperationFactory opFactory = ReferencingFactoryFinder.getCoordinateOperationFactory(HINTS);
        char[] separator = null;
        for (int i = 0; i < args.length; ++i) {
            CoordinateReferenceSystem crs1 = factory.createCoordinateReferenceSystem(args[i]);
            for (int j = i + 1; j < args.length; ++j) {
                CoordinateOperation op;
                CoordinateReferenceSystem crs2 = factory.createCoordinateReferenceSystem(args[j]);
                try {
                    op = opFactory.createOperation(crs1, crs2);
                }
                catch (OperationNotFoundException exception) {
                    out.println(exception.getLocalizedMessage());
                    continue;
                }
                if (separator == null) {
                    separator = Command.getSeparator();
                } else {
                    out.println(separator);
                }
                out.println(this.formatter.format(op.getMathTransform()));
            }
        }
    }

    private static void printAuthorityFactoryDependencies(PrintWriter out, boolean colors) {
        FactoryDependencies printer = new FactoryDependencies(CRS.getAuthorityFactory(false));
        printer.setAttributeEnabled(true);
        printer.setColorEnabled(colors);
        printer.print(out);
        out.flush();
    }

    private void dispose() throws FactoryException {
        if (this.factory instanceof AbstractAuthorityFactory) {
            ((AbstractAuthorityFactory)this.factory).dispose();
        }
    }

    public static void execute(String[] args) {
        Arguments arguments = new Arguments(args);
        PrintWriter out = arguments.out;
        Locale.setDefault(arguments.locale);
        if (arguments.getFlag("-forcexy")) {
            Hints.putSystemDefault(Hints.FORCE_LONGITUDE_FIRST_AXIS_ORDER, Boolean.TRUE);
        }
        if (arguments.getFlag("-help")) {
            args = arguments.getRemainingArguments(0);
            Command.help(out);
            return;
        }
        if (arguments.getFlag("-factories")) {
            args = arguments.getRemainingArguments(0);
            Command.factories(out);
            return;
        }
        if (arguments.getFlag("-dependencies")) {
            boolean colors = arguments.getFlag("-colors");
            args = arguments.getRemainingArguments(0);
            Command.printAuthorityFactoryDependencies(out, colors);
            return;
        }
        String authority = arguments.getOptionalString("-authority");
        Command command = new Command(authority);
        command.formatter.setColorEnabled(arguments.getFlag("-colors"));
        try {
            if (arguments.getFlag("-codes")) {
                args = arguments.getRemainingArguments(0);
                command.codes(out);
            } else if (arguments.getFlag("-bursawolfs")) {
                args = arguments.getRemainingArguments(Integer.MAX_VALUE, '-');
                command.bursaWolfs(out, args);
            } else if (arguments.getFlag("-operations")) {
                args = arguments.getRemainingArguments(2, '-');
                command.operations(out, args);
            } else if (arguments.getFlag("-transform")) {
                args = arguments.getRemainingArguments(2, '-');
                command.transform(out, args);
            } else {
                args = arguments.getRemainingArguments(Integer.MAX_VALUE, '-');
                command.list(out, args);
            }
            out.flush();
            command.dispose();
        }
        catch (FactoryException exception) {
            out.flush();
            arguments.err.println(exception.getLocalizedMessage());
        }
        catch (Exception exception) {
            out.flush();
            exception.printStackTrace(arguments.err);
        }
    }
}

