/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.cache;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import org.neo4j.kernel.impl.cache.Cache;
import org.neo4j.kernel.impl.cache.EntityWithSize;
import org.neo4j.kernel.impl.util.StringLogger;
import org.neo4j.kernel.info.DiagnosticsPhase;
import org.neo4j.kernel.info.DiagnosticsProvider;

public class AtomicArrayCache<E extends EntityWithSize>
implements Cache<E>,
DiagnosticsProvider {
    public static final long MIN_SIZE = 1L;
    private final AtomicReferenceArray<E> cache;
    private final long maxSize;
    private final AtomicLong currentSize = new AtomicLong(0L);
    private final long minLogInterval;
    private final String name;
    private long hitCount = 0L;
    private long missCount = 0L;
    private long totalPuts = 0L;
    private long collisions = 0L;
    private long purgeCount = 0L;
    private final StringLogger logger;
    private long putTimeStamp = 0L;
    private long lastPurgeLogTimestamp = 0L;

    AtomicArrayCache(AtomicReferenceArray<E> cache) {
        this.cache = cache;
        this.minLogInterval = Long.MAX_VALUE;
        this.maxSize = 0x40000000L;
        this.name = "test cache";
        this.logger = null;
    }

    public AtomicArrayCache(long maxSizeInBytes, float arrayHeapFraction, long minLogInterval, String name, StringLogger logger) {
        this.minLogInterval = minLogInterval;
        if (arrayHeapFraction < 1.0f || arrayHeapFraction > 10.0f) {
            throw new IllegalArgumentException("The heap fraction used by an array cache must be between 1% and 10%, not " + arrayHeapFraction + "%");
        }
        long memToUse = (long)((double)arrayHeapFraction * (double)Runtime.getRuntime().maxMemory() / 100.0);
        long maxElementCount = (int)(memToUse / 8L);
        if (memToUse > Integer.MAX_VALUE) {
            maxElementCount = Integer.MAX_VALUE;
        }
        if (maxSizeInBytes < 1L) {
            throw new IllegalArgumentException("Max size can not be " + maxSizeInBytes);
        }
        this.cache = new AtomicReferenceArray((int)maxElementCount);
        this.maxSize = maxSizeInBytes;
        this.name = name == null ? super.toString() : name;
        this.logger = logger == null ? StringLogger.SYSTEM : logger;
    }

    private int getPosition(EntityWithSize obj) {
        return (int)(obj.getId() % (long)this.cache.length());
    }

    private int getPosition(long id) {
        return (int)(id % (long)this.cache.length());
    }

    @Override
    public void put(E obj) {
        int pos;
        EntityWithSize oldObj;
        long time = System.currentTimeMillis();
        if (time - this.putTimeStamp > this.minLogInterval) {
            this.putTimeStamp = time;
            this.printStatistics();
        }
        if ((oldObj = (EntityWithSize)this.cache.get(pos = this.getPosition((EntityWithSize)obj))) != obj) {
            int objectSize = obj.size();
            if (this.cache.compareAndSet(pos, oldObj, (EntityWithSize)obj)) {
                int oldObjSize = 0;
                if (oldObj != null) {
                    oldObjSize = oldObj.size();
                }
                long size = this.currentSize.addAndGet(objectSize - oldObjSize);
                if (oldObj != null) {
                    ++this.collisions;
                }
                ++this.totalPuts;
                if (size > this.maxSize) {
                    this.purgeFrom(pos);
                }
            }
        }
    }

    @Override
    public E remove(long id) {
        int pos = this.getPosition(id);
        EntityWithSize obj = (EntityWithSize)this.cache.get(pos);
        if (obj != null && this.cache.compareAndSet(pos, obj, null)) {
            int objSize = obj.size();
            this.currentSize.addAndGet(objSize * -1);
        }
        return (E)obj;
    }

    @Override
    public E get(long id) {
        int pos = this.getPosition(id);
        EntityWithSize obj = (EntityWithSize)this.cache.get(pos);
        if (obj != null && obj.getId() == id) {
            ++this.hitCount;
            return (E)obj;
        }
        ++this.missCount;
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void purgeFrom(int pos) {
        if ((float)this.currentSize.get() <= (float)this.maxSize * 0.95f) {
            return;
        }
        ++this.purgeCount;
        long sizeBefore = this.currentSize.get();
        try {
            int index = 1;
            do {
                if (pos - index >= 0) {
                    int minusPos = pos - index;
                    this.remove(minusPos);
                    if ((float)this.currentSize.get() <= (float)this.maxSize * 0.9f) {
                        return;
                    }
                }
                if (pos + index >= this.cache.length()) continue;
                int plusPos = pos + index;
                this.remove(plusPos);
                if (!((float)this.currentSize.get() <= (float)this.maxSize * 0.9f)) continue;
                return;
            } while (pos - ++index >= 0 || pos + index < this.cache.length());
            this.remove(pos);
        }
        finally {
            long timestamp = System.currentTimeMillis();
            if (timestamp - this.lastPurgeLogTimestamp > this.minLogInterval) {
                this.lastPurgeLogTimestamp = timestamp;
                long sizeAfter = this.currentSize.get();
                String sizeBeforeStr = this.getSize(sizeBefore);
                String sizeAfterStr = this.getSize(sizeAfter);
                String diffStr = this.getSize(sizeBefore - sizeAfter);
                String missPercentage = (float)this.missCount / (float)(this.hitCount + this.missCount) * 100.0f + "%";
                String colPercentage = (float)this.collisions / (float)this.totalPuts * 100.0f + "%";
                this.logger.logMessage(this.name + " purge (nr " + this.purgeCount + ") " + sizeBeforeStr + " -> " + sizeAfterStr + " (" + diffStr + ") " + missPercentage + " misses, " + colPercentage + " collisions.", true);
                int elementCount = 0;
                long size = 0L;
                for (int i = 0; i < this.cache.length(); ++i) {
                    EntityWithSize obj = (EntityWithSize)this.cache.get(i);
                    if (obj == null) continue;
                    ++elementCount;
                    size += (long)obj.size();
                }
                this.logger.logMessage(this.name + " purge (nr " + this.purgeCount + "): elementCount now=" + elementCount + " and size=" + this.getSize(size) + " (" + this.getSize(this.currentSize.get()) + ") [" + this.getSize(this.currentSize.get() - size) + "]", true);
            }
        }
    }

    public void printStatistics() {
        this.logStatistics(this.logger);
    }

    @Override
    public String getDiagnosticsIdentifier() {
        return this.getName();
    }

    @Override
    public void acceptDiagnosticsVisitor(Object visitor) {
    }

    @Override
    public void dump(DiagnosticsPhase phase, StringLogger log) {
        if (phase.isExplicitlyRequested()) {
            this.logStatistics(log);
        }
    }

    private void logStatistics(StringLogger log) {
        String currentSizeStr = this.getSize(this.currentSize.get());
        String missPercentage = (float)this.missCount / (float)(this.hitCount + this.missCount) * 100.0f + "%";
        String colPercentage = (float)this.collisions / (float)this.totalPuts * 100.0f + "%";
        log.logMessage(this.name + " array size: " + this.cache.length() + " purge count: " + this.purgeCount + " size is: " + currentSizeStr + ", " + missPercentage + " misses, " + colPercentage + " collisions.", true);
    }

    private String getSize(long size) {
        if (size > 0x40000000L) {
            float value = (float)size / 1024.0f / 1024.0f / 1024.0f;
            return value + "Gb";
        }
        if (size > 0x100000L) {
            float value = (float)size / 1024.0f / 1024.0f;
            return value + "Mb";
        }
        if (size > 1024L) {
            float value = (float)size / 1024.0f / 1024.0f;
            return value + "kb";
        }
        return size + "b";
    }

    @Override
    public void clear() {
        for (int i = 0; i < this.cache.length(); ++i) {
            this.cache.set(i, null);
        }
        this.currentSize.set(0L);
    }

    @Override
    public void putAll(Collection<E> objects) {
        for (EntityWithSize obj : objects) {
            this.put(obj);
        }
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public long size() {
        return this.currentSize.get();
    }

    @Override
    public long hitCount() {
        return this.hitCount;
    }

    @Override
    public long missCount() {
        return this.missCount;
    }

    @Override
    public void updateSize(E obj, int sizeBefore, int sizeAfter) {
        int pos = this.getPosition((EntityWithSize)obj);
        EntityWithSize oldObj = (EntityWithSize)this.cache.get(pos);
        if (oldObj != obj) {
            return;
        }
        long size = this.currentSize.addAndGet(sizeAfter - sizeBefore);
        if (size > this.maxSize) {
            this.purgeFrom(pos);
        }
    }
}

