/* 
 * Licensed to Aduna under one or more contributor license agreements.  
 * See the NOTICE.txt file distributed with this work for additional 
 * information regarding copyright ownership. 
 *
 * Aduna licenses this file to you under the terms of the Aduna BSD 
 * License (the "License"); you may not use this file except in compliance 
 * with the License. See the LICENSE.txt file distributed with this work 
 * for the full License.
 *
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package org.openrdf.sail.rdbms.managers.base;

import java.sql.SQLException;
import java.util.concurrent.atomic.AtomicInteger;

import org.openrdf.sail.rdbms.managers.HashManager;
import org.openrdf.sail.rdbms.model.RdbmsValue;
import org.openrdf.sail.rdbms.schema.IdSequence;
import org.openrdf.sail.rdbms.util.LRUMap;

public abstract class ValueManagerBase<V extends RdbmsValue> extends ManagerBase {

	private LRUMap<Object, V> cache;

	private HashManager hashes;

	private AtomicInteger version = new AtomicInteger();

	private IdSequence ids;

	public void setHashManager(HashManager hashes) {
		this.hashes = hashes;
	}

	public IdSequence getIdSequence() {
		return ids;
	}

	public void setIdSequence(IdSequence ids) {
		this.ids = ids;
	}

	public void init() {
		cache = new LRUMap<Object, V>(getBatchSize());
	}

	@Override
	public void flush()
		throws SQLException, InterruptedException
	{
		if (hashes != null) {
			hashes.flush();
		}
		super.flush();
	}

	public V findInCache(Object key) {
		synchronized (cache) {
			if (cache.containsKey(key))
				return cache.get(key);
		}
		return null;
	}

	public void cache(V value)
		throws InterruptedException
	{
		if (value.isExpired(getIdVersion())) {
			synchronized (cache) {
				cache.put(key(value), value);
			}
			if (hashes != null) {
				hashes.lookupId(value);
			}
		}
	}

	public Number getInternalId(V value)
		throws SQLException, InterruptedException
	{
		if (value.isExpired(getIdVersion())) {
			if (hashes == null) {
				Number id = ids.idOf(value);
				value.setInternalId(id);
				value.setVersion(getIdVersion());
				insert(id, value);
			}
			else if (value.isExpired(getIdVersion())) {
				hashes.assignId(value, getIdVersion());
			}
		}
		return value.getInternalId();
	}

	public int getIdVersion() {
		return version.intValue() + (hashes == null ? 0 : hashes.getIdVersion());
	}

	public void removedStatements(String condition)
		throws SQLException
	{
		if (expunge(condition)) {
			version.addAndGet(1);
		}
	}

	protected abstract int getBatchSize();

	protected abstract void insert(Number id, V value)
		throws SQLException, InterruptedException;

	protected abstract Object key(V value);

	protected abstract boolean expunge(String condition)
		throws SQLException;

	@Override
	protected void optimize()
		throws SQLException
	{
		if (hashes != null) {
			hashes.optimize();
		}
	}

}
