/*
 * Copyright 2020 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.xml;

import org.antlr.v4.runtime.*;
import org.intellij.lang.annotations.Language;
import org.jspecify.annotations.Nullable;
import org.openrewrite.*;
import org.openrewrite.Parser;
import org.openrewrite.internal.EncodingDetectingInputStream;
import org.openrewrite.tree.ParseError;
import org.openrewrite.tree.ParsingEventListener;
import org.openrewrite.tree.ParsingExecutionContextView;
import org.openrewrite.xml.internal.XmlParserVisitor;
import org.openrewrite.xml.internal.grammar.XMLLexer;
import org.openrewrite.xml.internal.grammar.XMLParser;
import org.openrewrite.xml.tree.Xml;

import java.nio.file.Path;
import java.util.stream.Stream;

public class XmlParser implements Parser {
    @Override
    public Stream<SourceFile> parseInputs(Iterable<Input> sourceFiles, @Nullable Path relativeTo, ExecutionContext ctx) {
        ParsingEventListener parsingListener = ParsingExecutionContextView.view(ctx).getParsingListener();
        return acceptedInputs(sourceFiles).map(input -> {
            parsingListener.startedParsing(input);
            Path path = input.getRelativePath(relativeTo);
            try (EncodingDetectingInputStream is = input.getSource(ctx)) {
                String sourceStr = is.readFully();

                XMLLexer lexer = new XMLLexer(CharStreams.fromString(sourceStr));
                lexer.removeErrorListeners();
                lexer.addErrorListener(new ForwardingErrorListener(input.getPath(), ctx));

                XMLParser parser = new XMLParser(new CommonTokenStream(lexer));
                parser.removeErrorListeners();
                parser.addErrorListener(new ForwardingErrorListener(input.getPath(), ctx));

                Xml.Document document = new XmlParserVisitor(
                        path,
                        input.getFileAttributes(),
                        sourceStr,
                        is.getCharset(),
                        is.isCharsetBomMarked()
                ).visitDocument(parser.document());
                parsingListener.parsed(input, document);
                return requirePrintEqualsInput(document, input, relativeTo, ctx);
            } catch (Throwable t) {
                ctx.getOnError().accept(t);
                return ParseError.build(this, input, relativeTo, ctx, t);
            }
        });
    }

    @Override
    public Stream<SourceFile> parse(@Language("xml") String... sources) {
        return parse(new InMemoryExecutionContext(), sources);
    }

    @Override
    public boolean accept(Path path) {
        String p = path.toString();
        return p.endsWith(".xml") ||
               p.endsWith(".wsdl") ||
               p.endsWith(".xhtml") ||
               p.endsWith(".xsd") ||
               p.endsWith(".xsl") ||
               p.endsWith(".xslt") ||
               p.endsWith(".xmi") ||
               p.endsWith(".tld") ||
               p.endsWith(".xjb") ||
               p.endsWith(".jsp") ||
               // Datastage file formats that are all xml under the hood
               p.endsWith(".det") ||
               p.endsWith(".pjb") ||
               p.endsWith(".qjb") ||
               p.endsWith(".sjb") ||
               p.endsWith(".prt") ||
               p.endsWith(".srt") ||
               p.endsWith(".psc") ||
               p.endsWith(".ssc") ||
               p.endsWith(".tbd") ||
               p.endsWith(".tfm") ||
               p.endsWith(".dqs") ||
               p.endsWith(".stp") ||
               p.endsWith(".dcn") ||
               p.endsWith(".pst") ||
               // C# project files
               p.endsWith(".csproj") ||
               path.endsWith("packages.config");
    }

    @Override
    public Path sourcePathFromSourceText(Path prefix, String sourceCode) {
        return prefix.resolve("file.xml");
    }

    private static class ForwardingErrorListener extends BaseErrorListener {
        private final Path sourcePath;
        private final ExecutionContext ctx;

        private ForwardingErrorListener(Path sourcePath, ExecutionContext ctx) {
            this.sourcePath = sourcePath;
            this.ctx = ctx;
        }

        @Override
        public void syntaxError(Recognizer<?, ?> recognizer, Object offendingSymbol,
                                int line, int charPositionInLine, String msg, RecognitionException e) {
            ctx.getOnError().accept(new XmlParsingException(sourcePath,
                    String.format("Syntax error in %s at line %d:%d %s.", sourcePath, line, charPositionInLine, msg), e));
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder extends org.openrewrite.Parser.Builder {

        public Builder() {
            super(Xml.Document.class);
        }

        @Override
        public XmlParser build() {
            return new XmlParser();
        }

        @Override
        public String getDslName() {
            return "xml";
        }
    }
}
