/*
 * Copyright 2002-2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.validation;

/**
 * A validator for application-specific objects.
 * 
 * <p>This interface is totally divorced from any infrastructure
 * or context; that is to say it is not coupled to validating
 * only objects in the web tier, the data-access tier, or the
 * whatever-tier. As such it is amenable to being used in any layer
 * of an application, and supports the encapsulation of validation
 * logic as first-class citizens in their own right.
 * 
 * <p>Find below a simple but complete <code>Validator</code>
 * implementation, that validates that the various {@link String}
 * properties (fields) of a <code>UserLogin</code> instance are not
 * empty (that is they are not <code>null</code> and do not consist
 * wholly of whitespace), and that any password that is present is
 * at least <code>'MINIMUM_PASSWORD_LENGTH'</code> characters in length.
 * 
 * <pre class="code"> public class UserLoginValidator implements Validator {
 * 
 *    private static final int MINIMUM_PASSWORD_LENGTH = 6;
 * 
 *    public boolean supports(Class clazz) {
 *       return UserLogin.class.isAssignableFrom(clazz);
 *    }
 * 
 *    public void validate(Object target, Errors errors) {
 *       ValidationUtils.rejectIfEmptyOrWhitespace(errors, "userName", "field.required");
 *       ValidationUtils.rejectIfEmptyOrWhitespace(errors, "password", "field.required");
 *       UserLogin login = (UserLogin) target;
 *       if (login.getPassword() != null
 *             && login.getPassword().trim().length() < MINIMUM_PASSWORD_LENGTH) {
 *          errors.rejectValue("password", "field.min.length",
 *                new Object[]{Integer.valueOf(MINIMUM_PASSWORD_LENGTH)},
 *                "The password must be at least [" + MINIMUM_PASSWORD_LENGTH + "] characters in length.");
 *       }
 *    }
 * }</pre>
 * 
 * <p>See also the Spring reference manual for a fuller discussion of
 * the <code>Validator</code> interface and it's role in a enterprise
 * application. 
 * 
 * @author Rod Johnson
 * @see Errors
 * @see ValidationUtils
 */
public interface Validator {
	
	/**
	 * Can this {@link Validator} {@link #validate(Object, Errors) validate}
	 * instances of the supplied <code>clazz</code>?
	 * <p>This method is <i>'typically'</i> implemented like so:
	 * <pre class="code">return Foo.class.isAssignableFrom(clazz);</pre>
	 * (Where <code>Foo</code> is the class (or superclass) of the actual
	 * object instance that is to be {@link #validate(Object, Errors) validated}).)
	 * <p>However, implementations can (of course) choose to implement
	 * this method in any manner they please.
	 * @param clazz the {@link Class} that this {@link Validator} is
	 * being asked if it can {@link #validate(Object, Errors) validate}
	 * @return <code>true</code> if this {@link Validator} can indeed
	 * {@link #validate(Object, Errors) validate} instances of the
	 * supplied <code>clazz</code> 
	 */
	boolean supports(Class clazz);
	
	/**
	 * Validate the supplied <code>target</code> object, which must be
	 * of a {@link Class} for which the {@link #supports(Class)} method
	 * typically has (or would) return <code>true</code>.
	 * <p>The supplied {@link Errors errors} instance can be used to report
	 * any resulting validation errors.
	 * @param target the object that is to be validated (can be <code>null</code>) 
	 * @param errors contextual state about the validation process (never <code>null</code>) 
	 * @see ValidationUtils
	 */
	void validate(Object target, Errors errors);

}
