/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HAUtil;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LayoutVersion;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.InconsistentFSStateException;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.Util;
import org.apache.hadoop.hdfs.server.namenode.CheckpointFaultInjector;
import org.apache.hadoop.hdfs.server.namenode.CheckpointSignature;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.FSImageCompression;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormat;
import org.apache.hadoop.hdfs.server.namenode.FSImagePreTransactionalStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSImageStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectoryWithQuota;
import org.apache.hadoop.hdfs.server.namenode.MetaRecoveryContext;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Quota;
import org.apache.hadoop.hdfs.server.namenode.SaveNamespaceCancelledException;
import org.apache.hadoop.hdfs.server.namenode.SaveNamespaceContext;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Phase;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StartupProgress;
import org.apache.hadoop.hdfs.server.protocol.CheckpointCommand;
import org.apache.hadoop.hdfs.server.protocol.NamenodeCommand;
import org.apache.hadoop.hdfs.server.protocol.NamenodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.MD5FileUtils;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class FSImage
implements Closeable {
    public static final Log LOG = LogFactory.getLog((String)FSImage.class.getName());
    protected FSEditLog editLog = null;
    private boolean isUpgradeFinalized = false;
    protected NNStorage storage;
    protected long lastAppliedTxId = 0L;
    private final Configuration conf;
    protected NNStorageRetentionManager archivalManager;

    public FSImage(Configuration conf) throws IOException {
        this(conf, FSNamesystem.getNamespaceDirs(conf), FSNamesystem.getNamespaceEditsDirs(conf));
    }

    protected FSImage(Configuration conf, Collection<URI> imageDirs, List<URI> editsDirs) throws IOException {
        this.conf = conf;
        this.storage = new NNStorage(conf, imageDirs, editsDirs);
        if (conf.getBoolean("dfs.namenode.name.dir.restore", false)) {
            this.storage.setRestoreFailedStorage(true);
        }
        this.editLog = new FSEditLog(conf, this.storage, editsDirs);
        this.archivalManager = new NNStorageRetentionManager(conf, this.storage, this.editLog);
    }

    void format(FSNamesystem fsn, String clusterId) throws IOException {
        long fileCount = fsn.getTotalFiles();
        Preconditions.checkState((fileCount == 1L ? 1 : 0) != 0, (Object)("FSImage.format should be called with an uninitialized namesystem, has " + fileCount + " files"));
        NamespaceInfo ns = NNStorage.newNamespaceInfo();
        ns.clusterID = clusterId;
        this.storage.format(ns);
        this.editLog.formatNonFileJournals(ns);
        this.saveFSImageInAllDirs(fsn, 0L);
    }

    boolean confirmFormat(boolean force, boolean interactive) throws IOException {
        ArrayList confirms = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(null)) {
            confirms.add(sd);
        }
        confirms.addAll(this.editLog.getFormatConfirmables());
        return Storage.confirmFormat(confirms, force, interactive);
    }

    boolean recoverTransitionRead(HdfsServerConstants.StartupOption startOpt, FSNamesystem target, MetaRecoveryContext recovery) throws IOException {
        assert (startOpt != HdfsServerConstants.StartupOption.FORMAT) : "NameNode formatting should be performed before reading the image";
        Collection<URI> imageDirs = this.storage.getImageDirectories();
        Collection<URI> editsDirs = this.editLog.getEditURIs();
        if ((imageDirs.size() == 0 || editsDirs.size() == 0) && startOpt != HdfsServerConstants.StartupOption.IMPORT) {
            throw new IOException("All specified directories are not accessible or do not exist.");
        }
        HashMap<Storage.StorageDirectory, Storage.StorageState> dataDirStates = new HashMap<Storage.StorageDirectory, Storage.StorageState>();
        boolean isFormatted = this.recoverStorageDirs(startOpt, dataDirStates);
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Data dir states:\n  " + Joiner.on((String)"\n  ").withKeyValueSeparator(": ").join(dataDirStates)));
        }
        if (!isFormatted && startOpt != HdfsServerConstants.StartupOption.ROLLBACK && startOpt != HdfsServerConstants.StartupOption.IMPORT) {
            throw new IOException("NameNode is not formatted.");
        }
        int layoutVersion = this.storage.getLayoutVersion();
        if (layoutVersion < -3) {
            NNStorage.checkVersionUpgradable(this.storage.getLayoutVersion());
        }
        if (startOpt != HdfsServerConstants.StartupOption.UPGRADE && layoutVersion < -3 && layoutVersion != HdfsConstants.LAYOUT_VERSION) {
            throw new IOException("\nFile system image contains an old layout version " + this.storage.getLayoutVersion() + ".\nAn upgrade to version " + HdfsConstants.LAYOUT_VERSION + " is required.\n" + "Please restart NameNode with -upgrade option.");
        }
        this.storage.processStartupOptionsForUpgrade(startOpt, layoutVersion);
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            Storage.StorageState curState = (Storage.StorageState)((Object)dataDirStates.get(sd));
            switch (curState) {
                case NON_EXISTENT: {
                    throw new IOException((Object)((Object)Storage.StorageState.NON_EXISTENT) + " state cannot be here");
                }
                case NOT_FORMATTED: {
                    LOG.info((Object)("Storage directory " + sd.getRoot() + " is not formatted."));
                    LOG.info((Object)"Formatting ...");
                    sd.clearDirectory();
                    break;
                }
            }
        }
        switch (startOpt) {
            case UPGRADE: {
                this.doUpgrade(target);
                return false;
            }
            case IMPORT: {
                this.doImportCheckpoint(target);
                return false;
            }
            case ROLLBACK: {
                this.doRollback();
                break;
            }
        }
        return this.loadFSImage(target, recovery);
    }

    private boolean recoverStorageDirs(HdfsServerConstants.StartupOption startOpt, Map<Storage.StorageDirectory, Storage.StorageState> dataDirStates) throws IOException {
        boolean isFormatted = false;
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageState curState;
            Storage.StorageDirectory sd = it.next();
            try {
                curState = sd.analyzeStorage(startOpt, this.storage);
                String nameserviceId = DFSUtil.getNamenodeNameServiceId(this.conf);
                if (curState != Storage.StorageState.NORMAL && HAUtil.isHAEnabled(this.conf, nameserviceId)) {
                    throw new IOException("Cannot start an HA namenode with name dirs that need recovery. Dir: " + sd + " state: " + (Object)((Object)curState));
                }
                switch (curState) {
                    case NON_EXISTENT: {
                        throw new InconsistentFSStateException(sd.getRoot(), "storage directory does not exist or is not accessible.");
                    }
                    case NOT_FORMATTED: {
                        break;
                    }
                    case NORMAL: {
                        break;
                    }
                    default: {
                        sd.doRecover(curState);
                    }
                }
                if (curState != Storage.StorageState.NOT_FORMATTED && startOpt != HdfsServerConstants.StartupOption.ROLLBACK) {
                    this.storage.readProperties(sd);
                    isFormatted = true;
                }
                if (startOpt == HdfsServerConstants.StartupOption.IMPORT && isFormatted) {
                    throw new IOException("Cannot import image from a checkpoint.  NameNode already contains an image in " + sd.getRoot());
                }
            }
            catch (IOException ioe) {
                sd.unlock();
                throw ioe;
            }
            dataDirStates.put(sd, curState);
        }
        return isFormatted;
    }

    private void doUpgrade(FSNamesystem target) throws IOException {
        Storage.StorageDirectory sd;
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd2 = it.next();
            if (!sd2.getPreviousDir().exists()) continue;
            throw new InconsistentFSStateException(sd2.getRoot(), "previous fs state should not exist during upgrade. Finalize or rollback first.");
        }
        this.loadFSImage(target, null);
        long oldCTime = this.storage.getCTime();
        this.storage.cTime = Time.now();
        int oldLV = this.storage.getLayoutVersion();
        this.storage.layoutVersion = HdfsConstants.LAYOUT_VERSION;
        List<Storage.StorageDirectory> errorSDs = Collections.synchronizedList(new ArrayList());
        Iterator<Storage.StorageDirectory> it2 = this.storage.dirIterator();
        while (it2.hasNext()) {
            sd = it2.next();
            LOG.info((Object)("Starting upgrade of image directory " + sd.getRoot() + ".\n   old LV = " + oldLV + "; old CTime = " + oldCTime + ".\n   new LV = " + this.storage.getLayoutVersion() + "; new CTime = " + this.storage.getCTime()));
            try {
                File curDir = sd.getCurrentDir();
                File prevDir = sd.getPreviousDir();
                File tmpDir = sd.getPreviousTmp();
                assert (curDir.exists()) : "Current directory must exist.";
                assert (!prevDir.exists()) : "previous directory must not exist.";
                assert (!tmpDir.exists()) : "previous.tmp directory must not exist.";
                assert (!this.editLog.isSegmentOpen()) : "Edits log must not be open.";
                NNStorage.rename(curDir, tmpDir);
                if (curDir.mkdir()) continue;
                throw new IOException("Cannot create directory " + curDir);
            }
            catch (Exception e) {
                LOG.error((Object)("Failed to move aside pre-upgrade storage in image directory " + sd.getRoot()), (Throwable)e);
                errorSDs.add(sd);
            }
        }
        this.storage.reportErrorsOnDirectories(errorSDs);
        errorSDs.clear();
        this.saveFSImageInAllDirs(target, this.editLog.getLastWrittenTxId());
        it2 = this.storage.dirIterator();
        while (it2.hasNext()) {
            sd = it2.next();
            try {
                this.storage.writeProperties(sd);
                File prevDir = sd.getPreviousDir();
                File tmpDir = sd.getPreviousTmp();
                NNStorage.rename(tmpDir, prevDir);
            }
            catch (IOException ioe) {
                LOG.error((Object)("Unable to rename temp to previous for " + sd.getRoot()), (Throwable)ioe);
                errorSDs.add(sd);
                continue;
            }
            LOG.info((Object)("Upgrade of " + sd.getRoot() + " is complete."));
        }
        this.storage.reportErrorsOnDirectories(errorSDs);
        this.isUpgradeFinalized = false;
        if (!this.storage.getRemovedStorageDirs().isEmpty()) {
            throw new IOException("Upgrade failed in " + this.storage.getRemovedStorageDirs().size() + " storage directory(ies), previously logged.");
        }
    }

    private void doRollback() throws IOException {
        File prevDir;
        Storage.StorageDirectory sd;
        boolean canRollback = false;
        FSImage prevState = new FSImage(this.conf);
        prevState.getStorage().layoutVersion = HdfsConstants.LAYOUT_VERSION;
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            sd = it.next();
            prevDir = sd.getPreviousDir();
            if (!prevDir.exists()) {
                LOG.info((Object)("Storage directory " + sd.getRoot() + " does not contain previous fs state."));
                this.storage.readProperties(sd);
                continue;
            }
            prevState.getStorage().readPreviousVersionProperties(sd);
            if (prevState.getLayoutVersion() != HdfsConstants.LAYOUT_VERSION) {
                throw new IOException("Cannot rollback to storage version " + prevState.getLayoutVersion() + " using this version of the NameNode, which uses storage version " + HdfsConstants.LAYOUT_VERSION + ". " + "Please use the previous version of HDFS to perform the rollback.");
            }
            canRollback = true;
        }
        if (!canRollback) {
            throw new IOException("Cannot rollback. None of the storage directories contain previous fs state.");
        }
        it = this.storage.dirIterator();
        while (it.hasNext()) {
            sd = it.next();
            prevDir = sd.getPreviousDir();
            if (!prevDir.exists()) continue;
            LOG.info((Object)("Rolling back storage directory " + sd.getRoot() + ".\n   new LV = " + prevState.getStorage().getLayoutVersion() + "; new CTime = " + prevState.getStorage().getCTime()));
            File tmpDir = sd.getRemovedTmp();
            assert (!tmpDir.exists()) : "removed.tmp directory must not exist.";
            File curDir = sd.getCurrentDir();
            assert (curDir.exists()) : "Current directory must exist.";
            NNStorage.rename(curDir, tmpDir);
            NNStorage.rename(prevDir, curDir);
            NNStorage.deleteDir(tmpDir);
            LOG.info((Object)("Rollback of " + sd.getRoot() + " is complete."));
        }
        this.isUpgradeFinalized = true;
    }

    private void doFinalize(Storage.StorageDirectory sd) throws IOException {
        File prevDir = sd.getPreviousDir();
        if (!prevDir.exists()) {
            LOG.info((Object)("Directory " + prevDir + " does not exist."));
            LOG.info((Object)("Finalize upgrade for " + sd.getRoot() + " is not required."));
            return;
        }
        LOG.info((Object)("Finalizing upgrade for storage directory " + sd.getRoot() + "." + (this.storage.getLayoutVersion() == 0 ? "" : "\n   cur LV = " + this.storage.getLayoutVersion() + "; cur CTime = " + this.storage.getCTime())));
        assert (sd.getCurrentDir().exists()) : "Current directory must exist.";
        File tmpDir = sd.getFinalizedTmp();
        NNStorage.rename(prevDir, tmpDir);
        NNStorage.deleteDir(tmpDir);
        this.isUpgradeFinalized = true;
        LOG.info((Object)("Finalize upgrade for " + sd.getRoot() + " is complete."));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doImportCheckpoint(FSNamesystem target) throws IOException {
        FSImage ckptImage;
        Collection<URI> checkpointDirs = FSImage.getCheckpointDirs(this.conf, null);
        List<URI> checkpointEditsDirs = FSImage.getCheckpointEditsDirs(this.conf, null);
        if (checkpointDirs == null || checkpointDirs.isEmpty()) {
            throw new IOException("Cannot import image from a checkpoint. \"dfs.namenode.checkpoint.dir\" is not set.");
        }
        if (checkpointEditsDirs == null || checkpointEditsDirs.isEmpty()) {
            throw new IOException("Cannot import image from a checkpoint. \"dfs.namenode.checkpoint.dir\" is not set.");
        }
        FSImage realImage = target.getFSImage();
        target.dir.fsImage = ckptImage = new FSImage(this.conf, checkpointDirs, checkpointEditsDirs);
        try {
            ckptImage.recoverTransitionRead(HdfsServerConstants.StartupOption.REGULAR, target, null);
        }
        finally {
            ckptImage.close();
        }
        realImage.getStorage().setStorageInfo(ckptImage.getStorage());
        realImage.getEditLog().setNextTxId(ckptImage.getEditLog().getLastWrittenTxId() + 1L);
        realImage.initEditLog();
        target.dir.fsImage = realImage;
        realImage.getStorage().setBlockPoolID(ckptImage.getBlockPoolID());
        this.saveNamespace(target);
        this.getStorage().writeAll();
    }

    void finalizeUpgrade() throws IOException {
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            this.doFinalize(sd);
        }
    }

    boolean isUpgradeFinalized() {
        return this.isUpgradeFinalized;
    }

    public FSEditLog getEditLog() {
        return this.editLog;
    }

    @VisibleForTesting
    void setEditLogForTesting(FSEditLog newLog) {
        this.editLog = newLog;
    }

    void openEditLogForWrite() throws IOException {
        assert (this.editLog != null) : "editLog must be initialized";
        this.editLog.openForWrite();
        this.storage.writeTransactionIdFileToStorage(this.editLog.getCurSegmentTxId());
    }

    void reloadFromImageFile(File file, FSNamesystem target) throws IOException {
        target.clear();
        LOG.debug((Object)("Reloading namespace from " + file));
        this.loadFSImage(file, target, null);
    }

    boolean loadFSImage(FSNamesystem target, MetaRecoveryContext recovery) throws IOException {
        FSImageStorageInspector inspector = this.storage.readAndInspectDirs();
        FSImageStorageInspector.FSImageFile imageFile = null;
        this.isUpgradeFinalized = inspector.isUpgradeFinalized();
        List<FSImageStorageInspector.FSImageFile> imageFiles = inspector.getLatestImages();
        StartupProgress prog = NameNode.getStartupProgress();
        prog.beginPhase(Phase.LOADING_FSIMAGE);
        File phaseFile = imageFiles.get(0).getFile();
        prog.setFile(Phase.LOADING_FSIMAGE, phaseFile.getAbsolutePath());
        prog.setSize(Phase.LOADING_FSIMAGE, phaseFile.length());
        boolean needToSave = inspector.needToSave();
        Iterable<EditLogInputStream> editStreams = null;
        this.initEditLog();
        if (LayoutVersion.supports(LayoutVersion.Feature.TXID_BASED_LAYOUT, this.getLayoutVersion())) {
            long toAtLeastTxId = this.editLog.isOpenForWrite() ? inspector.getMaxSeenTxId() : 0L;
            editStreams = this.editLog.selectInputStreams(imageFiles.get(0).getCheckpointTxId() + 1L, toAtLeastTxId, recovery, false);
        } else {
            editStreams = FSImagePreTransactionalStorageInspector.getEditLogStreams(this.storage);
        }
        int maxOpSize = this.conf.getInt("dfs.namenode.max.op.size", 0x3200000);
        for (EditLogInputStream elis : editStreams) {
            elis.setMaxOpSize(maxOpSize);
        }
        for (EditLogInputStream l : editStreams) {
            LOG.debug((Object)("Planning to load edit log stream: " + l));
        }
        if (!editStreams.iterator().hasNext()) {
            LOG.info((Object)"No edit log streams selected.");
        }
        for (int i = 0; i < imageFiles.size(); ++i) {
            try {
                imageFile = imageFiles.get(i);
                this.loadFSImageFile(target, recovery, imageFile);
                break;
            }
            catch (IOException ioe) {
                LOG.error((Object)("Failed to load image from " + imageFile), (Throwable)ioe);
                target.clear();
                imageFile = null;
                continue;
            }
        }
        if (imageFile == null) {
            FSEditLog.closeAllStreams(editStreams);
            throw new IOException("Failed to load an FSImage file!");
        }
        prog.endPhase(Phase.LOADING_FSIMAGE);
        long txnsAdvanced = this.loadEdits(editStreams, target, recovery);
        this.editLog.setNextTxId(this.lastAppliedTxId + 1L);
        return needToSave |= this.needsResaveBasedOnStaleCheckpoint(imageFile.getFile(), txnsAdvanced);
    }

    void loadFSImageFile(FSNamesystem target, MetaRecoveryContext recovery, FSImageStorageInspector.FSImageFile imageFile) throws IOException {
        LOG.debug((Object)("Planning to load image :\n" + imageFile));
        Storage.StorageDirectory sdForProperties = imageFile.sd;
        this.storage.readProperties(sdForProperties);
        if (LayoutVersion.supports(LayoutVersion.Feature.TXID_BASED_LAYOUT, this.getLayoutVersion())) {
            this.loadFSImage(imageFile.getFile(), target, recovery);
        } else if (LayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_CHECKSUM, this.getLayoutVersion())) {
            String md5 = this.storage.getDeprecatedProperty("imageMD5Digest");
            if (md5 == null) {
                throw new InconsistentFSStateException(sdForProperties.getRoot(), "Message digest property imageMD5Digest not set for storage directory " + sdForProperties.getRoot());
            }
            this.loadFSImage(imageFile.getFile(), new MD5Hash(md5), target, recovery);
        } else {
            this.loadFSImage(imageFile.getFile(), null, target, recovery);
        }
    }

    public void initEditLog() {
        Preconditions.checkState((this.getNamespaceID() != 0 ? 1 : 0) != 0, (Object)"Must know namespace ID before initting edit log");
        String nameserviceId = DFSUtil.getNamenodeNameServiceId(this.conf);
        if (!HAUtil.isHAEnabled(this.conf, nameserviceId)) {
            this.editLog.initJournalsForWrite();
            this.editLog.recoverUnclosedStreams();
        } else {
            this.editLog.initSharedJournalsForRead();
        }
    }

    private boolean needsResaveBasedOnStaleCheckpoint(File imageFile, long numEditsLoaded) {
        long checkpointPeriod = this.conf.getLong("dfs.namenode.checkpoint.period", 3600L);
        long checkpointTxnCount = this.conf.getLong("dfs.namenode.checkpoint.txns", 1000000L);
        long checkpointAge = Time.now() - imageFile.lastModified();
        return checkpointAge > checkpointPeriod * 1000L || numEditsLoaded > checkpointTxnCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long loadEdits(Iterable<EditLogInputStream> editStreams, FSNamesystem target, MetaRecoveryContext recovery) throws IOException {
        LOG.debug((Object)("About to load edits:\n  " + Joiner.on((String)"\n  ").join(editStreams)));
        StartupProgress prog = NameNode.getStartupProgress();
        prog.beginPhase(Phase.LOADING_EDITS);
        long prevLastAppliedTxId = this.lastAppliedTxId;
        try {
            FSEditLogLoader loader = new FSEditLogLoader(target, this.lastAppliedTxId);
            for (EditLogInputStream editIn : editStreams) {
                LOG.info((Object)("Reading " + editIn + " expecting start txid #" + (this.lastAppliedTxId + 1L)));
                try {
                    loader.loadFSEdits(editIn, this.lastAppliedTxId + 1L, recovery);
                }
                finally {
                    this.lastAppliedTxId = loader.getLastAppliedTxId();
                }
                if (editIn.getLastTxId() == -12345L) continue;
                this.lastAppliedTxId = editIn.getLastTxId();
            }
        }
        finally {
            FSEditLog.closeAllStreams(editStreams);
            FSImage.updateCountForQuota(target.dir.rootDir);
        }
        prog.endPhase(Phase.LOADING_EDITS);
        return this.lastAppliedTxId - prevLastAppliedTxId;
    }

    static void updateCountForQuota(INodeDirectoryWithQuota root) {
        FSImage.updateCountForQuotaRecursively(root, Quota.Counts.newInstance());
    }

    private static void updateCountForQuotaRecursively(INodeDirectory dir, Quota.Counts counts) {
        long parentNamespace = counts.get(Quota.NAMESPACE);
        long parentDiskspace = counts.get(Quota.DISKSPACE);
        dir.computeQuotaUsage4CurrentDirectory(counts);
        for (INode child : dir.getChildrenList(null)) {
            if (child.isDirectory()) {
                FSImage.updateCountForQuotaRecursively(child.asDirectory(), counts);
                continue;
            }
            child.computeQuotaUsage(counts, false);
        }
        if (dir.isQuotaSet()) {
            long namespace = counts.get(Quota.NAMESPACE) - parentNamespace;
            if (Quota.isViolated(dir.getNsQuota(), namespace)) {
                LOG.error((Object)("BUG: Namespace quota violation in image for " + dir.getFullPathName() + " quota = " + dir.getNsQuota() + " < consumed = " + namespace));
            }
            long diskspace = counts.get(Quota.DISKSPACE) - parentDiskspace;
            if (Quota.isViolated(dir.getDsQuota(), diskspace)) {
                LOG.error((Object)("BUG: Diskspace quota violation in image for " + dir.getFullPathName() + " quota = " + dir.getDsQuota() + " < consumed = " + diskspace));
            }
            ((INodeDirectoryWithQuota)dir).setSpaceConsumed(namespace, diskspace);
        }
    }

    private void loadFSImage(File imageFile, FSNamesystem target, MetaRecoveryContext recovery) throws IOException {
        MD5Hash expectedMD5 = MD5FileUtils.readStoredMd5ForFile(imageFile);
        if (expectedMD5 == null) {
            throw new IOException("No MD5 file found corresponding to image file " + imageFile);
        }
        this.loadFSImage(imageFile, expectedMD5, target, recovery);
    }

    private void loadFSImage(File curFile, MD5Hash expectedMd5, FSNamesystem target, MetaRecoveryContext recovery) throws IOException {
        FSImageFormat.Loader loader = new FSImageFormat.Loader(this.conf, target);
        loader.load(curFile);
        target.setBlockPoolId(this.getBlockPoolID());
        MD5Hash readImageMd5 = loader.getLoadedImageMd5();
        if (expectedMd5 != null && !expectedMd5.equals(readImageMd5)) {
            throw new IOException("Image file " + curFile + " is corrupt with MD5 checksum of " + readImageMd5 + " but expecting " + expectedMd5);
        }
        long txId = loader.getLoadedImageTxId();
        LOG.info((Object)("Loaded image for txid " + txId + " from " + curFile));
        this.lastAppliedTxId = txId;
        this.storage.setMostRecentCheckpointInfo(txId, curFile.lastModified());
    }

    void saveFSImage(SaveNamespaceContext context, Storage.StorageDirectory sd) throws IOException {
        long txid = context.getTxId();
        File newFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
        File dstFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE, txid);
        FSImageFormat.Saver saver = new FSImageFormat.Saver(context);
        FSImageCompression compression = FSImageCompression.createCompression(this.conf);
        saver.save(newFile, compression);
        MD5FileUtils.saveMD5File(dstFile, saver.getSavedDigest());
        this.storage.setMostRecentCheckpointInfo(txid, Time.now());
    }

    private void waitForThreads(List<Thread> threads) {
        for (Thread thread : threads) {
            while (thread.isAlive()) {
                try {
                    thread.join();
                }
                catch (InterruptedException iex) {
                    LOG.error((Object)("Caught interrupted exception while waiting for thread " + thread.getName() + " to finish. Retrying join"));
                }
            }
        }
    }

    public synchronized void saveNamespace(FSNamesystem source) throws IOException {
        this.saveNamespace(source, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void saveNamespace(FSNamesystem source, Canceler canceler) throws IOException {
        assert (this.editLog != null) : "editLog must be initialized";
        this.storage.attemptRestoreRemovedStorage();
        boolean editLogWasOpen = this.editLog.isSegmentOpen();
        if (editLogWasOpen) {
            this.editLog.endCurrentLogSegment(true);
        }
        long imageTxId = this.getLastAppliedOrWrittenTxId();
        try {
            this.saveFSImageInAllDirs(source, imageTxId, canceler);
            this.storage.writeAll();
        }
        finally {
            if (editLogWasOpen) {
                this.editLog.startLogSegment(imageTxId + 1L, true);
                this.storage.writeTransactionIdFileToStorage(imageTxId + 1L);
            }
        }
    }

    protected synchronized void saveFSImageInAllDirs(FSNamesystem source, long txid) throws IOException {
        this.saveFSImageInAllDirs(source, txid, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected synchronized void saveFSImageInAllDirs(FSNamesystem source, long txid, Canceler canceler) throws IOException {
        StartupProgress prog = NameNode.getStartupProgress();
        prog.beginPhase(Phase.SAVING_CHECKPOINT);
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            throw new IOException("No image directories available!");
        }
        if (canceler == null) {
            canceler = new Canceler();
        }
        SaveNamespaceContext ctx = new SaveNamespaceContext(source, txid, canceler);
        try {
            ArrayList<Thread> saveThreads = new ArrayList<Thread>();
            Iterator<Storage.StorageDirectory> it = this.storage.dirIterator(NNStorage.NameNodeDirType.IMAGE);
            while (it.hasNext()) {
                Storage.StorageDirectory sd = it.next();
                FSImageSaver saver = new FSImageSaver(ctx, sd);
                Thread saveThread = new Thread((Runnable)saver, saver.toString());
                saveThreads.add(saveThread);
                saveThread.start();
            }
            this.waitForThreads(saveThreads);
            saveThreads.clear();
            this.storage.reportErrorsOnDirectories(ctx.getErrorSDs());
            if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
                throw new IOException("Failed to save in any storage directories while saving namespace.");
            }
            if (canceler.isCancelled()) {
                this.deleteCancelledCheckpoint(txid);
                ctx.checkCancelled();
                assert (false) : "should have thrown above!";
            }
            this.renameCheckpoint(txid);
            this.purgeOldStorage();
        }
        finally {
            ctx.markComplete();
            ctx = null;
        }
        prog.endPhase(Phase.SAVING_CHECKPOINT);
    }

    public void purgeOldStorage() {
        try {
            this.archivalManager.purgeOldStorage();
        }
        catch (Exception e) {
            LOG.warn((Object)"Unable to purge old storage", (Throwable)e);
        }
    }

    private void renameCheckpoint(long txid) throws IOException {
        ArrayList al = null;
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            try {
                this.renameCheckpointInDir(sd, txid);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Unable to rename checkpoint in " + sd), (Throwable)ioe);
                if (al == null) {
                    al = Lists.newArrayList();
                }
                al.add(sd);
            }
        }
        if (al != null) {
            this.storage.reportErrorsOnDirectories(al);
        }
    }

    private void deleteCancelledCheckpoint(long txid) throws IOException {
        ArrayList al = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            File ckpt = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
            if (!ckpt.exists() || ckpt.delete()) continue;
            LOG.warn((Object)("Unable to delete cancelled checkpoint in " + sd));
            al.add(sd);
        }
        this.storage.reportErrorsOnDirectories(al);
    }

    private void renameCheckpointInDir(Storage.StorageDirectory sd, long txid) throws IOException {
        File ckpt = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
        File curFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE, txid);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("renaming  " + ckpt.getAbsolutePath() + " to " + curFile.getAbsolutePath()));
        }
        if (!(ckpt.renameTo(curFile) || curFile.delete() && ckpt.renameTo(curFile))) {
            throw new IOException("renaming  " + ckpt.getAbsolutePath() + " to " + curFile.getAbsolutePath() + " FAILED");
        }
    }

    CheckpointSignature rollEditLog() throws IOException {
        this.getEditLog().rollEditLog();
        this.storage.writeTransactionIdFileToStorage(this.getEditLog().getCurSegmentTxId());
        return new CheckpointSignature(this);
    }

    NamenodeCommand startCheckpoint(NamenodeRegistration bnReg, NamenodeRegistration nnReg) throws IOException {
        LOG.info((Object)("Start checkpoint at txid " + this.getEditLog().getLastWrittenTxId()));
        String msg = null;
        if (bnReg.getNamespaceID() != this.storage.getNamespaceID()) {
            msg = "Name node " + bnReg.getAddress() + " has incompatible namespace id: " + bnReg.getNamespaceID() + " expected: " + this.storage.getNamespaceID();
        } else if (bnReg.isRole(HdfsServerConstants.NamenodeRole.NAMENODE)) {
            msg = "Name node " + bnReg.getAddress() + " role " + (Object)((Object)bnReg.getRole()) + ": checkpoint is not allowed.";
        } else if (bnReg.getLayoutVersion() < this.storage.getLayoutVersion() || bnReg.getLayoutVersion() == this.storage.getLayoutVersion() && bnReg.getCTime() > this.storage.getCTime()) {
            msg = "Name node " + bnReg.getAddress() + " has newer image layout version: LV = " + bnReg.getLayoutVersion() + " cTime = " + bnReg.getCTime() + ". Current version: LV = " + this.storage.getLayoutVersion() + " cTime = " + this.storage.getCTime();
        }
        if (msg != null) {
            LOG.error((Object)msg);
            return new NamenodeCommand(50);
        }
        boolean needToReturnImg = true;
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            needToReturnImg = false;
        }
        CheckpointSignature sig = this.rollEditLog();
        return new CheckpointCommand(sig, needToReturnImg);
    }

    void endCheckpoint(CheckpointSignature sig) throws IOException {
        LOG.info((Object)("End checkpoint at txid " + this.getEditLog().getLastWrittenTxId()));
        sig.validateStorageInfo(this);
    }

    public synchronized void saveDigestAndRenameCheckpointImage(long txid, MD5Hash digest) throws IOException {
        ArrayList badSds = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            File imageFile = NNStorage.getImageFile(sd, txid);
            try {
                MD5FileUtils.saveMD5File(imageFile, digest);
            }
            catch (IOException ioe) {
                badSds.add(sd);
            }
        }
        this.storage.reportErrorsOnDirectories(badSds);
        CheckpointFaultInjector.getInstance().afterMD5Rename();
        this.renameCheckpoint(txid);
        if (txid > this.storage.getMostRecentCheckpointTxId()) {
            this.storage.setMostRecentCheckpointInfo(txid, Time.now());
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.editLog != null) {
            this.getEditLog().close();
        }
        this.storage.close();
    }

    static Collection<URI> getCheckpointDirs(Configuration conf, String defaultValue) {
        Collection<String> dirNames = conf.getTrimmedStringCollection("dfs.namenode.checkpoint.dir");
        if (dirNames.size() == 0 && defaultValue != null) {
            dirNames.add(defaultValue);
        }
        return Util.stringCollectionAsURIs(dirNames);
    }

    static List<URI> getCheckpointEditsDirs(Configuration conf, String defaultName) {
        Collection<String> dirNames = conf.getTrimmedStringCollection("dfs.namenode.checkpoint.edits.dir");
        if (dirNames.size() == 0 && defaultName != null) {
            dirNames.add(defaultName);
        }
        return Util.stringCollectionAsURIs(dirNames);
    }

    public NNStorage getStorage() {
        return this.storage;
    }

    public int getLayoutVersion() {
        return this.storage.getLayoutVersion();
    }

    public int getNamespaceID() {
        return this.storage.getNamespaceID();
    }

    public String getClusterID() {
        return this.storage.getClusterID();
    }

    public String getBlockPoolID() {
        return this.storage.getBlockPoolID();
    }

    public synchronized long getLastAppliedTxId() {
        return this.lastAppliedTxId;
    }

    public long getLastAppliedOrWrittenTxId() {
        return Math.max(this.lastAppliedTxId, this.editLog != null ? this.editLog.getLastWrittenTxId() : 0L);
    }

    public void updateLastAppliedTxIdFromWritten() {
        this.lastAppliedTxId = this.editLog.getLastWrittenTxId();
    }

    public synchronized long getMostRecentCheckpointTxId() {
        return this.storage.getMostRecentCheckpointTxId();
    }

    private class FSImageSaver
    implements Runnable {
        private final SaveNamespaceContext context;
        private Storage.StorageDirectory sd;

        public FSImageSaver(SaveNamespaceContext context, Storage.StorageDirectory sd) {
            this.context = context;
            this.sd = sd;
        }

        @Override
        public void run() {
            try {
                FSImage.this.saveFSImage(this.context, this.sd);
            }
            catch (SaveNamespaceCancelledException snce) {
                LOG.info((Object)("Cancelled image saving for " + this.sd.getRoot() + ": " + snce.getMessage()));
            }
            catch (Throwable t) {
                LOG.error((Object)("Unable to save image for " + this.sd.getRoot()), t);
                this.context.reportErrorOnStorageDirectory(this.sd);
            }
        }

        public String toString() {
            return "FSImageSaver for " + this.sd.getRoot() + " of type " + this.sd.getStorageDirType();
        }
    }
}

