/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration information for a certificate revocation list (CRL). Your private certificate authority (CA)
 * creates base CRLs. Delta CRLs are not supported. You can enable CRLs for your new or an existing private CA by
 * setting the <b>Enabled</b> parameter to <code>true</code>. Your private CA writes CRLs to an S3 bucket that you
 * specify in the <b>S3BucketName</b> parameter. You can hide the name of your bucket by specifying a value for the
 * <b>CustomCname</b> parameter. Your private CA by default copies the CNAME or the S3 bucket name to the <b>CRL
 * Distribution Points</b> extension of each certificate it issues. If you want to configure this default behavior to be
 * something different, you can set the <b>CrlDistributionPointExtensionConfiguration</b> parameter. Your S3 bucket
 * policy must give write permission to Amazon Web Services Private CA.
 * </p>
 * <p>
 * Amazon Web Services Private CA assets that are stored in Amazon S3 can be protected with encryption. For more
 * information, see <a
 * href="https://docs.aws.amazon.com/privateca/latest/userguide/crl-planning.html#crl-encryption">Encrypting Your
 * CRLs</a>.
 * </p>
 * <p>
 * Your private CA uses the value in the <b>ExpirationInDays</b> parameter to calculate the <b>nextUpdate</b> field in
 * the CRL. The CRL is refreshed prior to a certificate's expiration date or when a certificate is revoked. When a
 * certificate is revoked, it appears in the CRL until the certificate expires, and then in one additional CRL after
 * expiration, and it always appears in the audit report.
 * </p>
 * <p>
 * A CRL is typically updated approximately 30 minutes after a certificate is revoked. If for any reason a CRL update
 * fails, Amazon Web Services Private CA makes further attempts every 15 minutes.
 * </p>
 * <p>
 * CRLs contain the following fields:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Version</b>: The current version number defined in RFC 5280 is V2. The integer value is 0x1.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Signature Algorithm</b>: The name of the algorithm used to sign the CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Issuer</b>: The X.500 distinguished name of your private CA that issued the CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Last Update</b>: The issue date and time of this CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Next Update</b>: The day and time by which the next CRL will be issued.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Revoked Certificates</b>: List of revoked certificates. Each list item contains the following information.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Serial Number</b>: The serial number, in hexadecimal format, of the revoked certificate.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Revocation Date</b>: Date and time the certificate was revoked.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>CRL Entry Extensions</b>: Optional extensions for the CRL entry.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>X509v3 CRL Reason Code</b>: Reason the certificate was revoked.
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>CRL Extensions</b>: Optional extensions for the CRL.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>X509v3 Authority Key Identifier</b>: Identifies the public key associated with the private key used to sign the
 * certificate.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>X509v3 CRL Number:</b>: Decimal sequence number for the CRL.
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <b>Signature Algorithm</b>: Algorithm used by your private CA to sign the CRL.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Signature Value</b>: Signature computed over the CRL.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Certificate revocation lists created by Amazon Web Services Private CA are DER-encoded. You can use the following
 * OpenSSL command to list a CRL.
 * </p>
 * <p>
 * <code>openssl crl -inform DER -text -in <i>crl_path</i> -noout</code>
 * </p>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/privateca/latest/userguide/crl-planning.html">Planning
 * a certificate revocation list (CRL)</a> in the <i>Amazon Web Services Private Certificate Authority User Guide</i>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CrlConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CrlConfiguration.Builder, CrlConfiguration> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(CrlConfiguration::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<Integer> EXPIRATION_IN_DAYS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExpirationInDays").getter(getter(CrlConfiguration::expirationInDays))
            .setter(setter(Builder::expirationInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationInDays").build()).build();

    private static final SdkField<String> CUSTOM_CNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomCname").getter(getter(CrlConfiguration::customCname)).setter(setter(Builder::customCname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomCname").build()).build();

    private static final SdkField<String> S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3BucketName").getter(getter(CrlConfiguration::s3BucketName)).setter(setter(Builder::s3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3BucketName").build()).build();

    private static final SdkField<String> S3_OBJECT_ACL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3ObjectAcl").getter(getter(CrlConfiguration::s3ObjectAclAsString)).setter(setter(Builder::s3ObjectAcl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3ObjectAcl").build()).build();

    private static final SdkField<CrlDistributionPointExtensionConfiguration> CRL_DISTRIBUTION_POINT_EXTENSION_CONFIGURATION_FIELD = SdkField
            .<CrlDistributionPointExtensionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("CrlDistributionPointExtensionConfiguration")
            .getter(getter(CrlConfiguration::crlDistributionPointExtensionConfiguration))
            .setter(setter(Builder::crlDistributionPointExtensionConfiguration))
            .constructor(CrlDistributionPointExtensionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CrlDistributionPointExtensionConfiguration").build()).build();

    private static final SdkField<String> CRL_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CrlType").getter(getter(CrlConfiguration::crlTypeAsString)).setter(setter(Builder::crlType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrlType").build()).build();

    private static final SdkField<String> CUSTOM_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomPath").getter(getter(CrlConfiguration::customPath)).setter(setter(Builder::customPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomPath").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD,
            EXPIRATION_IN_DAYS_FIELD, CUSTOM_CNAME_FIELD, S3_BUCKET_NAME_FIELD, S3_OBJECT_ACL_FIELD,
            CRL_DISTRIBUTION_POINT_EXTENSION_CONFIGURATION_FIELD, CRL_TYPE_FIELD, CUSTOM_PATH_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final Integer expirationInDays;

    private final String customCname;

    private final String s3BucketName;

    private final String s3ObjectAcl;

    private final CrlDistributionPointExtensionConfiguration crlDistributionPointExtensionConfiguration;

    private final String crlType;

    private final String customPath;

    private CrlConfiguration(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.expirationInDays = builder.expirationInDays;
        this.customCname = builder.customCname;
        this.s3BucketName = builder.s3BucketName;
        this.s3ObjectAcl = builder.s3ObjectAcl;
        this.crlDistributionPointExtensionConfiguration = builder.crlDistributionPointExtensionConfiguration;
        this.crlType = builder.crlType;
        this.customPath = builder.customPath;
    }

    /**
     * <p>
     * Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this value to
     * enable certificate revocation for a new CA when you call the <a
     * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
     * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
     * >UpdateCertificateAuthority</a> action.
     * </p>
     * 
     * @return Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this
     *         value to enable certificate revocation for a new CA when you call the <a
     *         href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html"
     *         >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
     *         href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
     *         >UpdateCertificateAuthority</a> action.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Validity period of the CRL in days.
     * </p>
     * 
     * @return Validity period of the CRL in days.
     */
    public final Integer expirationInDays() {
        return expirationInDays;
    }

    /**
     * <p>
     * Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an alias for
     * the CRL distribution point. Use this value if you don't want the name of your S3 bucket to be public.
     * </p>
     * <note>
     * <p>
     * The content of a Canonical Name (CNAME) record must conform to <a
     * href="https://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> restrictions on the use of special characters in URIs.
     * Additionally, the value of the CNAME must not include a protocol prefix such as "http://" or "https://".
     * </p>
     * </note>
     * 
     * @return Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an
     *         alias for the CRL distribution point. Use this value if you don't want the name of your S3 bucket to be
     *         public.</p> <note>
     *         <p>
     *         The content of a Canonical Name (CNAME) record must conform to <a
     *         href="https://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> restrictions on the use of special characters in
     *         URIs. Additionally, the value of the CNAME must not include a protocol prefix such as "http://" or
     *         "https://".
     *         </p>
     */
    public final String customCname() {
        return customCname;
    }

    /**
     * <p>
     * Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b> argument,
     * the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of the issued certificate.
     * You can change the name of your bucket by calling the <a
     * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
     * >UpdateCertificateAuthority</a> operation. You must specify a <a
     * href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-policies">bucket policy</a> that
     * allows Amazon Web Services Private CA to write the CRL to your bucket.
     * </p>
     * <note>
     * <p>
     * The <code>S3BucketName</code> parameter must conform to the <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html">S3 bucket naming rules</a>.
     * </p>
     * </note>
     * 
     * @return Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b>
     *         argument, the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of the
     *         issued certificate. You can change the name of your bucket by calling the <a
     *         href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
     *         >UpdateCertificateAuthority</a> operation. You must specify a <a
     *         href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-policies">bucket
     *         policy</a> that allows Amazon Web Services Private CA to write the CRL to your bucket.</p> <note>
     *         <p>
     *         The <code>S3BucketName</code> parameter must conform to the <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html">S3 bucket naming
     *         rules</a>.
     *         </p>
     */
    public final String s3BucketName() {
        return s3BucketName;
    }

    /**
     * <p>
     * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you choose
     * PUBLIC_READ, the CRL will be accessible over the public internet. If you choose BUCKET_OWNER_FULL_CONTROL, only
     * the owner of the CRL S3 bucket can access the CRL, and your PKI clients may need an alternative method of access.
     * </p>
     * <p>
     * If no value is specified, the default is <code>PUBLIC_READ</code>.
     * </p>
     * <p>
     * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the Block
     * Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
     * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
     * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access to
     * the S3 bucket</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3ObjectAcl} will
     * return {@link S3ObjectAcl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3ObjectAclAsString}.
     * </p>
     * 
     * @return Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
     *         you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
     *         BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients
     *         may need an alternative method of access. </p>
     *         <p>
     *         If no value is specified, the default is <code>PUBLIC_READ</code>.
     *         </p>
     *         <p>
     *         <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled
     *         the Block Public Access (BPA) feature in your S3 account, then you must specify the value of this
     *         parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have
     *         disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
     *         <code>PUBLIC_READ</code> as the value.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
     *         access to the S3 bucket</a>.
     * @see S3ObjectAcl
     */
    public final S3ObjectAcl s3ObjectAcl() {
        return S3ObjectAcl.fromValue(s3ObjectAcl);
    }

    /**
     * <p>
     * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you choose
     * PUBLIC_READ, the CRL will be accessible over the public internet. If you choose BUCKET_OWNER_FULL_CONTROL, only
     * the owner of the CRL S3 bucket can access the CRL, and your PKI clients may need an alternative method of access.
     * </p>
     * <p>
     * If no value is specified, the default is <code>PUBLIC_READ</code>.
     * </p>
     * <p>
     * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the Block
     * Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
     * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
     * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access to
     * the S3 bucket</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3ObjectAcl} will
     * return {@link S3ObjectAcl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3ObjectAclAsString}.
     * </p>
     * 
     * @return Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
     *         you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
     *         BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients
     *         may need an alternative method of access. </p>
     *         <p>
     *         If no value is specified, the default is <code>PUBLIC_READ</code>.
     *         </p>
     *         <p>
     *         <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled
     *         the Block Public Access (BPA) feature in your S3 account, then you must specify the value of this
     *         parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have
     *         disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
     *         <code>PUBLIC_READ</code> as the value.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
     *         access to the S3 bucket</a>.
     * @see S3ObjectAcl
     */
    public final String s3ObjectAclAsString() {
        return s3ObjectAcl;
    }

    /**
     * <p>
     * Configures the behavior of the CRL Distribution Point extension for certificates issued by your certificate
     * authority. If this field is not provided, then the CRl Distribution Point Extension will be present and contain
     * the default CRL URL.
     * </p>
     * 
     * @return Configures the behavior of the CRL Distribution Point extension for certificates issued by your
     *         certificate authority. If this field is not provided, then the CRl Distribution Point Extension will be
     *         present and contain the default CRL URL.
     */
    public final CrlDistributionPointExtensionConfiguration crlDistributionPointExtensionConfiguration() {
        return crlDistributionPointExtensionConfiguration;
    }

    /**
     * <p>
     * Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number of
     * certificates that the certificate authority can issue and revoke. For more information, see <a
     * href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle for all
     * unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate that Amazon Web
     * Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution point (CDP) extension, deﬁned
     * in <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC 5280</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the number of
     * certiﬁcates your private CA can issue.
     * </p>
     * <important>
     * <p>
     * When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point (IDP) URI
     * matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web Services Private CA marks
     * the IDP extension as critical, which your client must be able to process.
     * </p>
     * </important></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #crlType} will
     * return {@link CrlType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #crlTypeAsString}.
     * </p>
     * 
     * @return Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number of
     *         certificates that the certificate authority can issue and revoke. For more information, see <a
     *         href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle
     *         for all unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate that
     *         Amazon Web Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution point (CDP)
     *         extension, deﬁned in <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC
     *         5280</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the number
     *         of certiﬁcates your private CA can issue.
     *         </p>
     *         <important>
     *         <p>
     *         When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point (IDP)
     *         URI matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web Services
     *         Private CA marks the IDP extension as critical, which your client must be able to process.
     *         </p>
     *         </important></li>
     * @see CrlType
     */
    public final CrlType crlType() {
        return CrlType.fromValue(crlType);
    }

    /**
     * <p>
     * Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number of
     * certificates that the certificate authority can issue and revoke. For more information, see <a
     * href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle for all
     * unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate that Amazon Web
     * Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution point (CDP) extension, deﬁned
     * in <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC 5280</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the number of
     * certiﬁcates your private CA can issue.
     * </p>
     * <important>
     * <p>
     * When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point (IDP) URI
     * matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web Services Private CA marks
     * the IDP extension as critical, which your client must be able to process.
     * </p>
     * </important></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #crlType} will
     * return {@link CrlType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #crlTypeAsString}.
     * </p>
     * 
     * @return Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number of
     *         certificates that the certificate authority can issue and revoke. For more information, see <a
     *         href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle
     *         for all unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate that
     *         Amazon Web Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution point (CDP)
     *         extension, deﬁned in <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC
     *         5280</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the number
     *         of certiﬁcates your private CA can issue.
     *         </p>
     *         <important>
     *         <p>
     *         When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point (IDP)
     *         URI matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web Services
     *         Private CA marks the IDP extension as critical, which your client must be able to process.
     *         </p>
     *         </important></li>
     * @see CrlType
     */
    public final String crlTypeAsString() {
        return crlType;
    }

    /**
     * <p>
     * Designates a custom ﬁle path in S3 for CRL(s). For example,
     * <code>http://&lt;CustomName&gt;/ &lt;CustomPath&gt;/&lt;CrlPartition_GUID&gt;.crl</code>.
     * </p>
     * 
     * @return Designates a custom ﬁle path in S3 for CRL(s). For example,
     *         <code>http://&lt;CustomName&gt;/ &lt;CustomPath&gt;/&lt;CrlPartition_GUID&gt;.crl</code>.
     */
    public final String customPath() {
        return customPath;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(expirationInDays());
        hashCode = 31 * hashCode + Objects.hashCode(customCname());
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(s3ObjectAclAsString());
        hashCode = 31 * hashCode + Objects.hashCode(crlDistributionPointExtensionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(crlTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customPath());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CrlConfiguration)) {
            return false;
        }
        CrlConfiguration other = (CrlConfiguration) obj;
        return Objects.equals(enabled(), other.enabled())
                && Objects.equals(expirationInDays(), other.expirationInDays())
                && Objects.equals(customCname(), other.customCname())
                && Objects.equals(s3BucketName(), other.s3BucketName())
                && Objects.equals(s3ObjectAclAsString(), other.s3ObjectAclAsString())
                && Objects.equals(crlDistributionPointExtensionConfiguration(),
                        other.crlDistributionPointExtensionConfiguration())
                && Objects.equals(crlTypeAsString(), other.crlTypeAsString()) && Objects.equals(customPath(), other.customPath());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CrlConfiguration").add("Enabled", enabled()).add("ExpirationInDays", expirationInDays())
                .add("CustomCname", customCname()).add("S3BucketName", s3BucketName()).add("S3ObjectAcl", s3ObjectAclAsString())
                .add("CrlDistributionPointExtensionConfiguration", crlDistributionPointExtensionConfiguration())
                .add("CrlType", crlTypeAsString()).add("CustomPath", customPath()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "ExpirationInDays":
            return Optional.ofNullable(clazz.cast(expirationInDays()));
        case "CustomCname":
            return Optional.ofNullable(clazz.cast(customCname()));
        case "S3BucketName":
            return Optional.ofNullable(clazz.cast(s3BucketName()));
        case "S3ObjectAcl":
            return Optional.ofNullable(clazz.cast(s3ObjectAclAsString()));
        case "CrlDistributionPointExtensionConfiguration":
            return Optional.ofNullable(clazz.cast(crlDistributionPointExtensionConfiguration()));
        case "CrlType":
            return Optional.ofNullable(clazz.cast(crlTypeAsString()));
        case "CustomPath":
            return Optional.ofNullable(clazz.cast(customPath()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Enabled", ENABLED_FIELD);
        map.put("ExpirationInDays", EXPIRATION_IN_DAYS_FIELD);
        map.put("CustomCname", CUSTOM_CNAME_FIELD);
        map.put("S3BucketName", S3_BUCKET_NAME_FIELD);
        map.put("S3ObjectAcl", S3_OBJECT_ACL_FIELD);
        map.put("CrlDistributionPointExtensionConfiguration", CRL_DISTRIBUTION_POINT_EXTENSION_CONFIGURATION_FIELD);
        map.put("CrlType", CRL_TYPE_FIELD);
        map.put("CustomPath", CUSTOM_PATH_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CrlConfiguration, T> g) {
        return obj -> g.apply((CrlConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CrlConfiguration> {
        /**
         * <p>
         * Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this value
         * to enable certificate revocation for a new CA when you call the <a
         * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html"
         * >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
         * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
         * >UpdateCertificateAuthority</a> action.
         * </p>
         * 
         * @param enabled
         *        Boolean value that specifies whether certificate revocation lists (CRLs) are enabled. You can use this
         *        value to enable certificate revocation for a new CA when you call the <a
         *        href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html"
         *        >CreateCertificateAuthority</a> action or for an existing CA when you call the <a
         *        href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
         *        >UpdateCertificateAuthority</a> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Validity period of the CRL in days.
         * </p>
         * 
         * @param expirationInDays
         *        Validity period of the CRL in days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationInDays(Integer expirationInDays);

        /**
         * <p>
         * Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an alias
         * for the CRL distribution point. Use this value if you don't want the name of your S3 bucket to be public.
         * </p>
         * <note>
         * <p>
         * The content of a Canonical Name (CNAME) record must conform to <a
         * href="https://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> restrictions on the use of special characters in
         * URIs. Additionally, the value of the CNAME must not include a protocol prefix such as "http://" or
         * "https://".
         * </p>
         * </note>
         * 
         * @param customCname
         *        Name inserted into the certificate <b>CRL Distribution Points</b> extension that enables the use of an
         *        alias for the CRL distribution point. Use this value if you don't want the name of your S3 bucket to
         *        be public.</p> <note>
         *        <p>
         *        The content of a Canonical Name (CNAME) record must conform to <a
         *        href="https://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> restrictions on the use of special characters
         *        in URIs. Additionally, the value of the CNAME must not include a protocol prefix such as "http://" or
         *        "https://".
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customCname(String customCname);

        /**
         * <p>
         * Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b>
         * argument, the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of the
         * issued certificate. You can change the name of your bucket by calling the <a
         * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
         * >UpdateCertificateAuthority</a> operation. You must specify a <a
         * href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-policies">bucket policy</a>
         * that allows Amazon Web Services Private CA to write the CRL to your bucket.
         * </p>
         * <note>
         * <p>
         * The <code>S3BucketName</code> parameter must conform to the <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html">S3 bucket naming
         * rules</a>.
         * </p>
         * </note>
         * 
         * @param s3BucketName
         *        Name of the S3 bucket that contains the CRL. If you do not provide a value for the <b>CustomCname</b>
         *        argument, the name of your S3 bucket is placed into the <b>CRL Distribution Points</b> extension of
         *        the issued certificate. You can change the name of your bucket by calling the <a
         *        href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html"
         *        >UpdateCertificateAuthority</a> operation. You must specify a <a
         *        href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-policies">bucket
         *        policy</a> that allows Amazon Web Services Private CA to write the CRL to your bucket.</p> <note>
         *        <p>
         *        The <code>S3BucketName</code> parameter must conform to the <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/bucketnamingrules.html">S3 bucket naming
         *        rules</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketName(String s3BucketName);

        /**
         * <p>
         * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you
         * choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         * BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients may
         * need an alternative method of access.
         * </p>
         * <p>
         * If no value is specified, the default is <code>PUBLIC_READ</code>.
         * </p>
         * <p>
         * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the
         * Block Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
         * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
         * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access
         * to the S3 bucket</a>.
         * </p>
         * 
         * @param s3ObjectAcl
         *        Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
         *        you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         *        BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI
         *        clients may need an alternative method of access. </p>
         *        <p>
         *        If no value is specified, the default is <code>PUBLIC_READ</code>.
         *        </p>
         *        <p>
         *        <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have
         *        enabled the Block Public Access (BPA) feature in your S3 account, then you must specify the value of
         *        this parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you
         *        have disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
         *        <code>PUBLIC_READ</code> as the value.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
         *        access to the S3 bucket</a>.
         * @see S3ObjectAcl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectAcl
         */
        Builder s3ObjectAcl(String s3ObjectAcl);

        /**
         * <p>
         * Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If you
         * choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         * BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI clients may
         * need an alternative method of access.
         * </p>
         * <p>
         * If no value is specified, the default is <code>PUBLIC_READ</code>.
         * </p>
         * <p>
         * <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have enabled the
         * Block Public Access (BPA) feature in your S3 account, then you must specify the value of this parameter as
         * <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you have disabled BPA in S3,
         * then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or <code>PUBLIC_READ</code> as the value.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public access
         * to the S3 bucket</a>.
         * </p>
         * 
         * @param s3ObjectAcl
         *        Determines whether the CRL will be publicly readable or privately held in the CRL Amazon S3 bucket. If
         *        you choose PUBLIC_READ, the CRL will be accessible over the public internet. If you choose
         *        BUCKET_OWNER_FULL_CONTROL, only the owner of the CRL S3 bucket can access the CRL, and your PKI
         *        clients may need an alternative method of access. </p>
         *        <p>
         *        If no value is specified, the default is <code>PUBLIC_READ</code>.
         *        </p>
         *        <p>
         *        <i>Note:</i> This default can cause CA creation to fail in some circumstances. If you have have
         *        enabled the Block Public Access (BPA) feature in your S3 account, then you must specify the value of
         *        this parameter as <code>BUCKET_OWNER_FULL_CONTROL</code>, and not doing so results in an error. If you
         *        have disabled BPA in S3, then you can specify either <code>BUCKET_OWNER_FULL_CONTROL</code> or
         *        <code>PUBLIC_READ</code> as the value.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/privateca/latest/userguide/PcaCreateCa.html#s3-bpa">Blocking public
         *        access to the S3 bucket</a>.
         * @see S3ObjectAcl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectAcl
         */
        Builder s3ObjectAcl(S3ObjectAcl s3ObjectAcl);

        /**
         * <p>
         * Configures the behavior of the CRL Distribution Point extension for certificates issued by your certificate
         * authority. If this field is not provided, then the CRl Distribution Point Extension will be present and
         * contain the default CRL URL.
         * </p>
         * 
         * @param crlDistributionPointExtensionConfiguration
         *        Configures the behavior of the CRL Distribution Point extension for certificates issued by your
         *        certificate authority. If this field is not provided, then the CRl Distribution Point Extension will
         *        be present and contain the default CRL URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crlDistributionPointExtensionConfiguration(
                CrlDistributionPointExtensionConfiguration crlDistributionPointExtensionConfiguration);

        /**
         * <p>
         * Configures the behavior of the CRL Distribution Point extension for certificates issued by your certificate
         * authority. If this field is not provided, then the CRl Distribution Point Extension will be present and
         * contain the default CRL URL.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CrlDistributionPointExtensionConfiguration.Builder} avoiding the need to create one manually via
         * {@link CrlDistributionPointExtensionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CrlDistributionPointExtensionConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #crlDistributionPointExtensionConfiguration(CrlDistributionPointExtensionConfiguration)}.
         * 
         * @param crlDistributionPointExtensionConfiguration
         *        a consumer that will call methods on {@link CrlDistributionPointExtensionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #crlDistributionPointExtensionConfiguration(CrlDistributionPointExtensionConfiguration)
         */
        default Builder crlDistributionPointExtensionConfiguration(
                Consumer<CrlDistributionPointExtensionConfiguration.Builder> crlDistributionPointExtensionConfiguration) {
            return crlDistributionPointExtensionConfiguration(CrlDistributionPointExtensionConfiguration.builder()
                    .applyMutation(crlDistributionPointExtensionConfiguration).build());
        }

        /**
         * <p>
         * Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number of
         * certificates that the certificate authority can issue and revoke. For more information, see <a
         * href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle for
         * all unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate that Amazon
         * Web Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution point (CDP) extension,
         * deﬁned in <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC 5280</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the number of
         * certiﬁcates your private CA can issue.
         * </p>
         * <important>
         * <p>
         * When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point (IDP) URI
         * matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web Services Private CA
         * marks the IDP extension as critical, which your client must be able to process.
         * </p>
         * </important></li>
         * </ul>
         * 
         * @param crlType
         *        Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number
         *        of certificates that the certificate authority can issue and revoke. For more information, see <a
         *        href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle
         *        for all unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate
         *        that Amazon Web Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution
         *        point (CDP) extension, deﬁned in <a
         *        href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC 5280</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the
         *        number of certiﬁcates your private CA can issue.
         *        </p>
         *        <important>
         *        <p>
         *        When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point
         *        (IDP) URI matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web
         *        Services Private CA marks the IDP extension as critical, which your client must be able to process.
         *        </p>
         *        </important></li>
         * @see CrlType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CrlType
         */
        Builder crlType(String crlType);

        /**
         * <p>
         * Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number of
         * certificates that the certificate authority can issue and revoke. For more information, see <a
         * href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle for
         * all unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate that Amazon
         * Web Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution point (CDP) extension,
         * deﬁned in <a href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC 5280</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the number of
         * certiﬁcates your private CA can issue.
         * </p>
         * <important>
         * <p>
         * When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point (IDP) URI
         * matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web Services Private CA
         * marks the IDP extension as critical, which your client must be able to process.
         * </p>
         * </important></li>
         * </ul>
         * 
         * @param crlType
         *        Specifies whether to create a complete or partitioned CRL. This setting determines the maximum number
         *        of certificates that the certificate authority can issue and revoke. For more information, see <a
         *        href="privateca/latest/userguide/pca.html#limits_pca">Amazon Web Services Private CA quotas</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPLETE</code> - The default setting. Amazon Web Services Private CA maintains a single CRL ﬁle
         *        for all unexpired certiﬁcates issued by a CA that have been revoked for any reason. Each certiﬁcate
         *        that Amazon Web Services Private CA issues is bound to a speciﬁc CRL through its CRL distribution
         *        point (CDP) extension, deﬁned in <a
         *        href="https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9"> RFC 5280</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PARTITIONED</code> - Compared to complete CRLs, partitioned CRLs dramatically increase the
         *        number of certiﬁcates your private CA can issue.
         *        </p>
         *        <important>
         *        <p>
         *        When using partitioned CRLs, you must validate that the CRL's associated issuing distribution point
         *        (IDP) URI matches the certiﬁcate's CDP URI to ensure the right CRL has been fetched. Amazon Web
         *        Services Private CA marks the IDP extension as critical, which your client must be able to process.
         *        </p>
         *        </important></li>
         * @see CrlType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CrlType
         */
        Builder crlType(CrlType crlType);

        /**
         * <p>
         * Designates a custom ﬁle path in S3 for CRL(s). For example,
         * <code>http://&lt;CustomName&gt;/ &lt;CustomPath&gt;/&lt;CrlPartition_GUID&gt;.crl</code>.
         * </p>
         * 
         * @param customPath
         *        Designates a custom ﬁle path in S3 for CRL(s). For example,
         *        <code>http://&lt;CustomName&gt;/ &lt;CustomPath&gt;/&lt;CrlPartition_GUID&gt;.crl</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customPath(String customPath);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private Integer expirationInDays;

        private String customCname;

        private String s3BucketName;

        private String s3ObjectAcl;

        private CrlDistributionPointExtensionConfiguration crlDistributionPointExtensionConfiguration;

        private String crlType;

        private String customPath;

        private BuilderImpl() {
        }

        private BuilderImpl(CrlConfiguration model) {
            enabled(model.enabled);
            expirationInDays(model.expirationInDays);
            customCname(model.customCname);
            s3BucketName(model.s3BucketName);
            s3ObjectAcl(model.s3ObjectAcl);
            crlDistributionPointExtensionConfiguration(model.crlDistributionPointExtensionConfiguration);
            crlType(model.crlType);
            customPath(model.customPath);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Integer getExpirationInDays() {
            return expirationInDays;
        }

        public final void setExpirationInDays(Integer expirationInDays) {
            this.expirationInDays = expirationInDays;
        }

        @Override
        public final Builder expirationInDays(Integer expirationInDays) {
            this.expirationInDays = expirationInDays;
            return this;
        }

        public final String getCustomCname() {
            return customCname;
        }

        public final void setCustomCname(String customCname) {
            this.customCname = customCname;
        }

        @Override
        public final Builder customCname(String customCname) {
            this.customCname = customCname;
            return this;
        }

        public final String getS3BucketName() {
            return s3BucketName;
        }

        public final void setS3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
        }

        @Override
        public final Builder s3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
            return this;
        }

        public final String getS3ObjectAcl() {
            return s3ObjectAcl;
        }

        public final void setS3ObjectAcl(String s3ObjectAcl) {
            this.s3ObjectAcl = s3ObjectAcl;
        }

        @Override
        public final Builder s3ObjectAcl(String s3ObjectAcl) {
            this.s3ObjectAcl = s3ObjectAcl;
            return this;
        }

        @Override
        public final Builder s3ObjectAcl(S3ObjectAcl s3ObjectAcl) {
            this.s3ObjectAcl(s3ObjectAcl == null ? null : s3ObjectAcl.toString());
            return this;
        }

        public final CrlDistributionPointExtensionConfiguration.Builder getCrlDistributionPointExtensionConfiguration() {
            return crlDistributionPointExtensionConfiguration != null ? crlDistributionPointExtensionConfiguration.toBuilder()
                    : null;
        }

        public final void setCrlDistributionPointExtensionConfiguration(
                CrlDistributionPointExtensionConfiguration.BuilderImpl crlDistributionPointExtensionConfiguration) {
            this.crlDistributionPointExtensionConfiguration = crlDistributionPointExtensionConfiguration != null ? crlDistributionPointExtensionConfiguration
                    .build() : null;
        }

        @Override
        public final Builder crlDistributionPointExtensionConfiguration(
                CrlDistributionPointExtensionConfiguration crlDistributionPointExtensionConfiguration) {
            this.crlDistributionPointExtensionConfiguration = crlDistributionPointExtensionConfiguration;
            return this;
        }

        public final String getCrlType() {
            return crlType;
        }

        public final void setCrlType(String crlType) {
            this.crlType = crlType;
        }

        @Override
        public final Builder crlType(String crlType) {
            this.crlType = crlType;
            return this;
        }

        @Override
        public final Builder crlType(CrlType crlType) {
            this.crlType(crlType == null ? null : crlType.toString());
            return this;
        }

        public final String getCustomPath() {
            return customPath;
        }

        public final void setCustomPath(String customPath) {
            this.customPath = customPath;
        }

        @Override
        public final Builder customPath(String customPath) {
            this.customPath = customPath;
            return this;
        }

        @Override
        public CrlConfiguration build() {
            return new CrlConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
