/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the cache for the build project.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectCache implements SdkPojo, Serializable, ToCopyableBuilder<ProjectCache.Builder, ProjectCache> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ProjectCache::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("location").getter(getter(ProjectCache::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<List<String>> MODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("modes")
            .getter(getter(ProjectCache::modesAsStrings))
            .setter(setter(Builder::modesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, LOCATION_FIELD,
            MODES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String location;

    private final List<String> modes;

    private ProjectCache(BuilderImpl builder) {
        this.type = builder.type;
        this.location = builder.location;
        this.modes = builder.modes;
    }

    /**
     * <p>
     * The type of cache used by the build project. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NO_CACHE</code>: The build project does not use any cache.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The build project reads and writes from and to S3.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that build
     * host.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of cache used by the build project. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NO_CACHE</code>: The build project does not use any cache.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The build project reads and writes from and to S3.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
     *         that build host.
     *         </p>
     *         </li>
     * @see CacheType
     */
    public final CacheType type() {
        return CacheType.fromValue(type);
    }

    /**
     * <p>
     * The type of cache used by the build project. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NO_CACHE</code>: The build project does not use any cache.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The build project reads and writes from and to S3.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that build
     * host.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of cache used by the build project. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NO_CACHE</code>: The build project does not use any cache.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The build project reads and writes from and to S3.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
     *         that build host.
     *         </p>
     *         </li>
     * @see CacheType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Information about the cache location:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: This is the S3 bucket name/prefix.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Information about the cache location: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: This is the S3 bucket name/prefix.
     *         </p>
     *         </li>
     */
    public final String location() {
        return location;
    }

    /**
     * <p>
     * An array of strings that specify the local cache modes. You can use one or more local cache modes at the same
     * time. This is only used for <code>LOCAL</code> cache types.
     * </p>
     * <p>
     * Possible values are:
     * </p>
     * <dl>
     * <dt>LOCAL_SOURCE_CACHE</dt>
     * <dd>
     * <p>
     * Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull only
     * the change between commits. This mode is a good choice for projects with a clean working directory and a source
     * that is a large Git repository. If you choose this option and your project does not use a Git repository (GitHub,
     * GitHub Enterprise, or Bitbucket), the option is ignored.
     * </p>
     * </dd>
     * <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
     * <dd>
     * <p>
     * Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker images. It
     * can prevent the performance issues caused by pulling large Docker images down from the network.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You can use a Docker layer cache in the Linux environment only.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     * </p>
     * </li>
     * <li>
     * <p>
     * You should consider the security implications before you use a Docker layer cache.
     * </p>
     * </li>
     * </ul>
     * </note></dd>
     * <dt>LOCAL_CUSTOM_CACHE</dt>
     * <dd>
     * <p>
     * Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is not
     * suited to one of the other three local cache modes. If you use a custom cache:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Only directories can be specified for caching. You cannot specify individual files.
     * </p>
     * </li>
     * <li>
     * <p>
     * Symlinks are used to reference cached directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cached directories are linked to your build before it downloads its project sources. Cached items are overridden
     * if a source item has the same name. Directories are specified using cache paths in the buildspec file.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasModes} method.
     * </p>
     * 
     * @return An array of strings that specify the local cache modes. You can use one or more local cache modes at the
     *         same time. This is only used for <code>LOCAL</code> cache types.</p>
     *         <p>
     *         Possible values are:
     *         </p>
     *         <dl>
     *         <dt>LOCAL_SOURCE_CACHE</dt>
     *         <dd>
     *         <p>
     *         Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull
     *         only the change between commits. This mode is a good choice for projects with a clean working directory
     *         and a source that is a large Git repository. If you choose this option and your project does not use a
     *         Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
     *         </p>
     *         </dd>
     *         <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
     *         <dd>
     *         <p>
     *         Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
     *         images. It can prevent the performance issues caused by pulling large Docker images down from the
     *         network.
     *         </p>
     *         <note>
     *         <ul>
     *         <li>
     *         <p>
     *         You can use a Docker layer cache in the Linux environment only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You should consider the security implications before you use a Docker layer cache.
     *         </p>
     *         </li>
     *         </ul>
     *         </note></dd>
     *         <dt>LOCAL_CUSTOM_CACHE</dt>
     *         <dd>
     *         <p>
     *         Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario
     *         is not suited to one of the other three local cache modes. If you use a custom cache:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Only directories can be specified for caching. You cannot specify individual files.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Symlinks are used to reference cached directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cached directories are linked to your build before it downloads its project sources. Cached items are
     *         overridden if a source item has the same name. Directories are specified using cache paths in the
     *         buildspec file.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     */
    public final List<CacheMode> modes() {
        return ProjectCacheModesCopier.copyStringToEnum(modes);
    }

    /**
     * For responses, this returns true if the service returned a value for the Modes property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasModes() {
        return modes != null && !(modes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings that specify the local cache modes. You can use one or more local cache modes at the same
     * time. This is only used for <code>LOCAL</code> cache types.
     * </p>
     * <p>
     * Possible values are:
     * </p>
     * <dl>
     * <dt>LOCAL_SOURCE_CACHE</dt>
     * <dd>
     * <p>
     * Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull only
     * the change between commits. This mode is a good choice for projects with a clean working directory and a source
     * that is a large Git repository. If you choose this option and your project does not use a Git repository (GitHub,
     * GitHub Enterprise, or Bitbucket), the option is ignored.
     * </p>
     * </dd>
     * <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
     * <dd>
     * <p>
     * Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker images. It
     * can prevent the performance issues caused by pulling large Docker images down from the network.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You can use a Docker layer cache in the Linux environment only.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     * </p>
     * </li>
     * <li>
     * <p>
     * You should consider the security implications before you use a Docker layer cache.
     * </p>
     * </li>
     * </ul>
     * </note></dd>
     * <dt>LOCAL_CUSTOM_CACHE</dt>
     * <dd>
     * <p>
     * Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is not
     * suited to one of the other three local cache modes. If you use a custom cache:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Only directories can be specified for caching. You cannot specify individual files.
     * </p>
     * </li>
     * <li>
     * <p>
     * Symlinks are used to reference cached directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cached directories are linked to your build before it downloads its project sources. Cached items are overridden
     * if a source item has the same name. Directories are specified using cache paths in the buildspec file.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasModes} method.
     * </p>
     * 
     * @return An array of strings that specify the local cache modes. You can use one or more local cache modes at the
     *         same time. This is only used for <code>LOCAL</code> cache types.</p>
     *         <p>
     *         Possible values are:
     *         </p>
     *         <dl>
     *         <dt>LOCAL_SOURCE_CACHE</dt>
     *         <dd>
     *         <p>
     *         Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull
     *         only the change between commits. This mode is a good choice for projects with a clean working directory
     *         and a source that is a large Git repository. If you choose this option and your project does not use a
     *         Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
     *         </p>
     *         </dd>
     *         <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
     *         <dd>
     *         <p>
     *         Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
     *         images. It can prevent the performance issues caused by pulling large Docker images down from the
     *         network.
     *         </p>
     *         <note>
     *         <ul>
     *         <li>
     *         <p>
     *         You can use a Docker layer cache in the Linux environment only.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You should consider the security implications before you use a Docker layer cache.
     *         </p>
     *         </li>
     *         </ul>
     *         </note></dd>
     *         <dt>LOCAL_CUSTOM_CACHE</dt>
     *         <dd>
     *         <p>
     *         Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario
     *         is not suited to one of the other three local cache modes. If you use a custom cache:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Only directories can be specified for caching. You cannot specify individual files.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Symlinks are used to reference cached directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cached directories are linked to your build before it downloads its project sources. Cached items are
     *         overridden if a source item has the same name. Directories are specified using cache paths in the
     *         buildspec file.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     */
    public final List<String> modesAsStrings() {
        return modes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(hasModes() ? modesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectCache)) {
            return false;
        }
        ProjectCache other = (ProjectCache) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(location(), other.location())
                && hasModes() == other.hasModes() && Objects.equals(modesAsStrings(), other.modesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProjectCache").add("Type", typeAsString()).add("Location", location())
                .add("Modes", hasModes() ? modesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "modes":
            return Optional.ofNullable(clazz.cast(modesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("location", LOCATION_FIELD);
        map.put("modes", MODES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProjectCache, T> g) {
        return obj -> g.apply((ProjectCache) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectCache> {
        /**
         * <p>
         * The type of cache used by the build project. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NO_CACHE</code>: The build project does not use any cache.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The build project reads and writes from and to S3.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that
         * build host.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of cache used by the build project. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NO_CACHE</code>: The build project does not use any cache.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The build project reads and writes from and to S3.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
         *        that build host.
         *        </p>
         *        </li>
         * @see CacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CacheType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of cache used by the build project. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NO_CACHE</code>: The build project does not use any cache.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The build project reads and writes from and to S3.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to that
         * build host.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of cache used by the build project. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NO_CACHE</code>: The build project does not use any cache.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The build project reads and writes from and to S3.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCAL</code>: The build project stores a cache locally on a build host that is only available to
         *        that build host.
         *        </p>
         *        </li>
         * @see CacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CacheType
         */
        Builder type(CacheType type);

        /**
         * <p>
         * Information about the cache location:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: This is the S3 bucket name/prefix.
         * </p>
         * </li>
         * </ul>
         * 
         * @param location
         *        Information about the cache location: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NO_CACHE</code> or <code>LOCAL</code>: This value is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: This is the S3 bucket name/prefix.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * An array of strings that specify the local cache modes. You can use one or more local cache modes at the same
         * time. This is only used for <code>LOCAL</code> cache types.
         * </p>
         * <p>
         * Possible values are:
         * </p>
         * <dl>
         * <dt>LOCAL_SOURCE_CACHE</dt>
         * <dd>
         * <p>
         * Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull
         * only the change between commits. This mode is a good choice for projects with a clean working directory and a
         * source that is a large Git repository. If you choose this option and your project does not use a Git
         * repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </dd>
         * <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         * <dd>
         * <p>
         * Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         * images. It can prevent the performance issues caused by pulling large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></dd>
         * <dt>LOCAL_CUSTOM_CACHE</dt>
         * <dd>
         * <p>
         * Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is
         * not suited to one of the other three local cache modes. If you use a custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param modes
         *        An array of strings that specify the local cache modes. You can use one or more local cache modes at
         *        the same time. This is only used for <code>LOCAL</code> cache types.</p>
         *        <p>
         *        Possible values are:
         *        </p>
         *        <dl>
         *        <dt>LOCAL_SOURCE_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds
         *        pull only the change between commits. This mode is a good choice for projects with a clean working
         *        directory and a source that is a large Git repository. If you choose this option and your project does
         *        not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         *        </p>
         *        </dd>
         *        <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         *        images. It can prevent the performance issues caused by pulling large Docker images down from the
         *        network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></dd>
         *        <dt>LOCAL_CUSTOM_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches directories you specify in the buildspec file. This mode is a good choice if your build
         *        scenario is not suited to one of the other three local cache modes. If you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modesWithStrings(Collection<String> modes);

        /**
         * <p>
         * An array of strings that specify the local cache modes. You can use one or more local cache modes at the same
         * time. This is only used for <code>LOCAL</code> cache types.
         * </p>
         * <p>
         * Possible values are:
         * </p>
         * <dl>
         * <dt>LOCAL_SOURCE_CACHE</dt>
         * <dd>
         * <p>
         * Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull
         * only the change between commits. This mode is a good choice for projects with a clean working directory and a
         * source that is a large Git repository. If you choose this option and your project does not use a Git
         * repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </dd>
         * <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         * <dd>
         * <p>
         * Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         * images. It can prevent the performance issues caused by pulling large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></dd>
         * <dt>LOCAL_CUSTOM_CACHE</dt>
         * <dd>
         * <p>
         * Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is
         * not suited to one of the other three local cache modes. If you use a custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param modes
         *        An array of strings that specify the local cache modes. You can use one or more local cache modes at
         *        the same time. This is only used for <code>LOCAL</code> cache types.</p>
         *        <p>
         *        Possible values are:
         *        </p>
         *        <dl>
         *        <dt>LOCAL_SOURCE_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds
         *        pull only the change between commits. This mode is a good choice for projects with a clean working
         *        directory and a source that is a large Git repository. If you choose this option and your project does
         *        not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         *        </p>
         *        </dd>
         *        <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         *        images. It can prevent the performance issues caused by pulling large Docker images down from the
         *        network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></dd>
         *        <dt>LOCAL_CUSTOM_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches directories you specify in the buildspec file. This mode is a good choice if your build
         *        scenario is not suited to one of the other three local cache modes. If you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modesWithStrings(String... modes);

        /**
         * <p>
         * An array of strings that specify the local cache modes. You can use one or more local cache modes at the same
         * time. This is only used for <code>LOCAL</code> cache types.
         * </p>
         * <p>
         * Possible values are:
         * </p>
         * <dl>
         * <dt>LOCAL_SOURCE_CACHE</dt>
         * <dd>
         * <p>
         * Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull
         * only the change between commits. This mode is a good choice for projects with a clean working directory and a
         * source that is a large Git repository. If you choose this option and your project does not use a Git
         * repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </dd>
         * <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         * <dd>
         * <p>
         * Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         * images. It can prevent the performance issues caused by pulling large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></dd>
         * <dt>LOCAL_CUSTOM_CACHE</dt>
         * <dd>
         * <p>
         * Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is
         * not suited to one of the other three local cache modes. If you use a custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param modes
         *        An array of strings that specify the local cache modes. You can use one or more local cache modes at
         *        the same time. This is only used for <code>LOCAL</code> cache types.</p>
         *        <p>
         *        Possible values are:
         *        </p>
         *        <dl>
         *        <dt>LOCAL_SOURCE_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds
         *        pull only the change between commits. This mode is a good choice for projects with a clean working
         *        directory and a source that is a large Git repository. If you choose this option and your project does
         *        not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         *        </p>
         *        </dd>
         *        <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         *        images. It can prevent the performance issues caused by pulling large Docker images down from the
         *        network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></dd>
         *        <dt>LOCAL_CUSTOM_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches directories you specify in the buildspec file. This mode is a good choice if your build
         *        scenario is not suited to one of the other three local cache modes. If you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modes(Collection<CacheMode> modes);

        /**
         * <p>
         * An array of strings that specify the local cache modes. You can use one or more local cache modes at the same
         * time. This is only used for <code>LOCAL</code> cache types.
         * </p>
         * <p>
         * Possible values are:
         * </p>
         * <dl>
         * <dt>LOCAL_SOURCE_CACHE</dt>
         * <dd>
         * <p>
         * Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull
         * only the change between commits. This mode is a good choice for projects with a clean working directory and a
         * source that is a large Git repository. If you choose this option and your project does not use a Git
         * repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * </p>
         * </dd>
         * <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         * <dd>
         * <p>
         * Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         * images. It can prevent the performance issues caused by pulling large Docker images down from the network.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You can use a Docker layer cache in the Linux environment only.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         * </p>
         * </li>
         * <li>
         * <p>
         * You should consider the security implications before you use a Docker layer cache.
         * </p>
         * </li>
         * </ul>
         * </note></dd>
         * <dt>LOCAL_CUSTOM_CACHE</dt>
         * <dd>
         * <p>
         * Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is
         * not suited to one of the other three local cache modes. If you use a custom cache:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Only directories can be specified for caching. You cannot specify individual files.
         * </p>
         * </li>
         * <li>
         * <p>
         * Symlinks are used to reference cached directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cached directories are linked to your build before it downloads its project sources. Cached items are
         * overridden if a source item has the same name. Directories are specified using cache paths in the buildspec
         * file.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param modes
         *        An array of strings that specify the local cache modes. You can use one or more local cache modes at
         *        the same time. This is only used for <code>LOCAL</code> cache types.</p>
         *        <p>
         *        Possible values are:
         *        </p>
         *        <dl>
         *        <dt>LOCAL_SOURCE_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds
         *        pull only the change between commits. This mode is a good choice for projects with a clean working
         *        directory and a source that is a large Git repository. If you choose this option and your project does
         *        not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         *        </p>
         *        </dd>
         *        <dt>LOCAL_DOCKER_LAYER_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker
         *        images. It can prevent the performance issues caused by pulling large Docker images down from the
         *        network.
         *        </p>
         *        <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You can use a Docker layer cache in the Linux environment only.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>privileged</code> flag must be set so that your project has the required Docker permissions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You should consider the security implications before you use a Docker layer cache.
         *        </p>
         *        </li>
         *        </ul>
         *        </note></dd>
         *        <dt>LOCAL_CUSTOM_CACHE</dt>
         *        <dd>
         *        <p>
         *        Caches directories you specify in the buildspec file. This mode is a good choice if your build
         *        scenario is not suited to one of the other three local cache modes. If you use a custom cache:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Only directories can be specified for caching. You cannot specify individual files.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Symlinks are used to reference cached directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cached directories are linked to your build before it downloads its project sources. Cached items are
         *        overridden if a source item has the same name. Directories are specified using cache paths in the
         *        buildspec file.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modes(CacheMode... modes);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String location;

        private List<String> modes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectCache model) {
            type(model.type);
            location(model.location);
            modesWithStrings(model.modes);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CacheType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final Collection<String> getModes() {
            if (modes instanceof SdkAutoConstructList) {
                return null;
            }
            return modes;
        }

        public final void setModes(Collection<String> modes) {
            this.modes = ProjectCacheModesCopier.copy(modes);
        }

        @Override
        public final Builder modesWithStrings(Collection<String> modes) {
            this.modes = ProjectCacheModesCopier.copy(modes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modesWithStrings(String... modes) {
            modesWithStrings(Arrays.asList(modes));
            return this;
        }

        @Override
        public final Builder modes(Collection<CacheMode> modes) {
            this.modes = ProjectCacheModesCopier.copyEnumToString(modes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modes(CacheMode... modes) {
            modes(Arrays.asList(modes));
            return this;
        }

        @Override
        public ProjectCache build() {
            return new ProjectCache(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
