/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a webhook that connects repository events to a build project in CodeBuild.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Webhook implements SdkPojo, Serializable, ToCopyableBuilder<Webhook.Builder, Webhook> {
    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("url")
            .getter(getter(Webhook::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final SdkField<String> PAYLOAD_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("payloadUrl").getter(getter(Webhook::payloadUrl)).setter(setter(Builder::payloadUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payloadUrl").build()).build();

    private static final SdkField<String> SECRET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("secret")
            .getter(getter(Webhook::secret)).setter(setter(Builder::secret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secret").build()).build();

    private static final SdkField<String> BRANCH_FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("branchFilter").getter(getter(Webhook::branchFilter)).setter(setter(Builder::branchFilter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("branchFilter").build()).build();

    private static final SdkField<List<List<WebhookFilter>>> FILTER_GROUPS_FIELD = SdkField
            .<List<List<WebhookFilter>>> builder(MarshallingType.LIST)
            .memberName("filterGroups")
            .getter(getter(Webhook::filterGroups))
            .setter(setter(Builder::filterGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<WebhookFilter>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<WebhookFilter> builder(MarshallingType.SDK_POJO)
                                                                            .constructor(WebhookFilter::builder)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> BUILD_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildType").getter(getter(Webhook::buildTypeAsString)).setter(setter(Builder::buildType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildType").build()).build();

    private static final SdkField<Boolean> MANUAL_CREATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("manualCreation").getter(getter(Webhook::manualCreation)).setter(setter(Builder::manualCreation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("manualCreation").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_SECRET_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastModifiedSecret").getter(getter(Webhook::lastModifiedSecret))
            .setter(setter(Builder::lastModifiedSecret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModifiedSecret").build())
            .build();

    private static final SdkField<ScopeConfiguration> SCOPE_CONFIGURATION_FIELD = SdkField
            .<ScopeConfiguration> builder(MarshallingType.SDK_POJO).memberName("scopeConfiguration")
            .getter(getter(Webhook::scopeConfiguration)).setter(setter(Builder::scopeConfiguration))
            .constructor(ScopeConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scopeConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URL_FIELD, PAYLOAD_URL_FIELD,
            SECRET_FIELD, BRANCH_FILTER_FIELD, FILTER_GROUPS_FIELD, BUILD_TYPE_FIELD, MANUAL_CREATION_FIELD,
            LAST_MODIFIED_SECRET_FIELD, SCOPE_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String url;

    private final String payloadUrl;

    private final String secret;

    private final String branchFilter;

    private final List<List<WebhookFilter>> filterGroups;

    private final String buildType;

    private final Boolean manualCreation;

    private final Instant lastModifiedSecret;

    private final ScopeConfiguration scopeConfiguration;

    private Webhook(BuilderImpl builder) {
        this.url = builder.url;
        this.payloadUrl = builder.payloadUrl;
        this.secret = builder.secret;
        this.branchFilter = builder.branchFilter;
        this.filterGroups = builder.filterGroups;
        this.buildType = builder.buildType;
        this.manualCreation = builder.manualCreation;
        this.lastModifiedSecret = builder.lastModifiedSecret;
        this.scopeConfiguration = builder.scopeConfiguration;
    }

    /**
     * <p>
     * The URL to the webhook.
     * </p>
     * 
     * @return The URL to the webhook.
     */
    public final String url() {
        return url;
    }

    /**
     * <p>
     * The CodeBuild endpoint where webhook events are sent.
     * </p>
     * 
     * @return The CodeBuild endpoint where webhook events are sent.
     */
    public final String payloadUrl() {
        return payloadUrl;
    }

    /**
     * <p>
     * The secret token of the associated repository.
     * </p>
     * <note>
     * <p>
     * A Bitbucket webhook does not support <code>secret</code>.
     * </p>
     * </note>
     * 
     * @return The secret token of the associated repository. </p> <note>
     *         <p>
     *         A Bitbucket webhook does not support <code>secret</code>.
     *         </p>
     */
    public final String secret() {
        return secret;
    }

    /**
     * <p>
     * A regular expression used to determine which repository branches are built when a webhook is triggered. If the
     * name of a branch matches the regular expression, then it is built. If <code>branchFilter</code> is empty, then
     * all branches are built.
     * </p>
     * <note>
     * <p>
     * It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
     * </p>
     * </note>
     * 
     * @return A regular expression used to determine which repository branches are built when a webhook is triggered.
     *         If the name of a branch matches the regular expression, then it is built. If <code>branchFilter</code> is
     *         empty, then all branches are built.</p> <note>
     *         <p>
     *         It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
     *         </p>
     */
    public final String branchFilter() {
        return branchFilter;
    }

    /**
     * For responses, this returns true if the service returned a value for the FilterGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFilterGroups() {
        return filterGroups != null && !(filterGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered. At least
     * one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its <code>type</code>.
     * </p>
     * <p>
     * For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass. For a
     * filter group to pass, each of its filters must pass.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilterGroups} method.
     * </p>
     * 
     * @return An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered.
     *         At least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its
     *         <code>type</code>. </p>
     *         <p>
     *         For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass.
     *         For a filter group to pass, each of its filters must pass.
     */
    public final List<List<WebhookFilter>> filterGroups() {
        return filterGroups;
    }

    /**
     * <p>
     * Specifies the type of build this webhook will trigger.
     * </p>
     * <note>
     * <p>
     * <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects configured
     * for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner builds, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial: Configure a
     * CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildType} will
     * return {@link WebhookBuildType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of build this webhook will trigger.</p> <note>
     *         <p>
     *         <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects
     *         configured for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner
     *         builds, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial:
     *         Configure a CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
     *         </p>
     * @see WebhookBuildType
     */
    public final WebhookBuildType buildType() {
        return WebhookBuildType.fromValue(buildType);
    }

    /**
     * <p>
     * Specifies the type of build this webhook will trigger.
     * </p>
     * <note>
     * <p>
     * <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects configured
     * for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner builds, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial: Configure a
     * CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildType} will
     * return {@link WebhookBuildType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of build this webhook will trigger.</p> <note>
     *         <p>
     *         <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects
     *         configured for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner
     *         builds, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial:
     *         Configure a CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
     *         </p>
     * @see WebhookBuildType
     */
    public final String buildTypeAsString() {
        return buildType;
    }

    /**
     * <p>
     * If manualCreation is true, CodeBuild doesn't create a webhook in GitHub and instead returns
     * <code>payloadUrl</code> and <code>secret</code> values for the webhook. The <code>payloadUrl</code> and
     * <code>secret</code> values in the output can be used to manually create a webhook within GitHub.
     * </p>
     * <note>
     * <p>
     * manualCreation is only available for GitHub webhooks.
     * </p>
     * </note>
     * 
     * @return If manualCreation is true, CodeBuild doesn't create a webhook in GitHub and instead returns
     *         <code>payloadUrl</code> and <code>secret</code> values for the webhook. The <code>payloadUrl</code> and
     *         <code>secret</code> values in the output can be used to manually create a webhook within GitHub.</p>
     *         <note>
     *         <p>
     *         manualCreation is only available for GitHub webhooks.
     *         </p>
     */
    public final Boolean manualCreation() {
        return manualCreation;
    }

    /**
     * <p>
     * A timestamp that indicates the last time a repository's secret token was modified.
     * </p>
     * 
     * @return A timestamp that indicates the last time a repository's secret token was modified.
     */
    public final Instant lastModifiedSecret() {
        return lastModifiedSecret;
    }

    /**
     * <p>
     * The scope configuration for global or organization webhooks.
     * </p>
     * <note>
     * <p>
     * Global or organization webhooks are only available for GitHub and Github Enterprise webhooks.
     * </p>
     * </note>
     * 
     * @return The scope configuration for global or organization webhooks.</p> <note>
     *         <p>
     *         Global or organization webhooks are only available for GitHub and Github Enterprise webhooks.
     *         </p>
     */
    public final ScopeConfiguration scopeConfiguration() {
        return scopeConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(url());
        hashCode = 31 * hashCode + Objects.hashCode(payloadUrl());
        hashCode = 31 * hashCode + Objects.hashCode(secret());
        hashCode = 31 * hashCode + Objects.hashCode(branchFilter());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilterGroups() ? filterGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(buildTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(manualCreation());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedSecret());
        hashCode = 31 * hashCode + Objects.hashCode(scopeConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Webhook)) {
            return false;
        }
        Webhook other = (Webhook) obj;
        return Objects.equals(url(), other.url()) && Objects.equals(payloadUrl(), other.payloadUrl())
                && Objects.equals(secret(), other.secret()) && Objects.equals(branchFilter(), other.branchFilter())
                && hasFilterGroups() == other.hasFilterGroups() && Objects.equals(filterGroups(), other.filterGroups())
                && Objects.equals(buildTypeAsString(), other.buildTypeAsString())
                && Objects.equals(manualCreation(), other.manualCreation())
                && Objects.equals(lastModifiedSecret(), other.lastModifiedSecret())
                && Objects.equals(scopeConfiguration(), other.scopeConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Webhook").add("Url", url()).add("PayloadUrl", payloadUrl()).add("Secret", secret())
                .add("BranchFilter", branchFilter()).add("FilterGroups", hasFilterGroups() ? filterGroups() : null)
                .add("BuildType", buildTypeAsString()).add("ManualCreation", manualCreation())
                .add("LastModifiedSecret", lastModifiedSecret()).add("ScopeConfiguration", scopeConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "url":
            return Optional.ofNullable(clazz.cast(url()));
        case "payloadUrl":
            return Optional.ofNullable(clazz.cast(payloadUrl()));
        case "secret":
            return Optional.ofNullable(clazz.cast(secret()));
        case "branchFilter":
            return Optional.ofNullable(clazz.cast(branchFilter()));
        case "filterGroups":
            return Optional.ofNullable(clazz.cast(filterGroups()));
        case "buildType":
            return Optional.ofNullable(clazz.cast(buildTypeAsString()));
        case "manualCreation":
            return Optional.ofNullable(clazz.cast(manualCreation()));
        case "lastModifiedSecret":
            return Optional.ofNullable(clazz.cast(lastModifiedSecret()));
        case "scopeConfiguration":
            return Optional.ofNullable(clazz.cast(scopeConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("url", URL_FIELD);
        map.put("payloadUrl", PAYLOAD_URL_FIELD);
        map.put("secret", SECRET_FIELD);
        map.put("branchFilter", BRANCH_FILTER_FIELD);
        map.put("filterGroups", FILTER_GROUPS_FIELD);
        map.put("buildType", BUILD_TYPE_FIELD);
        map.put("manualCreation", MANUAL_CREATION_FIELD);
        map.put("lastModifiedSecret", LAST_MODIFIED_SECRET_FIELD);
        map.put("scopeConfiguration", SCOPE_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Webhook, T> g) {
        return obj -> g.apply((Webhook) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Webhook> {
        /**
         * <p>
         * The URL to the webhook.
         * </p>
         * 
         * @param url
         *        The URL to the webhook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);

        /**
         * <p>
         * The CodeBuild endpoint where webhook events are sent.
         * </p>
         * 
         * @param payloadUrl
         *        The CodeBuild endpoint where webhook events are sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payloadUrl(String payloadUrl);

        /**
         * <p>
         * The secret token of the associated repository.
         * </p>
         * <note>
         * <p>
         * A Bitbucket webhook does not support <code>secret</code>.
         * </p>
         * </note>
         * 
         * @param secret
         *        The secret token of the associated repository. </p> <note>
         *        <p>
         *        A Bitbucket webhook does not support <code>secret</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secret(String secret);

        /**
         * <p>
         * A regular expression used to determine which repository branches are built when a webhook is triggered. If
         * the name of a branch matches the regular expression, then it is built. If <code>branchFilter</code> is empty,
         * then all branches are built.
         * </p>
         * <note>
         * <p>
         * It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
         * </p>
         * </note>
         * 
         * @param branchFilter
         *        A regular expression used to determine which repository branches are built when a webhook is
         *        triggered. If the name of a branch matches the regular expression, then it is built. If
         *        <code>branchFilter</code> is empty, then all branches are built.</p> <note>
         *        <p>
         *        It is recommended that you use <code>filterGroups</code> instead of <code>branchFilter</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branchFilter(String branchFilter);

        /**
         * <p>
         * An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered. At
         * least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its <code>type</code>.
         * </p>
         * <p>
         * For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass. For
         * a filter group to pass, each of its filters must pass.
         * </p>
         * 
         * @param filterGroups
         *        An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are
         *        triggered. At least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its
         *        <code>type</code>. </p>
         *        <p>
         *        For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must
         *        pass. For a filter group to pass, each of its filters must pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterGroups(Collection<? extends Collection<WebhookFilter>> filterGroups);

        /**
         * <p>
         * An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are triggered. At
         * least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its <code>type</code>.
         * </p>
         * <p>
         * For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must pass. For
         * a filter group to pass, each of its filters must pass.
         * </p>
         * 
         * @param filterGroups
         *        An array of arrays of <code>WebhookFilter</code> objects used to determine which webhooks are
         *        triggered. At least one <code>WebhookFilter</code> in the array must specify <code>EVENT</code> as its
         *        <code>type</code>. </p>
         *        <p>
         *        For a build to be triggered, at least one filter group in the <code>filterGroups</code> array must
         *        pass. For a filter group to pass, each of its filters must pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterGroups(Collection<WebhookFilter>... filterGroups);

        /**
         * <p>
         * Specifies the type of build this webhook will trigger.
         * </p>
         * <note>
         * <p>
         * <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects
         * configured for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner builds,
         * see <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial:
         * Configure a CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
         * </p>
         * </note>
         * 
         * @param buildType
         *        Specifies the type of build this webhook will trigger.</p> <note>
         *        <p>
         *        <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects
         *        configured for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner
         *        builds, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial:
         *        Configure a CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
         *        </p>
         * @see WebhookBuildType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebhookBuildType
         */
        Builder buildType(String buildType);

        /**
         * <p>
         * Specifies the type of build this webhook will trigger.
         * </p>
         * <note>
         * <p>
         * <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects
         * configured for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner builds,
         * see <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial:
         * Configure a CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
         * </p>
         * </note>
         * 
         * @param buildType
         *        Specifies the type of build this webhook will trigger.</p> <note>
         *        <p>
         *        <code>RUNNER_BUILDKITE_BUILD</code> is only available for <code>NO_SOURCE</code> source type projects
         *        configured for Buildkite runner builds. For more information about CodeBuild-hosted Buildkite runner
         *        builds, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-runner-buildkite.html">Tutorial:
         *        Configure a CodeBuild-hosted Buildkite runner</a> in the <i>CodeBuild user guide</i>.
         *        </p>
         * @see WebhookBuildType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebhookBuildType
         */
        Builder buildType(WebhookBuildType buildType);

        /**
         * <p>
         * If manualCreation is true, CodeBuild doesn't create a webhook in GitHub and instead returns
         * <code>payloadUrl</code> and <code>secret</code> values for the webhook. The <code>payloadUrl</code> and
         * <code>secret</code> values in the output can be used to manually create a webhook within GitHub.
         * </p>
         * <note>
         * <p>
         * manualCreation is only available for GitHub webhooks.
         * </p>
         * </note>
         * 
         * @param manualCreation
         *        If manualCreation is true, CodeBuild doesn't create a webhook in GitHub and instead returns
         *        <code>payloadUrl</code> and <code>secret</code> values for the webhook. The <code>payloadUrl</code>
         *        and <code>secret</code> values in the output can be used to manually create a webhook within
         *        GitHub.</p> <note>
         *        <p>
         *        manualCreation is only available for GitHub webhooks.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manualCreation(Boolean manualCreation);

        /**
         * <p>
         * A timestamp that indicates the last time a repository's secret token was modified.
         * </p>
         * 
         * @param lastModifiedSecret
         *        A timestamp that indicates the last time a repository's secret token was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedSecret(Instant lastModifiedSecret);

        /**
         * <p>
         * The scope configuration for global or organization webhooks.
         * </p>
         * <note>
         * <p>
         * Global or organization webhooks are only available for GitHub and Github Enterprise webhooks.
         * </p>
         * </note>
         * 
         * @param scopeConfiguration
         *        The scope configuration for global or organization webhooks.</p> <note>
         *        <p>
         *        Global or organization webhooks are only available for GitHub and Github Enterprise webhooks.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeConfiguration(ScopeConfiguration scopeConfiguration);

        /**
         * <p>
         * The scope configuration for global or organization webhooks.
         * </p>
         * <note>
         * <p>
         * Global or organization webhooks are only available for GitHub and Github Enterprise webhooks.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ScopeConfiguration.Builder}
         * avoiding the need to create one manually via {@link ScopeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScopeConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #scopeConfiguration(ScopeConfiguration)}.
         * 
         * @param scopeConfiguration
         *        a consumer that will call methods on {@link ScopeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scopeConfiguration(ScopeConfiguration)
         */
        default Builder scopeConfiguration(Consumer<ScopeConfiguration.Builder> scopeConfiguration) {
            return scopeConfiguration(ScopeConfiguration.builder().applyMutation(scopeConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String url;

        private String payloadUrl;

        private String secret;

        private String branchFilter;

        private List<List<WebhookFilter>> filterGroups = DefaultSdkAutoConstructList.getInstance();

        private String buildType;

        private Boolean manualCreation;

        private Instant lastModifiedSecret;

        private ScopeConfiguration scopeConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(Webhook model) {
            url(model.url);
            payloadUrl(model.payloadUrl);
            secret(model.secret);
            branchFilter(model.branchFilter);
            filterGroups(model.filterGroups);
            buildType(model.buildType);
            manualCreation(model.manualCreation);
            lastModifiedSecret(model.lastModifiedSecret);
            scopeConfiguration(model.scopeConfiguration);
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final String getPayloadUrl() {
            return payloadUrl;
        }

        public final void setPayloadUrl(String payloadUrl) {
            this.payloadUrl = payloadUrl;
        }

        @Override
        public final Builder payloadUrl(String payloadUrl) {
            this.payloadUrl = payloadUrl;
            return this;
        }

        public final String getSecret() {
            return secret;
        }

        public final void setSecret(String secret) {
            this.secret = secret;
        }

        @Override
        public final Builder secret(String secret) {
            this.secret = secret;
            return this;
        }

        public final String getBranchFilter() {
            return branchFilter;
        }

        public final void setBranchFilter(String branchFilter) {
            this.branchFilter = branchFilter;
        }

        @Override
        public final Builder branchFilter(String branchFilter) {
            this.branchFilter = branchFilter;
            return this;
        }

        public final List<List<WebhookFilter.Builder>> getFilterGroups() {
            List<List<WebhookFilter.Builder>> result = FilterGroupsCopier.copyToBuilder(this.filterGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilterGroups(Collection<? extends Collection<WebhookFilter.BuilderImpl>> filterGroups) {
            this.filterGroups = FilterGroupsCopier.copyFromBuilder(filterGroups);
        }

        @Override
        public final Builder filterGroups(Collection<? extends Collection<WebhookFilter>> filterGroups) {
            this.filterGroups = FilterGroupsCopier.copy(filterGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filterGroups(Collection<WebhookFilter>... filterGroups) {
            filterGroups(Arrays.asList(filterGroups));
            return this;
        }

        public final String getBuildType() {
            return buildType;
        }

        public final void setBuildType(String buildType) {
            this.buildType = buildType;
        }

        @Override
        public final Builder buildType(String buildType) {
            this.buildType = buildType;
            return this;
        }

        @Override
        public final Builder buildType(WebhookBuildType buildType) {
            this.buildType(buildType == null ? null : buildType.toString());
            return this;
        }

        public final Boolean getManualCreation() {
            return manualCreation;
        }

        public final void setManualCreation(Boolean manualCreation) {
            this.manualCreation = manualCreation;
        }

        @Override
        public final Builder manualCreation(Boolean manualCreation) {
            this.manualCreation = manualCreation;
            return this;
        }

        public final Instant getLastModifiedSecret() {
            return lastModifiedSecret;
        }

        public final void setLastModifiedSecret(Instant lastModifiedSecret) {
            this.lastModifiedSecret = lastModifiedSecret;
        }

        @Override
        public final Builder lastModifiedSecret(Instant lastModifiedSecret) {
            this.lastModifiedSecret = lastModifiedSecret;
            return this;
        }

        public final ScopeConfiguration.Builder getScopeConfiguration() {
            return scopeConfiguration != null ? scopeConfiguration.toBuilder() : null;
        }

        public final void setScopeConfiguration(ScopeConfiguration.BuilderImpl scopeConfiguration) {
            this.scopeConfiguration = scopeConfiguration != null ? scopeConfiguration.build() : null;
        }

        @Override
        public final Builder scopeConfiguration(ScopeConfiguration scopeConfiguration) {
            this.scopeConfiguration = scopeConfiguration;
            return this;
        }

        @Override
        public Webhook build() {
            return new Webhook(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
