/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that specifies metadata for your organization's Config Custom Policy rule. The metadata includes the
 * runtime system in use, which accounts have debug logging enabled, and other custom rule metadata, such as resource
 * type, resource ID of Amazon Web Services resource, and organization trigger types that initiate Config to evaluate
 * Amazon Web Services resources against a rule.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrganizationCustomPolicyRuleMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<OrganizationCustomPolicyRuleMetadata.Builder, OrganizationCustomPolicyRuleMetadata> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(OrganizationCustomPolicyRuleMetadata::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<String>> ORGANIZATION_CONFIG_RULE_TRIGGER_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OrganizationConfigRuleTriggerTypes")
            .getter(getter(OrganizationCustomPolicyRuleMetadata::organizationConfigRuleTriggerTypesAsStrings))
            .setter(setter(Builder::organizationConfigRuleTriggerTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationConfigRuleTriggerTypes")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INPUT_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputParameters").getter(getter(OrganizationCustomPolicyRuleMetadata::inputParameters))
            .setter(setter(Builder::inputParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputParameters").build()).build();

    private static final SdkField<String> MAXIMUM_EXECUTION_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaximumExecutionFrequency")
            .getter(getter(OrganizationCustomPolicyRuleMetadata::maximumExecutionFrequencyAsString))
            .setter(setter(Builder::maximumExecutionFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumExecutionFrequency").build())
            .build();

    private static final SdkField<List<String>> RESOURCE_TYPES_SCOPE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypesScope")
            .getter(getter(OrganizationCustomPolicyRuleMetadata::resourceTypesScope))
            .setter(setter(Builder::resourceTypesScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypesScope").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_ID_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceIdScope").getter(getter(OrganizationCustomPolicyRuleMetadata::resourceIdScope))
            .setter(setter(Builder::resourceIdScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdScope").build()).build();

    private static final SdkField<String> TAG_KEY_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TagKeyScope").getter(getter(OrganizationCustomPolicyRuleMetadata::tagKeyScope))
            .setter(setter(Builder::tagKeyScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagKeyScope").build()).build();

    private static final SdkField<String> TAG_VALUE_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TagValueScope").getter(getter(OrganizationCustomPolicyRuleMetadata::tagValueScope))
            .setter(setter(Builder::tagValueScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagValueScope").build()).build();

    private static final SdkField<String> POLICY_RUNTIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyRuntime").getter(getter(OrganizationCustomPolicyRuleMetadata::policyRuntime))
            .setter(setter(Builder::policyRuntime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyRuntime").build()).build();

    private static final SdkField<String> POLICY_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyText").getter(getter(OrganizationCustomPolicyRuleMetadata::policyText))
            .setter(setter(Builder::policyText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyText").build()).build();

    private static final SdkField<List<String>> DEBUG_LOG_DELIVERY_ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DebugLogDeliveryAccounts")
            .getter(getter(OrganizationCustomPolicyRuleMetadata::debugLogDeliveryAccounts))
            .setter(setter(Builder::debugLogDeliveryAccounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DebugLogDeliveryAccounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            ORGANIZATION_CONFIG_RULE_TRIGGER_TYPES_FIELD, INPUT_PARAMETERS_FIELD, MAXIMUM_EXECUTION_FREQUENCY_FIELD,
            RESOURCE_TYPES_SCOPE_FIELD, RESOURCE_ID_SCOPE_FIELD, TAG_KEY_SCOPE_FIELD, TAG_VALUE_SCOPE_FIELD,
            POLICY_RUNTIME_FIELD, POLICY_TEXT_FIELD, DEBUG_LOG_DELIVERY_ACCOUNTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String description;

    private final List<String> organizationConfigRuleTriggerTypes;

    private final String inputParameters;

    private final String maximumExecutionFrequency;

    private final List<String> resourceTypesScope;

    private final String resourceIdScope;

    private final String tagKeyScope;

    private final String tagValueScope;

    private final String policyRuntime;

    private final String policyText;

    private final List<String> debugLogDeliveryAccounts;

    private OrganizationCustomPolicyRuleMetadata(BuilderImpl builder) {
        this.description = builder.description;
        this.organizationConfigRuleTriggerTypes = builder.organizationConfigRuleTriggerTypes;
        this.inputParameters = builder.inputParameters;
        this.maximumExecutionFrequency = builder.maximumExecutionFrequency;
        this.resourceTypesScope = builder.resourceTypesScope;
        this.resourceIdScope = builder.resourceIdScope;
        this.tagKeyScope = builder.tagKeyScope;
        this.tagValueScope = builder.tagValueScope;
        this.policyRuntime = builder.policyRuntime;
        this.policyText = builder.policyText;
        this.debugLogDeliveryAccounts = builder.debugLogDeliveryAccounts;
    }

    /**
     * <p>
     * The description that you provide for your organization Config Custom Policy rule.
     * </p>
     * 
     * @return The description that you provide for your organization Config Custom Policy rule.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy rules,
     * Config supports change-initiated notification types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a configuration
     * item as a result of a resource change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers an
     * oversized configuration item. Config may generate this notification type when a resource changes and the
     * notification exceeds the maximum size allowed by Amazon SNS.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrganizationConfigRuleTriggerTypes}
     * method.
     * </p>
     * 
     * @return The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy
     *         rules, Config supports change-initiated notification types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
     *         configuration item as a result of a resource change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers
     *         an oversized configuration item. Config may generate this notification type when a resource changes and
     *         the notification exceeds the maximum size allowed by Amazon SNS.
     *         </p>
     *         </li>
     */
    public final List<OrganizationConfigRuleTriggerTypeNoSN> organizationConfigRuleTriggerTypes() {
        return OrganizationConfigRuleTriggerTypeNoSNsCopier.copyStringToEnum(organizationConfigRuleTriggerTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the OrganizationConfigRuleTriggerTypes
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasOrganizationConfigRuleTriggerTypes() {
        return organizationConfigRuleTriggerTypes != null
                && !(organizationConfigRuleTriggerTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy rules,
     * Config supports change-initiated notification types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a configuration
     * item as a result of a resource change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers an
     * oversized configuration item. Config may generate this notification type when a resource changes and the
     * notification exceeds the maximum size allowed by Amazon SNS.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrganizationConfigRuleTriggerTypes}
     * method.
     * </p>
     * 
     * @return The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy
     *         rules, Config supports change-initiated notification types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
     *         configuration item as a result of a resource change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers
     *         an oversized configuration item. Config may generate this notification type when a resource changes and
     *         the notification exceeds the maximum size allowed by Amazon SNS.
     *         </p>
     *         </li>
     */
    public final List<String> organizationConfigRuleTriggerTypesAsStrings() {
        return organizationConfigRuleTriggerTypes;
    }

    /**
     * <p>
     * A string, in JSON format, that is passed to your organization Config Custom Policy rule.
     * </p>
     * 
     * @return A string, in JSON format, that is passed to your organization Config Custom Policy rule.
     */
    public final String inputParameters() {
        return inputParameters;
    }

    /**
     * <p>
     * The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is triggered
     * when Config delivers the configuration snapshot. For more information, see
     * <a>ConfigSnapshotDeliveryProperties</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is
     *         triggered when Config delivers the configuration snapshot. For more information, see
     *         <a>ConfigSnapshotDeliveryProperties</a>.
     * @see MaximumExecutionFrequency
     */
    public final MaximumExecutionFrequency maximumExecutionFrequency() {
        return MaximumExecutionFrequency.fromValue(maximumExecutionFrequency);
    }

    /**
     * <p>
     * The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is triggered
     * when Config delivers the configuration snapshot. For more information, see
     * <a>ConfigSnapshotDeliveryProperties</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is
     *         triggered when Config delivers the configuration snapshot. For more information, see
     *         <a>ConfigSnapshotDeliveryProperties</a>.
     * @see MaximumExecutionFrequency
     */
    public final String maximumExecutionFrequencyAsString() {
        return maximumExecutionFrequency;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypesScope property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypesScope() {
        return resourceTypesScope != null && !(resourceTypesScope instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of the Amazon Web Services resource that was evaluated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypesScope} method.
     * </p>
     * 
     * @return The type of the Amazon Web Services resource that was evaluated.
     */
    public final List<String> resourceTypesScope() {
        return resourceTypesScope;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services resource that was evaluated.
     * </p>
     * 
     * @return The ID of the Amazon Web Services resource that was evaluated.
     */
    public final String resourceIdScope() {
        return resourceIdScope;
    }

    /**
     * <p>
     * One part of a key-value pair that make up a tag. A key is a general label that acts like a category for more
     * specific tag values.
     * </p>
     * 
     * @return One part of a key-value pair that make up a tag. A key is a general label that acts like a category for
     *         more specific tag values.
     */
    public final String tagKeyScope() {
        return tagKeyScope;
    }

    /**
     * <p>
     * The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag category
     * (key).
     * </p>
     * 
     * @return The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag
     *         category (key).
     */
    public final String tagValueScope() {
        return tagValueScope;
    }

    /**
     * <p>
     * The runtime system for your organization Config Custom Policy rules. Guard is a policy-as-code language that
     * allows you to write policies that are enforced by Config Custom Policy rules. For more information about Guard,
     * see the <a href="https://github.com/aws-cloudformation/cloudformation-guard">Guard GitHub Repository</a>.
     * </p>
     * 
     * @return The runtime system for your organization Config Custom Policy rules. Guard is a policy-as-code language
     *         that allows you to write policies that are enforced by Config Custom Policy rules. For more information
     *         about Guard, see the <a href="https://github.com/aws-cloudformation/cloudformation-guard">Guard GitHub
     *         Repository</a>.
     */
    public final String policyRuntime() {
        return policyRuntime;
    }

    /**
     * <p>
     * The policy definition containing the logic for your organization Config Custom Policy rule.
     * </p>
     * 
     * @return The policy definition containing the logic for your organization Config Custom Policy rule.
     */
    public final String policyText() {
        return policyText;
    }

    /**
     * For responses, this returns true if the service returned a value for the DebugLogDeliveryAccounts property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDebugLogDeliveryAccounts() {
        return debugLogDeliveryAccounts != null && !(debugLogDeliveryAccounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of accounts that you can enable debug logging for your organization Config Custom Policy rule. List is
     * null when debug logging is enabled for all accounts.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDebugLogDeliveryAccounts} method.
     * </p>
     * 
     * @return A list of accounts that you can enable debug logging for your organization Config Custom Policy rule.
     *         List is null when debug logging is enabled for all accounts.
     */
    public final List<String> debugLogDeliveryAccounts() {
        return debugLogDeliveryAccounts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31
                * hashCode
                + Objects
                        .hashCode(hasOrganizationConfigRuleTriggerTypes() ? organizationConfigRuleTriggerTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(inputParameters());
        hashCode = 31 * hashCode + Objects.hashCode(maximumExecutionFrequencyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypesScope() ? resourceTypesScope() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdScope());
        hashCode = 31 * hashCode + Objects.hashCode(tagKeyScope());
        hashCode = 31 * hashCode + Objects.hashCode(tagValueScope());
        hashCode = 31 * hashCode + Objects.hashCode(policyRuntime());
        hashCode = 31 * hashCode + Objects.hashCode(policyText());
        hashCode = 31 * hashCode + Objects.hashCode(hasDebugLogDeliveryAccounts() ? debugLogDeliveryAccounts() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrganizationCustomPolicyRuleMetadata)) {
            return false;
        }
        OrganizationCustomPolicyRuleMetadata other = (OrganizationCustomPolicyRuleMetadata) obj;
        return Objects.equals(description(), other.description())
                && hasOrganizationConfigRuleTriggerTypes() == other.hasOrganizationConfigRuleTriggerTypes()
                && Objects.equals(organizationConfigRuleTriggerTypesAsStrings(),
                        other.organizationConfigRuleTriggerTypesAsStrings())
                && Objects.equals(inputParameters(), other.inputParameters())
                && Objects.equals(maximumExecutionFrequencyAsString(), other.maximumExecutionFrequencyAsString())
                && hasResourceTypesScope() == other.hasResourceTypesScope()
                && Objects.equals(resourceTypesScope(), other.resourceTypesScope())
                && Objects.equals(resourceIdScope(), other.resourceIdScope())
                && Objects.equals(tagKeyScope(), other.tagKeyScope()) && Objects.equals(tagValueScope(), other.tagValueScope())
                && Objects.equals(policyRuntime(), other.policyRuntime()) && Objects.equals(policyText(), other.policyText())
                && hasDebugLogDeliveryAccounts() == other.hasDebugLogDeliveryAccounts()
                && Objects.equals(debugLogDeliveryAccounts(), other.debugLogDeliveryAccounts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("OrganizationCustomPolicyRuleMetadata")
                .add("Description", description())
                .add("OrganizationConfigRuleTriggerTypes",
                        hasOrganizationConfigRuleTriggerTypes() ? organizationConfigRuleTriggerTypesAsStrings() : null)
                .add("InputParameters", inputParameters()).add("MaximumExecutionFrequency", maximumExecutionFrequencyAsString())
                .add("ResourceTypesScope", hasResourceTypesScope() ? resourceTypesScope() : null)
                .add("ResourceIdScope", resourceIdScope()).add("TagKeyScope", tagKeyScope())
                .add("TagValueScope", tagValueScope()).add("PolicyRuntime", policyRuntime()).add("PolicyText", policyText())
                .add("DebugLogDeliveryAccounts", hasDebugLogDeliveryAccounts() ? debugLogDeliveryAccounts() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OrganizationConfigRuleTriggerTypes":
            return Optional.ofNullable(clazz.cast(organizationConfigRuleTriggerTypesAsStrings()));
        case "InputParameters":
            return Optional.ofNullable(clazz.cast(inputParameters()));
        case "MaximumExecutionFrequency":
            return Optional.ofNullable(clazz.cast(maximumExecutionFrequencyAsString()));
        case "ResourceTypesScope":
            return Optional.ofNullable(clazz.cast(resourceTypesScope()));
        case "ResourceIdScope":
            return Optional.ofNullable(clazz.cast(resourceIdScope()));
        case "TagKeyScope":
            return Optional.ofNullable(clazz.cast(tagKeyScope()));
        case "TagValueScope":
            return Optional.ofNullable(clazz.cast(tagValueScope()));
        case "PolicyRuntime":
            return Optional.ofNullable(clazz.cast(policyRuntime()));
        case "PolicyText":
            return Optional.ofNullable(clazz.cast(policyText()));
        case "DebugLogDeliveryAccounts":
            return Optional.ofNullable(clazz.cast(debugLogDeliveryAccounts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Description", DESCRIPTION_FIELD);
        map.put("OrganizationConfigRuleTriggerTypes", ORGANIZATION_CONFIG_RULE_TRIGGER_TYPES_FIELD);
        map.put("InputParameters", INPUT_PARAMETERS_FIELD);
        map.put("MaximumExecutionFrequency", MAXIMUM_EXECUTION_FREQUENCY_FIELD);
        map.put("ResourceTypesScope", RESOURCE_TYPES_SCOPE_FIELD);
        map.put("ResourceIdScope", RESOURCE_ID_SCOPE_FIELD);
        map.put("TagKeyScope", TAG_KEY_SCOPE_FIELD);
        map.put("TagValueScope", TAG_VALUE_SCOPE_FIELD);
        map.put("PolicyRuntime", POLICY_RUNTIME_FIELD);
        map.put("PolicyText", POLICY_TEXT_FIELD);
        map.put("DebugLogDeliveryAccounts", DEBUG_LOG_DELIVERY_ACCOUNTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OrganizationCustomPolicyRuleMetadata, T> g) {
        return obj -> g.apply((OrganizationCustomPolicyRuleMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrganizationCustomPolicyRuleMetadata> {
        /**
         * <p>
         * The description that you provide for your organization Config Custom Policy rule.
         * </p>
         * 
         * @param description
         *        The description that you provide for your organization Config Custom Policy rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy
         * rules, Config supports change-initiated notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers an
         * oversized configuration item. Config may generate this notification type when a resource changes and the
         * notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that initiates Config to run an evaluation for a rule. For Config Custom
         *        Policy rules, Config supports change-initiated notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config
         *        delivers an oversized configuration item. Config may generate this notification type when a resource
         *        changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypesWithStrings(Collection<String> organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy
         * rules, Config supports change-initiated notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers an
         * oversized configuration item. Config may generate this notification type when a resource changes and the
         * notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that initiates Config to run an evaluation for a rule. For Config Custom
         *        Policy rules, Config supports change-initiated notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config
         *        delivers an oversized configuration item. Config may generate this notification type when a resource
         *        changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypesWithStrings(String... organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy
         * rules, Config supports change-initiated notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers an
         * oversized configuration item. Config may generate this notification type when a resource changes and the
         * notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that initiates Config to run an evaluation for a rule. For Config Custom
         *        Policy rules, Config supports change-initiated notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config
         *        delivers an oversized configuration item. Config may generate this notification type when a resource
         *        changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypes(
                Collection<OrganizationConfigRuleTriggerTypeNoSN> organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * The type of notification that initiates Config to run an evaluation for a rule. For Config Custom Policy
         * rules, Config supports change-initiated notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers an
         * oversized configuration item. Config may generate this notification type when a resource changes and the
         * notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that initiates Config to run an evaluation for a rule. For Config Custom
         *        Policy rules, Config supports change-initiated notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Initiates an evaluation when Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Initiates an evaluation when Config
         *        delivers an oversized configuration item. Config may generate this notification type when a resource
         *        changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypes(OrganizationConfigRuleTriggerTypeNoSN... organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * A string, in JSON format, that is passed to your organization Config Custom Policy rule.
         * </p>
         * 
         * @param inputParameters
         *        A string, in JSON format, that is passed to your organization Config Custom Policy rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputParameters(String inputParameters);

        /**
         * <p>
         * The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is
         * triggered when Config delivers the configuration snapshot. For more information, see
         * <a>ConfigSnapshotDeliveryProperties</a>.
         * </p>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is
         *        triggered when Config delivers the configuration snapshot. For more information, see
         *        <a>ConfigSnapshotDeliveryProperties</a>.
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(String maximumExecutionFrequency);

        /**
         * <p>
         * The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is
         * triggered when Config delivers the configuration snapshot. For more information, see
         * <a>ConfigSnapshotDeliveryProperties</a>.
         * </p>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which Config runs evaluations for a rule. Your Config Custom Policy rule is
         *        triggered when Config delivers the configuration snapshot. For more information, see
         *        <a>ConfigSnapshotDeliveryProperties</a>.
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency);

        /**
         * <p>
         * The type of the Amazon Web Services resource that was evaluated.
         * </p>
         * 
         * @param resourceTypesScope
         *        The type of the Amazon Web Services resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesScope(Collection<String> resourceTypesScope);

        /**
         * <p>
         * The type of the Amazon Web Services resource that was evaluated.
         * </p>
         * 
         * @param resourceTypesScope
         *        The type of the Amazon Web Services resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesScope(String... resourceTypesScope);

        /**
         * <p>
         * The ID of the Amazon Web Services resource that was evaluated.
         * </p>
         * 
         * @param resourceIdScope
         *        The ID of the Amazon Web Services resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdScope(String resourceIdScope);

        /**
         * <p>
         * One part of a key-value pair that make up a tag. A key is a general label that acts like a category for more
         * specific tag values.
         * </p>
         * 
         * @param tagKeyScope
         *        One part of a key-value pair that make up a tag. A key is a general label that acts like a category
         *        for more specific tag values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagKeyScope(String tagKeyScope);

        /**
         * <p>
         * The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag category
         * (key).
         * </p>
         * 
         * @param tagValueScope
         *        The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag
         *        category (key).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagValueScope(String tagValueScope);

        /**
         * <p>
         * The runtime system for your organization Config Custom Policy rules. Guard is a policy-as-code language that
         * allows you to write policies that are enforced by Config Custom Policy rules. For more information about
         * Guard, see the <a href="https://github.com/aws-cloudformation/cloudformation-guard">Guard GitHub
         * Repository</a>.
         * </p>
         * 
         * @param policyRuntime
         *        The runtime system for your organization Config Custom Policy rules. Guard is a policy-as-code
         *        language that allows you to write policies that are enforced by Config Custom Policy rules. For more
         *        information about Guard, see the <a
         *        href="https://github.com/aws-cloudformation/cloudformation-guard">Guard GitHub Repository</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyRuntime(String policyRuntime);

        /**
         * <p>
         * The policy definition containing the logic for your organization Config Custom Policy rule.
         * </p>
         * 
         * @param policyText
         *        The policy definition containing the logic for your organization Config Custom Policy rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyText(String policyText);

        /**
         * <p>
         * A list of accounts that you can enable debug logging for your organization Config Custom Policy rule. List is
         * null when debug logging is enabled for all accounts.
         * </p>
         * 
         * @param debugLogDeliveryAccounts
         *        A list of accounts that you can enable debug logging for your organization Config Custom Policy rule.
         *        List is null when debug logging is enabled for all accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder debugLogDeliveryAccounts(Collection<String> debugLogDeliveryAccounts);

        /**
         * <p>
         * A list of accounts that you can enable debug logging for your organization Config Custom Policy rule. List is
         * null when debug logging is enabled for all accounts.
         * </p>
         * 
         * @param debugLogDeliveryAccounts
         *        A list of accounts that you can enable debug logging for your organization Config Custom Policy rule.
         *        List is null when debug logging is enabled for all accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder debugLogDeliveryAccounts(String... debugLogDeliveryAccounts);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private List<String> organizationConfigRuleTriggerTypes = DefaultSdkAutoConstructList.getInstance();

        private String inputParameters;

        private String maximumExecutionFrequency;

        private List<String> resourceTypesScope = DefaultSdkAutoConstructList.getInstance();

        private String resourceIdScope;

        private String tagKeyScope;

        private String tagValueScope;

        private String policyRuntime;

        private String policyText;

        private List<String> debugLogDeliveryAccounts = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OrganizationCustomPolicyRuleMetadata model) {
            description(model.description);
            organizationConfigRuleTriggerTypesWithStrings(model.organizationConfigRuleTriggerTypes);
            inputParameters(model.inputParameters);
            maximumExecutionFrequency(model.maximumExecutionFrequency);
            resourceTypesScope(model.resourceTypesScope);
            resourceIdScope(model.resourceIdScope);
            tagKeyScope(model.tagKeyScope);
            tagValueScope(model.tagValueScope);
            policyRuntime(model.policyRuntime);
            policyText(model.policyText);
            debugLogDeliveryAccounts(model.debugLogDeliveryAccounts);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getOrganizationConfigRuleTriggerTypes() {
            if (organizationConfigRuleTriggerTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return organizationConfigRuleTriggerTypes;
        }

        public final void setOrganizationConfigRuleTriggerTypes(Collection<String> organizationConfigRuleTriggerTypes) {
            this.organizationConfigRuleTriggerTypes = OrganizationConfigRuleTriggerTypeNoSNsCopier
                    .copy(organizationConfigRuleTriggerTypes);
        }

        @Override
        public final Builder organizationConfigRuleTriggerTypesWithStrings(Collection<String> organizationConfigRuleTriggerTypes) {
            this.organizationConfigRuleTriggerTypes = OrganizationConfigRuleTriggerTypeNoSNsCopier
                    .copy(organizationConfigRuleTriggerTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationConfigRuleTriggerTypesWithStrings(String... organizationConfigRuleTriggerTypes) {
            organizationConfigRuleTriggerTypesWithStrings(Arrays.asList(organizationConfigRuleTriggerTypes));
            return this;
        }

        @Override
        public final Builder organizationConfigRuleTriggerTypes(
                Collection<OrganizationConfigRuleTriggerTypeNoSN> organizationConfigRuleTriggerTypes) {
            this.organizationConfigRuleTriggerTypes = OrganizationConfigRuleTriggerTypeNoSNsCopier
                    .copyEnumToString(organizationConfigRuleTriggerTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationConfigRuleTriggerTypes(
                OrganizationConfigRuleTriggerTypeNoSN... organizationConfigRuleTriggerTypes) {
            organizationConfigRuleTriggerTypes(Arrays.asList(organizationConfigRuleTriggerTypes));
            return this;
        }

        public final String getInputParameters() {
            return inputParameters;
        }

        public final void setInputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
        }

        @Override
        public final Builder inputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
            return this;
        }

        public final String getMaximumExecutionFrequency() {
            return maximumExecutionFrequency;
        }

        public final void setMaximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
        }

        @Override
        public final Builder maximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
            return this;
        }

        @Override
        public final Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency) {
            this.maximumExecutionFrequency(maximumExecutionFrequency == null ? null : maximumExecutionFrequency.toString());
            return this;
        }

        public final Collection<String> getResourceTypesScope() {
            if (resourceTypesScope instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypesScope;
        }

        public final void setResourceTypesScope(Collection<String> resourceTypesScope) {
            this.resourceTypesScope = ResourceTypesScopeCopier.copy(resourceTypesScope);
        }

        @Override
        public final Builder resourceTypesScope(Collection<String> resourceTypesScope) {
            this.resourceTypesScope = ResourceTypesScopeCopier.copy(resourceTypesScope);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesScope(String... resourceTypesScope) {
            resourceTypesScope(Arrays.asList(resourceTypesScope));
            return this;
        }

        public final String getResourceIdScope() {
            return resourceIdScope;
        }

        public final void setResourceIdScope(String resourceIdScope) {
            this.resourceIdScope = resourceIdScope;
        }

        @Override
        public final Builder resourceIdScope(String resourceIdScope) {
            this.resourceIdScope = resourceIdScope;
            return this;
        }

        public final String getTagKeyScope() {
            return tagKeyScope;
        }

        public final void setTagKeyScope(String tagKeyScope) {
            this.tagKeyScope = tagKeyScope;
        }

        @Override
        public final Builder tagKeyScope(String tagKeyScope) {
            this.tagKeyScope = tagKeyScope;
            return this;
        }

        public final String getTagValueScope() {
            return tagValueScope;
        }

        public final void setTagValueScope(String tagValueScope) {
            this.tagValueScope = tagValueScope;
        }

        @Override
        public final Builder tagValueScope(String tagValueScope) {
            this.tagValueScope = tagValueScope;
            return this;
        }

        public final String getPolicyRuntime() {
            return policyRuntime;
        }

        public final void setPolicyRuntime(String policyRuntime) {
            this.policyRuntime = policyRuntime;
        }

        @Override
        public final Builder policyRuntime(String policyRuntime) {
            this.policyRuntime = policyRuntime;
            return this;
        }

        public final String getPolicyText() {
            return policyText;
        }

        public final void setPolicyText(String policyText) {
            this.policyText = policyText;
        }

        @Override
        public final Builder policyText(String policyText) {
            this.policyText = policyText;
            return this;
        }

        public final Collection<String> getDebugLogDeliveryAccounts() {
            if (debugLogDeliveryAccounts instanceof SdkAutoConstructList) {
                return null;
            }
            return debugLogDeliveryAccounts;
        }

        public final void setDebugLogDeliveryAccounts(Collection<String> debugLogDeliveryAccounts) {
            this.debugLogDeliveryAccounts = DebugLogDeliveryAccountsCopier.copy(debugLogDeliveryAccounts);
        }

        @Override
        public final Builder debugLogDeliveryAccounts(Collection<String> debugLogDeliveryAccounts) {
            this.debugLogDeliveryAccounts = DebugLogDeliveryAccountsCopier.copy(debugLogDeliveryAccounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder debugLogDeliveryAccounts(String... debugLogDeliveryAccounts) {
            debugLogDeliveryAccounts(Arrays.asList(debugLogDeliveryAccounts));
            return this;
        }

        @Override
        public OrganizationCustomPolicyRuleMetadata build() {
            return new OrganizationCustomPolicyRuleMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
