/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the default recording frequency that Config uses to record configuration changes. Config supports
 * <i>Continuous recording</i> and <i>Daily recording</i>.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Continuous recording allows you to record configuration changes continuously whenever a change occurs.
 * </p>
 * </li>
 * <li>
 * <p>
 * Daily recording allows you to receive a configuration item (CI) representing the most recent state of your resources
 * over the last 24-hour period, only if it’s different from the previous CI recorded.
 * </p>
 * </li>
 * </ul>
 * <note>
 * <p>
 * Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall Manager, it is
 * recommended that you set the recording frequency to Continuous.
 * </p>
 * </note>
 * <p>
 * You can also override the recording frequency for specific resource types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecordingMode implements SdkPojo, Serializable, ToCopyableBuilder<RecordingMode.Builder, RecordingMode> {
    private static final SdkField<String> RECORDING_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recordingFrequency").getter(getter(RecordingMode::recordingFrequencyAsString))
            .setter(setter(Builder::recordingFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordingFrequency").build())
            .build();

    private static final SdkField<List<RecordingModeOverride>> RECORDING_MODE_OVERRIDES_FIELD = SdkField
            .<List<RecordingModeOverride>> builder(MarshallingType.LIST)
            .memberName("recordingModeOverrides")
            .getter(getter(RecordingMode::recordingModeOverrides))
            .setter(setter(Builder::recordingModeOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordingModeOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RecordingModeOverride> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecordingModeOverride::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RECORDING_FREQUENCY_FIELD,
            RECORDING_MODE_OVERRIDES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String recordingFrequency;

    private final List<RecordingModeOverride> recordingModeOverrides;

    private RecordingMode(BuilderImpl builder) {
        this.recordingFrequency = builder.recordingFrequency;
        this.recordingModeOverrides = builder.recordingModeOverrides;
    }

    /**
     * <p>
     * The default recording frequency that Config uses to record configuration changes.
     * </p>
     * <important>
     * <p>
     * Daily recording cannot be specified for the following resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::Config::ResourceCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConformancePackCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConfigurationRecorder</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these resource types
     * will be set to Continuous recording.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recordingFrequency} will return {@link RecordingFrequency#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #recordingFrequencyAsString}.
     * </p>
     * 
     * @return The default recording frequency that Config uses to record configuration changes.</p> <important>
     *         <p>
     *         Daily recording cannot be specified for the following resource types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ResourceCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConformancePackCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConfigurationRecorder</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these
     *         resource types will be set to Continuous recording.
     *         </p>
     * @see RecordingFrequency
     */
    public final RecordingFrequency recordingFrequency() {
        return RecordingFrequency.fromValue(recordingFrequency);
    }

    /**
     * <p>
     * The default recording frequency that Config uses to record configuration changes.
     * </p>
     * <important>
     * <p>
     * Daily recording cannot be specified for the following resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::Config::ResourceCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConformancePackCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConfigurationRecorder</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these resource types
     * will be set to Continuous recording.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recordingFrequency} will return {@link RecordingFrequency#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #recordingFrequencyAsString}.
     * </p>
     * 
     * @return The default recording frequency that Config uses to record configuration changes.</p> <important>
     *         <p>
     *         Daily recording cannot be specified for the following resource types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ResourceCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConformancePackCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConfigurationRecorder</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these
     *         resource types will be set to Continuous recording.
     *         </p>
     * @see RecordingFrequency
     */
    public final String recordingFrequencyAsString() {
        return recordingFrequency;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecordingModeOverrides property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRecordingModeOverrides() {
        return recordingModeOverrides != null && !(recordingModeOverrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the recording mode.
     * The <code>recordingModeOverride</code> object in the <code>recordingModeOverrides</code> array consists of three
     * fields: a <code>description</code>, the new <code>recordingFrequency</code>, and an array of
     * <code>resourceTypes</code> to override.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecordingModeOverrides} method.
     * </p>
     * 
     * @return An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the
     *         recording mode. The <code>recordingModeOverride</code> object in the <code>recordingModeOverrides</code>
     *         array consists of three fields: a <code>description</code>, the new <code>recordingFrequency</code>, and
     *         an array of <code>resourceTypes</code> to override.
     */
    public final List<RecordingModeOverride> recordingModeOverrides() {
        return recordingModeOverrides;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(recordingFrequencyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecordingModeOverrides() ? recordingModeOverrides() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecordingMode)) {
            return false;
        }
        RecordingMode other = (RecordingMode) obj;
        return Objects.equals(recordingFrequencyAsString(), other.recordingFrequencyAsString())
                && hasRecordingModeOverrides() == other.hasRecordingModeOverrides()
                && Objects.equals(recordingModeOverrides(), other.recordingModeOverrides());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecordingMode").add("RecordingFrequency", recordingFrequencyAsString())
                .add("RecordingModeOverrides", hasRecordingModeOverrides() ? recordingModeOverrides() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "recordingFrequency":
            return Optional.ofNullable(clazz.cast(recordingFrequencyAsString()));
        case "recordingModeOverrides":
            return Optional.ofNullable(clazz.cast(recordingModeOverrides()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("recordingFrequency", RECORDING_FREQUENCY_FIELD);
        map.put("recordingModeOverrides", RECORDING_MODE_OVERRIDES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RecordingMode, T> g) {
        return obj -> g.apply((RecordingMode) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecordingMode> {
        /**
         * <p>
         * The default recording frequency that Config uses to record configuration changes.
         * </p>
         * <important>
         * <p>
         * Daily recording cannot be specified for the following resource types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::Config::ResourceCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConformancePackCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConfigurationRecorder</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these resource
         * types will be set to Continuous recording.
         * </p>
         * </important>
         * 
         * @param recordingFrequency
         *        The default recording frequency that Config uses to record configuration changes.</p> <important>
         *        <p>
         *        Daily recording cannot be specified for the following resource types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ResourceCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConformancePackCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConfigurationRecorder</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these
         *        resource types will be set to Continuous recording.
         *        </p>
         * @see RecordingFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordingFrequency
         */
        Builder recordingFrequency(String recordingFrequency);

        /**
         * <p>
         * The default recording frequency that Config uses to record configuration changes.
         * </p>
         * <important>
         * <p>
         * Daily recording cannot be specified for the following resource types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::Config::ResourceCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConformancePackCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConfigurationRecorder</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these resource
         * types will be set to Continuous recording.
         * </p>
         * </important>
         * 
         * @param recordingFrequency
         *        The default recording frequency that Config uses to record configuration changes.</p> <important>
         *        <p>
         *        Daily recording cannot be specified for the following resource types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ResourceCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConformancePackCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConfigurationRecorder</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For the <b>allSupported</b> (<code>ALL_SUPPORTED_RESOURCE_TYPES</code>) recording strategy, these
         *        resource types will be set to Continuous recording.
         *        </p>
         * @see RecordingFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordingFrequency
         */
        Builder recordingFrequency(RecordingFrequency recordingFrequency);

        /**
         * <p>
         * An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the recording
         * mode. The <code>recordingModeOverride</code> object in the <code>recordingModeOverrides</code> array consists
         * of three fields: a <code>description</code>, the new <code>recordingFrequency</code>, and an array of
         * <code>resourceTypes</code> to override.
         * </p>
         * 
         * @param recordingModeOverrides
         *        An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the
         *        recording mode. The <code>recordingModeOverride</code> object in the
         *        <code>recordingModeOverrides</code> array consists of three fields: a <code>description</code>, the
         *        new <code>recordingFrequency</code>, and an array of <code>resourceTypes</code> to override.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordingModeOverrides(Collection<RecordingModeOverride> recordingModeOverrides);

        /**
         * <p>
         * An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the recording
         * mode. The <code>recordingModeOverride</code> object in the <code>recordingModeOverrides</code> array consists
         * of three fields: a <code>description</code>, the new <code>recordingFrequency</code>, and an array of
         * <code>resourceTypes</code> to override.
         * </p>
         * 
         * @param recordingModeOverrides
         *        An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the
         *        recording mode. The <code>recordingModeOverride</code> object in the
         *        <code>recordingModeOverrides</code> array consists of three fields: a <code>description</code>, the
         *        new <code>recordingFrequency</code>, and an array of <code>resourceTypes</code> to override.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordingModeOverrides(RecordingModeOverride... recordingModeOverrides);

        /**
         * <p>
         * An array of <code>recordingModeOverride</code> objects for you to specify your overrides for the recording
         * mode. The <code>recordingModeOverride</code> object in the <code>recordingModeOverrides</code> array consists
         * of three fields: a <code>description</code>, the new <code>recordingFrequency</code>, and an array of
         * <code>resourceTypes</code> to override.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.config.model.RecordingModeOverride.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.config.model.RecordingModeOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.config.model.RecordingModeOverride.Builder#build()} is called
         * immediately and its result is passed to {@link #recordingModeOverrides(List<RecordingModeOverride>)}.
         * 
         * @param recordingModeOverrides
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.config.model.RecordingModeOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordingModeOverrides(java.util.Collection<RecordingModeOverride>)
         */
        Builder recordingModeOverrides(Consumer<RecordingModeOverride.Builder>... recordingModeOverrides);
    }

    static final class BuilderImpl implements Builder {
        private String recordingFrequency;

        private List<RecordingModeOverride> recordingModeOverrides = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RecordingMode model) {
            recordingFrequency(model.recordingFrequency);
            recordingModeOverrides(model.recordingModeOverrides);
        }

        public final String getRecordingFrequency() {
            return recordingFrequency;
        }

        public final void setRecordingFrequency(String recordingFrequency) {
            this.recordingFrequency = recordingFrequency;
        }

        @Override
        public final Builder recordingFrequency(String recordingFrequency) {
            this.recordingFrequency = recordingFrequency;
            return this;
        }

        @Override
        public final Builder recordingFrequency(RecordingFrequency recordingFrequency) {
            this.recordingFrequency(recordingFrequency == null ? null : recordingFrequency.toString());
            return this;
        }

        public final List<RecordingModeOverride.Builder> getRecordingModeOverrides() {
            List<RecordingModeOverride.Builder> result = RecordingModeOverridesCopier.copyToBuilder(this.recordingModeOverrides);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecordingModeOverrides(Collection<RecordingModeOverride.BuilderImpl> recordingModeOverrides) {
            this.recordingModeOverrides = RecordingModeOverridesCopier.copyFromBuilder(recordingModeOverrides);
        }

        @Override
        public final Builder recordingModeOverrides(Collection<RecordingModeOverride> recordingModeOverrides) {
            this.recordingModeOverrides = RecordingModeOverridesCopier.copy(recordingModeOverrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordingModeOverrides(RecordingModeOverride... recordingModeOverrides) {
            recordingModeOverrides(Arrays.asList(recordingModeOverrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordingModeOverrides(Consumer<RecordingModeOverride.Builder>... recordingModeOverrides) {
            recordingModeOverrides(Stream.of(recordingModeOverrides)
                    .map(c -> RecordingModeOverride.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public RecordingMode build() {
            return new RecordingMode(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
