/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.controlcatalog;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.controlcatalog.model.GetControlRequest;
import software.amazon.awssdk.services.controlcatalog.model.GetControlResponse;
import software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse;
import software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListControlsResponse;
import software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse;
import software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse;
import software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsPublisher;
import software.amazon.awssdk.services.controlcatalog.paginators.ListControlsPublisher;
import software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsPublisher;
import software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesPublisher;

/**
 * Service client for accessing AWS Control Catalog asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Welcome to the Amazon Web Services Control Catalog API reference. This guide is for developers who need detailed
 * information about how to programmatically identify and filter the common controls and related metadata that are
 * available to Amazon Web Services customers. This API reference provides descriptions, syntax, and usage examples for
 * each of the actions and data types that are supported by Amazon Web Services Control Catalog.
 * </p>
 * <p>
 * Use the following links to get started with the Amazon Web Services Control Catalog API:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_Operations.html">Actions</a>: An
 * alphabetical list of all Control Catalog API operations.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_Types.html">Data types</a>: An
 * alphabetical list of all Control Catalog data types.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/CommonParameters.html">Common parameters</a>:
 * Parameters that all operations can use.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/CommonErrors.html">Common errors</a>: Client
 * and server errors that all operations can return.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ControlCatalogAsyncClient extends AwsClient {
    String SERVICE_NAME = "controlcatalog";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "controlcatalog";

    /**
     * <p>
     * Returns details about a specific control, most notably a list of Amazon Web Services Regions where this control
     * is supported. Input a value for the <i>ControlArn</i> parameter, in ARN form. <code>GetControl</code> accepts
     * <i>controltower</i> or <i>controlcatalog</i> control ARNs as input. Returns a <i>controlcatalog</i> ARN format.
     * </p>
     * <p>
     * In the API response, controls that have the value <code>GLOBAL</code> in the <code>Scope</code> field do not show
     * the <code>DeployableRegions</code> field, because it does not apply. Controls that have the value
     * <code>REGIONAL</code> in the <code>Scope</code> field return a value for the <code>DeployableRegions</code>
     * field, as shown in the example.
     * </p>
     *
     * @param getControlRequest
     * @return A Java Future containing the result of the GetControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.GetControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/GetControl" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetControlResponse> getControl(GetControlRequest getControlRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns details about a specific control, most notably a list of Amazon Web Services Regions where this control
     * is supported. Input a value for the <i>ControlArn</i> parameter, in ARN form. <code>GetControl</code> accepts
     * <i>controltower</i> or <i>controlcatalog</i> control ARNs as input. Returns a <i>controlcatalog</i> ARN format.
     * </p>
     * <p>
     * In the API response, controls that have the value <code>GLOBAL</code> in the <code>Scope</code> field do not show
     * the <code>DeployableRegions</code> field, because it does not apply. Controls that have the value
     * <code>REGIONAL</code> in the <code>Scope</code> field return a value for the <code>DeployableRegions</code>
     * field, as shown in the example.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetControlRequest.Builder} avoiding the need to
     * create one manually via {@link GetControlRequest#builder()}
     * </p>
     *
     * @param getControlRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.GetControlRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.GetControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/GetControl" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetControlResponse> getControl(Consumer<GetControlRequest.Builder> getControlRequest) {
        return getControl(GetControlRequest.builder().applyMutation(getControlRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of common controls from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see common controls that have a specific objective. If you don’t provide a
     * filter, the operation returns all common controls.
     * </p>
     *
     * @param listCommonControlsRequest
     * @return A Java Future containing the result of the ListCommonControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListCommonControlsResponse> listCommonControls(ListCommonControlsRequest listCommonControlsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of common controls from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see common controls that have a specific objective. If you don’t provide a
     * filter, the operation returns all common controls.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListCommonControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListCommonControlsRequest#builder()}
     * </p>
     *
     * @param listCommonControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListCommonControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListCommonControlsResponse> listCommonControls(
            Consumer<ListCommonControlsRequest.Builder> listCommonControlsRequest) {
        return listCommonControls(ListCommonControlsRequest.builder().applyMutation(listCommonControlsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsPublisher publisher = client.listCommonControlsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsPublisher publisher = client.listCommonControlsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCommonControlsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCommonControlsPublisher listCommonControlsPaginator(ListCommonControlsRequest listCommonControlsRequest) {
        return new ListCommonControlsPublisher(this, listCommonControlsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsPublisher publisher = client.listCommonControlsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsPublisher publisher = client.listCommonControlsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListCommonControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListCommonControlsRequest#builder()}
     * </p>
     *
     * @param listCommonControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCommonControlsPublisher listCommonControlsPaginator(
            Consumer<ListCommonControlsRequest.Builder> listCommonControlsRequest) {
        return listCommonControlsPaginator(ListCommonControlsRequest.builder().applyMutation(listCommonControlsRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of all available controls in the Amazon Web Services Control Catalog library. Allows you
     * to discover available controls. The list of controls is given as structures of type <i>controlSummary</i>. The
     * ARN is returned in the global <i>controlcatalog</i> format, as shown in the examples.
     * </p>
     *
     * @param listControlsRequest
     * @return A Java Future containing the result of the ListControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListControls" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListControlsResponse> listControls(ListControlsRequest listControlsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of all available controls in the Amazon Web Services Control Catalog library. Allows you
     * to discover available controls. The list of controls is given as structures of type <i>controlSummary</i>. The
     * ARN is returned in the global <i>controlcatalog</i> format, as shown in the examples.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListControlsRequest.Builder} avoiding the need to
     * create one manually via {@link ListControlsRequest#builder()}
     * </p>
     *
     * @param listControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListControls" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListControlsResponse> listControls(Consumer<ListControlsRequest.Builder> listControlsRequest) {
        return listControls(ListControlsRequest.builder().applyMutation(listControlsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listControls(software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListControlsPublisher publisher = client.listControlsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListControlsPublisher publisher = client.listControlsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListControlsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListControlsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listControls(software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest)} operation.</b>
     * </p>
     *
     * @param listControlsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListControls" target="_top">AWS
     *      API Documentation</a>
     */
    default ListControlsPublisher listControlsPaginator(ListControlsRequest listControlsRequest) {
        return new ListControlsPublisher(this, listControlsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listControls(software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListControlsPublisher publisher = client.listControlsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListControlsPublisher publisher = client.listControlsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListControlsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListControlsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listControls(software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListControlsRequest.Builder} avoiding the need to
     * create one manually via {@link ListControlsRequest#builder()}
     * </p>
     *
     * @param listControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListControlsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListControls" target="_top">AWS
     *      API Documentation</a>
     */
    default ListControlsPublisher listControlsPaginator(Consumer<ListControlsRequest.Builder> listControlsRequest) {
        return listControlsPaginator(ListControlsRequest.builder().applyMutation(listControlsRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of domains from the Amazon Web Services Control Catalog.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of domains from the Amazon Web Services Control Catalog.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDomainsRequest.Builder} avoiding the need to
     * create one manually via {@link ListDomainsRequest#builder()}
     * </p>
     *
     * @param listDomainsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListDomainsResponse> listDomains(Consumer<ListDomainsRequest.Builder> listDomainsRequest) {
        return listDomains(ListDomainsRequest.builder().applyMutation(listDomainsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDomainsPublisher listDomainsPaginator(ListDomainsRequest listDomainsRequest) {
        return new ListDomainsPublisher(this, listDomainsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDomainsRequest.Builder} avoiding the need to
     * create one manually via {@link ListDomainsRequest#builder()}
     * </p>
     *
     * @param listDomainsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDomainsPublisher listDomainsPaginator(Consumer<ListDomainsRequest.Builder> listDomainsRequest) {
        return listDomainsPaginator(ListDomainsRequest.builder().applyMutation(listDomainsRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of objectives from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see the objectives that belong to a specific domain. If you don’t provide a
     * filter, the operation returns all objectives.
     * </p>
     *
     * @param listObjectivesRequest
     * @return A Java Future containing the result of the ListObjectives operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListObjectivesResponse> listObjectives(ListObjectivesRequest listObjectivesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of objectives from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see the objectives that belong to a specific domain. If you don’t provide a
     * filter, the operation returns all objectives.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectivesRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectivesRequest#builder()}
     * </p>
     *
     * @param listObjectivesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListObjectives operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListObjectivesResponse> listObjectives(Consumer<ListObjectivesRequest.Builder> listObjectivesRequest) {
        return listObjectives(ListObjectivesRequest.builder().applyMutation(listObjectivesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesPublisher publisher = client.listObjectivesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesPublisher publisher = client.listObjectivesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listObjectivesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default ListObjectivesPublisher listObjectivesPaginator(ListObjectivesRequest listObjectivesRequest) {
        return new ListObjectivesPublisher(this, listObjectivesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesPublisher publisher = client.listObjectivesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesPublisher publisher = client.listObjectivesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectivesRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectivesRequest#builder()}
     * </p>
     *
     * @param listObjectivesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlCatalogException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ControlCatalogAsyncClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default ListObjectivesPublisher listObjectivesPaginator(Consumer<ListObjectivesRequest.Builder> listObjectivesRequest) {
        return listObjectivesPaginator(ListObjectivesRequest.builder().applyMutation(listObjectivesRequest).build());
    }

    @Override
    default ControlCatalogServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link ControlCatalogAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ControlCatalogAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ControlCatalogAsyncClient}.
     */
    static ControlCatalogAsyncClientBuilder builder() {
        return new DefaultControlCatalogAsyncClientBuilder();
    }
}
