/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The type and amount of a resource to assign to a container. The supported resource types are GPUs and Elastic
 * Inference accelerators. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-gpu.html">Working with GPUs on Amazon ECS</a>
 * or <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-inference.html">Working with Amazon
 * Elastic Inference on Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceRequirement implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceRequirement.Builder, ResourceRequirement> {
    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("value")
            .getter(getter(ResourceRequirement::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("value").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ResourceRequirement::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VALUE_FIELD, TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String value;

    private final String type;

    private ResourceRequirement(BuilderImpl builder) {
        this.value = builder.value;
        this.type = builder.type;
    }

    /**
     * <p>
     * The value for the specified resource type.
     * </p>
     * <p>
     * When the type is <code>GPU</code>, the value is the number of physical <code>GPUs</code> the Amazon ECS container
     * agent reserves for the container. The number of GPUs that's reserved for all containers in a task can't exceed
     * the number of available GPUs on the container instance that the task is launched on.
     * </p>
     * <p>
     * When the type is <code>InferenceAccelerator</code>, the <code>value</code> matches the <code>deviceName</code>
     * for an <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_InferenceAccelerator.html">
     * InferenceAccelerator</a> specified in a task definition.
     * </p>
     * 
     * @return The value for the specified resource type.</p>
     *         <p>
     *         When the type is <code>GPU</code>, the value is the number of physical <code>GPUs</code> the Amazon ECS
     *         container agent reserves for the container. The number of GPUs that's reserved for all containers in a
     *         task can't exceed the number of available GPUs on the container instance that the task is launched on.
     *         </p>
     *         <p>
     *         When the type is <code>InferenceAccelerator</code>, the <code>value</code> matches the
     *         <code>deviceName</code> for an <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_InferenceAccelerator.html"
     *         >InferenceAccelerator</a> specified in a task definition.
     */
    public final String value() {
        return value;
    }

    /**
     * <p>
     * The type of resource to assign to a container.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of resource to assign to a container.
     * @see ResourceType
     */
    public final ResourceType type() {
        return ResourceType.fromValue(type);
    }

    /**
     * <p>
     * The type of resource to assign to a container.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of resource to assign to a container.
     * @see ResourceType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceRequirement)) {
            return false;
        }
        ResourceRequirement other = (ResourceRequirement) obj;
        return Objects.equals(value(), other.value()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceRequirement").add("Value", value()).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "value":
            return Optional.ofNullable(clazz.cast(value()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("value", VALUE_FIELD);
        map.put("type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceRequirement, T> g) {
        return obj -> g.apply((ResourceRequirement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceRequirement> {
        /**
         * <p>
         * The value for the specified resource type.
         * </p>
         * <p>
         * When the type is <code>GPU</code>, the value is the number of physical <code>GPUs</code> the Amazon ECS
         * container agent reserves for the container. The number of GPUs that's reserved for all containers in a task
         * can't exceed the number of available GPUs on the container instance that the task is launched on.
         * </p>
         * <p>
         * When the type is <code>InferenceAccelerator</code>, the <code>value</code> matches the
         * <code>deviceName</code> for an <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_InferenceAccelerator.html"
         * >InferenceAccelerator</a> specified in a task definition.
         * </p>
         * 
         * @param value
         *        The value for the specified resource type.</p>
         *        <p>
         *        When the type is <code>GPU</code>, the value is the number of physical <code>GPUs</code> the Amazon
         *        ECS container agent reserves for the container. The number of GPUs that's reserved for all containers
         *        in a task can't exceed the number of available GPUs on the container instance that the task is
         *        launched on.
         *        </p>
         *        <p>
         *        When the type is <code>InferenceAccelerator</code>, the <code>value</code> matches the
         *        <code>deviceName</code> for an <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_InferenceAccelerator.html"
         *        >InferenceAccelerator</a> specified in a task definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The type of resource to assign to a container.
         * </p>
         * 
         * @param type
         *        The type of resource to assign to a container.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of resource to assign to a container.
         * </p>
         * 
         * @param type
         *        The type of resource to assign to a container.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder type(ResourceType type);
    }

    static final class BuilderImpl implements Builder {
        private String value;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceRequirement model) {
            value(model.value);
            type(model.type);
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ResourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public ResourceRequirement build() {
            return new ResourceRequirement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
