/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the circuit breaker used to determine when a service deployment has failed.
 * </p>
 * <p>
 * The deployment circuit breaker is the rolling update mechanism that determines if the tasks reach a steady state. The
 * deployment circuit breaker has an option that will automatically roll back a failed deployment to the last cpompleted
 * service revision. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-circuit-breaker.html">How the Amazon ECS
 * deployment circuit breaker detects failures</a> in the<i> Amazon ECS Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceDeploymentCircuitBreaker implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceDeploymentCircuitBreaker.Builder, ServiceDeploymentCircuitBreaker> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ServiceDeploymentCircuitBreaker::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Integer> FAILURE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("failureCount").getter(getter(ServiceDeploymentCircuitBreaker::failureCount))
            .setter(setter(Builder::failureCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureCount").build()).build();

    private static final SdkField<Integer> THRESHOLD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("threshold").getter(getter(ServiceDeploymentCircuitBreaker::threshold))
            .setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threshold").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            FAILURE_COUNT_FIELD, THRESHOLD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final Integer failureCount;

    private final Integer threshold;

    private ServiceDeploymentCircuitBreaker(BuilderImpl builder) {
        this.status = builder.status;
        this.failureCount = builder.failureCount;
        this.threshold = builder.threshold;
    }

    /**
     * <p>
     * The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment failures when the
     * status is <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceDeploymentRollbackMonitorsStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #statusAsString}.
     * </p>
     * 
     * @return The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment failures
     *         when the status is <code>DISABLED</code>.
     * @see ServiceDeploymentRollbackMonitorsStatus
     */
    public final ServiceDeploymentRollbackMonitorsStatus status() {
        return ServiceDeploymentRollbackMonitorsStatus.fromValue(status);
    }

    /**
     * <p>
     * The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment failures when the
     * status is <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceDeploymentRollbackMonitorsStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #statusAsString}.
     * </p>
     * 
     * @return The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment failures
     *         when the status is <code>DISABLED</code>.
     * @see ServiceDeploymentRollbackMonitorsStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The number of times the circuit breaker detected a service deploymeny failure.
     * </p>
     * 
     * @return The number of times the circuit breaker detected a service deploymeny failure.
     */
    public final Integer failureCount() {
        return failureCount;
    }

    /**
     * <p>
     * The threshhold which determines that the service deployment failed.
     * </p>
     * <p>
     * The deployment circuit breaker calculates the threshold value, and then uses the value to determine when to move
     * the deployment to a FAILED state. The deployment circuit breaker has a minimum threshold of 3 and a maximum
     * threshold of 200. and uses the values in the following formula to determine the deployment failure.
     * </p>
     * <p>
     * <code>0.5 * desired task count</code>
     * </p>
     * 
     * @return The threshhold which determines that the service deployment failed.</p>
     *         <p>
     *         The deployment circuit breaker calculates the threshold value, and then uses the value to determine when
     *         to move the deployment to a FAILED state. The deployment circuit breaker has a minimum threshold of 3 and
     *         a maximum threshold of 200. and uses the values in the following formula to determine the deployment
     *         failure.
     *         </p>
     *         <p>
     *         <code>0.5 * desired task count</code>
     */
    public final Integer threshold() {
        return threshold;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureCount());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceDeploymentCircuitBreaker)) {
            return false;
        }
        ServiceDeploymentCircuitBreaker other = (ServiceDeploymentCircuitBreaker) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(failureCount(), other.failureCount())
                && Objects.equals(threshold(), other.threshold());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceDeploymentCircuitBreaker").add("Status", statusAsString())
                .add("FailureCount", failureCount()).add("Threshold", threshold()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "failureCount":
            return Optional.ofNullable(clazz.cast(failureCount()));
        case "threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("status", STATUS_FIELD);
        map.put("failureCount", FAILURE_COUNT_FIELD);
        map.put("threshold", THRESHOLD_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ServiceDeploymentCircuitBreaker, T> g) {
        return obj -> g.apply((ServiceDeploymentCircuitBreaker) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceDeploymentCircuitBreaker> {
        /**
         * <p>
         * The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment failures when
         * the status is <code>DISABLED</code>.
         * </p>
         * 
         * @param status
         *        The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment
         *        failures when the status is <code>DISABLED</code>.
         * @see ServiceDeploymentRollbackMonitorsStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceDeploymentRollbackMonitorsStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment failures when
         * the status is <code>DISABLED</code>.
         * </p>
         * 
         * @param status
         *        The circuit breaker status. Amazon ECS is not using the circuit breaker for service deployment
         *        failures when the status is <code>DISABLED</code>.
         * @see ServiceDeploymentRollbackMonitorsStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceDeploymentRollbackMonitorsStatus
         */
        Builder status(ServiceDeploymentRollbackMonitorsStatus status);

        /**
         * <p>
         * The number of times the circuit breaker detected a service deploymeny failure.
         * </p>
         * 
         * @param failureCount
         *        The number of times the circuit breaker detected a service deploymeny failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureCount(Integer failureCount);

        /**
         * <p>
         * The threshhold which determines that the service deployment failed.
         * </p>
         * <p>
         * The deployment circuit breaker calculates the threshold value, and then uses the value to determine when to
         * move the deployment to a FAILED state. The deployment circuit breaker has a minimum threshold of 3 and a
         * maximum threshold of 200. and uses the values in the following formula to determine the deployment failure.
         * </p>
         * <p>
         * <code>0.5 * desired task count</code>
         * </p>
         * 
         * @param threshold
         *        The threshhold which determines that the service deployment failed.</p>
         *        <p>
         *        The deployment circuit breaker calculates the threshold value, and then uses the value to determine
         *        when to move the deployment to a FAILED state. The deployment circuit breaker has a minimum threshold
         *        of 3 and a maximum threshold of 200. and uses the values in the following formula to determine the
         *        deployment failure.
         *        </p>
         *        <p>
         *        <code>0.5 * desired task count</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Integer threshold);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private Integer failureCount;

        private Integer threshold;

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceDeploymentCircuitBreaker model) {
            status(model.status);
            failureCount(model.failureCount);
            threshold(model.threshold);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ServiceDeploymentRollbackMonitorsStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Integer getFailureCount() {
            return failureCount;
        }

        public final void setFailureCount(Integer failureCount) {
            this.failureCount = failureCount;
        }

        @Override
        public final Builder failureCount(Integer failureCount) {
            this.failureCount = failureCount;
            return this;
        }

        public final Integer getThreshold() {
            return threshold;
        }

        public final void setThreshold(Integer threshold) {
            this.threshold = threshold;
        }

        @Override
        public final Builder threshold(Integer threshold) {
            this.threshold = threshold;
            return this;
        }

        @Override
        public ServiceDeploymentCircuitBreaker build() {
            return new ServiceDeploymentCircuitBreaker(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
