/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateDataRepositoryTaskRequest extends FSxRequest implements
        ToCopyableBuilder<CreateDataRepositoryTaskRequest.Builder, CreateDataRepositoryTaskRequest> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(CreateDataRepositoryTaskRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<String>> PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Paths")
            .getter(getter(CreateDataRepositoryTaskRequest::paths))
            .setter(setter(Builder::paths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Paths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(CreateDataRepositoryTaskRequest::fileSystemId))
            .setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<CompletionReport> REPORT_FIELD = SdkField.<CompletionReport> builder(MarshallingType.SDK_POJO)
            .memberName("Report").getter(getter(CreateDataRepositoryTaskRequest::report)).setter(setter(Builder::report))
            .constructor(CompletionReport::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Report").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(CreateDataRepositoryTaskRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateDataRepositoryTaskRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> CAPACITY_TO_RELEASE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CapacityToRelease").getter(getter(CreateDataRepositoryTaskRequest::capacityToRelease))
            .setter(setter(Builder::capacityToRelease))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityToRelease").build()).build();

    private static final SdkField<ReleaseConfiguration> RELEASE_CONFIGURATION_FIELD = SdkField
            .<ReleaseConfiguration> builder(MarshallingType.SDK_POJO).memberName("ReleaseConfiguration")
            .getter(getter(CreateDataRepositoryTaskRequest::releaseConfiguration)).setter(setter(Builder::releaseConfiguration))
            .constructor(ReleaseConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, PATHS_FIELD,
            FILE_SYSTEM_ID_FIELD, REPORT_FIELD, CLIENT_REQUEST_TOKEN_FIELD, TAGS_FIELD, CAPACITY_TO_RELEASE_FIELD,
            RELEASE_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String type;

    private final List<String> paths;

    private final String fileSystemId;

    private final CompletionReport report;

    private final String clientRequestToken;

    private final List<Tag> tags;

    private final Long capacityToRelease;

    private final ReleaseConfiguration releaseConfiguration;

    private CreateDataRepositoryTaskRequest(BuilderImpl builder) {
        super(builder);
        this.type = builder.type;
        this.paths = builder.paths;
        this.fileSystemId = builder.fileSystemId;
        this.report = builder.report;
        this.clientRequestToken = builder.clientRequestToken;
        this.tags = builder.tags;
        this.capacityToRelease = builder.capacityToRelease;
        this.releaseConfiguration = builder.releaseConfiguration;
    }

    /**
     * <p>
     * Specifies the type of data repository task to create.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked data
     * repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to your Amazon
     * FSx for Lustre file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file system that have
     * been exported to a linked S3 bucket and that meet your specified release criteria.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataRepositoryTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies the type of data repository task to create.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked
     *         data repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to
     *         your Amazon FSx for Lustre file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file system
     *         that have been exported to a linked S3 bucket and that meet your specified release criteria.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
     *         </p>
     *         </li>
     * @see DataRepositoryTaskType
     */
    public final DataRepositoryTaskType type() {
        return DataRepositoryTaskType.fromValue(type);
    }

    /**
     * <p>
     * Specifies the type of data repository task to create.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked data
     * repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to your Amazon
     * FSx for Lustre file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file system that have
     * been exported to a linked S3 bucket and that meet your specified release criteria.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataRepositoryTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies the type of data repository task to create.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked
     *         data repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to
     *         your Amazon FSx for Lustre file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file system
     *         that have been exported to a linked S3 bucket and that meet your specified release criteria.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
     *         </p>
     *         </li>
     * @see DataRepositoryTaskType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * For responses, this returns true if the service returned a value for the Paths property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasPaths() {
        return paths != null && !(paths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of paths for the data repository task to use when the task is processed. If a path that you provide isn't
     * valid, the task fails. If you don't provide paths, the default behavior is to export all files to S3 (for export
     * tasks), import all files from S3 (for import tasks), or release all exported files that meet the last accessed
     * time criteria (for release tasks).
     * </p>
     * <ul>
     * <li>
     * <p>
     * For export tasks, the list contains paths on the FSx for Lustre file system from which the files are exported to
     * the Amazon S3 bucket. The default path is the file system root directory. The paths you provide need to be
     * relative to the mount point of the file system. If the mount point is <code>/mnt/fsx</code> and
     * <code>/mnt/fsx/path1</code> is a directory or file on the file system you want to export, then the path to
     * provide is <code>path1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For import tasks, the list contains paths in the Amazon S3 bucket from which POSIX metadata changes are imported
     * to the FSx for Lustre file system. The path can be an S3 bucket or prefix in the format
     * <code>s3://bucket-name/prefix</code> (where <code>prefix</code> is optional).
     * </p>
     * </li>
     * <li>
     * <p>
     * For release tasks, the list contains directory or file paths on the FSx for Lustre file system from which to
     * release exported files. If a directory is specified, files within the directory are released. If a file path is
     * specified, only that file is released. To release all exported files in the file system, specify a forward slash
     * (/) as the path.
     * </p>
     * <note>
     * <p>
     * A file must also meet the last accessed time criteria specified in for the file to be released.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPaths} method.
     * </p>
     * 
     * @return A list of paths for the data repository task to use when the task is processed. If a path that you
     *         provide isn't valid, the task fails. If you don't provide paths, the default behavior is to export all
     *         files to S3 (for export tasks), import all files from S3 (for import tasks), or release all exported
     *         files that meet the last accessed time criteria (for release tasks).</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For export tasks, the list contains paths on the FSx for Lustre file system from which the files are
     *         exported to the Amazon S3 bucket. The default path is the file system root directory. The paths you
     *         provide need to be relative to the mount point of the file system. If the mount point is
     *         <code>/mnt/fsx</code> and <code>/mnt/fsx/path1</code> is a directory or file on the file system you want
     *         to export, then the path to provide is <code>path1</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For import tasks, the list contains paths in the Amazon S3 bucket from which POSIX metadata changes are
     *         imported to the FSx for Lustre file system. The path can be an S3 bucket or prefix in the format
     *         <code>s3://bucket-name/prefix</code> (where <code>prefix</code> is optional).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For release tasks, the list contains directory or file paths on the FSx for Lustre file system from which
     *         to release exported files. If a directory is specified, files within the directory are released. If a
     *         file path is specified, only that file is released. To release all exported files in the file system,
     *         specify a forward slash (/) as the path.
     *         </p>
     *         <note>
     *         <p>
     *         A file must also meet the last accessed time criteria specified in for the file to be released.
     *         </p>
     *         </note></li>
     */
    public final List<String> paths() {
        return paths;
    }

    /**
     * Returns the value of the FileSystemId property for this object.
     * 
     * @return The value of the FileSystemId property for this object.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * Defines whether or not Amazon FSx provides a CompletionReport once the task has completed. A CompletionReport
     * provides a detailed report on the files that Amazon FSx processed that meet the criteria specified by the
     * <code>Scope</code> parameter. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/task-completion-report.html">Working with Task
     * Completion Reports</a>.
     * </p>
     * 
     * @return Defines whether or not Amazon FSx provides a CompletionReport once the task has completed. A
     *         CompletionReport provides a detailed report on the files that Amazon FSx processed that meet the criteria
     *         specified by the <code>Scope</code> parameter. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/task-completion-report.html">Working with Task
     *         Completion Reports</a>.
     */
    public final CompletionReport report() {
        return report;
    }

    /**
     * Returns the value of the ClientRequestToken property for this object.
     * 
     * @return The value of the ClientRequestToken property for this object.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies the amount of data to release, in GiB, by an Amazon File Cache <code>AUTO_RELEASE_DATA</code> task that
     * automatically releases files from the cache.
     * </p>
     * 
     * @return Specifies the amount of data to release, in GiB, by an Amazon File Cache <code>AUTO_RELEASE_DATA</code>
     *         task that automatically releases files from the cache.
     */
    public final Long capacityToRelease() {
        return capacityToRelease;
    }

    /**
     * <p>
     * The configuration that specifies the last accessed time criteria for files that will be released from an Amazon
     * FSx for Lustre file system.
     * </p>
     * 
     * @return The configuration that specifies the last accessed time criteria for files that will be released from an
     *         Amazon FSx for Lustre file system.
     */
    public final ReleaseConfiguration releaseConfiguration() {
        return releaseConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPaths() ? paths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(report());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(capacityToRelease());
        hashCode = 31 * hashCode + Objects.hashCode(releaseConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateDataRepositoryTaskRequest)) {
            return false;
        }
        CreateDataRepositoryTaskRequest other = (CreateDataRepositoryTaskRequest) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && hasPaths() == other.hasPaths()
                && Objects.equals(paths(), other.paths()) && Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(report(), other.report()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(capacityToRelease(), other.capacityToRelease())
                && Objects.equals(releaseConfiguration(), other.releaseConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateDataRepositoryTaskRequest").add("Type", typeAsString())
                .add("Paths", hasPaths() ? paths() : null).add("FileSystemId", fileSystemId()).add("Report", report())
                .add("ClientRequestToken", clientRequestToken()).add("Tags", hasTags() ? tags() : null)
                .add("CapacityToRelease", capacityToRelease()).add("ReleaseConfiguration", releaseConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Paths":
            return Optional.ofNullable(clazz.cast(paths()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "Report":
            return Optional.ofNullable(clazz.cast(report()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "CapacityToRelease":
            return Optional.ofNullable(clazz.cast(capacityToRelease()));
        case "ReleaseConfiguration":
            return Optional.ofNullable(clazz.cast(releaseConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("Paths", PATHS_FIELD);
        map.put("FileSystemId", FILE_SYSTEM_ID_FIELD);
        map.put("Report", REPORT_FIELD);
        map.put("ClientRequestToken", CLIENT_REQUEST_TOKEN_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("CapacityToRelease", CAPACITY_TO_RELEASE_FIELD);
        map.put("ReleaseConfiguration", RELEASE_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateDataRepositoryTaskRequest, T> g) {
        return obj -> g.apply((CreateDataRepositoryTaskRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends FSxRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateDataRepositoryTaskRequest> {
        /**
         * <p>
         * Specifies the type of data repository task to create.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked data
         * repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to your
         * Amazon FSx for Lustre file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file system that
         * have been exported to a linked S3 bucket and that meet your specified release criteria.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        Specifies the type of data repository task to create.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked
         *        data repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to
         *        your Amazon FSx for Lustre file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file
         *        system that have been exported to a linked S3 bucket and that meet your specified release criteria.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
         *        </p>
         *        </li>
         * @see DataRepositoryTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryTaskType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies the type of data repository task to create.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked data
         * repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to your
         * Amazon FSx for Lustre file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file system that
         * have been exported to a linked S3 bucket and that meet your specified release criteria.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        Specifies the type of data repository task to create.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EXPORT_TO_REPOSITORY</code> tasks export from your Amazon FSx for Lustre file system to a linked
         *        data repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IMPORT_METADATA_FROM_REPOSITORY</code> tasks import metadata changes from a linked S3 bucket to
         *        your Amazon FSx for Lustre file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RELEASE_DATA_FROM_FILESYSTEM</code> tasks release files in your Amazon FSx for Lustre file
         *        system that have been exported to a linked S3 bucket and that meet your specified release criteria.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUTO_RELEASE_DATA</code> tasks automatically release files from an Amazon File Cache resource.
         *        </p>
         *        </li>
         * @see DataRepositoryTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryTaskType
         */
        Builder type(DataRepositoryTaskType type);

        /**
         * <p>
         * A list of paths for the data repository task to use when the task is processed. If a path that you provide
         * isn't valid, the task fails. If you don't provide paths, the default behavior is to export all files to S3
         * (for export tasks), import all files from S3 (for import tasks), or release all exported files that meet the
         * last accessed time criteria (for release tasks).
         * </p>
         * <ul>
         * <li>
         * <p>
         * For export tasks, the list contains paths on the FSx for Lustre file system from which the files are exported
         * to the Amazon S3 bucket. The default path is the file system root directory. The paths you provide need to be
         * relative to the mount point of the file system. If the mount point is <code>/mnt/fsx</code> and
         * <code>/mnt/fsx/path1</code> is a directory or file on the file system you want to export, then the path to
         * provide is <code>path1</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For import tasks, the list contains paths in the Amazon S3 bucket from which POSIX metadata changes are
         * imported to the FSx for Lustre file system. The path can be an S3 bucket or prefix in the format
         * <code>s3://bucket-name/prefix</code> (where <code>prefix</code> is optional).
         * </p>
         * </li>
         * <li>
         * <p>
         * For release tasks, the list contains directory or file paths on the FSx for Lustre file system from which to
         * release exported files. If a directory is specified, files within the directory are released. If a file path
         * is specified, only that file is released. To release all exported files in the file system, specify a forward
         * slash (/) as the path.
         * </p>
         * <note>
         * <p>
         * A file must also meet the last accessed time criteria specified in for the file to be released.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param paths
         *        A list of paths for the data repository task to use when the task is processed. If a path that you
         *        provide isn't valid, the task fails. If you don't provide paths, the default behavior is to export all
         *        files to S3 (for export tasks), import all files from S3 (for import tasks), or release all exported
         *        files that meet the last accessed time criteria (for release tasks).</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For export tasks, the list contains paths on the FSx for Lustre file system from which the files are
         *        exported to the Amazon S3 bucket. The default path is the file system root directory. The paths you
         *        provide need to be relative to the mount point of the file system. If the mount point is
         *        <code>/mnt/fsx</code> and <code>/mnt/fsx/path1</code> is a directory or file on the file system you
         *        want to export, then the path to provide is <code>path1</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For import tasks, the list contains paths in the Amazon S3 bucket from which POSIX metadata changes
         *        are imported to the FSx for Lustre file system. The path can be an S3 bucket or prefix in the format
         *        <code>s3://bucket-name/prefix</code> (where <code>prefix</code> is optional).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For release tasks, the list contains directory or file paths on the FSx for Lustre file system from
         *        which to release exported files. If a directory is specified, files within the directory are released.
         *        If a file path is specified, only that file is released. To release all exported files in the file
         *        system, specify a forward slash (/) as the path.
         *        </p>
         *        <note>
         *        <p>
         *        A file must also meet the last accessed time criteria specified in for the file to be released.
         *        </p>
         *        </note></li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paths(Collection<String> paths);

        /**
         * <p>
         * A list of paths for the data repository task to use when the task is processed. If a path that you provide
         * isn't valid, the task fails. If you don't provide paths, the default behavior is to export all files to S3
         * (for export tasks), import all files from S3 (for import tasks), or release all exported files that meet the
         * last accessed time criteria (for release tasks).
         * </p>
         * <ul>
         * <li>
         * <p>
         * For export tasks, the list contains paths on the FSx for Lustre file system from which the files are exported
         * to the Amazon S3 bucket. The default path is the file system root directory. The paths you provide need to be
         * relative to the mount point of the file system. If the mount point is <code>/mnt/fsx</code> and
         * <code>/mnt/fsx/path1</code> is a directory or file on the file system you want to export, then the path to
         * provide is <code>path1</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For import tasks, the list contains paths in the Amazon S3 bucket from which POSIX metadata changes are
         * imported to the FSx for Lustre file system. The path can be an S3 bucket or prefix in the format
         * <code>s3://bucket-name/prefix</code> (where <code>prefix</code> is optional).
         * </p>
         * </li>
         * <li>
         * <p>
         * For release tasks, the list contains directory or file paths on the FSx for Lustre file system from which to
         * release exported files. If a directory is specified, files within the directory are released. If a file path
         * is specified, only that file is released. To release all exported files in the file system, specify a forward
         * slash (/) as the path.
         * </p>
         * <note>
         * <p>
         * A file must also meet the last accessed time criteria specified in for the file to be released.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param paths
         *        A list of paths for the data repository task to use when the task is processed. If a path that you
         *        provide isn't valid, the task fails. If you don't provide paths, the default behavior is to export all
         *        files to S3 (for export tasks), import all files from S3 (for import tasks), or release all exported
         *        files that meet the last accessed time criteria (for release tasks).</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For export tasks, the list contains paths on the FSx for Lustre file system from which the files are
         *        exported to the Amazon S3 bucket. The default path is the file system root directory. The paths you
         *        provide need to be relative to the mount point of the file system. If the mount point is
         *        <code>/mnt/fsx</code> and <code>/mnt/fsx/path1</code> is a directory or file on the file system you
         *        want to export, then the path to provide is <code>path1</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For import tasks, the list contains paths in the Amazon S3 bucket from which POSIX metadata changes
         *        are imported to the FSx for Lustre file system. The path can be an S3 bucket or prefix in the format
         *        <code>s3://bucket-name/prefix</code> (where <code>prefix</code> is optional).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For release tasks, the list contains directory or file paths on the FSx for Lustre file system from
         *        which to release exported files. If a directory is specified, files within the directory are released.
         *        If a file path is specified, only that file is released. To release all exported files in the file
         *        system, specify a forward slash (/) as the path.
         *        </p>
         *        <note>
         *        <p>
         *        A file must also meet the last accessed time criteria specified in for the file to be released.
         *        </p>
         *        </note></li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paths(String... paths);

        /**
         * Sets the value of the FileSystemId property for this object.
         *
         * @param fileSystemId
         *        The new value for the FileSystemId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * Defines whether or not Amazon FSx provides a CompletionReport once the task has completed. A CompletionReport
         * provides a detailed report on the files that Amazon FSx processed that meet the criteria specified by the
         * <code>Scope</code> parameter. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/task-completion-report.html">Working with Task
         * Completion Reports</a>.
         * </p>
         * 
         * @param report
         *        Defines whether or not Amazon FSx provides a CompletionReport once the task has completed. A
         *        CompletionReport provides a detailed report on the files that Amazon FSx processed that meet the
         *        criteria specified by the <code>Scope</code> parameter. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/task-completion-report.html">Working with
         *        Task Completion Reports</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder report(CompletionReport report);

        /**
         * <p>
         * Defines whether or not Amazon FSx provides a CompletionReport once the task has completed. A CompletionReport
         * provides a detailed report on the files that Amazon FSx processed that meet the criteria specified by the
         * <code>Scope</code> parameter. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/task-completion-report.html">Working with Task
         * Completion Reports</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CompletionReport.Builder} avoiding the
         * need to create one manually via {@link CompletionReport#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CompletionReport.Builder#build()} is called immediately and its
         * result is passed to {@link #report(CompletionReport)}.
         * 
         * @param report
         *        a consumer that will call methods on {@link CompletionReport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #report(CompletionReport)
         */
        default Builder report(Consumer<CompletionReport.Builder> report) {
            return report(CompletionReport.builder().applyMutation(report).build());
        }

        /**
         * Sets the value of the ClientRequestToken property for this object.
         *
         * @param clientRequestToken
         *        The new value for the ClientRequestToken property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.fsx.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.fsx.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.fsx.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Specifies the amount of data to release, in GiB, by an Amazon File Cache <code>AUTO_RELEASE_DATA</code> task
         * that automatically releases files from the cache.
         * </p>
         * 
         * @param capacityToRelease
         *        Specifies the amount of data to release, in GiB, by an Amazon File Cache
         *        <code>AUTO_RELEASE_DATA</code> task that automatically releases files from the cache.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityToRelease(Long capacityToRelease);

        /**
         * <p>
         * The configuration that specifies the last accessed time criteria for files that will be released from an
         * Amazon FSx for Lustre file system.
         * </p>
         * 
         * @param releaseConfiguration
         *        The configuration that specifies the last accessed time criteria for files that will be released from
         *        an Amazon FSx for Lustre file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseConfiguration(ReleaseConfiguration releaseConfiguration);

        /**
         * <p>
         * The configuration that specifies the last accessed time criteria for files that will be released from an
         * Amazon FSx for Lustre file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link ReleaseConfiguration.Builder} avoiding
         * the need to create one manually via {@link ReleaseConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ReleaseConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #releaseConfiguration(ReleaseConfiguration)}.
         * 
         * @param releaseConfiguration
         *        a consumer that will call methods on {@link ReleaseConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #releaseConfiguration(ReleaseConfiguration)
         */
        default Builder releaseConfiguration(Consumer<ReleaseConfiguration.Builder> releaseConfiguration) {
            return releaseConfiguration(ReleaseConfiguration.builder().applyMutation(releaseConfiguration).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends FSxRequest.BuilderImpl implements Builder {
        private String type;

        private List<String> paths = DefaultSdkAutoConstructList.getInstance();

        private String fileSystemId;

        private CompletionReport report;

        private String clientRequestToken;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Long capacityToRelease;

        private ReleaseConfiguration releaseConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateDataRepositoryTaskRequest model) {
            super(model);
            type(model.type);
            paths(model.paths);
            fileSystemId(model.fileSystemId);
            report(model.report);
            clientRequestToken(model.clientRequestToken);
            tags(model.tags);
            capacityToRelease(model.capacityToRelease);
            releaseConfiguration(model.releaseConfiguration);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DataRepositoryTaskType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Collection<String> getPaths() {
            if (paths instanceof SdkAutoConstructList) {
                return null;
            }
            return paths;
        }

        public final void setPaths(Collection<String> paths) {
            this.paths = DataRepositoryTaskPathsCopier.copy(paths);
        }

        @Override
        public final Builder paths(Collection<String> paths) {
            this.paths = DataRepositoryTaskPathsCopier.copy(paths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder paths(String... paths) {
            paths(Arrays.asList(paths));
            return this;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final CompletionReport.Builder getReport() {
            return report != null ? report.toBuilder() : null;
        }

        public final void setReport(CompletionReport.BuilderImpl report) {
            this.report = report != null ? report.build() : null;
        }

        @Override
        public final Builder report(CompletionReport report) {
            this.report = report;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Long getCapacityToRelease() {
            return capacityToRelease;
        }

        public final void setCapacityToRelease(Long capacityToRelease) {
            this.capacityToRelease = capacityToRelease;
        }

        @Override
        public final Builder capacityToRelease(Long capacityToRelease) {
            this.capacityToRelease = capacityToRelease;
            return this;
        }

        public final ReleaseConfiguration.Builder getReleaseConfiguration() {
            return releaseConfiguration != null ? releaseConfiguration.toBuilder() : null;
        }

        public final void setReleaseConfiguration(ReleaseConfiguration.BuilderImpl releaseConfiguration) {
            this.releaseConfiguration = releaseConfiguration != null ? releaseConfiguration.build() : null;
        }

        @Override
        public final Builder releaseConfiguration(ReleaseConfiguration releaseConfiguration) {
            this.releaseConfiguration = releaseConfiguration;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateDataRepositoryTaskRequest build() {
            return new CreateDataRepositoryTaskRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
