/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Lustre metadata performance configuration for the creation of an Amazon FSx for Lustre file system using a
 * <code>PERSISTENT_2</code> deployment type. The configuration uses a Metadata IOPS value to set the maximum rate of
 * metadata disk IOPS supported by the file system.
 * </p>
 * <p>
 * After creation, the file system supports increasing metadata performance. For more information on Metadata IOPS, see
 * <a
 * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/managing-metadata-performance.html#metadata-configuration"
 * >Lustre metadata performance configuration</a> in the <i>Amazon FSx for Lustre User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFileSystemLustreMetadataConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateFileSystemLustreMetadataConfiguration.Builder, CreateFileSystemLustreMetadataConfiguration> {
    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Iops")
            .getter(getter(CreateFileSystemLustreMetadataConfiguration::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops").build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mode")
            .getter(getter(CreateFileSystemLustreMetadataConfiguration::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IOPS_FIELD, MODE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer iops;

    private final String mode;

    private CreateFileSystemLustreMetadataConfiguration(BuilderImpl builder) {
        this.iops = builder.iops;
        this.mode = builder.mode;
    }

    /**
     * <p>
     * (USER_PROVISIONED mode only) Specifies the number of Metadata IOPS to provision for the file system. This
     * parameter sets the maximum rate of metadata disk IOPS supported by the file system. Valid values are
     * <code>1500</code>, <code>3000</code>, <code>6000</code>, <code>12000</code>, and multiples of <code>12000</code>
     * up to a maximum of <code>192000</code>.
     * </p>
     * <note>
     * <p>
     * Iops doesn’t have a default value. If you're using USER_PROVISIONED mode, you can choose to specify a valid
     * value. If you're using AUTOMATIC mode, you cannot specify a value because FSx for Lustre automatically sets the
     * value based on your file system storage capacity.
     * </p>
     * </note>
     * 
     * @return (USER_PROVISIONED mode only) Specifies the number of Metadata IOPS to provision for the file system. This
     *         parameter sets the maximum rate of metadata disk IOPS supported by the file system. Valid values are
     *         <code>1500</code>, <code>3000</code>, <code>6000</code>, <code>12000</code>, and multiples of
     *         <code>12000</code> up to a maximum of <code>192000</code>.</p> <note>
     *         <p>
     *         Iops doesn’t have a default value. If you're using USER_PROVISIONED mode, you can choose to specify a
     *         valid value. If you're using AUTOMATIC mode, you cannot specify a value because FSx for Lustre
     *         automatically sets the value based on your file system storage capacity.
     *         </p>
     */
    public final Integer iops() {
        return iops;
    }

    /**
     * <p>
     * The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using a
     * <code>PERSISTENT_2</code> deployment type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for your file
     * system based on your file system storage capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link MetadataConfigurationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using a
     *         <code>PERSISTENT_2</code> deployment type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for
     *         your file system based on your file system storage capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
     *         </p>
     *         </li>
     * @see MetadataConfigurationMode
     */
    public final MetadataConfigurationMode mode() {
        return MetadataConfigurationMode.fromValue(mode);
    }

    /**
     * <p>
     * The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using a
     * <code>PERSISTENT_2</code> deployment type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for your file
     * system based on your file system storage capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link MetadataConfigurationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using a
     *         <code>PERSISTENT_2</code> deployment type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for
     *         your file system based on your file system storage capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
     *         </p>
     *         </li>
     * @see MetadataConfigurationMode
     */
    public final String modeAsString() {
        return mode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFileSystemLustreMetadataConfiguration)) {
            return false;
        }
        CreateFileSystemLustreMetadataConfiguration other = (CreateFileSystemLustreMetadataConfiguration) obj;
        return Objects.equals(iops(), other.iops()) && Objects.equals(modeAsString(), other.modeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFileSystemLustreMetadataConfiguration").add("Iops", iops()).add("Mode", modeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Iops", IOPS_FIELD);
        map.put("Mode", MODE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateFileSystemLustreMetadataConfiguration, T> g) {
        return obj -> g.apply((CreateFileSystemLustreMetadataConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateFileSystemLustreMetadataConfiguration> {
        /**
         * <p>
         * (USER_PROVISIONED mode only) Specifies the number of Metadata IOPS to provision for the file system. This
         * parameter sets the maximum rate of metadata disk IOPS supported by the file system. Valid values are
         * <code>1500</code>, <code>3000</code>, <code>6000</code>, <code>12000</code>, and multiples of
         * <code>12000</code> up to a maximum of <code>192000</code>.
         * </p>
         * <note>
         * <p>
         * Iops doesn’t have a default value. If you're using USER_PROVISIONED mode, you can choose to specify a valid
         * value. If you're using AUTOMATIC mode, you cannot specify a value because FSx for Lustre automatically sets
         * the value based on your file system storage capacity.
         * </p>
         * </note>
         * 
         * @param iops
         *        (USER_PROVISIONED mode only) Specifies the number of Metadata IOPS to provision for the file system.
         *        This parameter sets the maximum rate of metadata disk IOPS supported by the file system. Valid values
         *        are <code>1500</code>, <code>3000</code>, <code>6000</code>, <code>12000</code>, and multiples of
         *        <code>12000</code> up to a maximum of <code>192000</code>.</p> <note>
         *        <p>
         *        Iops doesn’t have a default value. If you're using USER_PROVISIONED mode, you can choose to specify a
         *        valid value. If you're using AUTOMATIC mode, you cannot specify a value because FSx for Lustre
         *        automatically sets the value based on your file system storage capacity.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using a
         * <code>PERSISTENT_2</code> deployment type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for your
         * file system based on your file system storage capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mode
         *        The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using
         *        a <code>PERSISTENT_2</code> deployment type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for
         *        your file system based on your file system storage capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
         *        </p>
         *        </li>
         * @see MetadataConfigurationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetadataConfigurationMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using a
         * <code>PERSISTENT_2</code> deployment type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for your
         * file system based on your file system storage capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mode
         *        The metadata configuration mode for provisioning Metadata IOPS for an FSx for Lustre file system using
         *        a <code>PERSISTENT_2</code> deployment type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In AUTOMATIC mode, FSx for Lustre automatically provisions and scales the number of Metadata IOPS for
         *        your file system based on your file system storage capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In USER_PROVISIONED mode, you specify the number of Metadata IOPS to provision for your file system.
         *        </p>
         *        </li>
         * @see MetadataConfigurationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetadataConfigurationMode
         */
        Builder mode(MetadataConfigurationMode mode);
    }

    static final class BuilderImpl implements Builder {
        private Integer iops;

        private String mode;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFileSystemLustreMetadataConfiguration model) {
            iops(model.iops);
            mode(model.mode);
        }

        public final Integer getIops() {
            return iops;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(MetadataConfigurationMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        @Override
        public CreateFileSystemLustreMetadataConfiguration build() {
            return new CreateFileSystemLustreMetadataConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
