/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the configuration of the Amazon FSx for OpenZFS volume that you are creating.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateOpenZFSVolumeConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateOpenZFSVolumeConfiguration.Builder, CreateOpenZFSVolumeConfiguration> {
    private static final SdkField<String> PARENT_VOLUME_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParentVolumeId").getter(getter(CreateOpenZFSVolumeConfiguration::parentVolumeId))
            .setter(setter(Builder::parentVolumeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentVolumeId").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_RESERVATION_GIB_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacityReservationGiB")
            .getter(getter(CreateOpenZFSVolumeConfiguration::storageCapacityReservationGiB))
            .setter(setter(Builder::storageCapacityReservationGiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacityReservationGiB")
                    .build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_QUOTA_GIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacityQuotaGiB").getter(getter(CreateOpenZFSVolumeConfiguration::storageCapacityQuotaGiB))
            .setter(setter(Builder::storageCapacityQuotaGiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacityQuotaGiB").build())
            .build();

    private static final SdkField<Integer> RECORD_SIZE_KIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RecordSizeKiB").getter(getter(CreateOpenZFSVolumeConfiguration::recordSizeKiB))
            .setter(setter(Builder::recordSizeKiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordSizeKiB").build()).build();

    private static final SdkField<String> DATA_COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataCompressionType").getter(getter(CreateOpenZFSVolumeConfiguration::dataCompressionTypeAsString))
            .setter(setter(Builder::dataCompressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCompressionType").build())
            .build();

    private static final SdkField<Boolean> COPY_TAGS_TO_SNAPSHOTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToSnapshots").getter(getter(CreateOpenZFSVolumeConfiguration::copyTagsToSnapshots))
            .setter(setter(Builder::copyTagsToSnapshots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToSnapshots").build())
            .build();

    private static final SdkField<CreateOpenZFSOriginSnapshotConfiguration> ORIGIN_SNAPSHOT_FIELD = SdkField
            .<CreateOpenZFSOriginSnapshotConfiguration> builder(MarshallingType.SDK_POJO).memberName("OriginSnapshot")
            .getter(getter(CreateOpenZFSVolumeConfiguration::originSnapshot)).setter(setter(Builder::originSnapshot))
            .constructor(CreateOpenZFSOriginSnapshotConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginSnapshot").build()).build();

    private static final SdkField<Boolean> READ_ONLY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ReadOnly").getter(getter(CreateOpenZFSVolumeConfiguration::readOnly)).setter(setter(Builder::readOnly))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadOnly").build()).build();

    private static final SdkField<List<OpenZFSNfsExport>> NFS_EXPORTS_FIELD = SdkField
            .<List<OpenZFSNfsExport>> builder(MarshallingType.LIST)
            .memberName("NfsExports")
            .getter(getter(CreateOpenZFSVolumeConfiguration::nfsExports))
            .setter(setter(Builder::nfsExports))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NfsExports").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpenZFSNfsExport> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpenZFSNfsExport::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<OpenZFSUserOrGroupQuota>> USER_AND_GROUP_QUOTAS_FIELD = SdkField
            .<List<OpenZFSUserOrGroupQuota>> builder(MarshallingType.LIST)
            .memberName("UserAndGroupQuotas")
            .getter(getter(CreateOpenZFSVolumeConfiguration::userAndGroupQuotas))
            .setter(setter(Builder::userAndGroupQuotas))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAndGroupQuotas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpenZFSUserOrGroupQuota> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpenZFSUserOrGroupQuota::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PARENT_VOLUME_ID_FIELD,
            STORAGE_CAPACITY_RESERVATION_GIB_FIELD, STORAGE_CAPACITY_QUOTA_GIB_FIELD, RECORD_SIZE_KIB_FIELD,
            DATA_COMPRESSION_TYPE_FIELD, COPY_TAGS_TO_SNAPSHOTS_FIELD, ORIGIN_SNAPSHOT_FIELD, READ_ONLY_FIELD, NFS_EXPORTS_FIELD,
            USER_AND_GROUP_QUOTAS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String parentVolumeId;

    private final Integer storageCapacityReservationGiB;

    private final Integer storageCapacityQuotaGiB;

    private final Integer recordSizeKiB;

    private final String dataCompressionType;

    private final Boolean copyTagsToSnapshots;

    private final CreateOpenZFSOriginSnapshotConfiguration originSnapshot;

    private final Boolean readOnly;

    private final List<OpenZFSNfsExport> nfsExports;

    private final List<OpenZFSUserOrGroupQuota> userAndGroupQuotas;

    private CreateOpenZFSVolumeConfiguration(BuilderImpl builder) {
        this.parentVolumeId = builder.parentVolumeId;
        this.storageCapacityReservationGiB = builder.storageCapacityReservationGiB;
        this.storageCapacityQuotaGiB = builder.storageCapacityQuotaGiB;
        this.recordSizeKiB = builder.recordSizeKiB;
        this.dataCompressionType = builder.dataCompressionType;
        this.copyTagsToSnapshots = builder.copyTagsToSnapshots;
        this.originSnapshot = builder.originSnapshot;
        this.readOnly = builder.readOnly;
        this.nfsExports = builder.nfsExports;
        this.userAndGroupQuotas = builder.userAndGroupQuotas;
    }

    /**
     * <p>
     * The ID of the volume to use as the parent volume of the volume that you are creating.
     * </p>
     * 
     * @return The ID of the volume to use as the parent volume of the volume that you are creating.
     */
    public final String parentVolumeId() {
        return parentVolumeId;
    }

    /**
     * <p>
     * Specifies the amount of storage in gibibytes (GiB) to reserve from the parent volume. Setting
     * <code>StorageCapacityReservationGiB</code> guarantees that the specified amount of storage space on the parent
     * volume will always be available for the volume. You can't reserve more storage than the parent volume has. To
     * <i>not</i> specify a storage capacity reservation, set this to <code>0</code> or <code>-1</code>. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties">Volume
     * properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     * 
     * @return Specifies the amount of storage in gibibytes (GiB) to reserve from the parent volume. Setting
     *         <code>StorageCapacityReservationGiB</code> guarantees that the specified amount of storage space on the
     *         parent volume will always be available for the volume. You can't reserve more storage than the parent
     *         volume has. To <i>not</i> specify a storage capacity reservation, set this to <code>0</code> or
     *         <code>-1</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties">Volume
     *         properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
     */
    public final Integer storageCapacityReservationGiB() {
        return storageCapacityReservationGiB;
    }

    /**
     * <p>
     * Sets the maximum storage size in gibibytes (GiB) for the volume. You can specify a quota that is larger than the
     * storage on the parent volume. A volume quota limits the amount of storage that the volume can consume to the
     * configured amount, but does not guarantee the space will be available on the parent volume. To guarantee quota
     * space, you must also set <code>StorageCapacityReservationGiB</code>. To <i>not</i> specify a storage capacity
     * quota, set this to <code>-1</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties">Volume
     * properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     * 
     * @return Sets the maximum storage size in gibibytes (GiB) for the volume. You can specify a quota that is larger
     *         than the storage on the parent volume. A volume quota limits the amount of storage that the volume can
     *         consume to the configured amount, but does not guarantee the space will be available on the parent
     *         volume. To guarantee quota space, you must also set <code>StorageCapacityReservationGiB</code>. To
     *         <i>not</i> specify a storage capacity quota, set this to <code>-1</code>. </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties">Volume
     *         properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
     */
    public final Integer storageCapacityQuotaGiB() {
        return storageCapacityQuotaGiB;
    }

    /**
     * <p>
     * Specifies the suggested block size for a volume in a ZFS dataset, in kibibytes (KiB). For file systems using the
     * Intelligent-Tiering storage class, valid values are 128, 256, 512, 1024, 2048, or 4096 KiB, with a default of
     * 1024 KiB. For all other file systems, valid values are 4, 8, 16, 32, 64, 128, 256, 512, or 1024 KiB, with a
     * default of 128 KiB. We recommend using the default setting for the majority of use cases. Generally, workloads
     * that write in fixed small or large record sizes may benefit from setting a custom record size, like database
     * workloads (small record size) or media streaming workloads (large record size). For additional guidance on when
     * to set a custom record size, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#record-size-performance"> ZFS Record
     * size</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     * 
     * @return Specifies the suggested block size for a volume in a ZFS dataset, in kibibytes (KiB). For file systems
     *         using the Intelligent-Tiering storage class, valid values are 128, 256, 512, 1024, 2048, or 4096 KiB,
     *         with a default of 1024 KiB. For all other file systems, valid values are 4, 8, 16, 32, 64, 128, 256, 512,
     *         or 1024 KiB, with a default of 128 KiB. We recommend using the default setting for the majority of use
     *         cases. Generally, workloads that write in fixed small or large record sizes may benefit from setting a
     *         custom record size, like database workloads (small record size) or media streaming workloads (large
     *         record size). For additional guidance on when to set a custom record size, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#record-size-performance"> ZFS
     *         Record size</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
     */
    public final Integer recordSizeKiB() {
        return recordSizeKiB;
    }

    /**
     * <p>
     * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
     * default.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm. ZSTD
     * compression provides a higher level of data compression and higher read throughput performance than LZ4
     * compression.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4 compression
     * provides a lower level of compression and higher write throughput performance than ZSTD compression.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about volume compression types and the performance of your Amazon FSx for OpenZFS file
     * system, see <a href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs">
     * Tips for maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User
     * Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataCompressionType} will return {@link OpenZFSDataCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataCompressionTypeAsString}.
     * </p>
     * 
     * @return Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code>
     *         by default.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm.
     *         ZSTD compression provides a higher level of data compression and higher read throughput performance than
     *         LZ4 compression.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4 compression
     *         provides a lower level of compression and higher write throughput performance than ZSTD compression.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about volume compression types and the performance of your Amazon FSx for OpenZFS
     *         file system, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs"> Tips
     *         for maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User
     *         Guide</i>.
     * @see OpenZFSDataCompressionType
     */
    public final OpenZFSDataCompressionType dataCompressionType() {
        return OpenZFSDataCompressionType.fromValue(dataCompressionType);
    }

    /**
     * <p>
     * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
     * default.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm. ZSTD
     * compression provides a higher level of data compression and higher read throughput performance than LZ4
     * compression.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4 compression
     * provides a lower level of compression and higher write throughput performance than ZSTD compression.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about volume compression types and the performance of your Amazon FSx for OpenZFS file
     * system, see <a href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs">
     * Tips for maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User
     * Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataCompressionType} will return {@link OpenZFSDataCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataCompressionTypeAsString}.
     * </p>
     * 
     * @return Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code>
     *         by default.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm.
     *         ZSTD compression provides a higher level of data compression and higher read throughput performance than
     *         LZ4 compression.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4 compression
     *         provides a lower level of compression and higher write throughput performance than ZSTD compression.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about volume compression types and the performance of your Amazon FSx for OpenZFS
     *         file system, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs"> Tips
     *         for maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User
     *         Guide</i>.
     * @see OpenZFSDataCompressionType
     */
    public final String dataCompressionTypeAsString() {
        return dataCompressionType;
    }

    /**
     * <p>
     * A Boolean value indicating whether tags for the volume should be copied to snapshots. This value defaults to
     * <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied to snapshots where the
     * user doesn't specify tags. If this value is <code>true</code>, and you specify one or more tags, only the
     * specified tags are copied to snapshots. If you specify one or more tags when creating the snapshot, no tags are
     * copied from the volume, regardless of this value.
     * </p>
     * 
     * @return A Boolean value indicating whether tags for the volume should be copied to snapshots. This value defaults
     *         to <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied to snapshots
     *         where the user doesn't specify tags. If this value is <code>true</code>, and you specify one or more
     *         tags, only the specified tags are copied to snapshots. If you specify one or more tags when creating the
     *         snapshot, no tags are copied from the volume, regardless of this value.
     */
    public final Boolean copyTagsToSnapshots() {
        return copyTagsToSnapshots;
    }

    /**
     * <p>
     * The configuration object that specifies the snapshot to use as the origin of the data for the volume.
     * </p>
     * 
     * @return The configuration object that specifies the snapshot to use as the origin of the data for the volume.
     */
    public final CreateOpenZFSOriginSnapshotConfiguration originSnapshot() {
        return originSnapshot;
    }

    /**
     * <p>
     * A Boolean value indicating whether the volume is read-only.
     * </p>
     * 
     * @return A Boolean value indicating whether the volume is read-only.
     */
    public final Boolean readOnly() {
        return readOnly;
    }

    /**
     * For responses, this returns true if the service returned a value for the NfsExports property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNfsExports() {
        return nfsExports != null && !(nfsExports instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The configuration object for mounting a Network File System (NFS) file system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNfsExports} method.
     * </p>
     * 
     * @return The configuration object for mounting a Network File System (NFS) file system.
     */
    public final List<OpenZFSNfsExport> nfsExports() {
        return nfsExports;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserAndGroupQuotas property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUserAndGroupQuotas() {
        return userAndGroupQuotas != null && !(userAndGroupQuotas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Configures how much storage users and groups can use on the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserAndGroupQuotas} method.
     * </p>
     * 
     * @return Configures how much storage users and groups can use on the volume.
     */
    public final List<OpenZFSUserOrGroupQuota> userAndGroupQuotas() {
        return userAndGroupQuotas;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(parentVolumeId());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacityReservationGiB());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacityQuotaGiB());
        hashCode = 31 * hashCode + Objects.hashCode(recordSizeKiB());
        hashCode = 31 * hashCode + Objects.hashCode(dataCompressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToSnapshots());
        hashCode = 31 * hashCode + Objects.hashCode(originSnapshot());
        hashCode = 31 * hashCode + Objects.hashCode(readOnly());
        hashCode = 31 * hashCode + Objects.hashCode(hasNfsExports() ? nfsExports() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUserAndGroupQuotas() ? userAndGroupQuotas() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateOpenZFSVolumeConfiguration)) {
            return false;
        }
        CreateOpenZFSVolumeConfiguration other = (CreateOpenZFSVolumeConfiguration) obj;
        return Objects.equals(parentVolumeId(), other.parentVolumeId())
                && Objects.equals(storageCapacityReservationGiB(), other.storageCapacityReservationGiB())
                && Objects.equals(storageCapacityQuotaGiB(), other.storageCapacityQuotaGiB())
                && Objects.equals(recordSizeKiB(), other.recordSizeKiB())
                && Objects.equals(dataCompressionTypeAsString(), other.dataCompressionTypeAsString())
                && Objects.equals(copyTagsToSnapshots(), other.copyTagsToSnapshots())
                && Objects.equals(originSnapshot(), other.originSnapshot()) && Objects.equals(readOnly(), other.readOnly())
                && hasNfsExports() == other.hasNfsExports() && Objects.equals(nfsExports(), other.nfsExports())
                && hasUserAndGroupQuotas() == other.hasUserAndGroupQuotas()
                && Objects.equals(userAndGroupQuotas(), other.userAndGroupQuotas());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateOpenZFSVolumeConfiguration").add("ParentVolumeId", parentVolumeId())
                .add("StorageCapacityReservationGiB", storageCapacityReservationGiB())
                .add("StorageCapacityQuotaGiB", storageCapacityQuotaGiB()).add("RecordSizeKiB", recordSizeKiB())
                .add("DataCompressionType", dataCompressionTypeAsString()).add("CopyTagsToSnapshots", copyTagsToSnapshots())
                .add("OriginSnapshot", originSnapshot()).add("ReadOnly", readOnly())
                .add("NfsExports", hasNfsExports() ? nfsExports() : null)
                .add("UserAndGroupQuotas", hasUserAndGroupQuotas() ? userAndGroupQuotas() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ParentVolumeId":
            return Optional.ofNullable(clazz.cast(parentVolumeId()));
        case "StorageCapacityReservationGiB":
            return Optional.ofNullable(clazz.cast(storageCapacityReservationGiB()));
        case "StorageCapacityQuotaGiB":
            return Optional.ofNullable(clazz.cast(storageCapacityQuotaGiB()));
        case "RecordSizeKiB":
            return Optional.ofNullable(clazz.cast(recordSizeKiB()));
        case "DataCompressionType":
            return Optional.ofNullable(clazz.cast(dataCompressionTypeAsString()));
        case "CopyTagsToSnapshots":
            return Optional.ofNullable(clazz.cast(copyTagsToSnapshots()));
        case "OriginSnapshot":
            return Optional.ofNullable(clazz.cast(originSnapshot()));
        case "ReadOnly":
            return Optional.ofNullable(clazz.cast(readOnly()));
        case "NfsExports":
            return Optional.ofNullable(clazz.cast(nfsExports()));
        case "UserAndGroupQuotas":
            return Optional.ofNullable(clazz.cast(userAndGroupQuotas()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ParentVolumeId", PARENT_VOLUME_ID_FIELD);
        map.put("StorageCapacityReservationGiB", STORAGE_CAPACITY_RESERVATION_GIB_FIELD);
        map.put("StorageCapacityQuotaGiB", STORAGE_CAPACITY_QUOTA_GIB_FIELD);
        map.put("RecordSizeKiB", RECORD_SIZE_KIB_FIELD);
        map.put("DataCompressionType", DATA_COMPRESSION_TYPE_FIELD);
        map.put("CopyTagsToSnapshots", COPY_TAGS_TO_SNAPSHOTS_FIELD);
        map.put("OriginSnapshot", ORIGIN_SNAPSHOT_FIELD);
        map.put("ReadOnly", READ_ONLY_FIELD);
        map.put("NfsExports", NFS_EXPORTS_FIELD);
        map.put("UserAndGroupQuotas", USER_AND_GROUP_QUOTAS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateOpenZFSVolumeConfiguration, T> g) {
        return obj -> g.apply((CreateOpenZFSVolumeConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateOpenZFSVolumeConfiguration> {
        /**
         * <p>
         * The ID of the volume to use as the parent volume of the volume that you are creating.
         * </p>
         * 
         * @param parentVolumeId
         *        The ID of the volume to use as the parent volume of the volume that you are creating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentVolumeId(String parentVolumeId);

        /**
         * <p>
         * Specifies the amount of storage in gibibytes (GiB) to reserve from the parent volume. Setting
         * <code>StorageCapacityReservationGiB</code> guarantees that the specified amount of storage space on the
         * parent volume will always be available for the volume. You can't reserve more storage than the parent volume
         * has. To <i>not</i> specify a storage capacity reservation, set this to <code>0</code> or <code>-1</code>. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties">Volume
         * properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * </p>
         * 
         * @param storageCapacityReservationGiB
         *        Specifies the amount of storage in gibibytes (GiB) to reserve from the parent volume. Setting
         *        <code>StorageCapacityReservationGiB</code> guarantees that the specified amount of storage space on
         *        the parent volume will always be available for the volume. You can't reserve more storage than the
         *        parent volume has. To <i>not</i> specify a storage capacity reservation, set this to <code>0</code> or
         *        <code>-1</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties"
         *        >Volume properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacityReservationGiB(Integer storageCapacityReservationGiB);

        /**
         * <p>
         * Sets the maximum storage size in gibibytes (GiB) for the volume. You can specify a quota that is larger than
         * the storage on the parent volume. A volume quota limits the amount of storage that the volume can consume to
         * the configured amount, but does not guarantee the space will be available on the parent volume. To guarantee
         * quota space, you must also set <code>StorageCapacityReservationGiB</code>. To <i>not</i> specify a storage
         * capacity quota, set this to <code>-1</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties">Volume
         * properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * </p>
         * 
         * @param storageCapacityQuotaGiB
         *        Sets the maximum storage size in gibibytes (GiB) for the volume. You can specify a quota that is
         *        larger than the storage on the parent volume. A volume quota limits the amount of storage that the
         *        volume can consume to the configured amount, but does not guarantee the space will be available on the
         *        parent volume. To guarantee quota space, you must also set <code>StorageCapacityReservationGiB</code>.
         *        To <i>not</i> specify a storage capacity quota, set this to <code>-1</code>. </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/managing-volumes.html#volume-properties"
         *        >Volume properties</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacityQuotaGiB(Integer storageCapacityQuotaGiB);

        /**
         * <p>
         * Specifies the suggested block size for a volume in a ZFS dataset, in kibibytes (KiB). For file systems using
         * the Intelligent-Tiering storage class, valid values are 128, 256, 512, 1024, 2048, or 4096 KiB, with a
         * default of 1024 KiB. For all other file systems, valid values are 4, 8, 16, 32, 64, 128, 256, 512, or 1024
         * KiB, with a default of 128 KiB. We recommend using the default setting for the majority of use cases.
         * Generally, workloads that write in fixed small or large record sizes may benefit from setting a custom record
         * size, like database workloads (small record size) or media streaming workloads (large record size). For
         * additional guidance on when to set a custom record size, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#record-size-performance"> ZFS
         * Record size</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * </p>
         * 
         * @param recordSizeKiB
         *        Specifies the suggested block size for a volume in a ZFS dataset, in kibibytes (KiB). For file systems
         *        using the Intelligent-Tiering storage class, valid values are 128, 256, 512, 1024, 2048, or 4096 KiB,
         *        with a default of 1024 KiB. For all other file systems, valid values are 4, 8, 16, 32, 64, 128, 256,
         *        512, or 1024 KiB, with a default of 128 KiB. We recommend using the default setting for the majority
         *        of use cases. Generally, workloads that write in fixed small or large record sizes may benefit from
         *        setting a custom record size, like database workloads (small record size) or media streaming workloads
         *        (large record size). For additional guidance on when to set a custom record size, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#record-size-performance">
         *        ZFS Record size</a> in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordSizeKiB(Integer recordSizeKiB);

        /**
         * <p>
         * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
         * default.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm. ZSTD
         * compression provides a higher level of data compression and higher read throughput performance than LZ4
         * compression.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4 compression
         * provides a lower level of compression and higher write throughput performance than ZSTD compression.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about volume compression types and the performance of your Amazon FSx for OpenZFS file
         * system, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs"> Tips for
         * maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * </p>
         * 
         * @param dataCompressionType
         *        Specifies the method used to compress the data on the volume. The compression type is
         *        <code>NONE</code> by default.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression
         *        algorithm. ZSTD compression provides a higher level of data compression and higher read throughput
         *        performance than LZ4 compression.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4
         *        compression provides a lower level of compression and higher write throughput performance than ZSTD
         *        compression.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about volume compression types and the performance of your Amazon FSx for OpenZFS
         *        file system, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs"> Tips
         *        for maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User
         *        Guide</i>.
         * @see OpenZFSDataCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenZFSDataCompressionType
         */
        Builder dataCompressionType(String dataCompressionType);

        /**
         * <p>
         * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
         * default.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm. ZSTD
         * compression provides a higher level of data compression and higher read throughput performance than LZ4
         * compression.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4 compression
         * provides a lower level of compression and higher write throughput performance than ZSTD compression.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about volume compression types and the performance of your Amazon FSx for OpenZFS file
         * system, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs"> Tips for
         * maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User Guide</i>.
         * </p>
         * 
         * @param dataCompressionType
         *        Specifies the method used to compress the data on the volume. The compression type is
         *        <code>NONE</code> by default.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression
         *        algorithm. ZSTD compression provides a higher level of data compression and higher read throughput
         *        performance than LZ4 compression.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. LZ4
         *        compression provides a lower level of compression and higher write throughput performance than ZSTD
         *        compression.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about volume compression types and the performance of your Amazon FSx for OpenZFS
         *        file system, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/performance.html#performance-tips-zfs"> Tips
         *        for maximizing performance</a> File system and volume settings in the <i>Amazon FSx for OpenZFS User
         *        Guide</i>.
         * @see OpenZFSDataCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenZFSDataCompressionType
         */
        Builder dataCompressionType(OpenZFSDataCompressionType dataCompressionType);

        /**
         * <p>
         * A Boolean value indicating whether tags for the volume should be copied to snapshots. This value defaults to
         * <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied to snapshots where
         * the user doesn't specify tags. If this value is <code>true</code>, and you specify one or more tags, only the
         * specified tags are copied to snapshots. If you specify one or more tags when creating the snapshot, no tags
         * are copied from the volume, regardless of this value.
         * </p>
         * 
         * @param copyTagsToSnapshots
         *        A Boolean value indicating whether tags for the volume should be copied to snapshots. This value
         *        defaults to <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied
         *        to snapshots where the user doesn't specify tags. If this value is <code>true</code>, and you specify
         *        one or more tags, only the specified tags are copied to snapshots. If you specify one or more tags
         *        when creating the snapshot, no tags are copied from the volume, regardless of this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToSnapshots(Boolean copyTagsToSnapshots);

        /**
         * <p>
         * The configuration object that specifies the snapshot to use as the origin of the data for the volume.
         * </p>
         * 
         * @param originSnapshot
         *        The configuration object that specifies the snapshot to use as the origin of the data for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originSnapshot(CreateOpenZFSOriginSnapshotConfiguration originSnapshot);

        /**
         * <p>
         * The configuration object that specifies the snapshot to use as the origin of the data for the volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CreateOpenZFSOriginSnapshotConfiguration.Builder} avoiding the need to create one manually via
         * {@link CreateOpenZFSOriginSnapshotConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateOpenZFSOriginSnapshotConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #originSnapshot(CreateOpenZFSOriginSnapshotConfiguration)}.
         * 
         * @param originSnapshot
         *        a consumer that will call methods on {@link CreateOpenZFSOriginSnapshotConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #originSnapshot(CreateOpenZFSOriginSnapshotConfiguration)
         */
        default Builder originSnapshot(Consumer<CreateOpenZFSOriginSnapshotConfiguration.Builder> originSnapshot) {
            return originSnapshot(CreateOpenZFSOriginSnapshotConfiguration.builder().applyMutation(originSnapshot).build());
        }

        /**
         * <p>
         * A Boolean value indicating whether the volume is read-only.
         * </p>
         * 
         * @param readOnly
         *        A Boolean value indicating whether the volume is read-only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readOnly(Boolean readOnly);

        /**
         * <p>
         * The configuration object for mounting a Network File System (NFS) file system.
         * </p>
         * 
         * @param nfsExports
         *        The configuration object for mounting a Network File System (NFS) file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExports(Collection<OpenZFSNfsExport> nfsExports);

        /**
         * <p>
         * The configuration object for mounting a Network File System (NFS) file system.
         * </p>
         * 
         * @param nfsExports
         *        The configuration object for mounting a Network File System (NFS) file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExports(OpenZFSNfsExport... nfsExports);

        /**
         * <p>
         * The configuration object for mounting a Network File System (NFS) file system.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport.Builder#build()} is called immediately and
         * its result is passed to {@link #nfsExports(List<OpenZFSNfsExport>)}.
         * 
         * @param nfsExports
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nfsExports(java.util.Collection<OpenZFSNfsExport>)
         */
        Builder nfsExports(Consumer<OpenZFSNfsExport.Builder>... nfsExports);

        /**
         * <p>
         * Configures how much storage users and groups can use on the volume.
         * </p>
         * 
         * @param userAndGroupQuotas
         *        Configures how much storage users and groups can use on the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAndGroupQuotas(Collection<OpenZFSUserOrGroupQuota> userAndGroupQuotas);

        /**
         * <p>
         * Configures how much storage users and groups can use on the volume.
         * </p>
         * 
         * @param userAndGroupQuotas
         *        Configures how much storage users and groups can use on the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAndGroupQuotas(OpenZFSUserOrGroupQuota... userAndGroupQuotas);

        /**
         * <p>
         * Configures how much storage users and groups can use on the volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota.Builder#build()} is called
         * immediately and its result is passed to {@link #userAndGroupQuotas(List<OpenZFSUserOrGroupQuota>)}.
         * 
         * @param userAndGroupQuotas
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAndGroupQuotas(java.util.Collection<OpenZFSUserOrGroupQuota>)
         */
        Builder userAndGroupQuotas(Consumer<OpenZFSUserOrGroupQuota.Builder>... userAndGroupQuotas);
    }

    static final class BuilderImpl implements Builder {
        private String parentVolumeId;

        private Integer storageCapacityReservationGiB;

        private Integer storageCapacityQuotaGiB;

        private Integer recordSizeKiB;

        private String dataCompressionType;

        private Boolean copyTagsToSnapshots;

        private CreateOpenZFSOriginSnapshotConfiguration originSnapshot;

        private Boolean readOnly;

        private List<OpenZFSNfsExport> nfsExports = DefaultSdkAutoConstructList.getInstance();

        private List<OpenZFSUserOrGroupQuota> userAndGroupQuotas = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateOpenZFSVolumeConfiguration model) {
            parentVolumeId(model.parentVolumeId);
            storageCapacityReservationGiB(model.storageCapacityReservationGiB);
            storageCapacityQuotaGiB(model.storageCapacityQuotaGiB);
            recordSizeKiB(model.recordSizeKiB);
            dataCompressionType(model.dataCompressionType);
            copyTagsToSnapshots(model.copyTagsToSnapshots);
            originSnapshot(model.originSnapshot);
            readOnly(model.readOnly);
            nfsExports(model.nfsExports);
            userAndGroupQuotas(model.userAndGroupQuotas);
        }

        public final String getParentVolumeId() {
            return parentVolumeId;
        }

        public final void setParentVolumeId(String parentVolumeId) {
            this.parentVolumeId = parentVolumeId;
        }

        @Override
        public final Builder parentVolumeId(String parentVolumeId) {
            this.parentVolumeId = parentVolumeId;
            return this;
        }

        public final Integer getStorageCapacityReservationGiB() {
            return storageCapacityReservationGiB;
        }

        public final void setStorageCapacityReservationGiB(Integer storageCapacityReservationGiB) {
            this.storageCapacityReservationGiB = storageCapacityReservationGiB;
        }

        @Override
        public final Builder storageCapacityReservationGiB(Integer storageCapacityReservationGiB) {
            this.storageCapacityReservationGiB = storageCapacityReservationGiB;
            return this;
        }

        public final Integer getStorageCapacityQuotaGiB() {
            return storageCapacityQuotaGiB;
        }

        public final void setStorageCapacityQuotaGiB(Integer storageCapacityQuotaGiB) {
            this.storageCapacityQuotaGiB = storageCapacityQuotaGiB;
        }

        @Override
        public final Builder storageCapacityQuotaGiB(Integer storageCapacityQuotaGiB) {
            this.storageCapacityQuotaGiB = storageCapacityQuotaGiB;
            return this;
        }

        public final Integer getRecordSizeKiB() {
            return recordSizeKiB;
        }

        public final void setRecordSizeKiB(Integer recordSizeKiB) {
            this.recordSizeKiB = recordSizeKiB;
        }

        @Override
        public final Builder recordSizeKiB(Integer recordSizeKiB) {
            this.recordSizeKiB = recordSizeKiB;
            return this;
        }

        public final String getDataCompressionType() {
            return dataCompressionType;
        }

        public final void setDataCompressionType(String dataCompressionType) {
            this.dataCompressionType = dataCompressionType;
        }

        @Override
        public final Builder dataCompressionType(String dataCompressionType) {
            this.dataCompressionType = dataCompressionType;
            return this;
        }

        @Override
        public final Builder dataCompressionType(OpenZFSDataCompressionType dataCompressionType) {
            this.dataCompressionType(dataCompressionType == null ? null : dataCompressionType.toString());
            return this;
        }

        public final Boolean getCopyTagsToSnapshots() {
            return copyTagsToSnapshots;
        }

        public final void setCopyTagsToSnapshots(Boolean copyTagsToSnapshots) {
            this.copyTagsToSnapshots = copyTagsToSnapshots;
        }

        @Override
        public final Builder copyTagsToSnapshots(Boolean copyTagsToSnapshots) {
            this.copyTagsToSnapshots = copyTagsToSnapshots;
            return this;
        }

        public final CreateOpenZFSOriginSnapshotConfiguration.Builder getOriginSnapshot() {
            return originSnapshot != null ? originSnapshot.toBuilder() : null;
        }

        public final void setOriginSnapshot(CreateOpenZFSOriginSnapshotConfiguration.BuilderImpl originSnapshot) {
            this.originSnapshot = originSnapshot != null ? originSnapshot.build() : null;
        }

        @Override
        public final Builder originSnapshot(CreateOpenZFSOriginSnapshotConfiguration originSnapshot) {
            this.originSnapshot = originSnapshot;
            return this;
        }

        public final Boolean getReadOnly() {
            return readOnly;
        }

        public final void setReadOnly(Boolean readOnly) {
            this.readOnly = readOnly;
        }

        @Override
        public final Builder readOnly(Boolean readOnly) {
            this.readOnly = readOnly;
            return this;
        }

        public final List<OpenZFSNfsExport.Builder> getNfsExports() {
            List<OpenZFSNfsExport.Builder> result = OpenZFSNfsExportsCopier.copyToBuilder(this.nfsExports);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNfsExports(Collection<OpenZFSNfsExport.BuilderImpl> nfsExports) {
            this.nfsExports = OpenZFSNfsExportsCopier.copyFromBuilder(nfsExports);
        }

        @Override
        public final Builder nfsExports(Collection<OpenZFSNfsExport> nfsExports) {
            this.nfsExports = OpenZFSNfsExportsCopier.copy(nfsExports);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nfsExports(OpenZFSNfsExport... nfsExports) {
            nfsExports(Arrays.asList(nfsExports));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nfsExports(Consumer<OpenZFSNfsExport.Builder>... nfsExports) {
            nfsExports(Stream.of(nfsExports).map(c -> OpenZFSNfsExport.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<OpenZFSUserOrGroupQuota.Builder> getUserAndGroupQuotas() {
            List<OpenZFSUserOrGroupQuota.Builder> result = OpenZFSUserAndGroupQuotasCopier.copyToBuilder(this.userAndGroupQuotas);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUserAndGroupQuotas(Collection<OpenZFSUserOrGroupQuota.BuilderImpl> userAndGroupQuotas) {
            this.userAndGroupQuotas = OpenZFSUserAndGroupQuotasCopier.copyFromBuilder(userAndGroupQuotas);
        }

        @Override
        public final Builder userAndGroupQuotas(Collection<OpenZFSUserOrGroupQuota> userAndGroupQuotas) {
            this.userAndGroupQuotas = OpenZFSUserAndGroupQuotasCopier.copy(userAndGroupQuotas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAndGroupQuotas(OpenZFSUserOrGroupQuota... userAndGroupQuotas) {
            userAndGroupQuotas(Arrays.asList(userAndGroupQuotas));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAndGroupQuotas(Consumer<OpenZFSUserOrGroupQuota.Builder>... userAndGroupQuotas) {
            userAndGroupQuotas(Stream.of(userAndGroupQuotas).map(c -> OpenZFSUserOrGroupQuota.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CreateOpenZFSVolumeConfiguration build() {
            return new CreateOpenZFSVolumeConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
