/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The data repository configuration object for Lustre file systems returned in the response of the
 * <code>CreateFileSystem</code> operation.
 * </p>
 * <p>
 * This data type is not supported on file systems with a data repository association. For file systems with a data
 * repository association, see .
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataRepositoryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DataRepositoryConfiguration.Builder, DataRepositoryConfiguration> {
    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(DataRepositoryConfiguration::lifecycleAsString))
            .setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<String> IMPORT_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImportPath").getter(getter(DataRepositoryConfiguration::importPath)).setter(setter(Builder::importPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportPath").build()).build();

    private static final SdkField<String> EXPORT_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExportPath").getter(getter(DataRepositoryConfiguration::exportPath)).setter(setter(Builder::exportPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportPath").build()).build();

    private static final SdkField<Integer> IMPORTED_FILE_CHUNK_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ImportedFileChunkSize").getter(getter(DataRepositoryConfiguration::importedFileChunkSize))
            .setter(setter(Builder::importedFileChunkSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportedFileChunkSize").build())
            .build();

    private static final SdkField<String> AUTO_IMPORT_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoImportPolicy").getter(getter(DataRepositoryConfiguration::autoImportPolicyAsString))
            .setter(setter(Builder::autoImportPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoImportPolicy").build()).build();

    private static final SdkField<DataRepositoryFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<DataRepositoryFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(DataRepositoryConfiguration::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(DataRepositoryFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(LIFECYCLE_FIELD, IMPORT_PATH_FIELD, EXPORT_PATH_FIELD, IMPORTED_FILE_CHUNK_SIZE_FIELD,
                    AUTO_IMPORT_POLICY_FIELD, FAILURE_DETAILS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String lifecycle;

    private final String importPath;

    private final String exportPath;

    private final Integer importedFileChunkSize;

    private final String autoImportPolicy;

    private final DataRepositoryFailureDetails failureDetails;

    private DataRepositoryConfiguration(BuilderImpl builder) {
        this.lifecycle = builder.lifecycle;
        this.importPath = builder.importPath;
        this.exportPath = builder.exportPath;
        this.importedFileChunkSize = builder.importedFileChunkSize;
        this.autoImportPolicy = builder.autoImportPolicy;
        this.failureDetails = builder.failureDetails;
    }

    /**
     * <p>
     * Describes the state of the file system's S3 durable data repository, if it is configured with an S3 repository.
     * The lifecycle can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The data repository configuration between the FSx file system and the linked S3 data
     * repository is being created. The data repository is unavailable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The data repository is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until the data
     * repository configuration is corrected. For more information, see <a href=
     * "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
     * >Troubleshooting a Misconfigured linked S3 bucket</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability may be
     * impacted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the state of the file system's S3 durable data repository, if it is configured with an S3
     *         repository. The lifecycle can have the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The data repository configuration between the FSx file system and the linked S3
     *         data repository is being created. The data repository is unavailable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The data repository is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until the
     *         data repository configuration is corrected. For more information, see <a href=
     *         "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
     *         >Troubleshooting a Misconfigured linked S3 bucket</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability
     *         may be impacted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
     *         </p>
     *         </li>
     * @see DataRepositoryLifecycle
     */
    public final DataRepositoryLifecycle lifecycle() {
        return DataRepositoryLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * Describes the state of the file system's S3 durable data repository, if it is configured with an S3 repository.
     * The lifecycle can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The data repository configuration between the FSx file system and the linked S3 data
     * repository is being created. The data repository is unavailable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The data repository is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until the data
     * repository configuration is corrected. For more information, see <a href=
     * "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
     * >Troubleshooting a Misconfigured linked S3 bucket</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability may be
     * impacted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the state of the file system's S3 durable data repository, if it is configured with an S3
     *         repository. The lifecycle can have the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The data repository configuration between the FSx file system and the linked S3
     *         data repository is being created. The data repository is unavailable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The data repository is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until the
     *         data repository configuration is corrected. For more information, see <a href=
     *         "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
     *         >Troubleshooting a Misconfigured linked S3 bucket</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability
     *         may be impacted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
     *         </p>
     *         </li>
     * @see DataRepositoryLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * <p>
     * The import path to the Amazon S3 bucket (and optional prefix) that you're using as the data repository for your
     * FSx for Lustre file system, for example <code>s3://import-bucket/optional-prefix</code>. If a prefix is specified
     * after the Amazon S3 bucket name, only object keys with that prefix are loaded into the file system.
     * </p>
     * 
     * @return The import path to the Amazon S3 bucket (and optional prefix) that you're using as the data repository
     *         for your FSx for Lustre file system, for example <code>s3://import-bucket/optional-prefix</code>. If a
     *         prefix is specified after the Amazon S3 bucket name, only object keys with that prefix are loaded into
     *         the file system.
     */
    public final String importPath() {
        return importPath;
    }

    /**
     * <p>
     * The export path to the Amazon S3 bucket (and prefix) that you are using to store new and changed Lustre file
     * system files in S3.
     * </p>
     * 
     * @return The export path to the Amazon S3 bucket (and prefix) that you are using to store new and changed Lustre
     *         file system files in S3.
     */
    public final String exportPath() {
        return exportPath;
    }

    /**
     * <p>
     * For files imported from a data repository, this value determines the stripe count and maximum amount of data per
     * file (in MiB) stored on a single physical disk. The maximum number of disks that a single file can be striped
     * across is limited by the total number of disks that make up the file system.
     * </p>
     * <p>
     * The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3 objects have a
     * maximum size of 5 TB.
     * </p>
     * 
     * @return For files imported from a data repository, this value determines the stripe count and maximum amount of
     *         data per file (in MiB) stored on a single physical disk. The maximum number of disks that a single file
     *         can be striped across is limited by the total number of disks that make up the file system.</p>
     *         <p>
     *         The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3
     *         objects have a maximum size of 5 TB.
     */
    public final Integer importedFileChunkSize() {
        return importedFileChunkSize;
    }

    /**
     * <p>
     * Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The AutoImportPolicy
     * configures how Amazon FSx keeps your file and directory listings up to date as you add or modify objects in your
     * linked S3 bucket. <code>AutoImportPolicy</code> can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings from the
     * linked S3 bucket when the file system is created. FSx does not update file and directory listings for any new or
     * changed objects after choosing this option.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new objects added
     * to the linked S3 bucket that do not currently exist in the FSx file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings of any
     * new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket after you choose
     * this option.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings
     * of any new objects added to the S3 bucket, any existing objects that are changed in the S3 bucket, and any
     * objects that were deleted in the S3 bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoImportPolicy}
     * will return {@link AutoImportPolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #autoImportPolicyAsString}.
     * </p>
     * 
     * @return Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The
     *         AutoImportPolicy configures how Amazon FSx keeps your file and directory listings up to date as you add
     *         or modify objects in your linked S3 bucket. <code>AutoImportPolicy</code> can have the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings from
     *         the linked S3 bucket when the file system is created. FSx does not update file and directory listings for
     *         any new or changed objects after choosing this option.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new
     *         objects added to the linked S3 bucket that do not currently exist in the FSx file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings
     *         of any new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket
     *         after you choose this option.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and directory
     *         listings of any new objects added to the S3 bucket, any existing objects that are changed in the S3
     *         bucket, and any objects that were deleted in the S3 bucket.
     *         </p>
     *         </li>
     * @see AutoImportPolicyType
     */
    public final AutoImportPolicyType autoImportPolicy() {
        return AutoImportPolicyType.fromValue(autoImportPolicy);
    }

    /**
     * <p>
     * Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The AutoImportPolicy
     * configures how Amazon FSx keeps your file and directory listings up to date as you add or modify objects in your
     * linked S3 bucket. <code>AutoImportPolicy</code> can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings from the
     * linked S3 bucket when the file system is created. FSx does not update file and directory listings for any new or
     * changed objects after choosing this option.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new objects added
     * to the linked S3 bucket that do not currently exist in the FSx file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings of any
     * new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket after you choose
     * this option.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings
     * of any new objects added to the S3 bucket, any existing objects that are changed in the S3 bucket, and any
     * objects that were deleted in the S3 bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoImportPolicy}
     * will return {@link AutoImportPolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #autoImportPolicyAsString}.
     * </p>
     * 
     * @return Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The
     *         AutoImportPolicy configures how Amazon FSx keeps your file and directory listings up to date as you add
     *         or modify objects in your linked S3 bucket. <code>AutoImportPolicy</code> can have the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings from
     *         the linked S3 bucket when the file system is created. FSx does not update file and directory listings for
     *         any new or changed objects after choosing this option.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new
     *         objects added to the linked S3 bucket that do not currently exist in the FSx file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings
     *         of any new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket
     *         after you choose this option.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and directory
     *         listings of any new objects added to the S3 bucket, any existing objects that are changed in the S3
     *         bucket, and any objects that were deleted in the S3 bucket.
     *         </p>
     *         </li>
     * @see AutoImportPolicyType
     */
    public final String autoImportPolicyAsString() {
        return autoImportPolicy;
    }

    /**
     * Returns the value of the FailureDetails property for this object.
     * 
     * @return The value of the FailureDetails property for this object.
     */
    public final DataRepositoryFailureDetails failureDetails() {
        return failureDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(importPath());
        hashCode = 31 * hashCode + Objects.hashCode(exportPath());
        hashCode = 31 * hashCode + Objects.hashCode(importedFileChunkSize());
        hashCode = 31 * hashCode + Objects.hashCode(autoImportPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataRepositoryConfiguration)) {
            return false;
        }
        DataRepositoryConfiguration other = (DataRepositoryConfiguration) obj;
        return Objects.equals(lifecycleAsString(), other.lifecycleAsString()) && Objects.equals(importPath(), other.importPath())
                && Objects.equals(exportPath(), other.exportPath())
                && Objects.equals(importedFileChunkSize(), other.importedFileChunkSize())
                && Objects.equals(autoImportPolicyAsString(), other.autoImportPolicyAsString())
                && Objects.equals(failureDetails(), other.failureDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataRepositoryConfiguration").add("Lifecycle", lifecycleAsString())
                .add("ImportPath", importPath()).add("ExportPath", exportPath())
                .add("ImportedFileChunkSize", importedFileChunkSize()).add("AutoImportPolicy", autoImportPolicyAsString())
                .add("FailureDetails", failureDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "ImportPath":
            return Optional.ofNullable(clazz.cast(importPath()));
        case "ExportPath":
            return Optional.ofNullable(clazz.cast(exportPath()));
        case "ImportedFileChunkSize":
            return Optional.ofNullable(clazz.cast(importedFileChunkSize()));
        case "AutoImportPolicy":
            return Optional.ofNullable(clazz.cast(autoImportPolicyAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Lifecycle", LIFECYCLE_FIELD);
        map.put("ImportPath", IMPORT_PATH_FIELD);
        map.put("ExportPath", EXPORT_PATH_FIELD);
        map.put("ImportedFileChunkSize", IMPORTED_FILE_CHUNK_SIZE_FIELD);
        map.put("AutoImportPolicy", AUTO_IMPORT_POLICY_FIELD);
        map.put("FailureDetails", FAILURE_DETAILS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataRepositoryConfiguration, T> g) {
        return obj -> g.apply((DataRepositoryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataRepositoryConfiguration> {
        /**
         * <p>
         * Describes the state of the file system's S3 durable data repository, if it is configured with an S3
         * repository. The lifecycle can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The data repository configuration between the FSx file system and the linked S3 data
         * repository is being created. The data repository is unavailable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The data repository is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until the data
         * repository configuration is corrected. For more information, see <a href=
         * "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
         * >Troubleshooting a Misconfigured linked S3 bucket</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability may be
         * impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the state of the file system's S3 durable data repository, if it is configured with an S3
         *        repository. The lifecycle can have the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The data repository configuration between the FSx file system and the linked
         *        S3 data repository is being created. The data repository is unavailable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The data repository is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until
         *        the data repository configuration is corrected. For more information, see <a href=
         *        "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
         *        >Troubleshooting a Misconfigured linked S3 bucket</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability
         *        may be impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
         *        </p>
         *        </li>
         * @see DataRepositoryLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * Describes the state of the file system's S3 durable data repository, if it is configured with an S3
         * repository. The lifecycle can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The data repository configuration between the FSx file system and the linked S3 data
         * repository is being created. The data repository is unavailable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The data repository is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until the data
         * repository configuration is corrected. For more information, see <a href=
         * "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
         * >Troubleshooting a Misconfigured linked S3 bucket</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability may be
         * impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the state of the file system's S3 durable data repository, if it is configured with an S3
         *        repository. The lifecycle can have the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The data repository configuration between the FSx file system and the linked
         *        S3 data repository is being created. The data repository is unavailable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The data repository is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket until
         *        the data repository configuration is corrected. For more information, see <a href=
         *        "https://docs.aws.amazon.com/fsx/latest/LustreGuide/troubleshooting.html#troubleshooting-misconfigured-data-repository"
         *        >Troubleshooting a Misconfigured linked S3 bucket</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The data repository is undergoing a customer initiated update and availability
         *        may be impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The data repository is in a terminal state that cannot be recovered.
         *        </p>
         *        </li>
         * @see DataRepositoryLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryLifecycle
         */
        Builder lifecycle(DataRepositoryLifecycle lifecycle);

        /**
         * <p>
         * The import path to the Amazon S3 bucket (and optional prefix) that you're using as the data repository for
         * your FSx for Lustre file system, for example <code>s3://import-bucket/optional-prefix</code>. If a prefix is
         * specified after the Amazon S3 bucket name, only object keys with that prefix are loaded into the file system.
         * </p>
         * 
         * @param importPath
         *        The import path to the Amazon S3 bucket (and optional prefix) that you're using as the data repository
         *        for your FSx for Lustre file system, for example <code>s3://import-bucket/optional-prefix</code>. If a
         *        prefix is specified after the Amazon S3 bucket name, only object keys with that prefix are loaded into
         *        the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importPath(String importPath);

        /**
         * <p>
         * The export path to the Amazon S3 bucket (and prefix) that you are using to store new and changed Lustre file
         * system files in S3.
         * </p>
         * 
         * @param exportPath
         *        The export path to the Amazon S3 bucket (and prefix) that you are using to store new and changed
         *        Lustre file system files in S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportPath(String exportPath);

        /**
         * <p>
         * For files imported from a data repository, this value determines the stripe count and maximum amount of data
         * per file (in MiB) stored on a single physical disk. The maximum number of disks that a single file can be
         * striped across is limited by the total number of disks that make up the file system.
         * </p>
         * <p>
         * The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3 objects
         * have a maximum size of 5 TB.
         * </p>
         * 
         * @param importedFileChunkSize
         *        For files imported from a data repository, this value determines the stripe count and maximum amount
         *        of data per file (in MiB) stored on a single physical disk. The maximum number of disks that a single
         *        file can be striped across is limited by the total number of disks that make up the file system.</p>
         *        <p>
         *        The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3
         *        objects have a maximum size of 5 TB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importedFileChunkSize(Integer importedFileChunkSize);

        /**
         * <p>
         * Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The AutoImportPolicy
         * configures how Amazon FSx keeps your file and directory listings up to date as you add or modify objects in
         * your linked S3 bucket. <code>AutoImportPolicy</code> can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings from the
         * linked S3 bucket when the file system is created. FSx does not update file and directory listings for any new
         * or changed objects after choosing this option.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new objects
         * added to the linked S3 bucket that do not currently exist in the FSx file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings of
         * any new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket after you
         * choose this option.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and directory
         * listings of any new objects added to the S3 bucket, any existing objects that are changed in the S3 bucket,
         * and any objects that were deleted in the S3 bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param autoImportPolicy
         *        Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The
         *        AutoImportPolicy configures how Amazon FSx keeps your file and directory listings up to date as you
         *        add or modify objects in your linked S3 bucket. <code>AutoImportPolicy</code> can have the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings
         *        from the linked S3 bucket when the file system is created. FSx does not update file and directory
         *        listings for any new or changed objects after choosing this option.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new
         *        objects added to the linked S3 bucket that do not currently exist in the FSx file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory
         *        listings of any new objects added to the S3 bucket and any existing objects that are changed in the S3
         *        bucket after you choose this option.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and
         *        directory listings of any new objects added to the S3 bucket, any existing objects that are changed in
         *        the S3 bucket, and any objects that were deleted in the S3 bucket.
         *        </p>
         *        </li>
         * @see AutoImportPolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoImportPolicyType
         */
        Builder autoImportPolicy(String autoImportPolicy);

        /**
         * <p>
         * Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The AutoImportPolicy
         * configures how Amazon FSx keeps your file and directory listings up to date as you add or modify objects in
         * your linked S3 bucket. <code>AutoImportPolicy</code> can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings from the
         * linked S3 bucket when the file system is created. FSx does not update file and directory listings for any new
         * or changed objects after choosing this option.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new objects
         * added to the linked S3 bucket that do not currently exist in the FSx file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory listings of
         * any new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket after you
         * choose this option.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and directory
         * listings of any new objects added to the S3 bucket, any existing objects that are changed in the S3 bucket,
         * and any objects that were deleted in the S3 bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param autoImportPolicy
         *        Describes the file system's linked S3 data repository's <code>AutoImportPolicy</code>. The
         *        AutoImportPolicy configures how Amazon FSx keeps your file and directory listings up to date as you
         *        add or modify objects in your linked S3 bucket. <code>AutoImportPolicy</code> can have the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - (Default) AutoImport is off. Amazon FSx only updates file and directory listings
         *        from the linked S3 bucket when the file system is created. FSx does not update file and directory
         *        listings for any new or changed objects after choosing this option.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW</code> - AutoImport is on. Amazon FSx automatically imports directory listings of any new
         *        objects added to the linked S3 bucket that do not currently exist in the FSx file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_CHANGED</code> - AutoImport is on. Amazon FSx automatically imports file and directory
         *        listings of any new objects added to the S3 bucket and any existing objects that are changed in the S3
         *        bucket after you choose this option.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_CHANGED_DELETED</code> - AutoImport is on. Amazon FSx automatically imports file and
         *        directory listings of any new objects added to the S3 bucket, any existing objects that are changed in
         *        the S3 bucket, and any objects that were deleted in the S3 bucket.
         *        </p>
         *        </li>
         * @see AutoImportPolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoImportPolicyType
         */
        Builder autoImportPolicy(AutoImportPolicyType autoImportPolicy);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * @param failureDetails
         *        The new value for the FailureDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(DataRepositoryFailureDetails failureDetails);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DataRepositoryFailureDetails.Builder}
         * avoiding the need to create one manually via {@link DataRepositoryFailureDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataRepositoryFailureDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #failureDetails(DataRepositoryFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link DataRepositoryFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(DataRepositoryFailureDetails)
         */
        default Builder failureDetails(Consumer<DataRepositoryFailureDetails.Builder> failureDetails) {
            return failureDetails(DataRepositoryFailureDetails.builder().applyMutation(failureDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String lifecycle;

        private String importPath;

        private String exportPath;

        private Integer importedFileChunkSize;

        private String autoImportPolicy;

        private DataRepositoryFailureDetails failureDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(DataRepositoryConfiguration model) {
            lifecycle(model.lifecycle);
            importPath(model.importPath);
            exportPath(model.exportPath);
            importedFileChunkSize(model.importedFileChunkSize);
            autoImportPolicy(model.autoImportPolicy);
            failureDetails(model.failureDetails);
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(DataRepositoryLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final String getImportPath() {
            return importPath;
        }

        public final void setImportPath(String importPath) {
            this.importPath = importPath;
        }

        @Override
        public final Builder importPath(String importPath) {
            this.importPath = importPath;
            return this;
        }

        public final String getExportPath() {
            return exportPath;
        }

        public final void setExportPath(String exportPath) {
            this.exportPath = exportPath;
        }

        @Override
        public final Builder exportPath(String exportPath) {
            this.exportPath = exportPath;
            return this;
        }

        public final Integer getImportedFileChunkSize() {
            return importedFileChunkSize;
        }

        public final void setImportedFileChunkSize(Integer importedFileChunkSize) {
            this.importedFileChunkSize = importedFileChunkSize;
        }

        @Override
        public final Builder importedFileChunkSize(Integer importedFileChunkSize) {
            this.importedFileChunkSize = importedFileChunkSize;
            return this;
        }

        public final String getAutoImportPolicy() {
            return autoImportPolicy;
        }

        public final void setAutoImportPolicy(String autoImportPolicy) {
            this.autoImportPolicy = autoImportPolicy;
        }

        @Override
        public final Builder autoImportPolicy(String autoImportPolicy) {
            this.autoImportPolicy = autoImportPolicy;
            return this;
        }

        @Override
        public final Builder autoImportPolicy(AutoImportPolicyType autoImportPolicy) {
            this.autoImportPolicy(autoImportPolicy == null ? null : autoImportPolicy.toString());
            return this;
        }

        public final DataRepositoryFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        public final void setFailureDetails(DataRepositoryFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        @Override
        public final Builder failureDetails(DataRepositoryFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        @Override
        public DataRepositoryConfiguration build() {
            return new DataRepositoryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
