/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of a specific Amazon FSx file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FileSystem implements SdkPojo, Serializable, ToCopyableBuilder<FileSystem.Builder, FileSystem> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerId").getter(getter(FileSystem::ownerId)).setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(FileSystem::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(FileSystem::fileSystemId)).setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemType").getter(getter(FileSystem::fileSystemTypeAsString))
            .setter(setter(Builder::fileSystemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemType").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(FileSystem::lifecycleAsString)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<FileSystemFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<FileSystemFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(FileSystem::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(FileSystemFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacity").getter(getter(FileSystem::storageCapacity)).setter(setter(Builder::storageCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacity").build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(FileSystem::storageTypeAsString)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(FileSystem::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(FileSystem::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> NETWORK_INTERFACE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaceIds")
            .getter(getter(FileSystem::networkInterfaceIds))
            .setter(setter(Builder::networkInterfaceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DNSName").getter(getter(FileSystem::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DNSName").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(FileSystem::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(FileSystem::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(FileSystem::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<WindowsFileSystemConfiguration> WINDOWS_CONFIGURATION_FIELD = SdkField
            .<WindowsFileSystemConfiguration> builder(MarshallingType.SDK_POJO).memberName("WindowsConfiguration")
            .getter(getter(FileSystem::windowsConfiguration)).setter(setter(Builder::windowsConfiguration))
            .constructor(WindowsFileSystemConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowsConfiguration").build())
            .build();

    private static final SdkField<LustreFileSystemConfiguration> LUSTRE_CONFIGURATION_FIELD = SdkField
            .<LustreFileSystemConfiguration> builder(MarshallingType.SDK_POJO).memberName("LustreConfiguration")
            .getter(getter(FileSystem::lustreConfiguration)).setter(setter(Builder::lustreConfiguration))
            .constructor(LustreFileSystemConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LustreConfiguration").build())
            .build();

    private static final SdkField<List<AdministrativeAction>> ADMINISTRATIVE_ACTIONS_FIELD = SdkField
            .<List<AdministrativeAction>> builder(MarshallingType.LIST)
            .memberName("AdministrativeActions")
            .getter(getter(FileSystem::administrativeActions))
            .setter(setter(Builder::administrativeActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdministrativeActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AdministrativeAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(AdministrativeAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<OntapFileSystemConfiguration> ONTAP_CONFIGURATION_FIELD = SdkField
            .<OntapFileSystemConfiguration> builder(MarshallingType.SDK_POJO).memberName("OntapConfiguration")
            .getter(getter(FileSystem::ontapConfiguration)).setter(setter(Builder::ontapConfiguration))
            .constructor(OntapFileSystemConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OntapConfiguration").build())
            .build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemTypeVersion").getter(getter(FileSystem::fileSystemTypeVersion))
            .setter(setter(Builder::fileSystemTypeVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemTypeVersion").build())
            .build();

    private static final SdkField<OpenZFSFileSystemConfiguration> OPEN_ZFS_CONFIGURATION_FIELD = SdkField
            .<OpenZFSFileSystemConfiguration> builder(MarshallingType.SDK_POJO).memberName("OpenZFSConfiguration")
            .getter(getter(FileSystem::openZFSConfiguration)).setter(setter(Builder::openZFSConfiguration))
            .constructor(OpenZFSFileSystemConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpenZFSConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            CREATION_TIME_FIELD, FILE_SYSTEM_ID_FIELD, FILE_SYSTEM_TYPE_FIELD, LIFECYCLE_FIELD, FAILURE_DETAILS_FIELD,
            STORAGE_CAPACITY_FIELD, STORAGE_TYPE_FIELD, VPC_ID_FIELD, SUBNET_IDS_FIELD, NETWORK_INTERFACE_IDS_FIELD,
            DNS_NAME_FIELD, KMS_KEY_ID_FIELD, RESOURCE_ARN_FIELD, TAGS_FIELD, WINDOWS_CONFIGURATION_FIELD,
            LUSTRE_CONFIGURATION_FIELD, ADMINISTRATIVE_ACTIONS_FIELD, ONTAP_CONFIGURATION_FIELD, FILE_SYSTEM_TYPE_VERSION_FIELD,
            OPEN_ZFS_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final Instant creationTime;

    private final String fileSystemId;

    private final String fileSystemType;

    private final String lifecycle;

    private final FileSystemFailureDetails failureDetails;

    private final Integer storageCapacity;

    private final String storageType;

    private final String vpcId;

    private final List<String> subnetIds;

    private final List<String> networkInterfaceIds;

    private final String dnsName;

    private final String kmsKeyId;

    private final String resourceARN;

    private final List<Tag> tags;

    private final WindowsFileSystemConfiguration windowsConfiguration;

    private final LustreFileSystemConfiguration lustreConfiguration;

    private final List<AdministrativeAction> administrativeActions;

    private final OntapFileSystemConfiguration ontapConfiguration;

    private final String fileSystemTypeVersion;

    private final OpenZFSFileSystemConfiguration openZFSConfiguration;

    private FileSystem(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.creationTime = builder.creationTime;
        this.fileSystemId = builder.fileSystemId;
        this.fileSystemType = builder.fileSystemType;
        this.lifecycle = builder.lifecycle;
        this.failureDetails = builder.failureDetails;
        this.storageCapacity = builder.storageCapacity;
        this.storageType = builder.storageType;
        this.vpcId = builder.vpcId;
        this.subnetIds = builder.subnetIds;
        this.networkInterfaceIds = builder.networkInterfaceIds;
        this.dnsName = builder.dnsName;
        this.kmsKeyId = builder.kmsKeyId;
        this.resourceARN = builder.resourceARN;
        this.tags = builder.tags;
        this.windowsConfiguration = builder.windowsConfiguration;
        this.lustreConfiguration = builder.lustreConfiguration;
        this.administrativeActions = builder.administrativeActions;
        this.ontapConfiguration = builder.ontapConfiguration;
        this.fileSystemTypeVersion = builder.fileSystemTypeVersion;
        this.openZFSConfiguration = builder.openZFSConfiguration;
    }

    /**
     * <p>
     * The Amazon Web Services account that created the file system. If the file system was created by a user in IAM
     * Identity Center, the Amazon Web Services account to which the IAM user belongs is the owner.
     * </p>
     * 
     * @return The Amazon Web Services account that created the file system. If the file system was created by a user in
     *         IAM Identity Center, the Amazon Web Services account to which the IAM user belongs is the owner.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix time.
     * </p>
     * 
     * @return The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix
     *         time.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The system-generated, unique 17-digit ID of the file system.
     * </p>
     * 
     * @return The system-generated, unique 17-digit ID of the file system.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>, <code>ONTAP</code>,
     * or <code>OPENZFS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>,
     *         <code>ONTAP</code>, or <code>OPENZFS</code>.
     * @see FileSystemType
     */
    public final FileSystemType fileSystemType() {
        return FileSystemType.fromValue(fileSystemType);
    }

    /**
     * <p>
     * The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>, <code>ONTAP</code>,
     * or <code>OPENZFS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>,
     *         <code>ONTAP</code>, or <code>OPENZFS</code>.
     * @see FileSystemType
     */
    public final String fileSystemTypeAsString() {
        return fileSystemType;
    }

    /**
     * <p>
     * The lifecycle status of the file system. The following are the possible values and what they mean:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the new file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new file
     * system, Amazon FSx was unable to create the file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is currently
     * unavailable due to a change in your Active Directory configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link FileSystemLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the file system. The following are the possible values and what they mean:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the new file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a
     *         new file system, Amazon FSx was unable to create the file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is
     *         currently unavailable due to a change in your Active Directory configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
     *         </p>
     *         </li>
     * @see FileSystemLifecycle
     */
    public final FileSystemLifecycle lifecycle() {
        return FileSystemLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * The lifecycle status of the file system. The following are the possible values and what they mean:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the new file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new file
     * system, Amazon FSx was unable to create the file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is currently
     * unavailable due to a change in your Active Directory configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link FileSystemLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the file system. The following are the possible values and what they mean:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the new file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a
     *         new file system, Amazon FSx was unable to create the file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is
     *         currently unavailable due to a change in your Active Directory configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
     *         </p>
     *         </li>
     * @see FileSystemLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * Returns the value of the FailureDetails property for this object.
     * 
     * @return The value of the FailureDetails property for this object.
     */
    public final FileSystemFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * The storage capacity of the file system in gibibytes (GiB).
     * </p>
     * <p>
     * Amazon FSx responds with an HTTP status code 400 (Bad Request) if the value of <code>StorageCapacity</code> is
     * outside of the minimum or maximum values.
     * </p>
     * 
     * @return The storage capacity of the file system in gibibytes (GiB).</p>
     *         <p>
     *         Amazon FSx responds with an HTTP status code 400 (Bad Request) if the value of
     *         <code>StorageCapacity</code> is outside of the minimum or maximum values.
     */
    public final Integer storageCapacity() {
        return storageCapacity;
    }

    /**
     * <p>
     * The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid state drive
     * storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid
     *         state drive storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
     * @see StorageType
     */
    public final StorageType storageType() {
        return StorageType.fromValue(storageType);
    }

    /**
     * <p>
     * The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid state drive
     * storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid
     *         state drive storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
     * @see StorageType
     */
    public final String storageTypeAsString() {
        return storageType;
    }

    /**
     * <p>
     * The ID of the primary virtual private cloud (VPC) for the file system.
     * </p>
     * 
     * @return The ID of the primary virtual private cloud (VPC) for the file system.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the IDs of the subnets that the file system is accessible from. For the Amazon FSx Windows and ONTAP
     * <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the preferred file server
     * and one for the standby file server. The preferred file server subnet identified in the
     * <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.
     * </p>
     * <p>
     * For FSx for Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that contains
     * the file system's endpoint. For <code>MULTI_AZ_1</code> Windows and ONTAP file systems, the file system endpoint
     * is available in the <code>PreferredSubnetID</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return Specifies the IDs of the subnets that the file system is accessible from. For the Amazon FSx Windows and
     *         ONTAP <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the
     *         preferred file server and one for the standby file server. The preferred file server subnet identified in
     *         the <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.</p>
     *         <p>
     *         For FSx for Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
     *         contains the file system's endpoint. For <code>MULTI_AZ_1</code> Windows and ONTAP file systems, the file
     *         system endpoint is available in the <code>PreferredSubnetID</code>.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaceIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaceIds() {
        return networkInterfaceIds != null && !(networkInterfaceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the elastic network interfaces from which a specific file system is accessible. The elastic network
     * interface is automatically created in the same virtual private cloud (VPC) that the Amazon FSx file system was
     * created in. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network Interfaces</a> in the
     * <i>Amazon EC2 User Guide.</i>
     * </p>
     * <p>
     * For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an Amazon FSx
     * for Lustre file system, you can have more than one.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaceIds} method.
     * </p>
     * 
     * @return The IDs of the elastic network interfaces from which a specific file system is accessible. The elastic
     *         network interface is automatically created in the same virtual private cloud (VPC) that the Amazon FSx
     *         file system was created in. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network Interfaces</a>
     *         in the <i>Amazon EC2 User Guide.</i> </p>
     *         <p>
     *         For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an
     *         Amazon FSx for Lustre file system, you can have more than one.
     */
    public final List<String> networkInterfaceIds() {
        return networkInterfaceIds;
    }

    /**
     * <p>
     * The Domain Name System (DNS) name for the file system.
     * </p>
     * 
     * @return The Domain Name System (DNS) name for the file system.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The ID of the Key Management Service (KMS) key used to encrypt Amazon FSx file system data. Used as follows with
     * Amazon FSx file system types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon FSx for Lustre <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment types only.
     * </p>
     * <p>
     * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> types are encrypted using the Amazon FSx service KMS key for
     * your account.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for NetApp ONTAP
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for OpenZFS
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for Windows File Server
     * </p>
     * </li>
     * </ul>
     * 
     * @return The ID of the Key Management Service (KMS) key used to encrypt Amazon FSx file system data. Used as
     *         follows with Amazon FSx file system types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon FSx for Lustre <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment types only.
     *         </p>
     *         <p>
     *         <code>SCRATCH_1</code> and <code>SCRATCH_2</code> types are encrypted using the Amazon FSx service KMS
     *         key for your account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon FSx for NetApp ONTAP
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon FSx for OpenZFS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon FSx for Windows File Server
     *         </p>
     *         </li>
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the file system resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the file system resource.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to associate with the file system. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
     * resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags to associate with the file system. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
     *         resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The configuration for this Amazon FSx for Windows File Server file system.
     * </p>
     * 
     * @return The configuration for this Amazon FSx for Windows File Server file system.
     */
    public final WindowsFileSystemConfiguration windowsConfiguration() {
        return windowsConfiguration;
    }

    /**
     * Returns the value of the LustreConfiguration property for this object.
     * 
     * @return The value of the LustreConfiguration property for this object.
     */
    public final LustreFileSystemConfiguration lustreConfiguration() {
        return lustreConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdministrativeActions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAdministrativeActions() {
        return administrativeActions != null && !(administrativeActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of administrative actions for the file system that are in process or waiting to be processed.
     * Administrative actions describe changes to the Amazon FSx system that you have initiated using the
     * <code>UpdateFileSystem</code> operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdministrativeActions} method.
     * </p>
     * 
     * @return A list of administrative actions for the file system that are in process or waiting to be processed.
     *         Administrative actions describe changes to the Amazon FSx system that you have initiated using the
     *         <code>UpdateFileSystem</code> operation.
     */
    public final List<AdministrativeAction> administrativeActions() {
        return administrativeActions;
    }

    /**
     * <p>
     * The configuration for this Amazon FSx for NetApp ONTAP file system.
     * </p>
     * 
     * @return The configuration for this Amazon FSx for NetApp ONTAP file system.
     */
    public final OntapFileSystemConfiguration ontapConfiguration() {
        return ontapConfiguration;
    }

    /**
     * <p>
     * The Lustre version of the Amazon FSx for Lustre file system, which can be <code>2.10</code>, <code>2.12</code>,
     * or <code>2.15</code>.
     * </p>
     * 
     * @return The Lustre version of the Amazon FSx for Lustre file system, which can be <code>2.10</code>,
     *         <code>2.12</code>, or <code>2.15</code>.
     */
    public final String fileSystemTypeVersion() {
        return fileSystemTypeVersion;
    }

    /**
     * <p>
     * The configuration for this Amazon FSx for OpenZFS file system.
     * </p>
     * 
     * @return The configuration for this Amazon FSx for OpenZFS file system.
     */
    public final OpenZFSFileSystemConfiguration openZFSConfiguration() {
        return openZFSConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(storageTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaceIds() ? networkInterfaceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(windowsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(lustreConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdministrativeActions() ? administrativeActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ontapConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeVersion());
        hashCode = 31 * hashCode + Objects.hashCode(openZFSConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileSystem)) {
            return false;
        }
        FileSystem other = (FileSystem) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(fileSystemTypeAsString(), other.fileSystemTypeAsString())
                && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(storageCapacity(), other.storageCapacity())
                && Objects.equals(storageTypeAsString(), other.storageTypeAsString()) && Objects.equals(vpcId(), other.vpcId())
                && hasSubnetIds() == other.hasSubnetIds() && Objects.equals(subnetIds(), other.subnetIds())
                && hasNetworkInterfaceIds() == other.hasNetworkInterfaceIds()
                && Objects.equals(networkInterfaceIds(), other.networkInterfaceIds())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(resourceARN(), other.resourceARN()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(windowsConfiguration(), other.windowsConfiguration())
                && Objects.equals(lustreConfiguration(), other.lustreConfiguration())
                && hasAdministrativeActions() == other.hasAdministrativeActions()
                && Objects.equals(administrativeActions(), other.administrativeActions())
                && Objects.equals(ontapConfiguration(), other.ontapConfiguration())
                && Objects.equals(fileSystemTypeVersion(), other.fileSystemTypeVersion())
                && Objects.equals(openZFSConfiguration(), other.openZFSConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FileSystem").add("OwnerId", ownerId()).add("CreationTime", creationTime())
                .add("FileSystemId", fileSystemId()).add("FileSystemType", fileSystemTypeAsString())
                .add("Lifecycle", lifecycleAsString()).add("FailureDetails", failureDetails())
                .add("StorageCapacity", storageCapacity()).add("StorageType", storageTypeAsString()).add("VpcId", vpcId())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null)
                .add("NetworkInterfaceIds", hasNetworkInterfaceIds() ? networkInterfaceIds() : null).add("DNSName", dnsName())
                .add("KmsKeyId", kmsKeyId()).add("ResourceARN", resourceARN()).add("Tags", hasTags() ? tags() : null)
                .add("WindowsConfiguration", windowsConfiguration()).add("LustreConfiguration", lustreConfiguration())
                .add("AdministrativeActions", hasAdministrativeActions() ? administrativeActions() : null)
                .add("OntapConfiguration", ontapConfiguration()).add("FileSystemTypeVersion", fileSystemTypeVersion())
                .add("OpenZFSConfiguration", openZFSConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "FileSystemType":
            return Optional.ofNullable(clazz.cast(fileSystemTypeAsString()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "StorageCapacity":
            return Optional.ofNullable(clazz.cast(storageCapacity()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageTypeAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "NetworkInterfaceIds":
            return Optional.ofNullable(clazz.cast(networkInterfaceIds()));
        case "DNSName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "WindowsConfiguration":
            return Optional.ofNullable(clazz.cast(windowsConfiguration()));
        case "LustreConfiguration":
            return Optional.ofNullable(clazz.cast(lustreConfiguration()));
        case "AdministrativeActions":
            return Optional.ofNullable(clazz.cast(administrativeActions()));
        case "OntapConfiguration":
            return Optional.ofNullable(clazz.cast(ontapConfiguration()));
        case "FileSystemTypeVersion":
            return Optional.ofNullable(clazz.cast(fileSystemTypeVersion()));
        case "OpenZFSConfiguration":
            return Optional.ofNullable(clazz.cast(openZFSConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("OwnerId", OWNER_ID_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("FileSystemId", FILE_SYSTEM_ID_FIELD);
        map.put("FileSystemType", FILE_SYSTEM_TYPE_FIELD);
        map.put("Lifecycle", LIFECYCLE_FIELD);
        map.put("FailureDetails", FAILURE_DETAILS_FIELD);
        map.put("StorageCapacity", STORAGE_CAPACITY_FIELD);
        map.put("StorageType", STORAGE_TYPE_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("SubnetIds", SUBNET_IDS_FIELD);
        map.put("NetworkInterfaceIds", NETWORK_INTERFACE_IDS_FIELD);
        map.put("DNSName", DNS_NAME_FIELD);
        map.put("KmsKeyId", KMS_KEY_ID_FIELD);
        map.put("ResourceARN", RESOURCE_ARN_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("WindowsConfiguration", WINDOWS_CONFIGURATION_FIELD);
        map.put("LustreConfiguration", LUSTRE_CONFIGURATION_FIELD);
        map.put("AdministrativeActions", ADMINISTRATIVE_ACTIONS_FIELD);
        map.put("OntapConfiguration", ONTAP_CONFIGURATION_FIELD);
        map.put("FileSystemTypeVersion", FILE_SYSTEM_TYPE_VERSION_FIELD);
        map.put("OpenZFSConfiguration", OPEN_ZFS_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FileSystem, T> g) {
        return obj -> g.apply((FileSystem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FileSystem> {
        /**
         * <p>
         * The Amazon Web Services account that created the file system. If the file system was created by a user in IAM
         * Identity Center, the Amazon Web Services account to which the IAM user belongs is the owner.
         * </p>
         * 
         * @param ownerId
         *        The Amazon Web Services account that created the file system. If the file system was created by a user
         *        in IAM Identity Center, the Amazon Web Services account to which the IAM user belongs is the owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix time.
         * </p>
         * 
         * @param creationTime
         *        The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The system-generated, unique 17-digit ID of the file system.
         * </p>
         * 
         * @param fileSystemId
         *        The system-generated, unique 17-digit ID of the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>,
         * <code>ONTAP</code>, or <code>OPENZFS</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>,
         *        <code>ONTAP</code>, or <code>OPENZFS</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(String fileSystemType);

        /**
         * <p>
         * The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>,
         * <code>ONTAP</code>, or <code>OPENZFS</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system, which can be <code>LUSTRE</code>, <code>WINDOWS</code>,
         *        <code>ONTAP</code>, or <code>OPENZFS</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(FileSystemType fileSystemType);

        /**
         * <p>
         * The lifecycle status of the file system. The following are the possible values and what they mean:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the new file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new
         * file system, Amazon FSx was unable to create the file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is
         * currently unavailable due to a change in your Active Directory configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the file system. The following are the possible values and what they mean:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for
         *        use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the new file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating
         *        a new file system, Amazon FSx was unable to create the file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is
         *        currently unavailable due to a change in your Active Directory configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
         *        </p>
         *        </li>
         * @see FileSystemLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * The lifecycle status of the file system. The following are the possible values and what they mean:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the new file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new
         * file system, Amazon FSx was unable to create the file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is
         * currently unavailable due to a change in your Active Directory configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the file system. The following are the possible values and what they mean:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for
         *        use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the new file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating
         *        a new file system, Amazon FSx was unable to create the file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - The file system is in a failed but recoverable state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED_UNAVAILABLE</code> - (Amazon FSx for Windows File Server only) The file system is
         *        currently unavailable due to a change in your Active Directory configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The file system is undergoing a customer-initiated update.
         *        </p>
         *        </li>
         * @see FileSystemLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemLifecycle
         */
        Builder lifecycle(FileSystemLifecycle lifecycle);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * @param failureDetails
         *        The new value for the FailureDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(FileSystemFailureDetails failureDetails);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link FileSystemFailureDetails.Builder}
         * avoiding the need to create one manually via {@link FileSystemFailureDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FileSystemFailureDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #failureDetails(FileSystemFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link FileSystemFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(FileSystemFailureDetails)
         */
        default Builder failureDetails(Consumer<FileSystemFailureDetails.Builder> failureDetails) {
            return failureDetails(FileSystemFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * The storage capacity of the file system in gibibytes (GiB).
         * </p>
         * <p>
         * Amazon FSx responds with an HTTP status code 400 (Bad Request) if the value of <code>StorageCapacity</code>
         * is outside of the minimum or maximum values.
         * </p>
         * 
         * @param storageCapacity
         *        The storage capacity of the file system in gibibytes (GiB).</p>
         *        <p>
         *        Amazon FSx responds with an HTTP status code 400 (Bad Request) if the value of
         *        <code>StorageCapacity</code> is outside of the minimum or maximum values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacity(Integer storageCapacity);

        /**
         * <p>
         * The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid state
         * drive storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
         * </p>
         * 
         * @param storageType
         *        The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid
         *        state drive storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid state
         * drive storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
         * </p>
         * 
         * @param storageType
         *        The type of storage the file system is using. If set to <code>SSD</code>, the file system uses solid
         *        state drive storage. If set to <code>HDD</code>, the file system uses hard disk drive storage.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(StorageType storageType);

        /**
         * <p>
         * The ID of the primary virtual private cloud (VPC) for the file system.
         * </p>
         * 
         * @param vpcId
         *        The ID of the primary virtual private cloud (VPC) for the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system is accessible from. For the Amazon FSx Windows and
         * ONTAP <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the preferred
         * file server and one for the standby file server. The preferred file server subnet identified in the
         * <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.
         * </p>
         * <p>
         * For FSx for Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
         * contains the file system's endpoint. For <code>MULTI_AZ_1</code> Windows and ONTAP file systems, the file
         * system endpoint is available in the <code>PreferredSubnetID</code>.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system is accessible from. For the Amazon FSx Windows
         *        and ONTAP <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the
         *        preferred file server and one for the standby file server. The preferred file server subnet identified
         *        in the <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.</p>
         *        <p>
         *        For FSx for Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
         *        contains the file system's endpoint. For <code>MULTI_AZ_1</code> Windows and ONTAP file systems, the
         *        file system endpoint is available in the <code>PreferredSubnetID</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system is accessible from. For the Amazon FSx Windows and
         * ONTAP <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the preferred
         * file server and one for the standby file server. The preferred file server subnet identified in the
         * <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.
         * </p>
         * <p>
         * For FSx for Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
         * contains the file system's endpoint. For <code>MULTI_AZ_1</code> Windows and ONTAP file systems, the file
         * system endpoint is available in the <code>PreferredSubnetID</code>.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system is accessible from. For the Amazon FSx Windows
         *        and ONTAP <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the
         *        preferred file server and one for the standby file server. The preferred file server subnet identified
         *        in the <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.</p>
         *        <p>
         *        For FSx for Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
         *        contains the file system's endpoint. For <code>MULTI_AZ_1</code> Windows and ONTAP file systems, the
         *        file system endpoint is available in the <code>PreferredSubnetID</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The IDs of the elastic network interfaces from which a specific file system is accessible. The elastic
         * network interface is automatically created in the same virtual private cloud (VPC) that the Amazon FSx file
         * system was created in. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network Interfaces</a> in
         * the <i>Amazon EC2 User Guide.</i>
         * </p>
         * <p>
         * For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an Amazon
         * FSx for Lustre file system, you can have more than one.
         * </p>
         * 
         * @param networkInterfaceIds
         *        The IDs of the elastic network interfaces from which a specific file system is accessible. The elastic
         *        network interface is automatically created in the same virtual private cloud (VPC) that the Amazon FSx
         *        file system was created in. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
         *        Interfaces</a> in the <i>Amazon EC2 User Guide.</i> </p>
         *        <p>
         *        For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an
         *        Amazon FSx for Lustre file system, you can have more than one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(Collection<String> networkInterfaceIds);

        /**
         * <p>
         * The IDs of the elastic network interfaces from which a specific file system is accessible. The elastic
         * network interface is automatically created in the same virtual private cloud (VPC) that the Amazon FSx file
         * system was created in. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network Interfaces</a> in
         * the <i>Amazon EC2 User Guide.</i>
         * </p>
         * <p>
         * For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an Amazon
         * FSx for Lustre file system, you can have more than one.
         * </p>
         * 
         * @param networkInterfaceIds
         *        The IDs of the elastic network interfaces from which a specific file system is accessible. The elastic
         *        network interface is automatically created in the same virtual private cloud (VPC) that the Amazon FSx
         *        file system was created in. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
         *        Interfaces</a> in the <i>Amazon EC2 User Guide.</i> </p>
         *        <p>
         *        For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an
         *        Amazon FSx for Lustre file system, you can have more than one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(String... networkInterfaceIds);

        /**
         * <p>
         * The Domain Name System (DNS) name for the file system.
         * </p>
         * 
         * @param dnsName
         *        The Domain Name System (DNS) name for the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The ID of the Key Management Service (KMS) key used to encrypt Amazon FSx file system data. Used as follows
         * with Amazon FSx file system types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon FSx for Lustre <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment types only.
         * </p>
         * <p>
         * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> types are encrypted using the Amazon FSx service KMS key
         * for your account.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon FSx for NetApp ONTAP
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon FSx for OpenZFS
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon FSx for Windows File Server
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        The ID of the Key Management Service (KMS) key used to encrypt Amazon FSx file system data. Used as
         *        follows with Amazon FSx file system types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon FSx for Lustre <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment types only.
         *        </p>
         *        <p>
         *        <code>SCRATCH_1</code> and <code>SCRATCH_2</code> types are encrypted using the Amazon FSx service KMS
         *        key for your account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon FSx for NetApp ONTAP
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon FSx for OpenZFS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon FSx for Windows File Server
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the file system resource.
         * </p>
         * 
         * @param resourceARN
         *        The Amazon Resource Name (ARN) of the file system resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * The tags to associate with the file system. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
         * resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tags to associate with the file system. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
         *        resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags to associate with the file system. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
         * resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tags to associate with the file system. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
         *        resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags to associate with the file system. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/tag-resources.html">Tagging your Amazon FSx
         * resources</a> in the <i>Amazon FSx for Lustre User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.fsx.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.fsx.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.fsx.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The configuration for this Amazon FSx for Windows File Server file system.
         * </p>
         * 
         * @param windowsConfiguration
         *        The configuration for this Amazon FSx for Windows File Server file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowsConfiguration(WindowsFileSystemConfiguration windowsConfiguration);

        /**
         * <p>
         * The configuration for this Amazon FSx for Windows File Server file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link WindowsFileSystemConfiguration.Builder}
         * avoiding the need to create one manually via {@link WindowsFileSystemConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WindowsFileSystemConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #windowsConfiguration(WindowsFileSystemConfiguration)}.
         * 
         * @param windowsConfiguration
         *        a consumer that will call methods on {@link WindowsFileSystemConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #windowsConfiguration(WindowsFileSystemConfiguration)
         */
        default Builder windowsConfiguration(Consumer<WindowsFileSystemConfiguration.Builder> windowsConfiguration) {
            return windowsConfiguration(WindowsFileSystemConfiguration.builder().applyMutation(windowsConfiguration).build());
        }

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * @param lustreConfiguration
         *        The new value for the LustreConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lustreConfiguration(LustreFileSystemConfiguration lustreConfiguration);

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link LustreFileSystemConfiguration.Builder}
         * avoiding the need to create one manually via {@link LustreFileSystemConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LustreFileSystemConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #lustreConfiguration(LustreFileSystemConfiguration)}.
         * 
         * @param lustreConfiguration
         *        a consumer that will call methods on {@link LustreFileSystemConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lustreConfiguration(LustreFileSystemConfiguration)
         */
        default Builder lustreConfiguration(Consumer<LustreFileSystemConfiguration.Builder> lustreConfiguration) {
            return lustreConfiguration(LustreFileSystemConfiguration.builder().applyMutation(lustreConfiguration).build());
        }

        /**
         * <p>
         * A list of administrative actions for the file system that are in process or waiting to be processed.
         * Administrative actions describe changes to the Amazon FSx system that you have initiated using the
         * <code>UpdateFileSystem</code> operation.
         * </p>
         * 
         * @param administrativeActions
         *        A list of administrative actions for the file system that are in process or waiting to be processed.
         *        Administrative actions describe changes to the Amazon FSx system that you have initiated using the
         *        <code>UpdateFileSystem</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administrativeActions(Collection<AdministrativeAction> administrativeActions);

        /**
         * <p>
         * A list of administrative actions for the file system that are in process or waiting to be processed.
         * Administrative actions describe changes to the Amazon FSx system that you have initiated using the
         * <code>UpdateFileSystem</code> operation.
         * </p>
         * 
         * @param administrativeActions
         *        A list of administrative actions for the file system that are in process or waiting to be processed.
         *        Administrative actions describe changes to the Amazon FSx system that you have initiated using the
         *        <code>UpdateFileSystem</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administrativeActions(AdministrativeAction... administrativeActions);

        /**
         * <p>
         * A list of administrative actions for the file system that are in process or waiting to be processed.
         * Administrative actions describe changes to the Amazon FSx system that you have initiated using the
         * <code>UpdateFileSystem</code> operation.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.AdministrativeAction.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.fsx.model.AdministrativeAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fsx.model.AdministrativeAction.Builder#build()} is called immediately
         * and its result is passed to {@link #administrativeActions(List<AdministrativeAction>)}.
         * 
         * @param administrativeActions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fsx.model.AdministrativeAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #administrativeActions(java.util.Collection<AdministrativeAction>)
         */
        Builder administrativeActions(Consumer<AdministrativeAction.Builder>... administrativeActions);

        /**
         * <p>
         * The configuration for this Amazon FSx for NetApp ONTAP file system.
         * </p>
         * 
         * @param ontapConfiguration
         *        The configuration for this Amazon FSx for NetApp ONTAP file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ontapConfiguration(OntapFileSystemConfiguration ontapConfiguration);

        /**
         * <p>
         * The configuration for this Amazon FSx for NetApp ONTAP file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link OntapFileSystemConfiguration.Builder}
         * avoiding the need to create one manually via {@link OntapFileSystemConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OntapFileSystemConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #ontapConfiguration(OntapFileSystemConfiguration)}.
         * 
         * @param ontapConfiguration
         *        a consumer that will call methods on {@link OntapFileSystemConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ontapConfiguration(OntapFileSystemConfiguration)
         */
        default Builder ontapConfiguration(Consumer<OntapFileSystemConfiguration.Builder> ontapConfiguration) {
            return ontapConfiguration(OntapFileSystemConfiguration.builder().applyMutation(ontapConfiguration).build());
        }

        /**
         * <p>
         * The Lustre version of the Amazon FSx for Lustre file system, which can be <code>2.10</code>,
         * <code>2.12</code>, or <code>2.15</code>.
         * </p>
         * 
         * @param fileSystemTypeVersion
         *        The Lustre version of the Amazon FSx for Lustre file system, which can be <code>2.10</code>,
         *        <code>2.12</code>, or <code>2.15</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemTypeVersion(String fileSystemTypeVersion);

        /**
         * <p>
         * The configuration for this Amazon FSx for OpenZFS file system.
         * </p>
         * 
         * @param openZFSConfiguration
         *        The configuration for this Amazon FSx for OpenZFS file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder openZFSConfiguration(OpenZFSFileSystemConfiguration openZFSConfiguration);

        /**
         * <p>
         * The configuration for this Amazon FSx for OpenZFS file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link OpenZFSFileSystemConfiguration.Builder}
         * avoiding the need to create one manually via {@link OpenZFSFileSystemConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OpenZFSFileSystemConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #openZFSConfiguration(OpenZFSFileSystemConfiguration)}.
         * 
         * @param openZFSConfiguration
         *        a consumer that will call methods on {@link OpenZFSFileSystemConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #openZFSConfiguration(OpenZFSFileSystemConfiguration)
         */
        default Builder openZFSConfiguration(Consumer<OpenZFSFileSystemConfiguration.Builder> openZFSConfiguration) {
            return openZFSConfiguration(OpenZFSFileSystemConfiguration.builder().applyMutation(openZFSConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private Instant creationTime;

        private String fileSystemId;

        private String fileSystemType;

        private String lifecycle;

        private FileSystemFailureDetails failureDetails;

        private Integer storageCapacity;

        private String storageType;

        private String vpcId;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> networkInterfaceIds = DefaultSdkAutoConstructList.getInstance();

        private String dnsName;

        private String kmsKeyId;

        private String resourceARN;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private WindowsFileSystemConfiguration windowsConfiguration;

        private LustreFileSystemConfiguration lustreConfiguration;

        private List<AdministrativeAction> administrativeActions = DefaultSdkAutoConstructList.getInstance();

        private OntapFileSystemConfiguration ontapConfiguration;

        private String fileSystemTypeVersion;

        private OpenZFSFileSystemConfiguration openZFSConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(FileSystem model) {
            ownerId(model.ownerId);
            creationTime(model.creationTime);
            fileSystemId(model.fileSystemId);
            fileSystemType(model.fileSystemType);
            lifecycle(model.lifecycle);
            failureDetails(model.failureDetails);
            storageCapacity(model.storageCapacity);
            storageType(model.storageType);
            vpcId(model.vpcId);
            subnetIds(model.subnetIds);
            networkInterfaceIds(model.networkInterfaceIds);
            dnsName(model.dnsName);
            kmsKeyId(model.kmsKeyId);
            resourceARN(model.resourceARN);
            tags(model.tags);
            windowsConfiguration(model.windowsConfiguration);
            lustreConfiguration(model.lustreConfiguration);
            administrativeActions(model.administrativeActions);
            ontapConfiguration(model.ontapConfiguration);
            fileSystemTypeVersion(model.fileSystemTypeVersion);
            openZFSConfiguration(model.openZFSConfiguration);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getFileSystemType() {
            return fileSystemType;
        }

        public final void setFileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
        }

        @Override
        public final Builder fileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
            return this;
        }

        @Override
        public final Builder fileSystemType(FileSystemType fileSystemType) {
            this.fileSystemType(fileSystemType == null ? null : fileSystemType.toString());
            return this;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(FileSystemLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final FileSystemFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        public final void setFailureDetails(FileSystemFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        @Override
        public final Builder failureDetails(FileSystemFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final Integer getStorageCapacity() {
            return storageCapacity;
        }

        public final void setStorageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
        }

        @Override
        public final Builder storageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
            return this;
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        @Override
        public final Builder storageType(StorageType storageType) {
            this.storageType(storageType == null ? null : storageType.toString());
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final Collection<String> getNetworkInterfaceIds() {
            if (networkInterfaceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaceIds;
        }

        public final void setNetworkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdsCopier.copy(networkInterfaceIds);
        }

        @Override
        public final Builder networkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdsCopier.copy(networkInterfaceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaceIds(String... networkInterfaceIds) {
            networkInterfaceIds(Arrays.asList(networkInterfaceIds));
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final WindowsFileSystemConfiguration.Builder getWindowsConfiguration() {
            return windowsConfiguration != null ? windowsConfiguration.toBuilder() : null;
        }

        public final void setWindowsConfiguration(WindowsFileSystemConfiguration.BuilderImpl windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration != null ? windowsConfiguration.build() : null;
        }

        @Override
        public final Builder windowsConfiguration(WindowsFileSystemConfiguration windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration;
            return this;
        }

        public final LustreFileSystemConfiguration.Builder getLustreConfiguration() {
            return lustreConfiguration != null ? lustreConfiguration.toBuilder() : null;
        }

        public final void setLustreConfiguration(LustreFileSystemConfiguration.BuilderImpl lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration != null ? lustreConfiguration.build() : null;
        }

        @Override
        public final Builder lustreConfiguration(LustreFileSystemConfiguration lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration;
            return this;
        }

        public final List<AdministrativeAction.Builder> getAdministrativeActions() {
            List<AdministrativeAction.Builder> result = AdministrativeActionsCopier.copyToBuilder(this.administrativeActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAdministrativeActions(Collection<AdministrativeAction.BuilderImpl> administrativeActions) {
            this.administrativeActions = AdministrativeActionsCopier.copyFromBuilder(administrativeActions);
        }

        @Override
        public final Builder administrativeActions(Collection<AdministrativeAction> administrativeActions) {
            this.administrativeActions = AdministrativeActionsCopier.copy(administrativeActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder administrativeActions(AdministrativeAction... administrativeActions) {
            administrativeActions(Arrays.asList(administrativeActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder administrativeActions(Consumer<AdministrativeAction.Builder>... administrativeActions) {
            administrativeActions(Stream.of(administrativeActions)
                    .map(c -> AdministrativeAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final OntapFileSystemConfiguration.Builder getOntapConfiguration() {
            return ontapConfiguration != null ? ontapConfiguration.toBuilder() : null;
        }

        public final void setOntapConfiguration(OntapFileSystemConfiguration.BuilderImpl ontapConfiguration) {
            this.ontapConfiguration = ontapConfiguration != null ? ontapConfiguration.build() : null;
        }

        @Override
        public final Builder ontapConfiguration(OntapFileSystemConfiguration ontapConfiguration) {
            this.ontapConfiguration = ontapConfiguration;
            return this;
        }

        public final String getFileSystemTypeVersion() {
            return fileSystemTypeVersion;
        }

        public final void setFileSystemTypeVersion(String fileSystemTypeVersion) {
            this.fileSystemTypeVersion = fileSystemTypeVersion;
        }

        @Override
        public final Builder fileSystemTypeVersion(String fileSystemTypeVersion) {
            this.fileSystemTypeVersion = fileSystemTypeVersion;
            return this;
        }

        public final OpenZFSFileSystemConfiguration.Builder getOpenZFSConfiguration() {
            return openZFSConfiguration != null ? openZFSConfiguration.toBuilder() : null;
        }

        public final void setOpenZFSConfiguration(OpenZFSFileSystemConfiguration.BuilderImpl openZFSConfiguration) {
            this.openZFSConfiguration = openZFSConfiguration != null ? openZFSConfiguration.build() : null;
        }

        @Override
        public final Builder openZFSConfiguration(OpenZFSFileSystemConfiguration openZFSConfiguration) {
            this.openZFSConfiguration = openZFSConfiguration;
            return this;
        }

        @Override
        public FileSystem build() {
            return new FileSystem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
