/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for a data repository association that links an Amazon File Cache resource to an NFS data
 * repository.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NFSDataRepositoryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<NFSDataRepositoryConfiguration.Builder, NFSDataRepositoryConfiguration> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(NFSDataRepositoryConfiguration::versionAsString)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<String>> DNS_IPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIps")
            .getter(getter(NFSDataRepositoryConfiguration::dnsIps))
            .setter(setter(Builder::dnsIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AutoExportPolicy> AUTO_EXPORT_POLICY_FIELD = SdkField
            .<AutoExportPolicy> builder(MarshallingType.SDK_POJO).memberName("AutoExportPolicy")
            .getter(getter(NFSDataRepositoryConfiguration::autoExportPolicy)).setter(setter(Builder::autoExportPolicy))
            .constructor(AutoExportPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoExportPolicy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD, DNS_IPS_FIELD,
            AUTO_EXPORT_POLICY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String version;

    private final List<String> dnsIps;

    private final AutoExportPolicy autoExportPolicy;

    private NFSDataRepositoryConfiguration(BuilderImpl builder) {
        this.version = builder.version;
        this.dnsIps = builder.dnsIps;
        this.autoExportPolicy = builder.autoExportPolicy;
    }

    /**
     * <p>
     * The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only supported
     * value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3 protocol.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #version} will
     * return {@link NfsVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #versionAsString}.
     * </p>
     * 
     * @return The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only
     *         supported value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3
     *         protocol.
     * @see NfsVersion
     */
    public final NfsVersion version() {
        return NfsVersion.fromValue(version);
    }

    /**
     * <p>
     * The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only supported
     * value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3 protocol.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #version} will
     * return {@link NfsVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #versionAsString}.
     * </p>
     * 
     * @return The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only
     *         supported value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3
     *         protocol.
     * @see NfsVersion
     */
    public final String versionAsString() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIps property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIps() {
        return dnsIps != null && !(dnsIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of up to 2 IP addresses of DNS servers used to resolve the NFS file system domain name. The provided IP
     * addresses can either be the IP addresses of a DNS forwarder or resolver that the customer manages and runs inside
     * the customer VPC, or the IP addresses of the on-premises DNS servers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIps} method.
     * </p>
     * 
     * @return A list of up to 2 IP addresses of DNS servers used to resolve the NFS file system domain name. The
     *         provided IP addresses can either be the IP addresses of a DNS forwarder or resolver that the customer
     *         manages and runs inside the customer VPC, or the IP addresses of the on-premises DNS servers.
     */
    public final List<String> dnsIps() {
        return dnsIps;
    }

    /**
     * <p>
     * This parameter is not supported for Amazon File Cache.
     * </p>
     * 
     * @return This parameter is not supported for Amazon File Cache.
     */
    public final AutoExportPolicy autoExportPolicy() {
        return autoExportPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(versionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIps() ? dnsIps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(autoExportPolicy());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NFSDataRepositoryConfiguration)) {
            return false;
        }
        NFSDataRepositoryConfiguration other = (NFSDataRepositoryConfiguration) obj;
        return Objects.equals(versionAsString(), other.versionAsString()) && hasDnsIps() == other.hasDnsIps()
                && Objects.equals(dnsIps(), other.dnsIps()) && Objects.equals(autoExportPolicy(), other.autoExportPolicy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NFSDataRepositoryConfiguration").add("Version", versionAsString())
                .add("DnsIps", hasDnsIps() ? dnsIps() : null).add("AutoExportPolicy", autoExportPolicy()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Version":
            return Optional.ofNullable(clazz.cast(versionAsString()));
        case "DnsIps":
            return Optional.ofNullable(clazz.cast(dnsIps()));
        case "AutoExportPolicy":
            return Optional.ofNullable(clazz.cast(autoExportPolicy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Version", VERSION_FIELD);
        map.put("DnsIps", DNS_IPS_FIELD);
        map.put("AutoExportPolicy", AUTO_EXPORT_POLICY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NFSDataRepositoryConfiguration, T> g) {
        return obj -> g.apply((NFSDataRepositoryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NFSDataRepositoryConfiguration> {
        /**
         * <p>
         * The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only
         * supported value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3
         * protocol.
         * </p>
         * 
         * @param version
         *        The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only
         *        supported value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3
         *        protocol.
         * @see NfsVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NfsVersion
         */
        Builder version(String version);

        /**
         * <p>
         * The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only
         * supported value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3
         * protocol.
         * </p>
         * 
         * @param version
         *        The version of the NFS (Network File System) protocol of the NFS data repository. Currently, the only
         *        supported value is <code>NFS3</code>, which indicates that the data repository must support the NFSv3
         *        protocol.
         * @see NfsVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NfsVersion
         */
        Builder version(NfsVersion version);

        /**
         * <p>
         * A list of up to 2 IP addresses of DNS servers used to resolve the NFS file system domain name. The provided
         * IP addresses can either be the IP addresses of a DNS forwarder or resolver that the customer manages and runs
         * inside the customer VPC, or the IP addresses of the on-premises DNS servers.
         * </p>
         * 
         * @param dnsIps
         *        A list of up to 2 IP addresses of DNS servers used to resolve the NFS file system domain name. The
         *        provided IP addresses can either be the IP addresses of a DNS forwarder or resolver that the customer
         *        manages and runs inside the customer VPC, or the IP addresses of the on-premises DNS servers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(Collection<String> dnsIps);

        /**
         * <p>
         * A list of up to 2 IP addresses of DNS servers used to resolve the NFS file system domain name. The provided
         * IP addresses can either be the IP addresses of a DNS forwarder or resolver that the customer manages and runs
         * inside the customer VPC, or the IP addresses of the on-premises DNS servers.
         * </p>
         * 
         * @param dnsIps
         *        A list of up to 2 IP addresses of DNS servers used to resolve the NFS file system domain name. The
         *        provided IP addresses can either be the IP addresses of a DNS forwarder or resolver that the customer
         *        manages and runs inside the customer VPC, or the IP addresses of the on-premises DNS servers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(String... dnsIps);

        /**
         * <p>
         * This parameter is not supported for Amazon File Cache.
         * </p>
         * 
         * @param autoExportPolicy
         *        This parameter is not supported for Amazon File Cache.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoExportPolicy(AutoExportPolicy autoExportPolicy);

        /**
         * <p>
         * This parameter is not supported for Amazon File Cache.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoExportPolicy.Builder} avoiding the
         * need to create one manually via {@link AutoExportPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoExportPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #autoExportPolicy(AutoExportPolicy)}.
         * 
         * @param autoExportPolicy
         *        a consumer that will call methods on {@link AutoExportPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoExportPolicy(AutoExportPolicy)
         */
        default Builder autoExportPolicy(Consumer<AutoExportPolicy.Builder> autoExportPolicy) {
            return autoExportPolicy(AutoExportPolicy.builder().applyMutation(autoExportPolicy).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private List<String> dnsIps = DefaultSdkAutoConstructList.getInstance();

        private AutoExportPolicy autoExportPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(NFSDataRepositoryConfiguration model) {
            version(model.version);
            dnsIps(model.dnsIps);
            autoExportPolicy(model.autoExportPolicy);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        @Override
        public final Builder version(NfsVersion version) {
            this.version(version == null ? null : version.toString());
            return this;
        }

        public final Collection<String> getDnsIps() {
            if (dnsIps instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIps;
        }

        public final void setDnsIps(Collection<String> dnsIps) {
            this.dnsIps = RepositoryDnsIpsCopier.copy(dnsIps);
        }

        @Override
        public final Builder dnsIps(Collection<String> dnsIps) {
            this.dnsIps = RepositoryDnsIpsCopier.copy(dnsIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIps(String... dnsIps) {
            dnsIps(Arrays.asList(dnsIps));
            return this;
        }

        public final AutoExportPolicy.Builder getAutoExportPolicy() {
            return autoExportPolicy != null ? autoExportPolicy.toBuilder() : null;
        }

        public final void setAutoExportPolicy(AutoExportPolicy.BuilderImpl autoExportPolicy) {
            this.autoExportPolicy = autoExportPolicy != null ? autoExportPolicy.build() : null;
        }

        @Override
        public final Builder autoExportPolicy(AutoExportPolicy autoExportPolicy) {
            this.autoExportPolicy = autoExportPolicy;
            return this;
        }

        @Override
        public NFSDataRepositoryConfiguration build() {
            return new NFSDataRepositoryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
