/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration for the FSx for NetApp ONTAP file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OntapFileSystemConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<OntapFileSystemConfiguration.Builder, OntapFileSystemConfiguration> {
    private static final SdkField<Integer> AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AutomaticBackupRetentionDays")
            .getter(getter(OntapFileSystemConfiguration::automaticBackupRetentionDays))
            .setter(setter(Builder::automaticBackupRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticBackupRetentionDays")
                    .build()).build();

    private static final SdkField<String> DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DailyAutomaticBackupStartTime")
            .getter(getter(OntapFileSystemConfiguration::dailyAutomaticBackupStartTime))
            .setter(setter(Builder::dailyAutomaticBackupStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DailyAutomaticBackupStartTime")
                    .build()).build();

    private static final SdkField<String> DEPLOYMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentType").getter(getter(OntapFileSystemConfiguration::deploymentTypeAsString))
            .setter(setter(Builder::deploymentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentType").build()).build();

    private static final SdkField<String> ENDPOINT_IP_ADDRESS_RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointIpAddressRange").getter(getter(OntapFileSystemConfiguration::endpointIpAddressRange))
            .setter(setter(Builder::endpointIpAddressRange))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointIpAddressRange").build())
            .build();

    private static final SdkField<FileSystemEndpoints> ENDPOINTS_FIELD = SdkField
            .<FileSystemEndpoints> builder(MarshallingType.SDK_POJO).memberName("Endpoints")
            .getter(getter(OntapFileSystemConfiguration::endpoints)).setter(setter(Builder::endpoints))
            .constructor(FileSystemEndpoints::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints").build()).build();

    private static final SdkField<DiskIopsConfiguration> DISK_IOPS_CONFIGURATION_FIELD = SdkField
            .<DiskIopsConfiguration> builder(MarshallingType.SDK_POJO).memberName("DiskIopsConfiguration")
            .getter(getter(OntapFileSystemConfiguration::diskIopsConfiguration)).setter(setter(Builder::diskIopsConfiguration))
            .constructor(DiskIopsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DiskIopsConfiguration").build())
            .build();

    private static final SdkField<String> PREFERRED_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredSubnetId").getter(getter(OntapFileSystemConfiguration::preferredSubnetId))
            .setter(setter(Builder::preferredSubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredSubnetId").build()).build();

    private static final SdkField<List<String>> ROUTE_TABLE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RouteTableIds")
            .getter(getter(OntapFileSystemConfiguration::routeTableIds))
            .setter(setter(Builder::routeTableIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteTableIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> THROUGHPUT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ThroughputCapacity").getter(getter(OntapFileSystemConfiguration::throughputCapacity))
            .setter(setter(Builder::throughputCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThroughputCapacity").build())
            .build();

    private static final SdkField<String> WEEKLY_MAINTENANCE_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceStartTime")
            .getter(getter(OntapFileSystemConfiguration::weeklyMaintenanceStartTime))
            .setter(setter(Builder::weeklyMaintenanceStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceStartTime").build())
            .build();

    private static final SdkField<String> FSX_ADMIN_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FsxAdminPassword").getter(getter(OntapFileSystemConfiguration::fsxAdminPassword))
            .setter(setter(Builder::fsxAdminPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FsxAdminPassword").build()).build();

    private static final SdkField<Integer> HA_PAIRS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("HAPairs").getter(getter(OntapFileSystemConfiguration::haPairs)).setter(setter(Builder::haPairs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HAPairs").build()).build();

    private static final SdkField<Integer> THROUGHPUT_CAPACITY_PER_HA_PAIR_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ThroughputCapacityPerHAPair")
            .getter(getter(OntapFileSystemConfiguration::throughputCapacityPerHAPair))
            .setter(setter(Builder::throughputCapacityPerHAPair))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThroughputCapacityPerHAPair")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD, DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD, DEPLOYMENT_TYPE_FIELD,
            ENDPOINT_IP_ADDRESS_RANGE_FIELD, ENDPOINTS_FIELD, DISK_IOPS_CONFIGURATION_FIELD, PREFERRED_SUBNET_ID_FIELD,
            ROUTE_TABLE_IDS_FIELD, THROUGHPUT_CAPACITY_FIELD, WEEKLY_MAINTENANCE_START_TIME_FIELD, FSX_ADMIN_PASSWORD_FIELD,
            HA_PAIRS_FIELD, THROUGHPUT_CAPACITY_PER_HA_PAIR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer automaticBackupRetentionDays;

    private final String dailyAutomaticBackupStartTime;

    private final String deploymentType;

    private final String endpointIpAddressRange;

    private final FileSystemEndpoints endpoints;

    private final DiskIopsConfiguration diskIopsConfiguration;

    private final String preferredSubnetId;

    private final List<String> routeTableIds;

    private final Integer throughputCapacity;

    private final String weeklyMaintenanceStartTime;

    private final String fsxAdminPassword;

    private final Integer haPairs;

    private final Integer throughputCapacityPerHAPair;

    private OntapFileSystemConfiguration(BuilderImpl builder) {
        this.automaticBackupRetentionDays = builder.automaticBackupRetentionDays;
        this.dailyAutomaticBackupStartTime = builder.dailyAutomaticBackupStartTime;
        this.deploymentType = builder.deploymentType;
        this.endpointIpAddressRange = builder.endpointIpAddressRange;
        this.endpoints = builder.endpoints;
        this.diskIopsConfiguration = builder.diskIopsConfiguration;
        this.preferredSubnetId = builder.preferredSubnetId;
        this.routeTableIds = builder.routeTableIds;
        this.throughputCapacity = builder.throughputCapacity;
        this.weeklyMaintenanceStartTime = builder.weeklyMaintenanceStartTime;
        this.fsxAdminPassword = builder.fsxAdminPassword;
        this.haPairs = builder.haPairs;
        this.throughputCapacityPerHAPair = builder.throughputCapacityPerHAPair;
    }

    /**
     * Returns the value of the AutomaticBackupRetentionDays property for this object.
     * 
     * @return The value of the AutomaticBackupRetentionDays property for this object.
     */
    public final Integer automaticBackupRetentionDays() {
        return automaticBackupRetentionDays;
    }

    /**
     * Returns the value of the DailyAutomaticBackupStartTime property for this object.
     * 
     * @return The value of the DailyAutomaticBackupStartTime property for this object.
     */
    public final String dailyAutomaticBackupStartTime() {
        return dailyAutomaticBackupStartTime;
    }

    /**
     * <p>
     * Specifies the FSx for ONTAP file system deployment type in use in the file system.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     * temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     * temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a first-generation FSx for
     * ONTAP file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for Single-AZ
     * redundancy. This is a second-generation FSx for ONTAP file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing Multi-AZ or
     * Single-AZ file system deployment</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link OntapDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the FSx for ONTAP file system deployment type in use in the file system. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     *         temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     *         temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a first-generation
     *         FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for
     *         Single-AZ redundancy. This is a second-generation FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     *         href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing Multi-AZ
     *         or Single-AZ file system deployment</a>.
     * @see OntapDeploymentType
     */
    public final OntapDeploymentType deploymentType() {
        return OntapDeploymentType.fromValue(deploymentType);
    }

    /**
     * <p>
     * Specifies the FSx for ONTAP file system deployment type in use in the file system.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     * temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     * temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a first-generation FSx for
     * ONTAP file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for Single-AZ
     * redundancy. This is a second-generation FSx for ONTAP file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing Multi-AZ or
     * Single-AZ file system deployment</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link OntapDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the FSx for ONTAP file system deployment type in use in the file system. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     *         temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
     *         temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a first-generation
     *         FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for
     *         Single-AZ redundancy. This is a second-generation FSx for ONTAP file system.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     *         href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing Multi-AZ
     *         or Single-AZ file system deployment</a>.
     * @see OntapDeploymentType
     */
    public final String deploymentTypeAsString() {
        return deploymentType;
    }

    /**
     * <p>
     * (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will be created.
     * By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you from the 198.19.* range.
     * By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP addresses from the VPC’s primary CIDR
     * range to use as the endpoint IP address range for the file system. You can have overlapping endpoint IP addresses
     * for file systems deployed in the same VPC/route tables.
     * </p>
     * 
     * @return (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will be
     *         created. By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you from the
     *         198.19.* range. By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP addresses from
     *         the VPC’s primary CIDR range to use as the endpoint IP address range for the file system. You can have
     *         overlapping endpoint IP addresses for file systems deployed in the same VPC/route tables.
     */
    public final String endpointIpAddressRange() {
        return endpointIpAddressRange;
    }

    /**
     * <p>
     * The <code>Management</code> and <code>Intercluster</code> endpoints that are used to access data or to manage the
     * file system using the NetApp ONTAP CLI, REST API, or NetApp SnapMirror.
     * </p>
     * 
     * @return The <code>Management</code> and <code>Intercluster</code> endpoints that are used to access data or to
     *         manage the file system using the NetApp ONTAP CLI, REST API, or NetApp SnapMirror.
     */
    public final FileSystemEndpoints endpoints() {
        return endpoints;
    }

    /**
     * <p>
     * The SSD IOPS configuration for the ONTAP file system, specifying the number of provisioned IOPS and the provision
     * mode.
     * </p>
     * 
     * @return The SSD IOPS configuration for the ONTAP file system, specifying the number of provisioned IOPS and the
     *         provision mode.
     */
    public final DiskIopsConfiguration diskIopsConfiguration() {
        return diskIopsConfiguration;
    }

    /**
     * Returns the value of the PreferredSubnetId property for this object.
     * 
     * @return The value of the PreferredSubnetId property for this object.
     */
    public final String preferredSubnetId() {
        return preferredSubnetId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RouteTableIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRouteTableIds() {
        return routeTableIds != null && !(routeTableIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Multi-AZ only) The VPC route tables in which your file system's endpoints are created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRouteTableIds} method.
     * </p>
     * 
     * @return (Multi-AZ only) The VPC route tables in which your file system's endpoints are created.
     */
    public final List<String> routeTableIds() {
        return routeTableIds;
    }

    /**
     * Returns the value of the ThroughputCapacity property for this object.
     * 
     * @return The value of the ThroughputCapacity property for this object.
     */
    public final Integer throughputCapacity() {
        return throughputCapacity;
    }

    /**
     * Returns the value of the WeeklyMaintenanceStartTime property for this object.
     * 
     * @return The value of the WeeklyMaintenanceStartTime property for this object.
     */
    public final String weeklyMaintenanceStartTime() {
        return weeklyMaintenanceStartTime;
    }

    /**
     * <p>
     * You can use the <code>fsxadmin</code> user account to access the NetApp ONTAP CLI and REST API. The password
     * value is always redacted in the response.
     * </p>
     * 
     * @return You can use the <code>fsxadmin</code> user account to access the NetApp ONTAP CLI and REST API. The
     *         password value is always redacted in the response.
     */
    public final String fsxAdminPassword() {
        return fsxAdminPassword;
    }

    /**
     * <p>
     * Specifies how many high-availability (HA) file server pairs the file system will have. The default value is 1.
     * The value of this property affects the values of <code>StorageCapacity</code>, <code>Iops</code>, and
     * <code>ThroughputCapacity</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/HA-pairs.html">High-availability (HA) pairs</a> in the
     * FSx for ONTAP user guide.
     * </p>
     * <p>
     * Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The value of <code>HAPairs</code> is less than 1 or greater than 12.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>HAPairs</code> is greater than 1 and the value of <code>DeploymentType</code> is
     * <code>SINGLE_AZ_1</code>, <code>MULTI_AZ_1</code>, or <code>MULTI_AZ_2</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies how many high-availability (HA) file server pairs the file system will have. The default value
     *         is 1. The value of this property affects the values of <code>StorageCapacity</code>, <code>Iops</code>,
     *         and <code>ThroughputCapacity</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/HA-pairs.html">High-availability (HA) pairs</a>
     *         in the FSx for ONTAP user guide.</p>
     *         <p>
     *         Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The value of <code>HAPairs</code> is less than 1 or greater than 12.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>HAPairs</code> is greater than 1 and the value of <code>DeploymentType</code> is
     *         <code>SINGLE_AZ_1</code>, <code>MULTI_AZ_1</code>, or <code>MULTI_AZ_2</code>.
     *         </p>
     *         </li>
     */
    public final Integer haPairs() {
        return haPairs;
    }

    /**
     * <p>
     * Use to choose the throughput capacity per HA pair. When the value of <code>HAPairs</code> is equal to 1, the
     * value of <code>ThroughputCapacityPerHAPair</code> is the total throughput for the file system.
     * </p>
     * <p>
     * This field and <code>ThroughputCapacity</code> cannot be defined in the same API call, but one is required.
     * </p>
     * <p>
     * This field and <code>ThroughputCapacity</code> are the same for file systems with one HA pair.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>SINGLE_AZ_1</code> and <code>MULTI_AZ_1</code> file systems, valid values are 128, 256, 512, 1024,
     * 2048, or 4096 MBps.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>SINGLE_AZ_2</code>, valid values are 1536, 3072, or 6144 MBps.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>MULTI_AZ_2</code>, valid values are 384, 768, 1536, 3072, or 6144 MBps.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The value of <code>ThroughputCapacity</code> and <code>ThroughputCapacityPerHAPair</code> are not the same value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of deployment type is <code>SINGLE_AZ_2</code> and <code>ThroughputCapacity</code> /
     * <code>ThroughputCapacityPerHAPair</code> is not a valid HA pair (a value between 1 and 12).
     * </p>
     * </li>
     * <li>
     * <p>
     * The value of <code>ThroughputCapacityPerHAPair</code> is not a valid value.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Use to choose the throughput capacity per HA pair. When the value of <code>HAPairs</code> is equal to 1,
     *         the value of <code>ThroughputCapacityPerHAPair</code> is the total throughput for the file system.</p>
     *         <p>
     *         This field and <code>ThroughputCapacity</code> cannot be defined in the same API call, but one is
     *         required.
     *         </p>
     *         <p>
     *         This field and <code>ThroughputCapacity</code> are the same for file systems with one HA pair.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>SINGLE_AZ_1</code> and <code>MULTI_AZ_1</code> file systems, valid values are 128, 256, 512,
     *         1024, 2048, or 4096 MBps.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>SINGLE_AZ_2</code>, valid values are 1536, 3072, or 6144 MBps.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>MULTI_AZ_2</code>, valid values are 384, 768, 1536, 3072, or 6144 MBps.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The value of <code>ThroughputCapacity</code> and <code>ThroughputCapacityPerHAPair</code> are not the
     *         same value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of deployment type is <code>SINGLE_AZ_2</code> and <code>ThroughputCapacity</code> /
     *         <code>ThroughputCapacityPerHAPair</code> is not a valid HA pair (a value between 1 and 12).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The value of <code>ThroughputCapacityPerHAPair</code> is not a valid value.
     *         </p>
     *         </li>
     */
    public final Integer throughputCapacityPerHAPair() {
        return throughputCapacityPerHAPair;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(automaticBackupRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(dailyAutomaticBackupStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(endpointIpAddressRange());
        hashCode = 31 * hashCode + Objects.hashCode(endpoints());
        hashCode = 31 * hashCode + Objects.hashCode(diskIopsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(preferredSubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRouteTableIds() ? routeTableIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(throughputCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(fsxAdminPassword());
        hashCode = 31 * hashCode + Objects.hashCode(haPairs());
        hashCode = 31 * hashCode + Objects.hashCode(throughputCapacityPerHAPair());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OntapFileSystemConfiguration)) {
            return false;
        }
        OntapFileSystemConfiguration other = (OntapFileSystemConfiguration) obj;
        return Objects.equals(automaticBackupRetentionDays(), other.automaticBackupRetentionDays())
                && Objects.equals(dailyAutomaticBackupStartTime(), other.dailyAutomaticBackupStartTime())
                && Objects.equals(deploymentTypeAsString(), other.deploymentTypeAsString())
                && Objects.equals(endpointIpAddressRange(), other.endpointIpAddressRange())
                && Objects.equals(endpoints(), other.endpoints())
                && Objects.equals(diskIopsConfiguration(), other.diskIopsConfiguration())
                && Objects.equals(preferredSubnetId(), other.preferredSubnetId())
                && hasRouteTableIds() == other.hasRouteTableIds() && Objects.equals(routeTableIds(), other.routeTableIds())
                && Objects.equals(throughputCapacity(), other.throughputCapacity())
                && Objects.equals(weeklyMaintenanceStartTime(), other.weeklyMaintenanceStartTime())
                && Objects.equals(fsxAdminPassword(), other.fsxAdminPassword()) && Objects.equals(haPairs(), other.haPairs())
                && Objects.equals(throughputCapacityPerHAPair(), other.throughputCapacityPerHAPair());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OntapFileSystemConfiguration")
                .add("AutomaticBackupRetentionDays", automaticBackupRetentionDays())
                .add("DailyAutomaticBackupStartTime", dailyAutomaticBackupStartTime())
                .add("DeploymentType", deploymentTypeAsString()).add("EndpointIpAddressRange", endpointIpAddressRange())
                .add("Endpoints", endpoints()).add("DiskIopsConfiguration", diskIopsConfiguration())
                .add("PreferredSubnetId", preferredSubnetId()).add("RouteTableIds", hasRouteTableIds() ? routeTableIds() : null)
                .add("ThroughputCapacity", throughputCapacity()).add("WeeklyMaintenanceStartTime", weeklyMaintenanceStartTime())
                .add("FsxAdminPassword", fsxAdminPassword() == null ? null : "*** Sensitive Data Redacted ***")
                .add("HAPairs", haPairs()).add("ThroughputCapacityPerHAPair", throughputCapacityPerHAPair()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutomaticBackupRetentionDays":
            return Optional.ofNullable(clazz.cast(automaticBackupRetentionDays()));
        case "DailyAutomaticBackupStartTime":
            return Optional.ofNullable(clazz.cast(dailyAutomaticBackupStartTime()));
        case "DeploymentType":
            return Optional.ofNullable(clazz.cast(deploymentTypeAsString()));
        case "EndpointIpAddressRange":
            return Optional.ofNullable(clazz.cast(endpointIpAddressRange()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "DiskIopsConfiguration":
            return Optional.ofNullable(clazz.cast(diskIopsConfiguration()));
        case "PreferredSubnetId":
            return Optional.ofNullable(clazz.cast(preferredSubnetId()));
        case "RouteTableIds":
            return Optional.ofNullable(clazz.cast(routeTableIds()));
        case "ThroughputCapacity":
            return Optional.ofNullable(clazz.cast(throughputCapacity()));
        case "WeeklyMaintenanceStartTime":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceStartTime()));
        case "FsxAdminPassword":
            return Optional.ofNullable(clazz.cast(fsxAdminPassword()));
        case "HAPairs":
            return Optional.ofNullable(clazz.cast(haPairs()));
        case "ThroughputCapacityPerHAPair":
            return Optional.ofNullable(clazz.cast(throughputCapacityPerHAPair()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AutomaticBackupRetentionDays", AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD);
        map.put("DailyAutomaticBackupStartTime", DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD);
        map.put("DeploymentType", DEPLOYMENT_TYPE_FIELD);
        map.put("EndpointIpAddressRange", ENDPOINT_IP_ADDRESS_RANGE_FIELD);
        map.put("Endpoints", ENDPOINTS_FIELD);
        map.put("DiskIopsConfiguration", DISK_IOPS_CONFIGURATION_FIELD);
        map.put("PreferredSubnetId", PREFERRED_SUBNET_ID_FIELD);
        map.put("RouteTableIds", ROUTE_TABLE_IDS_FIELD);
        map.put("ThroughputCapacity", THROUGHPUT_CAPACITY_FIELD);
        map.put("WeeklyMaintenanceStartTime", WEEKLY_MAINTENANCE_START_TIME_FIELD);
        map.put("FsxAdminPassword", FSX_ADMIN_PASSWORD_FIELD);
        map.put("HAPairs", HA_PAIRS_FIELD);
        map.put("ThroughputCapacityPerHAPair", THROUGHPUT_CAPACITY_PER_HA_PAIR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OntapFileSystemConfiguration, T> g) {
        return obj -> g.apply((OntapFileSystemConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OntapFileSystemConfiguration> {
        /**
         * Sets the value of the AutomaticBackupRetentionDays property for this object.
         *
         * @param automaticBackupRetentionDays
         *        The new value for the AutomaticBackupRetentionDays property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays);

        /**
         * Sets the value of the DailyAutomaticBackupStartTime property for this object.
         *
         * @param dailyAutomaticBackupStartTime
         *        The new value for the DailyAutomaticBackupStartTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime);

        /**
         * <p>
         * Specifies the FSx for ONTAP file system deployment type in use in the file system.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
         * temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
         * temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a first-generation FSx
         * for ONTAP file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for Single-AZ
         * redundancy. This is a second-generation FSx for ONTAP file system.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing Multi-AZ or
         * Single-AZ file system deployment</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the FSx for ONTAP file system deployment type in use in the file system. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to
         *        tolerate temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP
         *        file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to
         *        tolerate temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a
         *        first-generation FSx for ONTAP file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for
         *        Single-AZ redundancy. This is a second-generation FSx for ONTAP file system.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing
         *        Multi-AZ or Single-AZ file system deployment</a>.
         * @see OntapDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OntapDeploymentType
         */
        Builder deploymentType(String deploymentType);

        /**
         * <p>
         * Specifies the FSx for ONTAP file system deployment type in use in the file system.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
         * temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to tolerate
         * temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a first-generation FSx
         * for ONTAP file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for Single-AZ
         * redundancy. This is a second-generation FSx for ONTAP file system.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing Multi-AZ or
         * Single-AZ file system deployment</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the FSx for ONTAP file system deployment type in use in the file system. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - A high availability file system configured for Multi-AZ redundancy to
         *        tolerate temporary Availability Zone (AZ) unavailability. This is a first-generation FSx for ONTAP
         *        file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_2</code> - A high availability file system configured for Multi-AZ redundancy to
         *        tolerate temporary AZ unavailability. This is a second-generation FSx for ONTAP file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy. This is a
         *        first-generation FSx for ONTAP file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_2</code> - A file system configured with multiple high-availability (HA) pairs for
         *        Single-AZ redundancy. This is a second-generation FSx for ONTAP file system.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">Choosing
         *        Multi-AZ or Single-AZ file system deployment</a>.
         * @see OntapDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OntapDeploymentType
         */
        Builder deploymentType(OntapDeploymentType deploymentType);

        /**
         * <p>
         * (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will be
         * created. By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you from the
         * 198.19.* range. By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP addresses from the
         * VPC’s primary CIDR range to use as the endpoint IP address range for the file system. You can have
         * overlapping endpoint IP addresses for file systems deployed in the same VPC/route tables.
         * </p>
         * 
         * @param endpointIpAddressRange
         *        (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will
         *        be created. By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you
         *        from the 198.19.* range. By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP
         *        addresses from the VPC’s primary CIDR range to use as the endpoint IP address range for the file
         *        system. You can have overlapping endpoint IP addresses for file systems deployed in the same VPC/route
         *        tables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIpAddressRange(String endpointIpAddressRange);

        /**
         * <p>
         * The <code>Management</code> and <code>Intercluster</code> endpoints that are used to access data or to manage
         * the file system using the NetApp ONTAP CLI, REST API, or NetApp SnapMirror.
         * </p>
         * 
         * @param endpoints
         *        The <code>Management</code> and <code>Intercluster</code> endpoints that are used to access data or to
         *        manage the file system using the NetApp ONTAP CLI, REST API, or NetApp SnapMirror.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(FileSystemEndpoints endpoints);

        /**
         * <p>
         * The <code>Management</code> and <code>Intercluster</code> endpoints that are used to access data or to manage
         * the file system using the NetApp ONTAP CLI, REST API, or NetApp SnapMirror.
         * </p>
         * This is a convenience method that creates an instance of the {@link FileSystemEndpoints.Builder} avoiding the
         * need to create one manually via {@link FileSystemEndpoints#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FileSystemEndpoints.Builder#build()} is called immediately and
         * its result is passed to {@link #endpoints(FileSystemEndpoints)}.
         * 
         * @param endpoints
         *        a consumer that will call methods on {@link FileSystemEndpoints.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpoints(FileSystemEndpoints)
         */
        default Builder endpoints(Consumer<FileSystemEndpoints.Builder> endpoints) {
            return endpoints(FileSystemEndpoints.builder().applyMutation(endpoints).build());
        }

        /**
         * <p>
         * The SSD IOPS configuration for the ONTAP file system, specifying the number of provisioned IOPS and the
         * provision mode.
         * </p>
         * 
         * @param diskIopsConfiguration
         *        The SSD IOPS configuration for the ONTAP file system, specifying the number of provisioned IOPS and
         *        the provision mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskIopsConfiguration(DiskIopsConfiguration diskIopsConfiguration);

        /**
         * <p>
         * The SSD IOPS configuration for the ONTAP file system, specifying the number of provisioned IOPS and the
         * provision mode.
         * </p>
         * This is a convenience method that creates an instance of the {@link DiskIopsConfiguration.Builder} avoiding
         * the need to create one manually via {@link DiskIopsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DiskIopsConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #diskIopsConfiguration(DiskIopsConfiguration)}.
         * 
         * @param diskIopsConfiguration
         *        a consumer that will call methods on {@link DiskIopsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #diskIopsConfiguration(DiskIopsConfiguration)
         */
        default Builder diskIopsConfiguration(Consumer<DiskIopsConfiguration.Builder> diskIopsConfiguration) {
            return diskIopsConfiguration(DiskIopsConfiguration.builder().applyMutation(diskIopsConfiguration).build());
        }

        /**
         * Sets the value of the PreferredSubnetId property for this object.
         *
         * @param preferredSubnetId
         *        The new value for the PreferredSubnetId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredSubnetId(String preferredSubnetId);

        /**
         * <p>
         * (Multi-AZ only) The VPC route tables in which your file system's endpoints are created.
         * </p>
         * 
         * @param routeTableIds
         *        (Multi-AZ only) The VPC route tables in which your file system's endpoints are created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableIds(Collection<String> routeTableIds);

        /**
         * <p>
         * (Multi-AZ only) The VPC route tables in which your file system's endpoints are created.
         * </p>
         * 
         * @param routeTableIds
         *        (Multi-AZ only) The VPC route tables in which your file system's endpoints are created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableIds(String... routeTableIds);

        /**
         * Sets the value of the ThroughputCapacity property for this object.
         *
         * @param throughputCapacity
         *        The new value for the ThroughputCapacity property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughputCapacity(Integer throughputCapacity);

        /**
         * Sets the value of the WeeklyMaintenanceStartTime property for this object.
         *
         * @param weeklyMaintenanceStartTime
         *        The new value for the WeeklyMaintenanceStartTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime);

        /**
         * <p>
         * You can use the <code>fsxadmin</code> user account to access the NetApp ONTAP CLI and REST API. The password
         * value is always redacted in the response.
         * </p>
         * 
         * @param fsxAdminPassword
         *        You can use the <code>fsxadmin</code> user account to access the NetApp ONTAP CLI and REST API. The
         *        password value is always redacted in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fsxAdminPassword(String fsxAdminPassword);

        /**
         * <p>
         * Specifies how many high-availability (HA) file server pairs the file system will have. The default value is
         * 1. The value of this property affects the values of <code>StorageCapacity</code>, <code>Iops</code>, and
         * <code>ThroughputCapacity</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/HA-pairs.html">High-availability (HA) pairs</a> in
         * the FSx for ONTAP user guide.
         * </p>
         * <p>
         * Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The value of <code>HAPairs</code> is less than 1 or greater than 12.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>HAPairs</code> is greater than 1 and the value of <code>DeploymentType</code> is
         * <code>SINGLE_AZ_1</code>, <code>MULTI_AZ_1</code>, or <code>MULTI_AZ_2</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param haPairs
         *        Specifies how many high-availability (HA) file server pairs the file system will have. The default
         *        value is 1. The value of this property affects the values of <code>StorageCapacity</code>,
         *        <code>Iops</code>, and <code>ThroughputCapacity</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/HA-pairs.html">High-availability (HA)
         *        pairs</a> in the FSx for ONTAP user guide.</p>
         *        <p>
         *        Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The value of <code>HAPairs</code> is less than 1 or greater than 12.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>HAPairs</code> is greater than 1 and the value of <code>DeploymentType</code> is
         *        <code>SINGLE_AZ_1</code>, <code>MULTI_AZ_1</code>, or <code>MULTI_AZ_2</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder haPairs(Integer haPairs);

        /**
         * <p>
         * Use to choose the throughput capacity per HA pair. When the value of <code>HAPairs</code> is equal to 1, the
         * value of <code>ThroughputCapacityPerHAPair</code> is the total throughput for the file system.
         * </p>
         * <p>
         * This field and <code>ThroughputCapacity</code> cannot be defined in the same API call, but one is required.
         * </p>
         * <p>
         * This field and <code>ThroughputCapacity</code> are the same for file systems with one HA pair.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>SINGLE_AZ_1</code> and <code>MULTI_AZ_1</code> file systems, valid values are 128, 256, 512, 1024,
         * 2048, or 4096 MBps.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>SINGLE_AZ_2</code>, valid values are 1536, 3072, or 6144 MBps.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>MULTI_AZ_2</code>, valid values are 384, 768, 1536, 3072, or 6144 MBps.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The value of <code>ThroughputCapacity</code> and <code>ThroughputCapacityPerHAPair</code> are not the same
         * value.
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of deployment type is <code>SINGLE_AZ_2</code> and <code>ThroughputCapacity</code> /
         * <code>ThroughputCapacityPerHAPair</code> is not a valid HA pair (a value between 1 and 12).
         * </p>
         * </li>
         * <li>
         * <p>
         * The value of <code>ThroughputCapacityPerHAPair</code> is not a valid value.
         * </p>
         * </li>
         * </ul>
         * 
         * @param throughputCapacityPerHAPair
         *        Use to choose the throughput capacity per HA pair. When the value of <code>HAPairs</code> is equal to
         *        1, the value of <code>ThroughputCapacityPerHAPair</code> is the total throughput for the file
         *        system.</p>
         *        <p>
         *        This field and <code>ThroughputCapacity</code> cannot be defined in the same API call, but one is
         *        required.
         *        </p>
         *        <p>
         *        This field and <code>ThroughputCapacity</code> are the same for file systems with one HA pair.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>SINGLE_AZ_1</code> and <code>MULTI_AZ_1</code> file systems, valid values are 128, 256, 512,
         *        1024, 2048, or 4096 MBps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>SINGLE_AZ_2</code>, valid values are 1536, 3072, or 6144 MBps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>MULTI_AZ_2</code>, valid values are 384, 768, 1536, 3072, or 6144 MBps.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon FSx responds with an HTTP status code 400 (Bad Request) for the following conditions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The value of <code>ThroughputCapacity</code> and <code>ThroughputCapacityPerHAPair</code> are not the
         *        same value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of deployment type is <code>SINGLE_AZ_2</code> and <code>ThroughputCapacity</code> /
         *        <code>ThroughputCapacityPerHAPair</code> is not a valid HA pair (a value between 1 and 12).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The value of <code>ThroughputCapacityPerHAPair</code> is not a valid value.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughputCapacityPerHAPair(Integer throughputCapacityPerHAPair);
    }

    static final class BuilderImpl implements Builder {
        private Integer automaticBackupRetentionDays;

        private String dailyAutomaticBackupStartTime;

        private String deploymentType;

        private String endpointIpAddressRange;

        private FileSystemEndpoints endpoints;

        private DiskIopsConfiguration diskIopsConfiguration;

        private String preferredSubnetId;

        private List<String> routeTableIds = DefaultSdkAutoConstructList.getInstance();

        private Integer throughputCapacity;

        private String weeklyMaintenanceStartTime;

        private String fsxAdminPassword;

        private Integer haPairs;

        private Integer throughputCapacityPerHAPair;

        private BuilderImpl() {
        }

        private BuilderImpl(OntapFileSystemConfiguration model) {
            automaticBackupRetentionDays(model.automaticBackupRetentionDays);
            dailyAutomaticBackupStartTime(model.dailyAutomaticBackupStartTime);
            deploymentType(model.deploymentType);
            endpointIpAddressRange(model.endpointIpAddressRange);
            endpoints(model.endpoints);
            diskIopsConfiguration(model.diskIopsConfiguration);
            preferredSubnetId(model.preferredSubnetId);
            routeTableIds(model.routeTableIds);
            throughputCapacity(model.throughputCapacity);
            weeklyMaintenanceStartTime(model.weeklyMaintenanceStartTime);
            fsxAdminPassword(model.fsxAdminPassword);
            haPairs(model.haPairs);
            throughputCapacityPerHAPair(model.throughputCapacityPerHAPair);
        }

        public final Integer getAutomaticBackupRetentionDays() {
            return automaticBackupRetentionDays;
        }

        public final void setAutomaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
        }

        @Override
        public final Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
            return this;
        }

        public final String getDailyAutomaticBackupStartTime() {
            return dailyAutomaticBackupStartTime;
        }

        public final void setDailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
        }

        @Override
        public final Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
            return this;
        }

        public final String getDeploymentType() {
            return deploymentType;
        }

        public final void setDeploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
        }

        @Override
        public final Builder deploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
            return this;
        }

        @Override
        public final Builder deploymentType(OntapDeploymentType deploymentType) {
            this.deploymentType(deploymentType == null ? null : deploymentType.toString());
            return this;
        }

        public final String getEndpointIpAddressRange() {
            return endpointIpAddressRange;
        }

        public final void setEndpointIpAddressRange(String endpointIpAddressRange) {
            this.endpointIpAddressRange = endpointIpAddressRange;
        }

        @Override
        public final Builder endpointIpAddressRange(String endpointIpAddressRange) {
            this.endpointIpAddressRange = endpointIpAddressRange;
            return this;
        }

        public final FileSystemEndpoints.Builder getEndpoints() {
            return endpoints != null ? endpoints.toBuilder() : null;
        }

        public final void setEndpoints(FileSystemEndpoints.BuilderImpl endpoints) {
            this.endpoints = endpoints != null ? endpoints.build() : null;
        }

        @Override
        public final Builder endpoints(FileSystemEndpoints endpoints) {
            this.endpoints = endpoints;
            return this;
        }

        public final DiskIopsConfiguration.Builder getDiskIopsConfiguration() {
            return diskIopsConfiguration != null ? diskIopsConfiguration.toBuilder() : null;
        }

        public final void setDiskIopsConfiguration(DiskIopsConfiguration.BuilderImpl diskIopsConfiguration) {
            this.diskIopsConfiguration = diskIopsConfiguration != null ? diskIopsConfiguration.build() : null;
        }

        @Override
        public final Builder diskIopsConfiguration(DiskIopsConfiguration diskIopsConfiguration) {
            this.diskIopsConfiguration = diskIopsConfiguration;
            return this;
        }

        public final String getPreferredSubnetId() {
            return preferredSubnetId;
        }

        public final void setPreferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
        }

        @Override
        public final Builder preferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
            return this;
        }

        public final Collection<String> getRouteTableIds() {
            if (routeTableIds instanceof SdkAutoConstructList) {
                return null;
            }
            return routeTableIds;
        }

        public final void setRouteTableIds(Collection<String> routeTableIds) {
            this.routeTableIds = RouteTableIdsCopier.copy(routeTableIds);
        }

        @Override
        public final Builder routeTableIds(Collection<String> routeTableIds) {
            this.routeTableIds = RouteTableIdsCopier.copy(routeTableIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routeTableIds(String... routeTableIds) {
            routeTableIds(Arrays.asList(routeTableIds));
            return this;
        }

        public final Integer getThroughputCapacity() {
            return throughputCapacity;
        }

        public final void setThroughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
        }

        @Override
        public final Builder throughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
            return this;
        }

        public final String getWeeklyMaintenanceStartTime() {
            return weeklyMaintenanceStartTime;
        }

        public final void setWeeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
        }

        @Override
        public final Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
            return this;
        }

        public final String getFsxAdminPassword() {
            return fsxAdminPassword;
        }

        public final void setFsxAdminPassword(String fsxAdminPassword) {
            this.fsxAdminPassword = fsxAdminPassword;
        }

        @Override
        public final Builder fsxAdminPassword(String fsxAdminPassword) {
            this.fsxAdminPassword = fsxAdminPassword;
            return this;
        }

        public final Integer getHaPairs() {
            return haPairs;
        }

        public final void setHaPairs(Integer haPairs) {
            this.haPairs = haPairs;
        }

        @Override
        public final Builder haPairs(Integer haPairs) {
            this.haPairs = haPairs;
            return this;
        }

        public final Integer getThroughputCapacityPerHAPair() {
            return throughputCapacityPerHAPair;
        }

        public final void setThroughputCapacityPerHAPair(Integer throughputCapacityPerHAPair) {
            this.throughputCapacityPerHAPair = throughputCapacityPerHAPair;
        }

        @Override
        public final Builder throughputCapacityPerHAPair(Integer throughputCapacityPerHAPair) {
            this.throughputCapacityPerHAPair = throughputCapacityPerHAPair;
            return this;
        }

        @Override
        public OntapFileSystemConfiguration build() {
            return new OntapFileSystemConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
