/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.geomaps;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.geomaps.internal.GeoMapsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.geomaps.model.AccessDeniedException;
import software.amazon.awssdk.services.geomaps.model.GeoMapsException;
import software.amazon.awssdk.services.geomaps.model.GetGlyphsRequest;
import software.amazon.awssdk.services.geomaps.model.GetGlyphsResponse;
import software.amazon.awssdk.services.geomaps.model.GetSpritesRequest;
import software.amazon.awssdk.services.geomaps.model.GetSpritesResponse;
import software.amazon.awssdk.services.geomaps.model.GetStaticMapRequest;
import software.amazon.awssdk.services.geomaps.model.GetStaticMapResponse;
import software.amazon.awssdk.services.geomaps.model.GetStyleDescriptorRequest;
import software.amazon.awssdk.services.geomaps.model.GetStyleDescriptorResponse;
import software.amazon.awssdk.services.geomaps.model.GetTileRequest;
import software.amazon.awssdk.services.geomaps.model.GetTileResponse;
import software.amazon.awssdk.services.geomaps.model.InternalServerException;
import software.amazon.awssdk.services.geomaps.model.ThrottlingException;
import software.amazon.awssdk.services.geomaps.model.ValidationException;
import software.amazon.awssdk.services.geomaps.transform.GetGlyphsRequestMarshaller;
import software.amazon.awssdk.services.geomaps.transform.GetSpritesRequestMarshaller;
import software.amazon.awssdk.services.geomaps.transform.GetStaticMapRequestMarshaller;
import software.amazon.awssdk.services.geomaps.transform.GetStyleDescriptorRequestMarshaller;
import software.amazon.awssdk.services.geomaps.transform.GetTileRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GeoMapsClient}.
 *
 * @see GeoMapsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGeoMapsClient implements GeoMapsClient {
    private static final Logger log = Logger.loggerFor(DefaultGeoMapsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGeoMapsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns the map's glyphs.
     * </p>
     *
     * @param getGlyphsRequest
     * @return Result of the GetGlyphs operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoMapsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoMapsClient.GetGlyphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-maps-2020-11-19/GetGlyphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGlyphsResponse getGlyphs(GetGlyphsRequest getGlyphsRequest) throws AwsServiceException, SdkClientException,
            GeoMapsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetGlyphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGlyphsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGlyphsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGlyphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Maps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGlyphs");

            return clientHandler.execute(new ClientExecutionParams<GetGlyphsRequest, GetGlyphsResponse>()
                    .withOperationName("GetGlyphs").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getGlyphsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGlyphsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the map's sprites.
     * </p>
     *
     * @param getSpritesRequest
     * @return Result of the GetSprites operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoMapsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoMapsClient.GetSprites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-maps-2020-11-19/GetSprites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSpritesResponse getSprites(GetSpritesRequest getSpritesRequest) throws AwsServiceException, SdkClientException,
            GeoMapsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetSpritesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSpritesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSpritesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSpritesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Maps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSprites");

            return clientHandler.execute(new ClientExecutionParams<GetSpritesRequest, GetSpritesResponse>()
                    .withOperationName("GetSprites").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSpritesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSpritesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides high-quality static map images with customizable options. You can modify the map's appearance and
     * overlay additional information. It's an ideal solution for applications requiring tailored static map snapshots.
     * </p>
     *
     * @param getStaticMapRequest
     * @return Result of the GetStaticMap operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         The request was denied because of insufficient access or permissions. Check with an administrator to
     *         verify your permissions.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoMapsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoMapsClient.GetStaticMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-maps-2020-11-19/GetStaticMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStaticMapResponse getStaticMap(GetStaticMapRequest getStaticMapRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            GeoMapsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetStaticMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStaticMapResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStaticMapRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStaticMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Maps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStaticMap");

            return clientHandler.execute(new ClientExecutionParams<GetStaticMapRequest, GetStaticMapResponse>()
                    .withOperationName("GetStaticMap").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getStaticMapRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStaticMapRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the style.
     * </p>
     *
     * @param getStyleDescriptorRequest
     * @return Result of the GetStyleDescriptor operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoMapsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoMapsClient.GetStyleDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-maps-2020-11-19/GetStyleDescriptor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetStyleDescriptorResponse getStyleDescriptor(GetStyleDescriptorRequest getStyleDescriptorRequest)
            throws AwsServiceException, SdkClientException, GeoMapsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetStyleDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetStyleDescriptorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStyleDescriptorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStyleDescriptorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Maps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStyleDescriptor");

            return clientHandler.execute(new ClientExecutionParams<GetStyleDescriptorRequest, GetStyleDescriptorResponse>()
                    .withOperationName("GetStyleDescriptor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getStyleDescriptorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStyleDescriptorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a tile. Map tiles are used by clients to render a map. they're addressed using a grid arrangement with an
     * X coordinate, Y coordinate, and Z (zoom) level.
     * </p>
     *
     * @param getTileRequest
     * @return Result of the GetTile operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         The request was denied because of insufficient access or permissions. Check with an administrator to
     *         verify your permissions.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoMapsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoMapsClient.GetTile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-maps-2020-11-19/GetTile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTileResponse getTile(GetTileRequest getTileRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, GeoMapsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetTileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Maps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTile");

            return clientHandler.execute(new ClientExecutionParams<GetTileRequest, GetTileResponse>()
                    .withOperationName("GetTile").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getTileRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        GeoMapsServiceClientConfigurationBuilder serviceConfigBuilder = new GeoMapsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GeoMapsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public final GeoMapsServiceClientConfiguration serviceClientConfiguration() {
        return new GeoMapsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
