/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Options related to the destination.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteMatrixDestinationOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<RouteMatrixDestinationOptions.Builder, RouteMatrixDestinationOptions> {
    private static final SdkField<Long> AVOID_ACTIONS_FOR_DISTANCE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("AvoidActionsForDistance").getter(getter(RouteMatrixDestinationOptions::avoidActionsForDistance))
            .setter(setter(Builder::avoidActionsForDistance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvoidActionsForDistance").build())
            .build();

    private static final SdkField<Double> HEADING_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Heading")
            .getter(getter(RouteMatrixDestinationOptions::heading)).setter(setter(Builder::heading))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Heading").build()).build();

    private static final SdkField<RouteMatrixMatchingOptions> MATCHING_FIELD = SdkField
            .<RouteMatrixMatchingOptions> builder(MarshallingType.SDK_POJO).memberName("Matching")
            .getter(getter(RouteMatrixDestinationOptions::matching)).setter(setter(Builder::matching))
            .constructor(RouteMatrixMatchingOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Matching").build()).build();

    private static final SdkField<RouteMatrixSideOfStreetOptions> SIDE_OF_STREET_FIELD = SdkField
            .<RouteMatrixSideOfStreetOptions> builder(MarshallingType.SDK_POJO).memberName("SideOfStreet")
            .getter(getter(RouteMatrixDestinationOptions::sideOfStreet)).setter(setter(Builder::sideOfStreet))
            .constructor(RouteMatrixSideOfStreetOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SideOfStreet").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AVOID_ACTIONS_FOR_DISTANCE_FIELD, HEADING_FIELD, MATCHING_FIELD, SIDE_OF_STREET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long avoidActionsForDistance;

    private final Double heading;

    private final RouteMatrixMatchingOptions matching;

    private final RouteMatrixSideOfStreetOptions sideOfStreet;

    private RouteMatrixDestinationOptions(BuilderImpl builder) {
        this.avoidActionsForDistance = builder.avoidActionsForDistance;
        this.heading = builder.heading;
        this.matching = builder.matching;
        this.sideOfStreet = builder.sideOfStreet;
    }

    /**
     * <p>
     * Avoids actions for the provided distance. This is typically to consider for users in moving vehicles who may not
     * have sufficient time to make an action at an origin or a destination.
     * </p>
     * 
     * @return Avoids actions for the provided distance. This is typically to consider for users in moving vehicles who
     *         may not have sufficient time to make an action at an origin or a destination.
     */
    public final Long avoidActionsForDistance() {
        return avoidActionsForDistance;
    }

    /**
     * <p>
     * GPS Heading at the position.
     * </p>
     * 
     * @return GPS Heading at the position.
     */
    public final Double heading() {
        return heading;
    }

    /**
     * <p>
     * Options to configure matching the provided position to the road network.
     * </p>
     * 
     * @return Options to configure matching the provided position to the road network.
     */
    public final RouteMatrixMatchingOptions matching() {
        return matching;
    }

    /**
     * <p>
     * Options to configure matching the provided position to a side of the street.
     * </p>
     * 
     * @return Options to configure matching the provided position to a side of the street.
     */
    public final RouteMatrixSideOfStreetOptions sideOfStreet() {
        return sideOfStreet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(avoidActionsForDistance());
        hashCode = 31 * hashCode + Objects.hashCode(heading());
        hashCode = 31 * hashCode + Objects.hashCode(matching());
        hashCode = 31 * hashCode + Objects.hashCode(sideOfStreet());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteMatrixDestinationOptions)) {
            return false;
        }
        RouteMatrixDestinationOptions other = (RouteMatrixDestinationOptions) obj;
        return Objects.equals(avoidActionsForDistance(), other.avoidActionsForDistance())
                && Objects.equals(heading(), other.heading()) && Objects.equals(matching(), other.matching())
                && Objects.equals(sideOfStreet(), other.sideOfStreet());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteMatrixDestinationOptions").add("AvoidActionsForDistance", avoidActionsForDistance())
                .add("Heading", heading()).add("Matching", matching()).add("SideOfStreet", sideOfStreet()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvoidActionsForDistance":
            return Optional.ofNullable(clazz.cast(avoidActionsForDistance()));
        case "Heading":
            return Optional.ofNullable(clazz.cast(heading()));
        case "Matching":
            return Optional.ofNullable(clazz.cast(matching()));
        case "SideOfStreet":
            return Optional.ofNullable(clazz.cast(sideOfStreet()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AvoidActionsForDistance", AVOID_ACTIONS_FOR_DISTANCE_FIELD);
        map.put("Heading", HEADING_FIELD);
        map.put("Matching", MATCHING_FIELD);
        map.put("SideOfStreet", SIDE_OF_STREET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteMatrixDestinationOptions, T> g) {
        return obj -> g.apply((RouteMatrixDestinationOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteMatrixDestinationOptions> {
        /**
         * <p>
         * Avoids actions for the provided distance. This is typically to consider for users in moving vehicles who may
         * not have sufficient time to make an action at an origin or a destination.
         * </p>
         * 
         * @param avoidActionsForDistance
         *        Avoids actions for the provided distance. This is typically to consider for users in moving vehicles
         *        who may not have sufficient time to make an action at an origin or a destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avoidActionsForDistance(Long avoidActionsForDistance);

        /**
         * <p>
         * GPS Heading at the position.
         * </p>
         * 
         * @param heading
         *        GPS Heading at the position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heading(Double heading);

        /**
         * <p>
         * Options to configure matching the provided position to the road network.
         * </p>
         * 
         * @param matching
         *        Options to configure matching the provided position to the road network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matching(RouteMatrixMatchingOptions matching);

        /**
         * <p>
         * Options to configure matching the provided position to the road network.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixMatchingOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixMatchingOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixMatchingOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #matching(RouteMatrixMatchingOptions)}.
         * 
         * @param matching
         *        a consumer that will call methods on {@link RouteMatrixMatchingOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matching(RouteMatrixMatchingOptions)
         */
        default Builder matching(Consumer<RouteMatrixMatchingOptions.Builder> matching) {
            return matching(RouteMatrixMatchingOptions.builder().applyMutation(matching).build());
        }

        /**
         * <p>
         * Options to configure matching the provided position to a side of the street.
         * </p>
         * 
         * @param sideOfStreet
         *        Options to configure matching the provided position to a side of the street.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sideOfStreet(RouteMatrixSideOfStreetOptions sideOfStreet);

        /**
         * <p>
         * Options to configure matching the provided position to a side of the street.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixSideOfStreetOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixSideOfStreetOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixSideOfStreetOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #sideOfStreet(RouteMatrixSideOfStreetOptions)}.
         * 
         * @param sideOfStreet
         *        a consumer that will call methods on {@link RouteMatrixSideOfStreetOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sideOfStreet(RouteMatrixSideOfStreetOptions)
         */
        default Builder sideOfStreet(Consumer<RouteMatrixSideOfStreetOptions.Builder> sideOfStreet) {
            return sideOfStreet(RouteMatrixSideOfStreetOptions.builder().applyMutation(sideOfStreet).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Long avoidActionsForDistance;

        private Double heading;

        private RouteMatrixMatchingOptions matching;

        private RouteMatrixSideOfStreetOptions sideOfStreet;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteMatrixDestinationOptions model) {
            avoidActionsForDistance(model.avoidActionsForDistance);
            heading(model.heading);
            matching(model.matching);
            sideOfStreet(model.sideOfStreet);
        }

        public final Long getAvoidActionsForDistance() {
            return avoidActionsForDistance;
        }

        public final void setAvoidActionsForDistance(Long avoidActionsForDistance) {
            this.avoidActionsForDistance = avoidActionsForDistance;
        }

        @Override
        public final Builder avoidActionsForDistance(Long avoidActionsForDistance) {
            this.avoidActionsForDistance = avoidActionsForDistance;
            return this;
        }

        public final Double getHeading() {
            return heading;
        }

        public final void setHeading(Double heading) {
            this.heading = heading;
        }

        @Override
        public final Builder heading(Double heading) {
            this.heading = heading;
            return this;
        }

        public final RouteMatrixMatchingOptions.Builder getMatching() {
            return matching != null ? matching.toBuilder() : null;
        }

        public final void setMatching(RouteMatrixMatchingOptions.BuilderImpl matching) {
            this.matching = matching != null ? matching.build() : null;
        }

        @Override
        public final Builder matching(RouteMatrixMatchingOptions matching) {
            this.matching = matching;
            return this;
        }

        public final RouteMatrixSideOfStreetOptions.Builder getSideOfStreet() {
            return sideOfStreet != null ? sideOfStreet.toBuilder() : null;
        }

        public final void setSideOfStreet(RouteMatrixSideOfStreetOptions.BuilderImpl sideOfStreet) {
            this.sideOfStreet = sideOfStreet != null ? sideOfStreet.build() : null;
        }

        @Override
        public final Builder sideOfStreet(RouteMatrixSideOfStreetOptions sideOfStreet) {
            this.sideOfStreet = sideOfStreet;
            return this;
        }

        @Override
        public RouteMatrixDestinationOptions build() {
            return new RouteMatrixDestinationOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
