/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The calculated route matrix containing the results for all pairs of Origins to Destination positions. Each row
 * corresponds to one entry in Origins. Each entry in the row corresponds to the route from that entry in Origins to an
 * entry in Destination positions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteMatrixEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<RouteMatrixEntry.Builder, RouteMatrixEntry> {
    private static final SdkField<Long> DISTANCE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Distance")
            .getter(getter(RouteMatrixEntry::distance)).setter(setter(Builder::distance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Distance").build()).build();

    private static final SdkField<Long> DURATION_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Duration")
            .getter(getter(RouteMatrixEntry::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<String> ERROR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Error")
            .getter(getter(RouteMatrixEntry::errorAsString)).setter(setter(Builder::error))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Error").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DISTANCE_FIELD,
            DURATION_FIELD, ERROR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long distance;

    private final Long duration;

    private final String error;

    private RouteMatrixEntry(BuilderImpl builder) {
        this.distance = builder.distance;
        this.duration = builder.duration;
        this.error = builder.error;
    }

    /**
     * <p>
     * The total distance of travel for the route.
     * </p>
     * 
     * @return The total distance of travel for the route.
     */
    public final Long distance() {
        return distance;
    }

    /**
     * <p>
     * The expected duration of travel for the route.
     * </p>
     * <p>
     * <b>Unit</b>: <code>seconds</code>
     * </p>
     * 
     * @return The expected duration of travel for the route.</p>
     *         <p>
     *         <b>Unit</b>: <code>seconds</code>
     */
    public final Long duration() {
        return duration;
    }

    /**
     * <p>
     * Error code that occurred during calculation of the route.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #error} will return
     * {@link RouteMatrixErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorAsString}.
     * </p>
     * 
     * @return Error code that occurred during calculation of the route.
     * @see RouteMatrixErrorCode
     */
    public final RouteMatrixErrorCode error() {
        return RouteMatrixErrorCode.fromValue(error);
    }

    /**
     * <p>
     * Error code that occurred during calculation of the route.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #error} will return
     * {@link RouteMatrixErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorAsString}.
     * </p>
     * 
     * @return Error code that occurred during calculation of the route.
     * @see RouteMatrixErrorCode
     */
    public final String errorAsString() {
        return error;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(distance());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(errorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteMatrixEntry)) {
            return false;
        }
        RouteMatrixEntry other = (RouteMatrixEntry) obj;
        return Objects.equals(distance(), other.distance()) && Objects.equals(duration(), other.duration())
                && Objects.equals(errorAsString(), other.errorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteMatrixEntry").add("Distance", distance()).add("Duration", duration())
                .add("Error", errorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Distance":
            return Optional.ofNullable(clazz.cast(distance()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "Error":
            return Optional.ofNullable(clazz.cast(errorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Distance", DISTANCE_FIELD);
        map.put("Duration", DURATION_FIELD);
        map.put("Error", ERROR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteMatrixEntry, T> g) {
        return obj -> g.apply((RouteMatrixEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteMatrixEntry> {
        /**
         * <p>
         * The total distance of travel for the route.
         * </p>
         * 
         * @param distance
         *        The total distance of travel for the route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distance(Long distance);

        /**
         * <p>
         * The expected duration of travel for the route.
         * </p>
         * <p>
         * <b>Unit</b>: <code>seconds</code>
         * </p>
         * 
         * @param duration
         *        The expected duration of travel for the route.</p>
         *        <p>
         *        <b>Unit</b>: <code>seconds</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Long duration);

        /**
         * <p>
         * Error code that occurred during calculation of the route.
         * </p>
         * 
         * @param error
         *        Error code that occurred during calculation of the route.
         * @see RouteMatrixErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteMatrixErrorCode
         */
        Builder error(String error);

        /**
         * <p>
         * Error code that occurred during calculation of the route.
         * </p>
         * 
         * @param error
         *        Error code that occurred during calculation of the route.
         * @see RouteMatrixErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteMatrixErrorCode
         */
        Builder error(RouteMatrixErrorCode error);
    }

    static final class BuilderImpl implements Builder {
        private Long distance;

        private Long duration;

        private String error;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteMatrixEntry model) {
            distance(model.distance);
            duration(model.duration);
            error(model.error);
        }

        public final Long getDistance() {
            return distance;
        }

        public final void setDistance(Long distance) {
            this.distance = distance;
        }

        @Override
        public final Builder distance(Long distance) {
            this.distance = distance;
            return this;
        }

        public final Long getDuration() {
            return duration;
        }

        public final void setDuration(Long duration) {
            this.duration = duration;
        }

        @Override
        public final Builder duration(Long duration) {
            this.duration = duration;
            return this;
        }

        public final String getError() {
            return error;
        }

        public final void setError(String error) {
            this.error = error;
        }

        @Override
        public final Builder error(String error) {
            this.error = error;
            return this;
        }

        @Override
        public final Builder error(RouteMatrixErrorCode error) {
            this.error(error == null ? null : error.toString());
            return this;
        }

        @Override
        public RouteMatrixEntry build() {
            return new RouteMatrixEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
