/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Waypoint between the Origin and Destination.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteWaypoint implements SdkPojo, Serializable, ToCopyableBuilder<RouteWaypoint.Builder, RouteWaypoint> {
    private static final SdkField<Long> AVOID_ACTIONS_FOR_DISTANCE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("AvoidActionsForDistance").getter(getter(RouteWaypoint::avoidActionsForDistance))
            .setter(setter(Builder::avoidActionsForDistance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvoidActionsForDistance").build())
            .build();

    private static final SdkField<Boolean> AVOID_U_TURNS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AvoidUTurns").getter(getter(RouteWaypoint::avoidUTurns)).setter(setter(Builder::avoidUTurns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvoidUTurns").build()).build();

    private static final SdkField<Double> HEADING_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Heading")
            .getter(getter(RouteWaypoint::heading)).setter(setter(Builder::heading))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Heading").build()).build();

    private static final SdkField<RouteMatchingOptions> MATCHING_FIELD = SdkField
            .<RouteMatchingOptions> builder(MarshallingType.SDK_POJO).memberName("Matching")
            .getter(getter(RouteWaypoint::matching)).setter(setter(Builder::matching)).constructor(RouteMatchingOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Matching").build()).build();

    private static final SdkField<Boolean> PASS_THROUGH_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PassThrough").getter(getter(RouteWaypoint::passThrough)).setter(setter(Builder::passThrough))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PassThrough").build()).build();

    private static final SdkField<List<Double>> POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Position")
            .getter(getter(RouteWaypoint::position))
            .setter(setter(Builder::position))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Position").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RouteSideOfStreetOptions> SIDE_OF_STREET_FIELD = SdkField
            .<RouteSideOfStreetOptions> builder(MarshallingType.SDK_POJO).memberName("SideOfStreet")
            .getter(getter(RouteWaypoint::sideOfStreet)).setter(setter(Builder::sideOfStreet))
            .constructor(RouteSideOfStreetOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SideOfStreet").build()).build();

    private static final SdkField<Long> STOP_DURATION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("StopDuration").getter(getter(RouteWaypoint::stopDuration)).setter(setter(Builder::stopDuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StopDuration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AVOID_ACTIONS_FOR_DISTANCE_FIELD, AVOID_U_TURNS_FIELD, HEADING_FIELD, MATCHING_FIELD, PASS_THROUGH_FIELD,
            POSITION_FIELD, SIDE_OF_STREET_FIELD, STOP_DURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long avoidActionsForDistance;

    private final Boolean avoidUTurns;

    private final Double heading;

    private final RouteMatchingOptions matching;

    private final Boolean passThrough;

    private final List<Double> position;

    private final RouteSideOfStreetOptions sideOfStreet;

    private final Long stopDuration;

    private RouteWaypoint(BuilderImpl builder) {
        this.avoidActionsForDistance = builder.avoidActionsForDistance;
        this.avoidUTurns = builder.avoidUTurns;
        this.heading = builder.heading;
        this.matching = builder.matching;
        this.passThrough = builder.passThrough;
        this.position = builder.position;
        this.sideOfStreet = builder.sideOfStreet;
        this.stopDuration = builder.stopDuration;
    }

    /**
     * <p>
     * Avoids actions for the provided distance. This is typically to consider for users in moving vehicles who may not
     * have sufficient time to make an action at an origin or a destination.
     * </p>
     * 
     * @return Avoids actions for the provided distance. This is typically to consider for users in moving vehicles who
     *         may not have sufficient time to make an action at an origin or a destination.
     */
    public final Long avoidActionsForDistance() {
        return avoidActionsForDistance;
    }

    /**
     * <p>
     * Avoid U-turns for calculation on highways and motorways.
     * </p>
     * 
     * @return Avoid U-turns for calculation on highways and motorways.
     */
    public final Boolean avoidUTurns() {
        return avoidUTurns;
    }

    /**
     * <p>
     * GPS Heading at the position.
     * </p>
     * 
     * @return GPS Heading at the position.
     */
    public final Double heading() {
        return heading;
    }

    /**
     * <p>
     * Options to configure matching the provided position to the road network.
     * </p>
     * 
     * @return Options to configure matching the provided position to the road network.
     */
    public final RouteMatchingOptions matching() {
        return matching;
    }

    /**
     * <p>
     * If the waypoint should not be treated as a stop. If yes, the waypoint is passed through and doesn't split the
     * route into different legs.
     * </p>
     * 
     * @return If the waypoint should not be treated as a stop. If yes, the waypoint is passed through and doesn't split
     *         the route into different legs.
     */
    public final Boolean passThrough() {
        return passThrough;
    }

    /**
     * For responses, this returns true if the service returned a value for the Position property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPosition() {
        return position != null && !(position instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Position defined as <code>[longitude, latitude]</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPosition} method.
     * </p>
     * 
     * @return Position defined as <code>[longitude, latitude]</code>.
     */
    public final List<Double> position() {
        return position;
    }

    /**
     * <p>
     * Options to configure matching the provided position to a side of the street.
     * </p>
     * 
     * @return Options to configure matching the provided position to a side of the street.
     */
    public final RouteSideOfStreetOptions sideOfStreet() {
        return sideOfStreet;
    }

    /**
     * <p>
     * Duration of the stop.
     * </p>
     * <p>
     * <b>Unit</b>: <code>seconds</code>
     * </p>
     * 
     * @return Duration of the stop.</p>
     *         <p>
     *         <b>Unit</b>: <code>seconds</code>
     */
    public final Long stopDuration() {
        return stopDuration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(avoidActionsForDistance());
        hashCode = 31 * hashCode + Objects.hashCode(avoidUTurns());
        hashCode = 31 * hashCode + Objects.hashCode(heading());
        hashCode = 31 * hashCode + Objects.hashCode(matching());
        hashCode = 31 * hashCode + Objects.hashCode(passThrough());
        hashCode = 31 * hashCode + Objects.hashCode(hasPosition() ? position() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sideOfStreet());
        hashCode = 31 * hashCode + Objects.hashCode(stopDuration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteWaypoint)) {
            return false;
        }
        RouteWaypoint other = (RouteWaypoint) obj;
        return Objects.equals(avoidActionsForDistance(), other.avoidActionsForDistance())
                && Objects.equals(avoidUTurns(), other.avoidUTurns()) && Objects.equals(heading(), other.heading())
                && Objects.equals(matching(), other.matching()) && Objects.equals(passThrough(), other.passThrough())
                && hasPosition() == other.hasPosition() && Objects.equals(position(), other.position())
                && Objects.equals(sideOfStreet(), other.sideOfStreet()) && Objects.equals(stopDuration(), other.stopDuration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteWaypoint").add("AvoidActionsForDistance", avoidActionsForDistance())
                .add("AvoidUTurns", avoidUTurns()).add("Heading", heading()).add("Matching", matching())
                .add("PassThrough", passThrough()).add("Position", position() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SideOfStreet", sideOfStreet()).add("StopDuration", stopDuration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvoidActionsForDistance":
            return Optional.ofNullable(clazz.cast(avoidActionsForDistance()));
        case "AvoidUTurns":
            return Optional.ofNullable(clazz.cast(avoidUTurns()));
        case "Heading":
            return Optional.ofNullable(clazz.cast(heading()));
        case "Matching":
            return Optional.ofNullable(clazz.cast(matching()));
        case "PassThrough":
            return Optional.ofNullable(clazz.cast(passThrough()));
        case "Position":
            return Optional.ofNullable(clazz.cast(position()));
        case "SideOfStreet":
            return Optional.ofNullable(clazz.cast(sideOfStreet()));
        case "StopDuration":
            return Optional.ofNullable(clazz.cast(stopDuration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AvoidActionsForDistance", AVOID_ACTIONS_FOR_DISTANCE_FIELD);
        map.put("AvoidUTurns", AVOID_U_TURNS_FIELD);
        map.put("Heading", HEADING_FIELD);
        map.put("Matching", MATCHING_FIELD);
        map.put("PassThrough", PASS_THROUGH_FIELD);
        map.put("Position", POSITION_FIELD);
        map.put("SideOfStreet", SIDE_OF_STREET_FIELD);
        map.put("StopDuration", STOP_DURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteWaypoint, T> g) {
        return obj -> g.apply((RouteWaypoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteWaypoint> {
        /**
         * <p>
         * Avoids actions for the provided distance. This is typically to consider for users in moving vehicles who may
         * not have sufficient time to make an action at an origin or a destination.
         * </p>
         * 
         * @param avoidActionsForDistance
         *        Avoids actions for the provided distance. This is typically to consider for users in moving vehicles
         *        who may not have sufficient time to make an action at an origin or a destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avoidActionsForDistance(Long avoidActionsForDistance);

        /**
         * <p>
         * Avoid U-turns for calculation on highways and motorways.
         * </p>
         * 
         * @param avoidUTurns
         *        Avoid U-turns for calculation on highways and motorways.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avoidUTurns(Boolean avoidUTurns);

        /**
         * <p>
         * GPS Heading at the position.
         * </p>
         * 
         * @param heading
         *        GPS Heading at the position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heading(Double heading);

        /**
         * <p>
         * Options to configure matching the provided position to the road network.
         * </p>
         * 
         * @param matching
         *        Options to configure matching the provided position to the road network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matching(RouteMatchingOptions matching);

        /**
         * <p>
         * Options to configure matching the provided position to the road network.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatchingOptions.Builder} avoiding
         * the need to create one manually via {@link RouteMatchingOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatchingOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #matching(RouteMatchingOptions)}.
         * 
         * @param matching
         *        a consumer that will call methods on {@link RouteMatchingOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matching(RouteMatchingOptions)
         */
        default Builder matching(Consumer<RouteMatchingOptions.Builder> matching) {
            return matching(RouteMatchingOptions.builder().applyMutation(matching).build());
        }

        /**
         * <p>
         * If the waypoint should not be treated as a stop. If yes, the waypoint is passed through and doesn't split the
         * route into different legs.
         * </p>
         * 
         * @param passThrough
         *        If the waypoint should not be treated as a stop. If yes, the waypoint is passed through and doesn't
         *        split the route into different legs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder passThrough(Boolean passThrough);

        /**
         * <p>
         * Position defined as <code>[longitude, latitude]</code>.
         * </p>
         * 
         * @param position
         *        Position defined as <code>[longitude, latitude]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Collection<Double> position);

        /**
         * <p>
         * Position defined as <code>[longitude, latitude]</code>.
         * </p>
         * 
         * @param position
         *        Position defined as <code>[longitude, latitude]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Double... position);

        /**
         * <p>
         * Options to configure matching the provided position to a side of the street.
         * </p>
         * 
         * @param sideOfStreet
         *        Options to configure matching the provided position to a side of the street.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sideOfStreet(RouteSideOfStreetOptions sideOfStreet);

        /**
         * <p>
         * Options to configure matching the provided position to a side of the street.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteSideOfStreetOptions.Builder}
         * avoiding the need to create one manually via {@link RouteSideOfStreetOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteSideOfStreetOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #sideOfStreet(RouteSideOfStreetOptions)}.
         * 
         * @param sideOfStreet
         *        a consumer that will call methods on {@link RouteSideOfStreetOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sideOfStreet(RouteSideOfStreetOptions)
         */
        default Builder sideOfStreet(Consumer<RouteSideOfStreetOptions.Builder> sideOfStreet) {
            return sideOfStreet(RouteSideOfStreetOptions.builder().applyMutation(sideOfStreet).build());
        }

        /**
         * <p>
         * Duration of the stop.
         * </p>
         * <p>
         * <b>Unit</b>: <code>seconds</code>
         * </p>
         * 
         * @param stopDuration
         *        Duration of the stop.</p>
         *        <p>
         *        <b>Unit</b>: <code>seconds</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopDuration(Long stopDuration);
    }

    static final class BuilderImpl implements Builder {
        private Long avoidActionsForDistance;

        private Boolean avoidUTurns;

        private Double heading;

        private RouteMatchingOptions matching;

        private Boolean passThrough;

        private List<Double> position = DefaultSdkAutoConstructList.getInstance();

        private RouteSideOfStreetOptions sideOfStreet;

        private Long stopDuration;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteWaypoint model) {
            avoidActionsForDistance(model.avoidActionsForDistance);
            avoidUTurns(model.avoidUTurns);
            heading(model.heading);
            matching(model.matching);
            passThrough(model.passThrough);
            position(model.position);
            sideOfStreet(model.sideOfStreet);
            stopDuration(model.stopDuration);
        }

        public final Long getAvoidActionsForDistance() {
            return avoidActionsForDistance;
        }

        public final void setAvoidActionsForDistance(Long avoidActionsForDistance) {
            this.avoidActionsForDistance = avoidActionsForDistance;
        }

        @Override
        public final Builder avoidActionsForDistance(Long avoidActionsForDistance) {
            this.avoidActionsForDistance = avoidActionsForDistance;
            return this;
        }

        public final Boolean getAvoidUTurns() {
            return avoidUTurns;
        }

        public final void setAvoidUTurns(Boolean avoidUTurns) {
            this.avoidUTurns = avoidUTurns;
        }

        @Override
        public final Builder avoidUTurns(Boolean avoidUTurns) {
            this.avoidUTurns = avoidUTurns;
            return this;
        }

        public final Double getHeading() {
            return heading;
        }

        public final void setHeading(Double heading) {
            this.heading = heading;
        }

        @Override
        public final Builder heading(Double heading) {
            this.heading = heading;
            return this;
        }

        public final RouteMatchingOptions.Builder getMatching() {
            return matching != null ? matching.toBuilder() : null;
        }

        public final void setMatching(RouteMatchingOptions.BuilderImpl matching) {
            this.matching = matching != null ? matching.build() : null;
        }

        @Override
        public final Builder matching(RouteMatchingOptions matching) {
            this.matching = matching;
            return this;
        }

        public final Boolean getPassThrough() {
            return passThrough;
        }

        public final void setPassThrough(Boolean passThrough) {
            this.passThrough = passThrough;
        }

        @Override
        public final Builder passThrough(Boolean passThrough) {
            this.passThrough = passThrough;
            return this;
        }

        public final Collection<Double> getPosition() {
            if (position instanceof SdkAutoConstructList) {
                return null;
            }
            return position;
        }

        public final void setPosition(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
        }

        @Override
        public final Builder position(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder position(Double... position) {
            position(Arrays.asList(position));
            return this;
        }

        public final RouteSideOfStreetOptions.Builder getSideOfStreet() {
            return sideOfStreet != null ? sideOfStreet.toBuilder() : null;
        }

        public final void setSideOfStreet(RouteSideOfStreetOptions.BuilderImpl sideOfStreet) {
            this.sideOfStreet = sideOfStreet != null ? sideOfStreet.build() : null;
        }

        @Override
        public final Builder sideOfStreet(RouteSideOfStreetOptions sideOfStreet) {
            this.sideOfStreet = sideOfStreet;
            return this;
        }

        public final Long getStopDuration() {
            return stopDuration;
        }

        public final void setStopDuration(Long stopDuration) {
            this.stopDuration = stopDuration;
        }

        @Override
        public final Builder stopDuration(Long stopDuration) {
            this.stopDuration = stopDuration;
            return this;
        }

        @Override
        public RouteWaypoint build() {
            return new RouteWaypoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
