/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glacier;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.glacier.internal.GlacierServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.glacier.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.glacier.model.AbortMultipartUploadResponse;
import software.amazon.awssdk.services.glacier.model.AbortVaultLockRequest;
import software.amazon.awssdk.services.glacier.model.AbortVaultLockResponse;
import software.amazon.awssdk.services.glacier.model.AddTagsToVaultRequest;
import software.amazon.awssdk.services.glacier.model.AddTagsToVaultResponse;
import software.amazon.awssdk.services.glacier.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.glacier.model.CompleteMultipartUploadResponse;
import software.amazon.awssdk.services.glacier.model.CompleteVaultLockRequest;
import software.amazon.awssdk.services.glacier.model.CompleteVaultLockResponse;
import software.amazon.awssdk.services.glacier.model.CreateVaultRequest;
import software.amazon.awssdk.services.glacier.model.CreateVaultResponse;
import software.amazon.awssdk.services.glacier.model.DeleteArchiveRequest;
import software.amazon.awssdk.services.glacier.model.DeleteArchiveResponse;
import software.amazon.awssdk.services.glacier.model.DeleteVaultAccessPolicyRequest;
import software.amazon.awssdk.services.glacier.model.DeleteVaultAccessPolicyResponse;
import software.amazon.awssdk.services.glacier.model.DeleteVaultNotificationsRequest;
import software.amazon.awssdk.services.glacier.model.DeleteVaultNotificationsResponse;
import software.amazon.awssdk.services.glacier.model.DeleteVaultRequest;
import software.amazon.awssdk.services.glacier.model.DeleteVaultResponse;
import software.amazon.awssdk.services.glacier.model.DescribeJobRequest;
import software.amazon.awssdk.services.glacier.model.DescribeJobResponse;
import software.amazon.awssdk.services.glacier.model.DescribeVaultRequest;
import software.amazon.awssdk.services.glacier.model.DescribeVaultResponse;
import software.amazon.awssdk.services.glacier.model.GetDataRetrievalPolicyRequest;
import software.amazon.awssdk.services.glacier.model.GetDataRetrievalPolicyResponse;
import software.amazon.awssdk.services.glacier.model.GetJobOutputRequest;
import software.amazon.awssdk.services.glacier.model.GetJobOutputResponse;
import software.amazon.awssdk.services.glacier.model.GetVaultAccessPolicyRequest;
import software.amazon.awssdk.services.glacier.model.GetVaultAccessPolicyResponse;
import software.amazon.awssdk.services.glacier.model.GetVaultLockRequest;
import software.amazon.awssdk.services.glacier.model.GetVaultLockResponse;
import software.amazon.awssdk.services.glacier.model.GetVaultNotificationsRequest;
import software.amazon.awssdk.services.glacier.model.GetVaultNotificationsResponse;
import software.amazon.awssdk.services.glacier.model.GlacierException;
import software.amazon.awssdk.services.glacier.model.InitiateJobRequest;
import software.amazon.awssdk.services.glacier.model.InitiateJobResponse;
import software.amazon.awssdk.services.glacier.model.InitiateMultipartUploadRequest;
import software.amazon.awssdk.services.glacier.model.InitiateMultipartUploadResponse;
import software.amazon.awssdk.services.glacier.model.InitiateVaultLockRequest;
import software.amazon.awssdk.services.glacier.model.InitiateVaultLockResponse;
import software.amazon.awssdk.services.glacier.model.InsufficientCapacityException;
import software.amazon.awssdk.services.glacier.model.InvalidParameterValueException;
import software.amazon.awssdk.services.glacier.model.LimitExceededException;
import software.amazon.awssdk.services.glacier.model.ListJobsRequest;
import software.amazon.awssdk.services.glacier.model.ListJobsResponse;
import software.amazon.awssdk.services.glacier.model.ListMultipartUploadsRequest;
import software.amazon.awssdk.services.glacier.model.ListMultipartUploadsResponse;
import software.amazon.awssdk.services.glacier.model.ListPartsRequest;
import software.amazon.awssdk.services.glacier.model.ListPartsResponse;
import software.amazon.awssdk.services.glacier.model.ListProvisionedCapacityRequest;
import software.amazon.awssdk.services.glacier.model.ListProvisionedCapacityResponse;
import software.amazon.awssdk.services.glacier.model.ListTagsForVaultRequest;
import software.amazon.awssdk.services.glacier.model.ListTagsForVaultResponse;
import software.amazon.awssdk.services.glacier.model.ListVaultsRequest;
import software.amazon.awssdk.services.glacier.model.ListVaultsResponse;
import software.amazon.awssdk.services.glacier.model.MissingParameterValueException;
import software.amazon.awssdk.services.glacier.model.PolicyEnforcedException;
import software.amazon.awssdk.services.glacier.model.PurchaseProvisionedCapacityRequest;
import software.amazon.awssdk.services.glacier.model.PurchaseProvisionedCapacityResponse;
import software.amazon.awssdk.services.glacier.model.RemoveTagsFromVaultRequest;
import software.amazon.awssdk.services.glacier.model.RemoveTagsFromVaultResponse;
import software.amazon.awssdk.services.glacier.model.RequestTimeoutException;
import software.amazon.awssdk.services.glacier.model.ResourceNotFoundException;
import software.amazon.awssdk.services.glacier.model.ServiceUnavailableException;
import software.amazon.awssdk.services.glacier.model.SetDataRetrievalPolicyRequest;
import software.amazon.awssdk.services.glacier.model.SetDataRetrievalPolicyResponse;
import software.amazon.awssdk.services.glacier.model.SetVaultAccessPolicyRequest;
import software.amazon.awssdk.services.glacier.model.SetVaultAccessPolicyResponse;
import software.amazon.awssdk.services.glacier.model.SetVaultNotificationsRequest;
import software.amazon.awssdk.services.glacier.model.SetVaultNotificationsResponse;
import software.amazon.awssdk.services.glacier.model.UploadArchiveRequest;
import software.amazon.awssdk.services.glacier.model.UploadArchiveResponse;
import software.amazon.awssdk.services.glacier.model.UploadMultipartPartRequest;
import software.amazon.awssdk.services.glacier.model.UploadMultipartPartResponse;
import software.amazon.awssdk.services.glacier.transform.AbortMultipartUploadRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.AbortVaultLockRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.AddTagsToVaultRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.CompleteMultipartUploadRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.CompleteVaultLockRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.CreateVaultRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.DeleteArchiveRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.DeleteVaultAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.DeleteVaultNotificationsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.DeleteVaultRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.DescribeVaultRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.GetDataRetrievalPolicyRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.GetJobOutputRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.GetVaultAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.GetVaultLockRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.GetVaultNotificationsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.InitiateJobRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.InitiateMultipartUploadRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.InitiateVaultLockRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.ListMultipartUploadsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.ListPartsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.ListProvisionedCapacityRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.ListTagsForVaultRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.ListVaultsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.PurchaseProvisionedCapacityRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.RemoveTagsFromVaultRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.SetDataRetrievalPolicyRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.SetVaultAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.SetVaultNotificationsRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.UploadArchiveRequestMarshaller;
import software.amazon.awssdk.services.glacier.transform.UploadMultipartPartRequestMarshaller;
import software.amazon.awssdk.services.glacier.waiters.GlacierWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GlacierClient}.
 *
 * @see GlacierClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGlacierClient implements GlacierClient {
    private static final Logger log = Logger.loggerFor(DefaultGlacierClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGlacierClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This operation aborts a multipart upload identified by the upload ID.
     * </p>
     * <p>
     * After the Abort Multipart Upload request succeeds, you cannot upload any more parts to the multipart upload or
     * complete the multipart upload. Aborting a completed upload fails. However, aborting an already-aborted upload
     * will succeed, for a short time. For more information about uploading a part and completing a multipart upload,
     * see <a>UploadMultipartPart</a> and <a>CompleteMultipartUpload</a>.
     * </p>
     * <p>
     * This operation is idempotent.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-archives.html">Working with Archives in
     * Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html">Abort Multipart
     * Upload</a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param abortMultipartUploadRequest
     *        Provides options to abort a multipart upload identified by the upload ID.</p>
     *        <p>
     *        For information about the underlying REST API, see <a
     *        href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html">Abort
     *        Multipart Upload</a>. For conceptual information, see <a
     *        href="https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-archives.html">Working with
     *        Archives in Amazon S3 Glacier</a>.
     * @return Result of the AbortMultipartUpload operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.AbortMultipartUpload
     */
    @Override
    public AbortMultipartUploadResponse abortMultipartUpload(AbortMultipartUploadRequest abortMultipartUploadRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AbortMultipartUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AbortMultipartUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(abortMultipartUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, abortMultipartUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AbortMultipartUpload");

            return clientHandler.execute(new ClientExecutionParams<AbortMultipartUploadRequest, AbortMultipartUploadResponse>()
                    .withOperationName("AbortMultipartUpload").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(abortMultipartUploadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AbortMultipartUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation aborts the vault locking process if the vault lock is not in the <code>Locked</code> state. If the
     * vault lock is in the <code>Locked</code> state when this operation is requested, the operation returns an
     * <code>AccessDeniedException</code> error. Aborting the vault locking process removes the vault lock policy from
     * the specified vault.
     * </p>
     * <p>
     * A vault lock is put into the <code>InProgress</code> state by calling <a>InitiateVaultLock</a>. A vault lock is
     * put into the <code>Locked</code> state by calling <a>CompleteVaultLock</a>. You can get the state of a vault lock
     * by calling <a>GetVaultLock</a>. For more information about the vault locking process, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock.html">Amazon Glacier Vault Lock</a>. For
     * more information about vault lock policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock-policy.html">Amazon Glacier Access Control
     * with Vault Lock Policies</a>.
     * </p>
     * <p>
     * This operation is idempotent. You can successfully invoke this operation multiple times, if the vault lock is in
     * the <code>InProgress</code> state or if there is no policy associated with the vault.
     * </p>
     *
     * @param abortVaultLockRequest
     *        The input values for <code>AbortVaultLock</code>.
     * @return Result of the AbortVaultLock operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.AbortVaultLock
     */
    @Override
    public AbortVaultLockResponse abortVaultLock(AbortVaultLockRequest abortVaultLockRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AbortVaultLockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AbortVaultLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(abortVaultLockRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, abortVaultLockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AbortVaultLock");

            return clientHandler.execute(new ClientExecutionParams<AbortVaultLockRequest, AbortVaultLockResponse>()
                    .withOperationName("AbortVaultLock").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(abortVaultLockRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AbortVaultLockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation adds the specified tags to a vault. Each tag is composed of a key and a value. Each vault can have
     * up to 10 tags. If your request would cause the tag limit for the vault to be exceeded, the operation throws the
     * <code>LimitExceededException</code> error. If a tag already exists on the vault under a specified key, the
     * existing key value will be overwritten. For more information about tags, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/tagging.html">Tagging Amazon S3 Glacier Resources</a>.
     * </p>
     *
     * @param addTagsToVaultRequest
     *        The input values for <code>AddTagsToVault</code>.
     * @return Result of the AddTagsToVault operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws LimitExceededException
     *         Returned if the request results in a vault or account limit being exceeded.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.AddTagsToVault
     */
    @Override
    public AddTagsToVaultResponse addTagsToVault(AddTagsToVaultRequest addTagsToVaultRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ResourceNotFoundException,
            LimitExceededException, ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsToVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsToVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addTagsToVaultRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsToVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTagsToVault");

            return clientHandler.execute(new ClientExecutionParams<AddTagsToVaultRequest, AddTagsToVaultResponse>()
                    .withOperationName("AddTagsToVault").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(addTagsToVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddTagsToVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You call this operation to inform Amazon S3 Glacier (Glacier) that all the archive parts have been uploaded and
     * that Glacier can now assemble the archive from the uploaded parts. After assembling and saving the archive to the
     * vault, Glacier returns the URI path of the newly created archive resource. Using the URI path, you can then
     * access the archive. After you upload an archive, you should save the archive ID returned to retrieve the archive
     * at a later point. You can also get the vault inventory to obtain a list of archive IDs in a vault. For more
     * information, see <a>InitiateJob</a>.
     * </p>
     * <p>
     * In the request, you must include the computed SHA256 tree hash of the entire archive you have uploaded. For
     * information about computing a SHA256 tree hash, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/checksum-calculations.html">Computing Checksums</a>.
     * On the server side, Glacier also constructs the SHA256 tree hash of the assembled archive. If the values match,
     * Glacier saves the archive to the vault; otherwise, it returns an error, and the operation fails. The
     * <a>ListParts</a> operation returns a list of parts uploaded for a specific multipart upload. It includes checksum
     * information for each uploaded part that can be used to debug a bad checksum issue.
     * </p>
     * <p>
     * Additionally, Glacier also checks for any missing content ranges when assembling the archive, if missing content
     * ranges are found, Glacier returns an error and the operation fails.
     * </p>
     * <p>
     * Complete Multipart Upload is an idempotent operation. After your first successful complete multipart upload, if
     * you call the operation again within a short period, the operation will succeed and return the same archive ID.
     * This is useful in the event you experience a network issue that causes an aborted connection or receive a 500
     * server error, in which case you can repeat your Complete Multipart Upload request and get the same archive ID
     * without creating duplicate archives. Note, however, that after the multipart upload completes, you cannot call
     * the List Parts operation and the multipart upload will not appear in List Multipart Uploads response, even if
     * idempotent complete is possible.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/uploading-archive-mpu.html">Uploading Large Archives
     * in Parts (Multipart Upload)</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html">Complete Multipart
     * Upload</a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param completeMultipartUploadRequest
     *        Provides options to complete a multipart upload operation. This informs Amazon Glacier that all the
     *        archive parts have been uploaded and Amazon S3 Glacier (Glacier) can now assemble the archive from the
     *        uploaded parts. After assembling and saving the archive to the vault, Glacier returns the URI path of the
     *        newly created archive resource.
     * @return Result of the CompleteMultipartUpload operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.CompleteMultipartUpload
     */
    @Override
    public CompleteMultipartUploadResponse completeMultipartUpload(CompleteMultipartUploadRequest completeMultipartUploadRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CompleteMultipartUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CompleteMultipartUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeMultipartUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeMultipartUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteMultipartUpload");

            return clientHandler
                    .execute(new ClientExecutionParams<CompleteMultipartUploadRequest, CompleteMultipartUploadResponse>()
                            .withOperationName("CompleteMultipartUpload").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(completeMultipartUploadRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CompleteMultipartUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation completes the vault locking process by transitioning the vault lock from the
     * <code>InProgress</code> state to the <code>Locked</code> state, which causes the vault lock policy to become
     * unchangeable. A vault lock is put into the <code>InProgress</code> state by calling <a>InitiateVaultLock</a>. You
     * can obtain the state of the vault lock by calling <a>GetVaultLock</a>. For more information about the vault
     * locking process, <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock.html">Amazon Glacier
     * Vault Lock</a>.
     * </p>
     * <p>
     * This operation is idempotent. This request is always successful if the vault lock is in the <code>Locked</code>
     * state and the provided lock ID matches the lock ID originally used to lock the vault.
     * </p>
     * <p>
     * If an invalid lock ID is passed in the request when the vault lock is in the <code>Locked</code> state, the
     * operation returns an <code>AccessDeniedException</code> error. If an invalid lock ID is passed in the request
     * when the vault lock is in the <code>InProgress</code> state, the operation throws an
     * <code>InvalidParameter</code> error.
     * </p>
     *
     * @param completeVaultLockRequest
     *        The input values for <code>CompleteVaultLock</code>.
     * @return Result of the CompleteVaultLock operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.CompleteVaultLock
     */
    @Override
    public CompleteVaultLockResponse completeVaultLock(CompleteVaultLockRequest completeVaultLockRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CompleteVaultLockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CompleteVaultLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeVaultLockRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeVaultLockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteVaultLock");

            return clientHandler.execute(new ClientExecutionParams<CompleteVaultLockRequest, CompleteVaultLockResponse>()
                    .withOperationName("CompleteVaultLock").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(completeVaultLockRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CompleteVaultLockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation creates a new vault with the specified name. The name of the vault must be unique within a region
     * for an AWS account. You can create up to 1,000 vaults per account. If you need to create more vaults, contact
     * Amazon S3 Glacier.
     * </p>
     * <p>
     * You must use the following guidelines when naming a vault.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Names can be between 1 and 255 characters long.
     * </p>
     * </li>
     * <li>
     * <p>
     * Allowed characters are a-z, A-Z, 0-9, '_' (underscore), '-' (hyphen), and '.' (period).
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation is idempotent.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/creating-vaults.html">Creating a Vault in Amazon
     * Glacier</a> and <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html">Create Vault
     * </a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param createVaultRequest
     *        Provides options to create a vault.
     * @return Result of the CreateVault operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws LimitExceededException
     *         Returned if the request results in a vault or account limit being exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.CreateVault
     */
    @Override
    public CreateVaultResponse createVault(CreateVaultRequest createVaultRequest) throws InvalidParameterValueException,
            MissingParameterValueException, ServiceUnavailableException, LimitExceededException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVaultRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVault");

            return clientHandler.execute(new ClientExecutionParams<CreateVaultRequest, CreateVaultResponse>()
                    .withOperationName("CreateVault").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createVaultRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes an archive from a vault. Subsequent requests to initiate a retrieval of this archive will
     * fail. Archive retrievals that are in progress for this archive ID may or may not succeed according to the
     * following scenarios:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the archive retrieval job is actively preparing the data for download when Amazon S3 Glacier receives the
     * delete archive request, the archival retrieval operation might fail.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the archive retrieval job has successfully prepared the archive for download when Amazon S3 Glacier receives
     * the delete archive request, you will be able to download the output.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation is idempotent. Attempting to delete an already-deleted archive does not result in an error.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/deleting-an-archive.html">Deleting an Archive in
     * Amazon Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html">Delete Archive</a> in the
     * <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param deleteArchiveRequest
     *        Provides options for deleting an archive from an Amazon S3 Glacier vault.
     * @return Result of the DeleteArchive operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.DeleteArchive
     */
    @Override
    public DeleteArchiveResponse deleteArchive(DeleteArchiveRequest deleteArchiveRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteArchiveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteArchiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteArchiveRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteArchiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteArchive");

            return clientHandler.execute(new ClientExecutionParams<DeleteArchiveRequest, DeleteArchiveResponse>()
                    .withOperationName("DeleteArchive").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteArchiveRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteArchiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes a vault. Amazon S3 Glacier will delete a vault only if there are no archives in the vault
     * as of the last inventory and there have been no writes to the vault since the last inventory. If either of these
     * conditions is not satisfied, the vault deletion fails (that is, the vault is not removed) and Amazon S3 Glacier
     * returns an error. You can use <a>DescribeVault</a> to return the number of archives in a vault, and you can use
     * <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html">Initiate a Job (POST
     * jobs)</a> to initiate a new inventory retrieval for a vault. The inventory contains the archive IDs you use to
     * delete archives using <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html">Delete Archive (DELETE
     * archive)</a>.
     * </p>
     * <p>
     * This operation is idempotent.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/deleting-vaults.html">Deleting a Vault in Amazon
     * Glacier</a> and <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html">Delete Vault
     * </a> in the <i>Amazon S3 Glacier Developer Guide</i>.
     * </p>
     *
     * @param deleteVaultRequest
     *        Provides options for deleting a vault from Amazon S3 Glacier.
     * @return Result of the DeleteVault operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.DeleteVault
     */
    @Override
    public DeleteVaultResponse deleteVault(DeleteVaultRequest deleteVaultRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVaultRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVault");

            return clientHandler.execute(new ClientExecutionParams<DeleteVaultRequest, DeleteVaultResponse>()
                    .withOperationName("DeleteVault").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteVaultRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes the access policy associated with the specified vault. The operation is eventually
     * consistent; that is, it might take some time for Amazon S3 Glacier to completely remove the access policy, and
     * you might still see the effect of the policy for a short time after you send the delete request.
     * </p>
     * <p>
     * This operation is idempotent. You can invoke delete multiple times, even if there is no policy associated with
     * the vault. For more information about vault access policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-access-policy.html">Amazon Glacier Access
     * Control with Vault Access Policies</a>.
     * </p>
     *
     * @param deleteVaultAccessPolicyRequest
     *        DeleteVaultAccessPolicy input.
     * @return Result of the DeleteVaultAccessPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.DeleteVaultAccessPolicy
     */
    @Override
    public DeleteVaultAccessPolicyResponse deleteVaultAccessPolicy(DeleteVaultAccessPolicyRequest deleteVaultAccessPolicyRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVaultAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVaultAccessPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVaultAccessPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVaultAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVaultAccessPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteVaultAccessPolicyRequest, DeleteVaultAccessPolicyResponse>()
                            .withOperationName("DeleteVaultAccessPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteVaultAccessPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteVaultAccessPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes the notification configuration set for a vault. The operation is eventually consistent;
     * that is, it might take some time for Amazon S3 Glacier to completely disable the notifications and you might
     * still receive some notifications for a short time after you send the delete request.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/configuring-notifications.html">Configuring Vault
     * Notifications in Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html">Delete Vault
     * Notification Configuration </a> in the Amazon S3 Glacier Developer Guide.
     * </p>
     *
     * @param deleteVaultNotificationsRequest
     *        Provides options for deleting a vault notification configuration from an Amazon Glacier vault.
     * @return Result of the DeleteVaultNotifications operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.DeleteVaultNotifications
     */
    @Override
    public DeleteVaultNotificationsResponse deleteVaultNotifications(
            DeleteVaultNotificationsRequest deleteVaultNotificationsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVaultNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVaultNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVaultNotificationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVaultNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVaultNotifications");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteVaultNotificationsRequest, DeleteVaultNotificationsResponse>()
                            .withOperationName("DeleteVaultNotifications").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteVaultNotificationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteVaultNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns information about a job you previously initiated, including the job initiation date, the
     * user who initiated the job, the job status code/message and the Amazon SNS topic to notify after Amazon S3
     * Glacier (Glacier) completes the job. For more information about initiating a job, see <a>InitiateJob</a>.
     * </p>
     * <note>
     * <p>
     * This operation enables you to check the status of your job. However, it is strongly recommended that you set up
     * an Amazon SNS topic and specify it in your initiate job request so that Glacier can notify the topic after it
     * completes the job.
     * </p>
     * </note>
     * <p>
     * A job ID will not expire for at least 24 hours after Glacier completes the job.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For more information about using this operation, see the documentation for the underlying REST API <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html">Describe Job</a> in the
     * <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param describeJobRequest
     *        Provides options for retrieving a job description.
     * @return Result of the DescribeJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.DescribeJob
     */
    @Override
    public DescribeJobResponse describeJob(DescribeJobRequest describeJobRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                    .withOperationName("DescribeJob").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeJobRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns information about a vault, including the vault's Amazon Resource Name (ARN), the date the
     * vault was created, the number of archives it contains, and the total size of all the archives in the vault. The
     * number of archives and their total size are as of the last inventory generation. This means that if you add or
     * remove an archive from a vault, and then immediately use Describe Vault, the change in contents will not be
     * immediately reflected. If you want to retrieve the latest inventory of the vault, use <a>InitiateJob</a>. Amazon
     * S3 Glacier generates vault inventories approximately daily. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-inventory.html">Downloading a Vault Inventory in
     * Amazon S3 Glacier</a>.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/retrieving-vault-info.html">Retrieving Vault Metadata
     * in Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get.html">Describe Vault </a> in the
     * <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param describeVaultRequest
     *        Provides options for retrieving metadata for a specific vault in Amazon Glacier.
     * @return Result of the DescribeVault operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.DescribeVault
     */
    @Override
    public DescribeVaultResponse describeVault(DescribeVaultRequest describeVaultRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVaultRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVault");

            return clientHandler.execute(new ClientExecutionParams<DescribeVaultRequest, DescribeVaultResponse>()
                    .withOperationName("DescribeVault").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns the current data retrieval policy for the account and region specified in the GET request.
     * For more information about data retrieval policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/data-retrieval-policy.html">Amazon Glacier Data
     * Retrieval Policies</a>.
     * </p>
     *
     * @param getDataRetrievalPolicyRequest
     *        Input for GetDataRetrievalPolicy.
     * @return Result of the GetDataRetrievalPolicy operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.GetDataRetrievalPolicy
     */
    @Override
    public GetDataRetrievalPolicyResponse getDataRetrievalPolicy(GetDataRetrievalPolicyRequest getDataRetrievalPolicyRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataRetrievalPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataRetrievalPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataRetrievalPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataRetrievalPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataRetrievalPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataRetrievalPolicyRequest, GetDataRetrievalPolicyResponse>()
                            .withOperationName("GetDataRetrievalPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDataRetrievalPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataRetrievalPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation downloads the output of the job you initiated using <a>InitiateJob</a>. Depending on the job type
     * you specified when you initiated the job, the output will be either the content of an archive or a vault
     * inventory.
     * </p>
     * <p>
     * You can download all the job output or download a portion of the output by specifying a byte range. In the case
     * of an archive retrieval job, depending on the byte range you specify, Amazon S3 Glacier (Glacier) returns the
     * checksum for the portion of the data. You can compute the checksum on the client and verify that the values match
     * to ensure the portion you downloaded is the correct data.
     * </p>
     * <p>
     * A job ID will not expire for at least 24 hours after Glacier completes the job. That a byte range. For both
     * archive and inventory retrieval jobs, you should verify the downloaded size against the size returned in the
     * headers from the <b>Get Job Output</b> response.
     * </p>
     * <p>
     * For archive retrieval jobs, you should also verify that the size is what you expected. If you download a portion
     * of the output, the expected size is based on the range of bytes you specified. For example, if you specify a
     * range of <code>bytes=0-1048575</code>, you should verify your download size is 1,048,576 bytes. If you download
     * an entire archive, the expected size is the size of the archive when you uploaded it to Amazon S3 Glacier The
     * expected size is also returned in the headers from the <b>Get Job Output</b> response.
     * </p>
     * <p>
     * In the case of an archive retrieval job, depending on the byte range you specify, Glacier returns the checksum
     * for the portion of the data. To ensure the portion you downloaded is the correct data, compute the checksum on
     * the client, verify that the values match, and verify that the size is what you expected.
     * </p>
     * <p>
     * A job ID does not expire for at least 24 hours after Glacier completes the job. That is, you can download the job
     * output within the 24 hours period after Amazon Glacier completes the job.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and the underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-inventory.html">Downloading a Vault
     * Inventory</a>, <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/downloading-an-archive.html">Downloading an
     * Archive</a>, and <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html">Get Job
     * Output </a>
     * </p>
     *
     * @param getJobOutputRequest
     *        Provides options for downloading output of an Amazon S3 Glacier job.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetJobOutputResponse
     *        and an InputStream to the response content are provided as parameters to the callback. The callback may
     *        return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The job data, either archive data or inventory data.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.GetJobOutput
     */
    @Override
    public <ReturnT> ReturnT getJobOutput(GetJobOutputRequest getJobOutputRequest,
            ResponseTransformer<GetJobOutputResponse, ReturnT> responseTransformer) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetJobOutputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobOutputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobOutputRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobOutput");

            return clientHandler.execute(
                    new ClientExecutionParams<GetJobOutputRequest, GetJobOutputResponse>().withOperationName("GetJobOutput")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getJobOutputRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetJobOutputRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation retrieves the <code>access-policy</code> subresource set on the vault; for more information on
     * setting this subresource, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html">Set Vault Access Policy
     * (PUT access-policy)</a>. If there is no access policy set on the vault, the operation returns a
     * <code>404 Not found</code> error. For more information about vault access policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-access-policy.html">Amazon Glacier Access
     * Control with Vault Access Policies</a>.
     * </p>
     *
     * @param getVaultAccessPolicyRequest
     *        Input for GetVaultAccessPolicy.
     * @return Result of the GetVaultAccessPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.GetVaultAccessPolicy
     */
    @Override
    public GetVaultAccessPolicyResponse getVaultAccessPolicy(GetVaultAccessPolicyRequest getVaultAccessPolicyRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVaultAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetVaultAccessPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVaultAccessPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVaultAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVaultAccessPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetVaultAccessPolicyRequest, GetVaultAccessPolicyResponse>()
                    .withOperationName("GetVaultAccessPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVaultAccessPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVaultAccessPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation retrieves the following attributes from the <code>lock-policy</code> subresource set on the
     * specified vault:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The vault lock policy set on the vault.
     * </p>
     * </li>
     * <li>
     * <p>
     * The state of the vault lock, which is either <code>InProgess</code> or <code>Locked</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * When the lock ID expires. The lock ID is used to complete the vault locking process.
     * </p>
     * </li>
     * <li>
     * <p>
     * When the vault lock was initiated and put into the <code>InProgress</code> state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A vault lock is put into the <code>InProgress</code> state by calling <a>InitiateVaultLock</a>. A vault lock is
     * put into the <code>Locked</code> state by calling <a>CompleteVaultLock</a>. You can abort the vault locking
     * process by calling <a>AbortVaultLock</a>. For more information about the vault locking process, <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock.html">Amazon Glacier Vault Lock</a>.
     * </p>
     * <p>
     * If there is no vault lock policy set on the vault, the operation returns a <code>404 Not found</code> error. For
     * more information about vault lock policies, <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock-policy.html">Amazon Glacier Access Control
     * with Vault Lock Policies</a>.
     * </p>
     *
     * @param getVaultLockRequest
     *        The input values for <code>GetVaultLock</code>.
     * @return Result of the GetVaultLock operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.GetVaultLock
     */
    @Override
    public GetVaultLockResponse getVaultLock(GetVaultLockRequest getVaultLockRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVaultLockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetVaultLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVaultLockRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVaultLockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVaultLock");

            return clientHandler.execute(new ClientExecutionParams<GetVaultLockRequest, GetVaultLockResponse>()
                    .withOperationName("GetVaultLock").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVaultLockRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVaultLockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation retrieves the <code>notification-configuration</code> subresource of the specified vault.
     * </p>
     * <p>
     * For information about setting a notification configuration on a vault, see <a>SetVaultNotifications</a>. If a
     * notification configuration for a vault is not set, the operation returns a <code>404 Not Found</code> error. For
     * more information about vault notifications, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/configuring-notifications.html">Configuring Vault
     * Notifications in Amazon S3 Glacier</a>.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/configuring-notifications.html">Configuring Vault
     * Notifications in Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html">Get Vault
     * Notification Configuration </a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param getVaultNotificationsRequest
     *        Provides options for retrieving the notification configuration set on an Amazon Glacier vault.
     * @return Result of the GetVaultNotifications operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.GetVaultNotifications
     */
    @Override
    public GetVaultNotificationsResponse getVaultNotifications(GetVaultNotificationsRequest getVaultNotificationsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVaultNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetVaultNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVaultNotificationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVaultNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVaultNotifications");

            return clientHandler.execute(new ClientExecutionParams<GetVaultNotificationsRequest, GetVaultNotificationsResponse>()
                    .withOperationName("GetVaultNotifications").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVaultNotificationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVaultNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation initiates a job of the specified type, which can be a select, an archival retrieval, or a vault
     * retrieval. For more information about using this operation, see the documentation for the underlying REST API <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html">Initiate a Job</a>.
     * </p>
     *
     * @param initiateJobRequest
     *        Provides options for initiating an Amazon S3 Glacier job.
     * @return Result of the InitiateJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws PolicyEnforcedException
     *         Returned if a retrieval job would exceed the current data policy's retrieval rate limit. For more
     *         information about data retrieval policies,
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws InsufficientCapacityException
     *         Returned if there is insufficient capacity to process this expedited request. This error only applies to
     *         expedited retrievals and not to standard or bulk retrievals.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.InitiateJob
     */
    @Override
    public InitiateJobResponse initiateJob(InitiateJobRequest initiateJobRequest) throws ResourceNotFoundException,
            PolicyEnforcedException, InvalidParameterValueException, MissingParameterValueException,
            InsufficientCapacityException, ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InitiateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateJob");

            return clientHandler.execute(new ClientExecutionParams<InitiateJobRequest, InitiateJobResponse>()
                    .withOperationName("InitiateJob").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(initiateJobRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitiateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation initiates a multipart upload. Amazon S3 Glacier creates a multipart upload resource and returns
     * its ID in the response. The multipart upload ID is used in subsequent requests to upload parts of an archive (see
     * <a>UploadMultipartPart</a>).
     * </p>
     * <p>
     * When you initiate a multipart upload, you specify the part size in number of bytes. The part size must be a
     * megabyte (1024 KB) multiplied by a power of 2-for example, 1048576 (1 MB), 2097152 (2 MB), 4194304 (4 MB),
     * 8388608 (8 MB), and so on. The minimum allowable part size is 1 MB, and the maximum is 4 GB.
     * </p>
     * <p>
     * Every part you upload to this resource (see <a>UploadMultipartPart</a>), except the last one, must have the same
     * size. The last one can be the same size or smaller. For example, suppose you want to upload a 16.2 MB file. If
     * you initiate the multipart upload with a part size of 4 MB, you will upload four parts of 4 MB each and one part
     * of 0.2 MB.
     * </p>
     * <note>
     * <p>
     * You don't need to know the size of the archive when you start a multipart upload because Amazon S3 Glacier does
     * not require you to specify the overall archive size.
     * </p>
     * </note>
     * <p>
     * After you complete the multipart upload, Amazon S3 Glacier (Glacier) removes the multipart upload resource
     * referenced by the ID. Glacier also removes the multipart upload resource if you cancel the multipart upload or it
     * may be removed if there is no activity for a period of 24 hours.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/uploading-archive-mpu.html">Uploading Large Archives
     * in Parts (Multipart Upload)</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html">Initiate Multipart
     * Upload</a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param initiateMultipartUploadRequest
     *        Provides options for initiating a multipart upload to an Amazon S3 Glacier vault.
     * @return Result of the InitiateMultipartUpload operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.InitiateMultipartUpload
     */
    @Override
    public InitiateMultipartUploadResponse initiateMultipartUpload(InitiateMultipartUploadRequest initiateMultipartUploadRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateMultipartUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InitiateMultipartUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateMultipartUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateMultipartUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateMultipartUpload");

            return clientHandler
                    .execute(new ClientExecutionParams<InitiateMultipartUploadRequest, InitiateMultipartUploadResponse>()
                            .withOperationName("InitiateMultipartUpload").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(initiateMultipartUploadRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new InitiateMultipartUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation initiates the vault locking process by doing the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Installing a vault lock policy on the specified vault.
     * </p>
     * </li>
     * <li>
     * <p>
     * Setting the lock state of vault lock to <code>InProgress</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returning a lock ID, which is used to complete the vault locking process.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can set one vault lock policy for each vault and this policy can be up to 20 KB in size. For more information
     * about vault lock policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock-policy.html">Amazon Glacier Access Control
     * with Vault Lock Policies</a>.
     * </p>
     * <p>
     * You must complete the vault locking process within 24 hours after the vault lock enters the
     * <code>InProgress</code> state. After the 24 hour window ends, the lock ID expires, the vault automatically exits
     * the <code>InProgress</code> state, and the vault lock policy is removed from the vault. You call
     * <a>CompleteVaultLock</a> to complete the vault locking process by setting the state of the vault lock to
     * <code>Locked</code>.
     * </p>
     * <p>
     * After a vault lock is in the <code>Locked</code> state, you cannot initiate a new vault lock for the vault.
     * </p>
     * <p>
     * You can abort the vault locking process by calling <a>AbortVaultLock</a>. You can get the state of the vault lock
     * by calling <a>GetVaultLock</a>. For more information about the vault locking process, <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-lock.html">Amazon Glacier Vault Lock</a>.
     * </p>
     * <p>
     * If this operation is called when the vault lock is in the <code>InProgress</code> state, the operation returns an
     * <code>AccessDeniedException</code> error. When the vault lock is in the <code>InProgress</code> state you must
     * call <a>AbortVaultLock</a> before you can initiate a new vault lock policy.
     * </p>
     *
     * @param initiateVaultLockRequest
     *        The input values for <code>InitiateVaultLock</code>.
     * @return Result of the InitiateVaultLock operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.InitiateVaultLock
     */
    @Override
    public InitiateVaultLockResponse initiateVaultLock(InitiateVaultLockRequest initiateVaultLockRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateVaultLockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InitiateVaultLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateVaultLockRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateVaultLockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateVaultLock");

            return clientHandler.execute(new ClientExecutionParams<InitiateVaultLockRequest, InitiateVaultLockResponse>()
                    .withOperationName("InitiateVaultLock").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(initiateVaultLockRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitiateVaultLockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists jobs for a vault, including jobs that are in-progress and jobs that have recently finished.
     * The List Job operation returns a list of these jobs sorted by job initiation time.
     * </p>
     * <note>
     * <p>
     * Amazon Glacier retains recently completed jobs for a period before deleting them; however, it eventually removes
     * completed jobs. The output of completed jobs can be retrieved. Retaining completed jobs for a period of time
     * after they have completed enables you to get a job output in the event you miss the job completion notification
     * or your first attempt to download it fails. For example, suppose you start an archive retrieval job to download
     * an archive. After the job completes, you start to download the archive but encounter a network error. In this
     * scenario, you can retry and download the archive while the job exists.
     * </p>
     * </note>
     * <p>
     * The List Jobs operation supports pagination. You should always check the response <code>Marker</code> field. If
     * there are no more jobs to list, the <code>Marker</code> field is set to <code>null</code>. If there are more jobs
     * to list, the <code>Marker</code> field is set to a non-null value, which you can use to continue the pagination
     * of the list. To return a list of jobs that begins at a specific job, set the marker request parameter to the
     * <code>Marker</code> value for that job that you obtained from a previous List Jobs request.
     * </p>
     * <p>
     * You can set a maximum limit for the number of jobs returned in the response by specifying the <code>limit</code>
     * parameter in the request. The default limit is 50. The number of jobs returned might be fewer than the limit, but
     * the number of returned jobs never exceeds the limit.
     * </p>
     * <p>
     * Additionally, you can filter the jobs list returned by specifying the optional <code>statuscode</code> parameter
     * or <code>completed</code> parameter, or both. Using the <code>statuscode</code> parameter, you can specify to
     * return only jobs that match either the <code>InProgress</code>, <code>Succeeded</code>, or <code>Failed</code>
     * status. Using the <code>completed</code> parameter, you can specify to return only jobs that were completed (
     * <code>true</code>) or jobs that were not completed (<code>false</code>).
     * </p>
     * <p>
     * For more information about using this operation, see the documentation for the underlying REST API <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html">List Jobs</a>.
     * </p>
     *
     * @param listJobsRequest
     *        Provides options for retrieving a job list for an Amazon S3 Glacier vault.
     * @return Result of the ListJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.ListJobs
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                    .withOperationName("ListJobs").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listJobsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists in-progress multipart uploads for the specified vault. An in-progress multipart upload is a
     * multipart upload that has been initiated by an <a>InitiateMultipartUpload</a> request, but has not yet been
     * completed or aborted. The list returned in the List Multipart Upload response has no guaranteed order.
     * </p>
     * <p>
     * The List Multipart Uploads operation supports pagination. By default, this operation returns up to 50 multipart
     * uploads in the response. You should always check the response for a <code>marker</code> at which to continue the
     * list; if there are no more items the <code>marker</code> is <code>null</code>. To return a list of multipart
     * uploads that begins at a specific upload, set the <code>marker</code> request parameter to the value you obtained
     * from a previous List Multipart Upload request. You can also limit the number of uploads returned in the response
     * by specifying the <code>limit</code> parameter in the request.
     * </p>
     * <p>
     * Note the difference between this operation and listing parts (<a>ListParts</a>). The List Multipart Uploads
     * operation lists all multipart uploads for a vault and does not require a multipart upload ID. The List Parts
     * operation requires a multipart upload ID since parts are associated with a single upload.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and the underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-archives.html">Working with Archives in
     * Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html">List Multipart
     * Uploads </a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param listMultipartUploadsRequest
     *        Provides options for retrieving list of in-progress multipart uploads for an Amazon Glacier vault.
     * @return Result of the ListMultipartUploads operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.ListMultipartUploads
     */
    @Override
    public ListMultipartUploadsResponse listMultipartUploads(ListMultipartUploadsRequest listMultipartUploadsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMultipartUploadsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMultipartUploadsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMultipartUploadsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMultipartUploadsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMultipartUploads");

            return clientHandler.execute(new ClientExecutionParams<ListMultipartUploadsRequest, ListMultipartUploadsResponse>()
                    .withOperationName("ListMultipartUploads").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMultipartUploadsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMultipartUploadsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists the parts of an archive that have been uploaded in a specific multipart upload. You can make
     * this request at any time during an in-progress multipart upload before you complete the upload (see
     * <a>CompleteMultipartUpload</a>. List Parts returns an error for completed uploads. The list returned in the List
     * Parts response is sorted by part range.
     * </p>
     * <p>
     * The List Parts operation supports pagination. By default, this operation returns up to 50 uploaded parts in the
     * response. You should always check the response for a <code>marker</code> at which to continue the list; if there
     * are no more items the <code>marker</code> is <code>null</code>. To return a list of parts that begins at a
     * specific part, set the <code>marker</code> request parameter to the value you obtained from a previous List Parts
     * request. You can also limit the number of parts returned in the response by specifying the <code>limit</code>
     * parameter in the request.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and the underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-archives.html">Working with Archives in
     * Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html">List Parts</a> in the
     * <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param listPartsRequest
     *        Provides options for retrieving a list of parts of an archive that have been uploaded in a specific
     *        multipart upload.
     * @return Result of the ListParts operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.ListParts
     */
    @Override
    public ListPartsResponse listParts(ListPartsRequest listPartsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPartsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPartsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPartsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPartsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParts");

            return clientHandler.execute(new ClientExecutionParams<ListPartsRequest, ListPartsResponse>()
                    .withOperationName("ListParts").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listPartsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPartsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists the provisioned capacity units for the specified AWS account.
     * </p>
     *
     * @param listProvisionedCapacityRequest
     * @return Result of the ListProvisionedCapacity operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.ListProvisionedCapacity
     */
    @Override
    public ListProvisionedCapacityResponse listProvisionedCapacity(ListProvisionedCapacityRequest listProvisionedCapacityRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProvisionedCapacityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProvisionedCapacityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProvisionedCapacityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProvisionedCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProvisionedCapacity");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProvisionedCapacityRequest, ListProvisionedCapacityResponse>()
                            .withOperationName("ListProvisionedCapacity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProvisionedCapacityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProvisionedCapacityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists all the tags attached to a vault. The operation returns an empty map if there are no tags.
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/tagging.html">Tagging Amazon S3 Glacier Resources</a>.
     * </p>
     *
     * @param listTagsForVaultRequest
     *        The input value for <code>ListTagsForVaultInput</code>.
     * @return Result of the ListTagsForVault operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.ListTagsForVault
     */
    @Override
    public ListTagsForVaultResponse listTagsForVault(ListTagsForVaultRequest listTagsForVaultRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ResourceNotFoundException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsForVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForVaultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForVault");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForVaultRequest, ListTagsForVaultResponse>()
                    .withOperationName("ListTagsForVault").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists all vaults owned by the calling user's account. The list returned in the response is
     * ASCII-sorted by vault name.
     * </p>
     * <p>
     * By default, this operation returns up to 10 items. If there are more vaults to list, the response
     * <code>marker</code> field contains the vault Amazon Resource Name (ARN) at which to continue the list with a new
     * List Vaults request; otherwise, the <code>marker</code> field is <code>null</code>. To return a list of vaults
     * that begins at a specific vault, set the <code>marker</code> request parameter to the vault ARN you obtained from
     * a previous List Vaults request. You can also limit the number of vaults returned in the response by specifying
     * the <code>limit</code> parameter in the request.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/retrieving-vault-info.html">Retrieving Vault Metadata
     * in Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html">List Vaults </a> in the <i>Amazon
     * Glacier Developer Guide</i>.
     * </p>
     *
     * @param listVaultsRequest
     *        Provides options to retrieve the vault list owned by the calling user's account. The list provides
     *        metadata information for each vault.
     * @return Result of the ListVaults operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.ListVaults
     */
    @Override
    public ListVaultsResponse listVaults(ListVaultsRequest listVaultsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVaultsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVaultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVaultsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVaultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVaults");

            return clientHandler.execute(new ClientExecutionParams<ListVaultsRequest, ListVaultsResponse>()
                    .withOperationName("ListVaults").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listVaultsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVaultsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation purchases a provisioned capacity unit for an AWS account.
     * </p>
     *
     * @param purchaseProvisionedCapacityRequest
     * @return Result of the PurchaseProvisionedCapacity operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws LimitExceededException
     *         Returned if the request results in a vault or account limit being exceeded.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.PurchaseProvisionedCapacity
     */
    @Override
    public PurchaseProvisionedCapacityResponse purchaseProvisionedCapacity(
            PurchaseProvisionedCapacityRequest purchaseProvisionedCapacityRequest) throws InvalidParameterValueException,
            MissingParameterValueException, LimitExceededException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PurchaseProvisionedCapacityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PurchaseProvisionedCapacityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(purchaseProvisionedCapacityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, purchaseProvisionedCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PurchaseProvisionedCapacity");

            return clientHandler
                    .execute(new ClientExecutionParams<PurchaseProvisionedCapacityRequest, PurchaseProvisionedCapacityResponse>()
                            .withOperationName("PurchaseProvisionedCapacity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(purchaseProvisionedCapacityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PurchaseProvisionedCapacityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation removes one or more tags from the set of tags attached to a vault. For more information about
     * tags, see <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/tagging.html">Tagging Amazon S3 Glacier
     * Resources</a>. This operation is idempotent. The operation will be successful, even if there are no tags attached
     * to the vault.
     * </p>
     *
     * @param removeTagsFromVaultRequest
     *        The input value for <code>RemoveTagsFromVaultInput</code>.
     * @return Result of the RemoveTagsFromVault operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.RemoveTagsFromVault
     */
    @Override
    public RemoveTagsFromVaultResponse removeTagsFromVault(RemoveTagsFromVaultRequest removeTagsFromVaultRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ResourceNotFoundException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsFromVaultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveTagsFromVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeTagsFromVaultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsFromVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTagsFromVault");

            return clientHandler.execute(new ClientExecutionParams<RemoveTagsFromVaultRequest, RemoveTagsFromVaultResponse>()
                    .withOperationName("RemoveTagsFromVault").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(removeTagsFromVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RemoveTagsFromVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation sets and then enacts a data retrieval policy in the region specified in the PUT request. You can
     * set one policy per region for an AWS account. The policy is enacted within a few minutes of a successful PUT
     * operation.
     * </p>
     * <p>
     * The set policy operation does not affect retrieval jobs that were in progress before the policy was enacted. For
     * more information about data retrieval policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/data-retrieval-policy.html">Amazon Glacier Data
     * Retrieval Policies</a>.
     * </p>
     *
     * @param setDataRetrievalPolicyRequest
     *        SetDataRetrievalPolicy input.
     * @return Result of the SetDataRetrievalPolicy operation returned by the service.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.SetDataRetrievalPolicy
     */
    @Override
    public SetDataRetrievalPolicyResponse setDataRetrievalPolicy(SetDataRetrievalPolicyRequest setDataRetrievalPolicyRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetDataRetrievalPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetDataRetrievalPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setDataRetrievalPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setDataRetrievalPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetDataRetrievalPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<SetDataRetrievalPolicyRequest, SetDataRetrievalPolicyResponse>()
                            .withOperationName("SetDataRetrievalPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(setDataRetrievalPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SetDataRetrievalPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation configures an access policy for a vault and will overwrite an existing policy. To configure a
     * vault access policy, send a PUT request to the <code>access-policy</code> subresource of the vault. An access
     * policy is specific to a vault and is also called a vault subresource. You can set one access policy per vault and
     * the policy can be up to 20 KB in size. For more information about vault access policies, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/vault-access-policy.html">Amazon Glacier Access
     * Control with Vault Access Policies</a>.
     * </p>
     *
     * @param setVaultAccessPolicyRequest
     *        SetVaultAccessPolicy input.
     * @return Result of the SetVaultAccessPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.SetVaultAccessPolicy
     */
    @Override
    public SetVaultAccessPolicyResponse setVaultAccessPolicy(SetVaultAccessPolicyRequest setVaultAccessPolicyRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetVaultAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetVaultAccessPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setVaultAccessPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setVaultAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetVaultAccessPolicy");

            return clientHandler.execute(new ClientExecutionParams<SetVaultAccessPolicyRequest, SetVaultAccessPolicyResponse>()
                    .withOperationName("SetVaultAccessPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(setVaultAccessPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetVaultAccessPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation configures notifications that will be sent when specific events happen to a vault. By default, you
     * don't get any notifications.
     * </p>
     * <p>
     * To configure vault notifications, send a PUT request to the <code>notification-configuration</code> subresource
     * of the vault. The request should include a JSON document that provides an Amazon SNS topic and specific events
     * for which you want Amazon S3 Glacier to send notifications to the topic.
     * </p>
     * <p>
     * Amazon SNS topics must grant permission to the vault to be allowed to publish notifications to the topic. You can
     * configure a vault to publish a notification for the following vault events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ArchiveRetrievalCompleted</b> This event occurs when a job that was initiated for an archive retrieval is
     * completed (<a>InitiateJob</a>). The status of the completed job can be "Succeeded" or "Failed". The notification
     * sent to the SNS topic is the same output as returned from <a>DescribeJob</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InventoryRetrievalCompleted</b> This event occurs when a job that was initiated for an inventory retrieval is
     * completed (<a>InitiateJob</a>). The status of the completed job can be "Succeeded" or "Failed". The notification
     * sent to the SNS topic is the same output as returned from <a>DescribeJob</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/configuring-notifications.html">Configuring Vault
     * Notifications in Amazon S3 Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html">Set Vault
     * Notification Configuration </a> in the <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param setVaultNotificationsRequest
     *        Provides options to configure notifications that will be sent when specific events happen to a vault.
     * @return Result of the SetVaultNotifications operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.SetVaultNotifications
     */
    @Override
    public SetVaultNotificationsResponse setVaultNotifications(SetVaultNotificationsRequest setVaultNotificationsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetVaultNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetVaultNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setVaultNotificationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setVaultNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetVaultNotifications");

            return clientHandler.execute(new ClientExecutionParams<SetVaultNotificationsRequest, SetVaultNotificationsResponse>()
                    .withOperationName("SetVaultNotifications").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(setVaultNotificationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetVaultNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation adds an archive to a vault. This is a synchronous operation, and for a successful upload, your
     * data is durably persisted. Amazon S3 Glacier returns the archive ID in the <code>x-amz-archive-id</code> header
     * of the response.
     * </p>
     * <p>
     * You must use the archive ID to access your data in Amazon S3 Glacier. After you upload an archive, you should
     * save the archive ID returned so that you can retrieve or delete the archive later. Besides saving the archive ID,
     * you can also index it and give it a friendly name to allow for better searching. You can also use the optional
     * archive description field to specify how the archive is referred to in an external index of archives, such as you
     * might create in Amazon DynamoDB. You can also get the vault inventory to obtain a list of archive IDs in a vault.
     * For more information, see <a>InitiateJob</a>.
     * </p>
     * <p>
     * You must provide a SHA256 tree hash of the data you are uploading. For information about computing a SHA256 tree
     * hash, see <a href="https://docs.aws.amazon.com/amazonglacier/latest/dev/checksum-calculations.html">Computing
     * Checksums</a>.
     * </p>
     * <p>
     * You can optionally specify an archive description of up to 1,024 printable ASCII characters. You can get the
     * archive description when you either retrieve the archive or get the vault inventory. For more information, see
     * <a>InitiateJob</a>. Amazon Glacier does not interpret the description in any way. An archive description does not
     * need to be unique. You cannot use the description to retrieve or sort the archive list.
     * </p>
     * <p>
     * Archives are immutable. After you upload an archive, you cannot edit the archive or its description.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/uploading-an-archive.html">Uploading an Archive in
     * Amazon Glacier</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html">Upload Archive</a> in the
     * <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param uploadArchiveRequest
     *        Provides options to add an archive to a vault.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The data to upload.
     *        </p>
     *        '
     * @return Result of the UploadArchive operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws RequestTimeoutException
     *         Returned if, when uploading an archive, Amazon S3 Glacier times out while receiving the upload.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.UploadArchive
     */
    @Override
    public UploadArchiveResponse uploadArchive(UploadArchiveRequest uploadArchiveRequest, RequestBody requestBody)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            RequestTimeoutException, ServiceUnavailableException, AwsServiceException, SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UploadArchiveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UploadArchiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(uploadArchiveRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, uploadArchiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UploadArchive");

            return clientHandler.execute(new ClientExecutionParams<UploadArchiveRequest, UploadArchiveResponse>()
                    .withOperationName("UploadArchive")
                    .withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration)
                    .withInput(uploadArchiveRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new UploadArchiveRequestMarshaller(protocolFactory))
                                    .requestBody(requestBody).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation uploads a part of an archive. You can upload archive parts in any order. You can also upload them
     * in parallel. You can upload up to 10,000 parts for a multipart upload.
     * </p>
     * <p>
     * Amazon Glacier rejects your upload part request if any of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>SHA256 tree hash does not match</b>To ensure that part data is not corrupted in transmission, you compute a
     * SHA256 tree hash of the part and include it in your request. Upon receiving the part data, Amazon S3 Glacier also
     * computes a SHA256 tree hash. If these hash values don't match, the operation fails. For information about
     * computing a SHA256 tree hash, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/checksum-calculations.html">Computing Checksums</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Part size does not match</b>The size of each part except the last must match the size specified in the
     * corresponding <a>InitiateMultipartUpload</a> request. The size of the last part must be the same size as, or
     * smaller than, the specified size.
     * </p>
     * <note>
     * <p>
     * If you upload a part whose size is smaller than the part size you specified in your initiate multipart upload
     * request and that part is not the last part, then the upload part request will succeed. However, the subsequent
     * Complete Multipart Upload request will fail.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b>Range does not align</b>The byte range value in the request does not align with the part size specified in the
     * corresponding initiate request. For example, if you specify a part size of 4194304 bytes (4 MB), then 0 to
     * 4194303 bytes (4 MB - 1) and 4194304 (4 MB) to 8388607 (8 MB - 1) are valid part ranges. However, if you set a
     * range value of 2 MB to 6 MB, the range does not align with the part size and the upload will fail.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation is idempotent. If you upload the same part multiple times, the data included in the most recent
     * request overwrites the previously uploaded data.
     * </p>
     * <p>
     * An AWS account has full permission to perform all operations (actions). However, AWS Identity and Access
     * Management (IAM) users don't have any permissions by default. You must grant them explicit permission to perform
     * specific actions. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/using-iam-with-amazon-glacier.html">Access Control
     * Using AWS Identity and Access Management (IAM)</a>.
     * </p>
     * <p>
     * For conceptual information and underlying REST API, see <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/uploading-archive-mpu.html">Uploading Large Archives
     * in Parts (Multipart Upload)</a> and <a
     * href="https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html">Upload Part </a> in the
     * <i>Amazon Glacier Developer Guide</i>.
     * </p>
     *
     * @param uploadMultipartPartRequest
     *        Provides options to upload a part of an archive in a multipart upload operation.
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The data to upload.
     *        </p>
     *        '
     * @return Result of the UploadMultipartPart operation returned by the service.
     * @throws ResourceNotFoundException
     *         Returned if the specified resource (such as a vault, upload ID, or job ID) doesn't exist.
     * @throws InvalidParameterValueException
     *         Returned if a parameter of the request is incorrectly specified.
     * @throws MissingParameterValueException
     *         Returned if a required header or parameter is missing from the request.
     * @throws RequestTimeoutException
     *         Returned if, when uploading an archive, Amazon S3 Glacier times out while receiving the upload.
     * @throws ServiceUnavailableException
     *         Returned if the service cannot complete the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlacierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlacierClient.UploadMultipartPart
     */
    @Override
    public UploadMultipartPartResponse uploadMultipartPart(UploadMultipartPartRequest uploadMultipartPartRequest,
            RequestBody requestBody) throws ResourceNotFoundException, InvalidParameterValueException,
            MissingParameterValueException, RequestTimeoutException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, GlacierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UploadMultipartPartResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UploadMultipartPartResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(uploadMultipartPartRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, uploadMultipartPartRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Glacier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UploadMultipartPart");

            return clientHandler.execute(new ClientExecutionParams<UploadMultipartPartRequest, UploadMultipartPartResponse>()
                    .withOperationName("UploadMultipartPart")
                    .withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration)
                    .withInput(uploadMultipartPartRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new UploadMultipartPartRequestMarshaller(protocolFactory))
                                    .requestBody(requestBody).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link GlacierWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link GlacierWaiter}
     */
    @Override
    public GlacierWaiter waiter() {
        return GlacierWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        GlacierServiceClientConfigurationBuilder serviceConfigBuilder = new GlacierServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GlacierException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .customErrorCodeFieldName("code")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTimeoutException")
                                .exceptionBuilderSupplier(RequestTimeoutException::builder).httpStatusCode(408).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingParameterValueException")
                                .exceptionBuilderSupplier(MissingParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientCapacityException")
                                .exceptionBuilderSupplier(InsufficientCapacityException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyEnforcedException")
                                .exceptionBuilderSupplier(PolicyEnforcedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build());
    }

    @Override
    public final GlacierServiceClientConfiguration serviceClientConfiguration() {
        return new GlacierServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
