/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an entity that is affected by a Health event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AffectedEntity implements SdkPojo, Serializable, ToCopyableBuilder<AffectedEntity.Builder, AffectedEntity> {
    private static final SdkField<String> ENTITY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("entityArn").getter(getter(AffectedEntity::entityArn)).setter(setter(Builder::entityArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityArn").build()).build();

    private static final SdkField<String> EVENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventArn").getter(getter(AffectedEntity::eventArn)).setter(setter(Builder::eventArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventArn").build()).build();

    private static final SdkField<String> ENTITY_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("entityValue").getter(getter(AffectedEntity::entityValue)).setter(setter(Builder::entityValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityValue").build()).build();

    private static final SdkField<String> ENTITY_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("entityUrl").getter(getter(AffectedEntity::entityUrl)).setter(setter(Builder::entityUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityUrl").build()).build();

    private static final SdkField<String> AWS_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awsAccountId").getter(getter(AffectedEntity::awsAccountId)).setter(setter(Builder::awsAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsAccountId").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime").getter(getter(AffectedEntity::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusCode").getter(getter(AffectedEntity::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(AffectedEntity::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ENTITY_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("entityMetadata")
            .getter(getter(AffectedEntity::entityMetadata))
            .setter(setter(Builder::entityMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENTITY_ARN_FIELD,
            EVENT_ARN_FIELD, ENTITY_VALUE_FIELD, ENTITY_URL_FIELD, AWS_ACCOUNT_ID_FIELD, LAST_UPDATED_TIME_FIELD,
            STATUS_CODE_FIELD, TAGS_FIELD, ENTITY_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String entityArn;

    private final String eventArn;

    private final String entityValue;

    private final String entityUrl;

    private final String awsAccountId;

    private final Instant lastUpdatedTime;

    private final String statusCodeValue;

    private final Map<String, String> tags;

    private final Map<String, String> entityMetadata;

    private AffectedEntity(BuilderImpl builder) {
        this.entityArn = builder.entityArn;
        this.eventArn = builder.eventArn;
        this.entityValue = builder.entityValue;
        this.entityUrl = builder.entityUrl;
        this.awsAccountId = builder.awsAccountId;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.statusCodeValue = builder.statusCodeValue;
        this.tags = builder.tags;
        this.entityMetadata = builder.entityMetadata;
    }

    /**
     * <p>
     * The unique identifier for the entity. Format:
     * <code>arn:aws:health:<i>entity-region</i>:<i>aws-account</i>:entity/<i>entity-id</i> </code>. Example:
     * <code>arn:aws:health:us-east-1:111222333444:entity/AVh5GGT7ul1arKr1sE1K</code>
     * </p>
     * 
     * @return The unique identifier for the entity. Format:
     *         <code>arn:aws:health:<i>entity-region</i>:<i>aws-account</i>:entity/<i>entity-id</i> </code>. Example:
     *         <code>arn:aws:health:us-east-1:111222333444:entity/AVh5GGT7ul1arKr1sE1K</code>
     */
    public final String entityArn() {
        return entityArn;
    }

    /**
     * <p>
     * The unique identifier for the event. The event ARN has the
     * <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
     * format.
     * </p>
     * <p>
     * For example, an event ARN might look like the following:
     * </p>
     * <p>
     * <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
     * </p>
     * 
     * @return The unique identifier for the event. The event ARN has the
     *         <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
     *         format.</p>
     *         <p>
     *         For example, an event ARN might look like the following:
     *         </p>
     *         <p>
     *         <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
     */
    public final String eventArn() {
        return eventArn;
    }

    /**
     * <p>
     * The ID of the affected entity.
     * </p>
     * 
     * @return The ID of the affected entity.
     */
    public final String entityValue() {
        return entityValue;
    }

    /**
     * <p>
     * The URL of the affected entity.
     * </p>
     * 
     * @return The URL of the affected entity.
     */
    public final String entityUrl() {
        return entityUrl;
    }

    /**
     * <p>
     * The 12-digit Amazon Web Services account number that contains the affected entity.
     * </p>
     * 
     * @return The 12-digit Amazon Web Services account number that contains the affected entity.
     */
    public final String awsAccountId() {
        return awsAccountId;
    }

    /**
     * <p>
     * The most recent time that the entity was updated.
     * </p>
     * 
     * @return The most recent time that the entity was updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The most recent status of the entity affected by the event. The possible values are <code>IMPAIRED</code>,
     * <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link EntityStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The most recent status of the entity affected by the event. The possible values are <code>IMPAIRED</code>
     *         , <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
     * @see EntityStatusCode
     */
    public final EntityStatusCode statusCode() {
        return EntityStatusCode.fromValue(statusCodeValue);
    }

    /**
     * <p>
     * The most recent status of the entity affected by the event. The possible values are <code>IMPAIRED</code>,
     * <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link EntityStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The most recent status of the entity affected by the event. The possible values are <code>IMPAIRED</code>
     *         , <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
     * @see EntityStatusCode
     */
    public final String statusCodeAsString() {
        return statusCodeValue;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of entity tags attached to the affected entity.
     * </p>
     * <note>
     * <p>
     * Currently, the <code>tags</code> property isn't supported.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A map of entity tags attached to the affected entity.</p> <note>
     *         <p>
     *         Currently, the <code>tags</code> property isn't supported.
     *         </p>
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntityMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEntityMetadata() {
        return entityMetadata != null && !(entityMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional metadata about the affected entity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityMetadata} method.
     * </p>
     * 
     * @return Additional metadata about the affected entity.
     */
    public final Map<String, String> entityMetadata() {
        return entityMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(entityArn());
        hashCode = 31 * hashCode + Objects.hashCode(eventArn());
        hashCode = 31 * hashCode + Objects.hashCode(entityValue());
        hashCode = 31 * hashCode + Objects.hashCode(entityUrl());
        hashCode = 31 * hashCode + Objects.hashCode(awsAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEntityMetadata() ? entityMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AffectedEntity)) {
            return false;
        }
        AffectedEntity other = (AffectedEntity) obj;
        return Objects.equals(entityArn(), other.entityArn()) && Objects.equals(eventArn(), other.eventArn())
                && Objects.equals(entityValue(), other.entityValue()) && Objects.equals(entityUrl(), other.entityUrl())
                && Objects.equals(awsAccountId(), other.awsAccountId())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasEntityMetadata() == other.hasEntityMetadata()
                && Objects.equals(entityMetadata(), other.entityMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AffectedEntity").add("EntityArn", entityArn()).add("EventArn", eventArn())
                .add("EntityValue", entityValue()).add("EntityUrl", entityUrl()).add("AwsAccountId", awsAccountId())
                .add("LastUpdatedTime", lastUpdatedTime()).add("StatusCode", statusCodeAsString())
                .add("Tags", hasTags() ? tags() : null).add("EntityMetadata", hasEntityMetadata() ? entityMetadata() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "entityArn":
            return Optional.ofNullable(clazz.cast(entityArn()));
        case "eventArn":
            return Optional.ofNullable(clazz.cast(eventArn()));
        case "entityValue":
            return Optional.ofNullable(clazz.cast(entityValue()));
        case "entityUrl":
            return Optional.ofNullable(clazz.cast(entityUrl()));
        case "awsAccountId":
            return Optional.ofNullable(clazz.cast(awsAccountId()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "entityMetadata":
            return Optional.ofNullable(clazz.cast(entityMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("entityArn", ENTITY_ARN_FIELD);
        map.put("eventArn", EVENT_ARN_FIELD);
        map.put("entityValue", ENTITY_VALUE_FIELD);
        map.put("entityUrl", ENTITY_URL_FIELD);
        map.put("awsAccountId", AWS_ACCOUNT_ID_FIELD);
        map.put("lastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("statusCode", STATUS_CODE_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("entityMetadata", ENTITY_METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AffectedEntity, T> g) {
        return obj -> g.apply((AffectedEntity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AffectedEntity> {
        /**
         * <p>
         * The unique identifier for the entity. Format:
         * <code>arn:aws:health:<i>entity-region</i>:<i>aws-account</i>:entity/<i>entity-id</i> </code>. Example:
         * <code>arn:aws:health:us-east-1:111222333444:entity/AVh5GGT7ul1arKr1sE1K</code>
         * </p>
         * 
         * @param entityArn
         *        The unique identifier for the entity. Format:
         *        <code>arn:aws:health:<i>entity-region</i>:<i>aws-account</i>:entity/<i>entity-id</i> </code>. Example:
         *        <code>arn:aws:health:us-east-1:111222333444:entity/AVh5GGT7ul1arKr1sE1K</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityArn(String entityArn);

        /**
         * <p>
         * The unique identifier for the event. The event ARN has the
         * <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
         * format.
         * </p>
         * <p>
         * For example, an event ARN might look like the following:
         * </p>
         * <p>
         * <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
         * </p>
         * 
         * @param eventArn
         *        The unique identifier for the event. The event ARN has the
         *        <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
         *        format.</p>
         *        <p>
         *        For example, an event ARN might look like the following:
         *        </p>
         *        <p>
         *        <code>arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventArn(String eventArn);

        /**
         * <p>
         * The ID of the affected entity.
         * </p>
         * 
         * @param entityValue
         *        The ID of the affected entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityValue(String entityValue);

        /**
         * <p>
         * The URL of the affected entity.
         * </p>
         * 
         * @param entityUrl
         *        The URL of the affected entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityUrl(String entityUrl);

        /**
         * <p>
         * The 12-digit Amazon Web Services account number that contains the affected entity.
         * </p>
         * 
         * @param awsAccountId
         *        The 12-digit Amazon Web Services account number that contains the affected entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsAccountId(String awsAccountId);

        /**
         * <p>
         * The most recent time that the entity was updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The most recent time that the entity was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The most recent status of the entity affected by the event. The possible values are <code>IMPAIRED</code>,
         * <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
         * </p>
         * 
         * @param statusCodeValue
         *        The most recent status of the entity affected by the event. The possible values are
         *        <code>IMPAIRED</code>, <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
         * @see EntityStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityStatusCode
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * The most recent status of the entity affected by the event. The possible values are <code>IMPAIRED</code>,
         * <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
         * </p>
         * 
         * @param statusCodeValue
         *        The most recent status of the entity affected by the event. The possible values are
         *        <code>IMPAIRED</code>, <code>UNIMPAIRED</code>, and <code>UNKNOWN</code>.
         * @see EntityStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityStatusCode
         */
        Builder statusCode(EntityStatusCode statusCodeValue);

        /**
         * <p>
         * A map of entity tags attached to the affected entity.
         * </p>
         * <note>
         * <p>
         * Currently, the <code>tags</code> property isn't supported.
         * </p>
         * </note>
         * 
         * @param tags
         *        A map of entity tags attached to the affected entity.</p> <note>
         *        <p>
         *        Currently, the <code>tags</code> property isn't supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Additional metadata about the affected entity.
         * </p>
         * 
         * @param entityMetadata
         *        Additional metadata about the affected entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityMetadata(Map<String, String> entityMetadata);
    }

    static final class BuilderImpl implements Builder {
        private String entityArn;

        private String eventArn;

        private String entityValue;

        private String entityUrl;

        private String awsAccountId;

        private Instant lastUpdatedTime;

        private String statusCodeValue;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> entityMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AffectedEntity model) {
            entityArn(model.entityArn);
            eventArn(model.eventArn);
            entityValue(model.entityValue);
            entityUrl(model.entityUrl);
            awsAccountId(model.awsAccountId);
            lastUpdatedTime(model.lastUpdatedTime);
            statusCode(model.statusCodeValue);
            tags(model.tags);
            entityMetadata(model.entityMetadata);
        }

        public final String getEntityArn() {
            return entityArn;
        }

        public final void setEntityArn(String entityArn) {
            this.entityArn = entityArn;
        }

        @Override
        public final Builder entityArn(String entityArn) {
            this.entityArn = entityArn;
            return this;
        }

        public final String getEventArn() {
            return eventArn;
        }

        public final void setEventArn(String eventArn) {
            this.eventArn = eventArn;
        }

        @Override
        public final Builder eventArn(String eventArn) {
            this.eventArn = eventArn;
            return this;
        }

        public final String getEntityValue() {
            return entityValue;
        }

        public final void setEntityValue(String entityValue) {
            this.entityValue = entityValue;
        }

        @Override
        public final Builder entityValue(String entityValue) {
            this.entityValue = entityValue;
            return this;
        }

        public final String getEntityUrl() {
            return entityUrl;
        }

        public final void setEntityUrl(String entityUrl) {
            this.entityUrl = entityUrl;
        }

        @Override
        public final Builder entityUrl(String entityUrl) {
            this.entityUrl = entityUrl;
            return this;
        }

        public final String getAwsAccountId() {
            return awsAccountId;
        }

        public final void setAwsAccountId(String awsAccountId) {
            this.awsAccountId = awsAccountId;
        }

        @Override
        public final Builder awsAccountId(String awsAccountId) {
            this.awsAccountId = awsAccountId;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        @Override
        public final Builder statusCode(EntityStatusCode statusCodeValue) {
            this.statusCode(statusCodeValue == null ? null : statusCodeValue.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = _tagSetCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = _tagSetCopier.copy(tags);
            return this;
        }

        public final Map<String, String> getEntityMetadata() {
            if (entityMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return entityMetadata;
        }

        public final void setEntityMetadata(Map<String, String> entityMetadata) {
            this.entityMetadata = _entityMetadataCopier.copy(entityMetadata);
        }

        @Override
        public final Builder entityMetadata(Map<String, String> entityMetadata) {
            this.entityMetadata = _entityMetadataCopier.copy(entityMetadata);
            return this;
        }

        @Override
        public AffectedEntity build() {
            return new AffectedEntity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
