/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspaces;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.keyspaces.internal.KeyspacesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.keyspaces.model.AccessDeniedException;
import software.amazon.awssdk.services.keyspaces.model.ConflictException;
import software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.CreateTableRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateTableResponse;
import software.amazon.awssdk.services.keyspaces.model.CreateTypeRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateTypeResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteTableRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteTableResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteTypeRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteTypeResponse;
import software.amazon.awssdk.services.keyspaces.model.GetKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.GetKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.GetTableAutoScalingSettingsRequest;
import software.amazon.awssdk.services.keyspaces.model.GetTableAutoScalingSettingsResponse;
import software.amazon.awssdk.services.keyspaces.model.GetTableRequest;
import software.amazon.awssdk.services.keyspaces.model.GetTableResponse;
import software.amazon.awssdk.services.keyspaces.model.GetTypeRequest;
import software.amazon.awssdk.services.keyspaces.model.GetTypeResponse;
import software.amazon.awssdk.services.keyspaces.model.InternalServerException;
import software.amazon.awssdk.services.keyspaces.model.KeyspacesException;
import software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTablesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTablesResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTypesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTypesResponse;
import software.amazon.awssdk.services.keyspaces.model.ResourceNotFoundException;
import software.amazon.awssdk.services.keyspaces.model.RestoreTableRequest;
import software.amazon.awssdk.services.keyspaces.model.RestoreTableResponse;
import software.amazon.awssdk.services.keyspaces.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.keyspaces.model.TagResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.TagResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.UntagResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.UntagResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.UpdateKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.UpdateKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.UpdateTableRequest;
import software.amazon.awssdk.services.keyspaces.model.UpdateTableResponse;
import software.amazon.awssdk.services.keyspaces.model.ValidationException;
import software.amazon.awssdk.services.keyspaces.transform.CreateKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.CreateTypeRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.DeleteKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.DeleteTypeRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.GetKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.GetTableAutoScalingSettingsRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.GetTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.GetTypeRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListKeyspacesRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListTypesRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.RestoreTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.UpdateKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.UpdateTableRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KeyspacesAsyncClient}.
 *
 * @see KeyspacesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKeyspacesAsyncClient implements KeyspacesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKeyspacesAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKeyspacesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * The <code>CreateKeyspace</code> operation adds a new keyspace to your account. In an Amazon Web Services account,
     * keyspace names must be unique within each Region.
     * </p>
     * <p>
     * <code>CreateKeyspace</code> is an asynchronous operation. You can monitor the creation status of the new keyspace
     * by using the <code>GetKeyspace</code> operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/getting-started.keyspaces.html">Create a keyspace</a>
     * in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createKeyspaceRequest
     * @return A Java Future containing the result of the CreateKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKeyspaceResponse> createKeyspace(CreateKeyspaceRequest createKeyspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKeyspaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKeyspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateKeyspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKeyspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKeyspaceRequest, CreateKeyspaceResponse>()
                            .withOperationName("CreateKeyspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKeyspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createKeyspaceRequest));
            CompletableFuture<CreateKeyspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateTable</code> operation adds a new table to the specified keyspace. Within a keyspace, table names
     * must be unique.
     * </p>
     * <p>
     * <code>CreateTable</code> is an asynchronous operation. When the request is received, the status of the table is
     * set to <code>CREATING</code>. You can monitor the creation status of the new table by using the
     * <code>GetTable</code> operation, which returns the current <code>status</code> of the table. You can start using
     * a table when the status is <code>ACTIVE</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/getting-started.tables.html">Create a table</a> in
     * the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createTableRequest
     * @return A Java Future containing the result of the CreateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTableResponse> createTable(CreateTableRequest createTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                            .withOperationName("CreateTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTableRequest));
            CompletableFuture<CreateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateType</code> operation creates a new user-defined type in the specified keyspace.
     * </p>
     * <p>
     * To configure the required permissions, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/configure-udt-permissions.html#udt-permissions-create"
     * >Permissions to create a UDT</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/udts.html">User-defined
     * types (UDTs)</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createTypeRequest
     * @return A Java Future containing the result of the CreateType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTypeResponse> createType(CreateTypeRequest createTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTypeRequest, CreateTypeResponse>().withOperationName("CreateType")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTypeRequest));
            CompletableFuture<CreateTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteKeyspace</code> operation deletes a keyspace and all of its tables.
     * </p>
     *
     * @param deleteKeyspaceRequest
     * @return A Java Future containing the result of the DeleteKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKeyspaceResponse> deleteKeyspace(DeleteKeyspaceRequest deleteKeyspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKeyspaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKeyspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteKeyspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteKeyspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKeyspaceRequest, DeleteKeyspaceResponse>()
                            .withOperationName("DeleteKeyspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKeyspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteKeyspaceRequest));
            CompletableFuture<DeleteKeyspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteTable</code> operation deletes a table and all of its data. After a <code>DeleteTable</code>
     * request is received, the specified table is in the <code>DELETING</code> state until Amazon Keyspaces completes
     * the deletion. If the table is in the <code>ACTIVE</code> state, you can delete it. If a table is either in the
     * <code>CREATING</code> or <code>UPDATING</code> states, then Amazon Keyspaces returns a
     * <code>ResourceInUseException</code>. If the specified table does not exist, Amazon Keyspaces returns a
     * <code>ResourceNotFoundException</code>. If the table is already in the <code>DELETING</code> state, no error is
     * returned.
     * </p>
     *
     * @param deleteTableRequest
     * @return A Java Future containing the result of the DeleteTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTableResponse> deleteTable(DeleteTableRequest deleteTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                            .withOperationName("DeleteTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTableRequest));
            CompletableFuture<DeleteTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteType</code> operation deletes a user-defined type (UDT). You can only delete a type that is not
     * used in a table or another UDT.
     * </p>
     * <p>
     * To configure the required permissions, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/configure-udt-permissions.html#udt-permissions-drop"
     * >Permissions to delete a UDT</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param deleteTypeRequest
     * @return A Java Future containing the result of the DeleteType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTypeResponse> deleteType(DeleteTypeRequest deleteTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTypeRequest, DeleteTypeResponse>().withOperationName("DeleteType")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTypeRequest));
            CompletableFuture<DeleteTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the name of the specified keyspace, the Amazon Resource Name (ARN), the replication strategy, the Amazon
     * Web Services Regions of a multi-Region keyspace, and the status of newly added Regions after an
     * <code>UpdateKeyspace</code> operation.
     * </p>
     *
     * @param getKeyspaceRequest
     * @return A Java Future containing the result of the GetKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetKeyspaceResponse> getKeyspace(GetKeyspaceRequest getKeyspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKeyspaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKeyspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetKeyspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKeyspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKeyspaceRequest, GetKeyspaceResponse>()
                            .withOperationName("GetKeyspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKeyspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKeyspaceRequest));
            CompletableFuture<GetKeyspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the table, including the table's name and current status, the keyspace name,
     * configuration settings, and metadata.
     * </p>
     * <p>
     * To read table metadata using <code>GetTable</code>, the IAM principal needs <code>Select</code> action
     * permissions for the table and the system keyspace.
     * </p>
     *
     * @param getTableRequest
     * @return A Java Future containing the result of the GetTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableResponse> getTable(GetTableRequest getTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableRequest, GetTableResponse>().withOperationName("GetTable")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTableRequest));
            CompletableFuture<GetTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns auto scaling related settings of the specified table in JSON format. If the table is a multi-Region
     * table, the Amazon Web Services Region specific auto scaling settings of the table are included.
     * </p>
     * <p>
     * Amazon Keyspaces auto scaling helps you provision throughput capacity for variable workloads efficiently by
     * increasing and decreasing your table's read and write capacity automatically in response to application traffic.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/autoscaling.html">Managing throughput capacity
     * automatically with Amazon Keyspaces auto scaling</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <code>GetTableAutoScalingSettings</code> can't be used as an action in an IAM policy.
     * </p>
     * </important>
     * <p>
     * To define permissions for <code>GetTableAutoScalingSettings</code>, you must allow the following two actions in
     * the IAM policy statement's <code>Action</code> element:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalableTargets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalingPolicies</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getTableAutoScalingSettingsRequest
     * @return A Java Future containing the result of the GetTableAutoScalingSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetTableAutoScalingSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTableAutoScalingSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableAutoScalingSettingsResponse> getTableAutoScalingSettings(
            GetTableAutoScalingSettingsRequest getTableAutoScalingSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableAutoScalingSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableAutoScalingSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableAutoScalingSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableAutoScalingSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableAutoScalingSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableAutoScalingSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableAutoScalingSettingsRequest, GetTableAutoScalingSettingsResponse>()
                            .withOperationName("GetTableAutoScalingSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableAutoScalingSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableAutoScalingSettingsRequest));
            CompletableFuture<GetTableAutoScalingSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetType</code> operation returns information about the type, for example the field definitions, the
     * timestamp when the type was last modified, the level of nesting, the status, and details about if the type is
     * used in other types and tables.
     * </p>
     * <p>
     * To read keyspace metadata using <code>GetType</code>, the IAM principal needs <code>Select</code> action
     * permissions for the system keyspace. To configure the required permissions, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/configure-udt-permissions.html#udt-permissions-view"
     * >Permissions to view a UDT</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param getTypeRequest
     * @return A Java Future containing the result of the GetType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTypeResponse> getType(GetTypeRequest getTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTypeRequest, GetTypeResponse>().withOperationName("GetType")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTypeRequest));
            CompletableFuture<GetTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListKeyspaces</code> operation returns a list of keyspaces.
     * </p>
     *
     * @param listKeyspacesRequest
     * @return A Java Future containing the result of the ListKeyspaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListKeyspacesResponse> listKeyspaces(ListKeyspacesRequest listKeyspacesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKeyspacesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKeyspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKeyspaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKeyspacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListKeyspacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKeyspacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKeyspacesRequest, ListKeyspacesResponse>()
                            .withOperationName("ListKeyspaces").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKeyspacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKeyspacesRequest));
            CompletableFuture<ListKeyspacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListTables</code> operation returns a list of tables for a specified keyspace.
     * </p>
     * <p>
     * To read keyspace metadata using <code>ListTables</code>, the IAM principal needs <code>Select</code> action
     * permissions for the system keyspace.
     * </p>
     *
     * @param listTablesRequest
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTablesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTablesRequest));
            CompletableFuture<ListTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all tags associated with the specified Amazon Keyspaces resource.
     * </p>
     * <p>
     * To read keyspace metadata using <code>ListTagsForResource</code>, the IAM principal needs <code>Select</code>
     * action permissions for the specified resource and the system keyspace.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListTypes</code> operation returns a list of types for a specified keyspace.
     * </p>
     * <p>
     * To read keyspace metadata using <code>ListTypes</code>, the IAM principal needs <code>Select</code> action
     * permissions for the system keyspace. To configure the required permissions, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/configure-udt-permissions.html#udt-permissions-view"
     * >Permissions to view a UDT</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param listTypesRequest
     * @return A Java Future containing the result of the ListTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTypesResponse> listTypes(ListTypesRequest listTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTypesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTypesRequest, ListTypesResponse>().withOperationName("ListTypes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTypesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTypesRequest));
            CompletableFuture<ListTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores the table to the specified point in time within the <code>earliest_restorable_timestamp</code> and the
     * current time. For more information about restore points, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_window"
     * > Time window for PITR continuous backups</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Any number of users can execute up to 4 concurrent restores (any type of restore) in a given account.
     * </p>
     * <p>
     * When you restore using point in time recovery, Amazon Keyspaces restores your source table's schema and data to
     * the state based on the selected timestamp <code>(day:hour:minute:second)</code> to a new table. The Time to Live
     * (TTL) settings are also restored to the state based on the selected timestamp.
     * </p>
     * <p>
     * In addition to the table's schema, data, and TTL settings, <code>RestoreTable</code> restores the capacity mode,
     * auto scaling settings, encryption settings, and point-in-time recovery settings from the source table. Unlike the
     * table's schema data and TTL settings, which are restored based on the selected timestamp, these settings are
     * always restored based on the table's settings as of the current time or when the table was deleted.
     * </p>
     * <p>
     * You can also overwrite these settings during restore:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Read/write capacity mode
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned throughput capacity units
     * </p>
     * </li>
     * <li>
     * <p>
     * Auto scaling settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Point-in-time (PITR) settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_settings"
     * >PITR restore settings</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Note that the following settings are not restored, and you must configure them manually for the new table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Identity and Access Management (IAM) policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableRequest
     * @return A Java Future containing the result of the RestoreTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.RestoreTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/RestoreTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreTableResponse> restoreTable(RestoreTableRequest restoreTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RestoreTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreTableRequest, RestoreTableResponse>()
                            .withOperationName("RestoreTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreTableRequest));
            CompletableFuture<RestoreTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a set of tags with a Amazon Keyspaces resource. You can then activate these user-defined tags so that
     * they appear on the Cost Management Console for cost allocation tracking. For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/tagging-keyspaces.html">Adding tags and labels to
     * Amazon Keyspaces resources</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * For IAM policy examples that show how to control access to Amazon Keyspaces resources based on tags, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-tags"
     * >Amazon Keyspaces resource access based on tags</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Amazon Keyspaces resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a new Amazon Web Services Region to the keyspace. You can add a new Region to a keyspace that is either a
     * single or a multi-Region keyspace. Amazon Keyspaces is going to replicate all tables in the keyspace to the new
     * Region. To successfully replicate all tables to the new Region, they must use client-side timestamps for conflict
     * resolution. To enable client-side timestamps, specify <code>clientSideTimestamps.status = enabled</code> when
     * invoking the API. For more information about client-side timestamps, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/client-side-timestamps.html">Client-side timestamps
     * in Amazon Keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * To add a Region to a keyspace using the <code>UpdateKeyspace</code> API, the IAM principal needs permissions for
     * the following IAM actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>cassandra:Alter</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:AlterMultiRegionResource</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:Create</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:CreateMultiRegionResource</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:Select</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:SelectMultiRegionResource</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:Modify</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:ModifyMultiRegionResource</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the keyspace contains a table that is configured in provisioned mode with auto scaling enabled, the following
     * additional IAM actions need to be allowed.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>application-autoscaling:RegisterScalableTarget</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:DeregisterScalableTarget</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalableTargets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:PutScalingPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalingPolicies</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To use the <code>UpdateKeyspace</code> API, the IAM principal also needs permissions to create a service-linked
     * role with the following elements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>iam:CreateServiceLinkedRole</code> - The <b>action</b> the principal can perform.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>arn:aws:iam::*:role/aws-service-role/replication.cassandra.amazonaws.com/AWSServiceRoleForKeyspacesReplication</code>
     * - The <b>resource</b> that the action can be performed on.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>iam:AWSServiceName: replication.cassandra.amazonaws.com</code> - The only Amazon Web Services service that
     * this role can be attached to is Amazon Keyspaces.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/howitworks_replication_permissions_addReplica.html"
     * >Configure the IAM permissions required to add an Amazon Web Services Region to a keyspace</a> in the <i>Amazon
     * Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param updateKeyspaceRequest
     * @return A Java Future containing the result of the UpdateKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UpdateKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UpdateKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKeyspaceResponse> updateKeyspace(UpdateKeyspaceRequest updateKeyspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKeyspaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKeyspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateKeyspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKeyspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKeyspaceRequest, UpdateKeyspaceResponse>()
                            .withOperationName("UpdateKeyspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKeyspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKeyspaceRequest));
            CompletableFuture<UpdateKeyspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds new columns to the table or updates one of the table's settings, for example capacity mode, auto scaling,
     * encryption, point-in-time recovery, or ttl settings. Note that you can only update one specific table setting per
     * update operation.
     * </p>
     *
     * @param updateTableRequest
     * @return A Java Future containing the result of the UpdateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace, table, or type that doesn't
     *         exist. The resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UpdateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableResponse> updateTable(UpdateTableRequest updateTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableRequest, UpdateTableResponse>()
                            .withOperationName("UpdateTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTableRequest));
            CompletableFuture<UpdateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final KeyspacesServiceClientConfiguration serviceClientConfiguration() {
        return new KeyspacesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KeyspacesException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        KeyspacesServiceClientConfigurationBuilder serviceConfigBuilder = new KeyspacesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
