/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the geofence geometry details.
 * </p>
 * <p>
 * A geofence geometry is made up of either a polygon or a circle. Can be a polygon, a circle or a polygon encoded in
 * Geobuf format. Including multiple selections will return a validation error.
 * </p>
 * <note>
 * <p>
 * Amazon Location doesn't currently support polygons with holes, multipolygons, polygons that are wound clockwise, or
 * that cross the antimeridian.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GeofenceGeometry implements SdkPojo, Serializable,
        ToCopyableBuilder<GeofenceGeometry.Builder, GeofenceGeometry> {
    private static final SdkField<List<List<List<Double>>>> POLYGON_FIELD = SdkField
            .<List<List<List<Double>>>> builder(MarshallingType.LIST)
            .memberName("Polygon")
            .getter(getter(GeofenceGeometry::polygon))
            .setter(setter(Builder::polygon))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Polygon").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<List<Double>>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<List<Double>> builder(MarshallingType.LIST)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build(),
                                                                                    ListTrait
                                                                                            .builder()
                                                                                            .memberLocationName(null)
                                                                                            .memberFieldInfo(
                                                                                                    SdkField.<Double> builder(
                                                                                                            MarshallingType.DOUBLE)
                                                                                                            .traits(LocationTrait
                                                                                                                    .builder()
                                                                                                                    .location(
                                                                                                                            MarshallLocation.PAYLOAD)
                                                                                                                    .locationName(
                                                                                                                            "member")
                                                                                                                    .build())
                                                                                                            .build()).build())
                                                                            .build()).build()).build()).build()).build();

    private static final SdkField<Circle> CIRCLE_FIELD = SdkField.<Circle> builder(MarshallingType.SDK_POJO).memberName("Circle")
            .getter(getter(GeofenceGeometry::circle)).setter(setter(Builder::circle)).constructor(Circle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Circle").build()).build();

    private static final SdkField<SdkBytes> GEOBUF_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Geobuf").getter(getter(GeofenceGeometry::geobuf)).setter(setter(Builder::geobuf))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geobuf").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLYGON_FIELD, CIRCLE_FIELD,
            GEOBUF_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<List<List<Double>>> polygon;

    private final Circle circle;

    private final SdkBytes geobuf;

    private GeofenceGeometry(BuilderImpl builder) {
        this.polygon = builder.polygon;
        this.circle = builder.circle;
        this.geobuf = builder.geobuf;
    }

    /**
     * For responses, this returns true if the service returned a value for the Polygon property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPolygon() {
        return polygon != null && !(polygon instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A polygon is a list of linear rings which are each made up of a list of vertices.
     * </p>
     * <p>
     * Each vertex is a 2-dimensional point of the form: <code>[longitude, latitude]</code>. This is represented as an
     * array of doubles of length 2 (so <code>[double, double]</code>).
     * </p>
     * <p>
     * An array of 4 or more vertices, where the first and last vertex are the same (to form a closed boundary), is
     * called a linear ring. The linear ring vertices must be listed in counter-clockwise order around the ring’s
     * interior. The linear ring is represented as an array of vertices, or an array of arrays of doubles (
     * <code>[[double, double], ...]</code>).
     * </p>
     * <p>
     * A geofence consists of a single linear ring. To allow for future expansion, the Polygon parameter takes an array
     * of linear rings, which is represented as an array of arrays of arrays of doubles (
     * <code>[[[double, double], ...], ...]</code>).
     * </p>
     * <p>
     * A linear ring for use in geofences can consist of between 4 and 1,000 vertices.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPolygon} method.
     * </p>
     * 
     * @return A polygon is a list of linear rings which are each made up of a list of vertices.</p>
     *         <p>
     *         Each vertex is a 2-dimensional point of the form: <code>[longitude, latitude]</code>. This is represented
     *         as an array of doubles of length 2 (so <code>[double, double]</code>).
     *         </p>
     *         <p>
     *         An array of 4 or more vertices, where the first and last vertex are the same (to form a closed boundary),
     *         is called a linear ring. The linear ring vertices must be listed in counter-clockwise order around the
     *         ring’s interior. The linear ring is represented as an array of vertices, or an array of arrays of doubles
     *         (<code>[[double, double], ...]</code>).
     *         </p>
     *         <p>
     *         A geofence consists of a single linear ring. To allow for future expansion, the Polygon parameter takes
     *         an array of linear rings, which is represented as an array of arrays of arrays of doubles (
     *         <code>[[[double, double], ...], ...]</code>).
     *         </p>
     *         <p>
     *         A linear ring for use in geofences can consist of between 4 and 1,000 vertices.
     */
    public final List<List<List<Double>>> polygon() {
        return polygon;
    }

    /**
     * <p>
     * A circle on the earth, as defined by a center point and a radius.
     * </p>
     * 
     * @return A circle on the earth, as defined by a center point and a radius.
     */
    public final Circle circle() {
        return circle;
    }

    /**
     * <p>
     * Geobuf is a compact binary encoding for geographic data that provides lossless compression of GeoJSON polygons.
     * The Geobuf must be Base64-encoded.
     * </p>
     * <p>
     * A polygon in Geobuf format can have up to 100,000 vertices.
     * </p>
     * 
     * @return Geobuf is a compact binary encoding for geographic data that provides lossless compression of GeoJSON
     *         polygons. The Geobuf must be Base64-encoded.</p>
     *         <p>
     *         A polygon in Geobuf format can have up to 100,000 vertices.
     */
    public final SdkBytes geobuf() {
        return geobuf;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasPolygon() ? polygon() : null);
        hashCode = 31 * hashCode + Objects.hashCode(circle());
        hashCode = 31 * hashCode + Objects.hashCode(geobuf());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GeofenceGeometry)) {
            return false;
        }
        GeofenceGeometry other = (GeofenceGeometry) obj;
        return hasPolygon() == other.hasPolygon() && Objects.equals(polygon(), other.polygon())
                && Objects.equals(circle(), other.circle()) && Objects.equals(geobuf(), other.geobuf());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GeofenceGeometry").add("Polygon", polygon() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Circle", circle() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Geobuf", geobuf() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Polygon":
            return Optional.ofNullable(clazz.cast(polygon()));
        case "Circle":
            return Optional.ofNullable(clazz.cast(circle()));
        case "Geobuf":
            return Optional.ofNullable(clazz.cast(geobuf()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Polygon", POLYGON_FIELD);
        map.put("Circle", CIRCLE_FIELD);
        map.put("Geobuf", GEOBUF_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GeofenceGeometry, T> g) {
        return obj -> g.apply((GeofenceGeometry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GeofenceGeometry> {
        /**
         * <p>
         * A polygon is a list of linear rings which are each made up of a list of vertices.
         * </p>
         * <p>
         * Each vertex is a 2-dimensional point of the form: <code>[longitude, latitude]</code>. This is represented as
         * an array of doubles of length 2 (so <code>[double, double]</code>).
         * </p>
         * <p>
         * An array of 4 or more vertices, where the first and last vertex are the same (to form a closed boundary), is
         * called a linear ring. The linear ring vertices must be listed in counter-clockwise order around the ring’s
         * interior. The linear ring is represented as an array of vertices, or an array of arrays of doubles (
         * <code>[[double, double], ...]</code>).
         * </p>
         * <p>
         * A geofence consists of a single linear ring. To allow for future expansion, the Polygon parameter takes an
         * array of linear rings, which is represented as an array of arrays of arrays of doubles (
         * <code>[[[double, double], ...], ...]</code>).
         * </p>
         * <p>
         * A linear ring for use in geofences can consist of between 4 and 1,000 vertices.
         * </p>
         * 
         * @param polygon
         *        A polygon is a list of linear rings which are each made up of a list of vertices.</p>
         *        <p>
         *        Each vertex is a 2-dimensional point of the form: <code>[longitude, latitude]</code>. This is
         *        represented as an array of doubles of length 2 (so <code>[double, double]</code>).
         *        </p>
         *        <p>
         *        An array of 4 or more vertices, where the first and last vertex are the same (to form a closed
         *        boundary), is called a linear ring. The linear ring vertices must be listed in counter-clockwise order
         *        around the ring’s interior. The linear ring is represented as an array of vertices, or an array of
         *        arrays of doubles (<code>[[double, double], ...]</code>).
         *        </p>
         *        <p>
         *        A geofence consists of a single linear ring. To allow for future expansion, the Polygon parameter
         *        takes an array of linear rings, which is represented as an array of arrays of arrays of doubles (
         *        <code>[[[double, double], ...], ...]</code>).
         *        </p>
         *        <p>
         *        A linear ring for use in geofences can consist of between 4 and 1,000 vertices.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder polygon(Collection<? extends Collection<? extends Collection<Double>>> polygon);

        /**
         * <p>
         * A polygon is a list of linear rings which are each made up of a list of vertices.
         * </p>
         * <p>
         * Each vertex is a 2-dimensional point of the form: <code>[longitude, latitude]</code>. This is represented as
         * an array of doubles of length 2 (so <code>[double, double]</code>).
         * </p>
         * <p>
         * An array of 4 or more vertices, where the first and last vertex are the same (to form a closed boundary), is
         * called a linear ring. The linear ring vertices must be listed in counter-clockwise order around the ring’s
         * interior. The linear ring is represented as an array of vertices, or an array of arrays of doubles (
         * <code>[[double, double], ...]</code>).
         * </p>
         * <p>
         * A geofence consists of a single linear ring. To allow for future expansion, the Polygon parameter takes an
         * array of linear rings, which is represented as an array of arrays of arrays of doubles (
         * <code>[[[double, double], ...], ...]</code>).
         * </p>
         * <p>
         * A linear ring for use in geofences can consist of between 4 and 1,000 vertices.
         * </p>
         * 
         * @param polygon
         *        A polygon is a list of linear rings which are each made up of a list of vertices.</p>
         *        <p>
         *        Each vertex is a 2-dimensional point of the form: <code>[longitude, latitude]</code>. This is
         *        represented as an array of doubles of length 2 (so <code>[double, double]</code>).
         *        </p>
         *        <p>
         *        An array of 4 or more vertices, where the first and last vertex are the same (to form a closed
         *        boundary), is called a linear ring. The linear ring vertices must be listed in counter-clockwise order
         *        around the ring’s interior. The linear ring is represented as an array of vertices, or an array of
         *        arrays of doubles (<code>[[double, double], ...]</code>).
         *        </p>
         *        <p>
         *        A geofence consists of a single linear ring. To allow for future expansion, the Polygon parameter
         *        takes an array of linear rings, which is represented as an array of arrays of arrays of doubles (
         *        <code>[[[double, double], ...], ...]</code>).
         *        </p>
         *        <p>
         *        A linear ring for use in geofences can consist of between 4 and 1,000 vertices.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder polygon(Collection<? extends Collection<Double>>... polygon);

        /**
         * <p>
         * A circle on the earth, as defined by a center point and a radius.
         * </p>
         * 
         * @param circle
         *        A circle on the earth, as defined by a center point and a radius.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder circle(Circle circle);

        /**
         * <p>
         * A circle on the earth, as defined by a center point and a radius.
         * </p>
         * This is a convenience method that creates an instance of the {@link Circle.Builder} avoiding the need to
         * create one manually via {@link Circle#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Circle.Builder#build()} is called immediately and its result is
         * passed to {@link #circle(Circle)}.
         * 
         * @param circle
         *        a consumer that will call methods on {@link Circle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #circle(Circle)
         */
        default Builder circle(Consumer<Circle.Builder> circle) {
            return circle(Circle.builder().applyMutation(circle).build());
        }

        /**
         * <p>
         * Geobuf is a compact binary encoding for geographic data that provides lossless compression of GeoJSON
         * polygons. The Geobuf must be Base64-encoded.
         * </p>
         * <p>
         * A polygon in Geobuf format can have up to 100,000 vertices.
         * </p>
         * 
         * @param geobuf
         *        Geobuf is a compact binary encoding for geographic data that provides lossless compression of GeoJSON
         *        polygons. The Geobuf must be Base64-encoded.</p>
         *        <p>
         *        A polygon in Geobuf format can have up to 100,000 vertices.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geobuf(SdkBytes geobuf);
    }

    static final class BuilderImpl implements Builder {
        private List<List<List<Double>>> polygon = DefaultSdkAutoConstructList.getInstance();

        private Circle circle;

        private SdkBytes geobuf;

        private BuilderImpl() {
        }

        private BuilderImpl(GeofenceGeometry model) {
            polygon(model.polygon);
            circle(model.circle);
            geobuf(model.geobuf);
        }

        public final Collection<? extends Collection<? extends Collection<Double>>> getPolygon() {
            if (polygon instanceof SdkAutoConstructList) {
                return null;
            }
            return polygon;
        }

        public final void setPolygon(Collection<? extends Collection<? extends Collection<Double>>> polygon) {
            this.polygon = LinearRingsCopier.copy(polygon);
        }

        @Override
        public final Builder polygon(Collection<? extends Collection<? extends Collection<Double>>> polygon) {
            this.polygon = LinearRingsCopier.copy(polygon);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder polygon(Collection<? extends Collection<Double>>... polygon) {
            polygon(Arrays.asList(polygon));
            return this;
        }

        public final Circle.Builder getCircle() {
            return circle != null ? circle.toBuilder() : null;
        }

        public final void setCircle(Circle.BuilderImpl circle) {
            this.circle = circle != null ? circle.build() : null;
        }

        @Override
        public final Builder circle(Circle circle) {
            this.circle = circle;
            return this;
        }

        public final ByteBuffer getGeobuf() {
            return geobuf == null ? null : geobuf.asByteBuffer();
        }

        public final void setGeobuf(ByteBuffer geobuf) {
            geobuf(geobuf == null ? null : SdkBytes.fromByteBuffer(geobuf));
        }

        @Override
        public final Builder geobuf(SdkBytes geobuf) {
            this.geobuf = geobuf;
            return this;
        }

        @Override
        public GeofenceGeometry build() {
            return new GeofenceGeometry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
