/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inferred state of the device, given the provided position, IP address, cellular signals, and Wi-Fi- access
 * points.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InferredState implements SdkPojo, Serializable, ToCopyableBuilder<InferredState.Builder, InferredState> {
    private static final SdkField<List<Double>> POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Position")
            .getter(getter(InferredState::position))
            .setter(setter(Builder::position))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Position").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PositionalAccuracy> ACCURACY_FIELD = SdkField
            .<PositionalAccuracy> builder(MarshallingType.SDK_POJO).memberName("Accuracy")
            .getter(getter(InferredState::accuracy)).setter(setter(Builder::accuracy)).constructor(PositionalAccuracy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accuracy").build()).build();

    private static final SdkField<Double> DEVIATION_DISTANCE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("DeviationDistance").getter(getter(InferredState::deviationDistance))
            .setter(setter(Builder::deviationDistance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviationDistance").build()).build();

    private static final SdkField<Boolean> PROXY_DETECTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ProxyDetected").getter(getter(InferredState::proxyDetected)).setter(setter(Builder::proxyDetected))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProxyDetected").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POSITION_FIELD,
            ACCURACY_FIELD, DEVIATION_DISTANCE_FIELD, PROXY_DETECTED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<Double> position;

    private final PositionalAccuracy accuracy;

    private final Double deviationDistance;

    private final Boolean proxyDetected;

    private InferredState(BuilderImpl builder) {
        this.position = builder.position;
        this.accuracy = builder.accuracy;
        this.deviationDistance = builder.deviationDistance;
        this.proxyDetected = builder.proxyDetected;
    }

    /**
     * For responses, this returns true if the service returned a value for the Position property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPosition() {
        return position != null && !(position instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The device position inferred by the provided position, IP address, cellular signals, and Wi-Fi- access points.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPosition} method.
     * </p>
     * 
     * @return The device position inferred by the provided position, IP address, cellular signals, and Wi-Fi- access
     *         points.
     */
    public final List<Double> position() {
        return position;
    }

    /**
     * <p>
     * The level of certainty of the inferred position.
     * </p>
     * 
     * @return The level of certainty of the inferred position.
     */
    public final PositionalAccuracy accuracy() {
        return accuracy;
    }

    /**
     * <p>
     * The distance between the inferred position and the device's self-reported position.
     * </p>
     * 
     * @return The distance between the inferred position and the device's self-reported position.
     */
    public final Double deviationDistance() {
        return deviationDistance;
    }

    /**
     * <p>
     * Indicates if a proxy was used.
     * </p>
     * 
     * @return Indicates if a proxy was used.
     */
    public final Boolean proxyDetected() {
        return proxyDetected;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasPosition() ? position() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accuracy());
        hashCode = 31 * hashCode + Objects.hashCode(deviationDistance());
        hashCode = 31 * hashCode + Objects.hashCode(proxyDetected());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InferredState)) {
            return false;
        }
        InferredState other = (InferredState) obj;
        return hasPosition() == other.hasPosition() && Objects.equals(position(), other.position())
                && Objects.equals(accuracy(), other.accuracy()) && Objects.equals(deviationDistance(), other.deviationDistance())
                && Objects.equals(proxyDetected(), other.proxyDetected());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InferredState").add("Position", position() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Accuracy", accuracy()).add("DeviationDistance", deviationDistance()).add("ProxyDetected", proxyDetected())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Position":
            return Optional.ofNullable(clazz.cast(position()));
        case "Accuracy":
            return Optional.ofNullable(clazz.cast(accuracy()));
        case "DeviationDistance":
            return Optional.ofNullable(clazz.cast(deviationDistance()));
        case "ProxyDetected":
            return Optional.ofNullable(clazz.cast(proxyDetected()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Position", POSITION_FIELD);
        map.put("Accuracy", ACCURACY_FIELD);
        map.put("DeviationDistance", DEVIATION_DISTANCE_FIELD);
        map.put("ProxyDetected", PROXY_DETECTED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InferredState, T> g) {
        return obj -> g.apply((InferredState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InferredState> {
        /**
         * <p>
         * The device position inferred by the provided position, IP address, cellular signals, and Wi-Fi- access
         * points.
         * </p>
         * 
         * @param position
         *        The device position inferred by the provided position, IP address, cellular signals, and Wi-Fi- access
         *        points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Collection<Double> position);

        /**
         * <p>
         * The device position inferred by the provided position, IP address, cellular signals, and Wi-Fi- access
         * points.
         * </p>
         * 
         * @param position
         *        The device position inferred by the provided position, IP address, cellular signals, and Wi-Fi- access
         *        points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Double... position);

        /**
         * <p>
         * The level of certainty of the inferred position.
         * </p>
         * 
         * @param accuracy
         *        The level of certainty of the inferred position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accuracy(PositionalAccuracy accuracy);

        /**
         * <p>
         * The level of certainty of the inferred position.
         * </p>
         * This is a convenience method that creates an instance of the {@link PositionalAccuracy.Builder} avoiding the
         * need to create one manually via {@link PositionalAccuracy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PositionalAccuracy.Builder#build()} is called immediately and its
         * result is passed to {@link #accuracy(PositionalAccuracy)}.
         * 
         * @param accuracy
         *        a consumer that will call methods on {@link PositionalAccuracy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accuracy(PositionalAccuracy)
         */
        default Builder accuracy(Consumer<PositionalAccuracy.Builder> accuracy) {
            return accuracy(PositionalAccuracy.builder().applyMutation(accuracy).build());
        }

        /**
         * <p>
         * The distance between the inferred position and the device's self-reported position.
         * </p>
         * 
         * @param deviationDistance
         *        The distance between the inferred position and the device's self-reported position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviationDistance(Double deviationDistance);

        /**
         * <p>
         * Indicates if a proxy was used.
         * </p>
         * 
         * @param proxyDetected
         *        Indicates if a proxy was used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proxyDetected(Boolean proxyDetected);
    }

    static final class BuilderImpl implements Builder {
        private List<Double> position = DefaultSdkAutoConstructList.getInstance();

        private PositionalAccuracy accuracy;

        private Double deviationDistance;

        private Boolean proxyDetected;

        private BuilderImpl() {
        }

        private BuilderImpl(InferredState model) {
            position(model.position);
            accuracy(model.accuracy);
            deviationDistance(model.deviationDistance);
            proxyDetected(model.proxyDetected);
        }

        public final Collection<Double> getPosition() {
            if (position instanceof SdkAutoConstructList) {
                return null;
            }
            return position;
        }

        public final void setPosition(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
        }

        @Override
        public final Builder position(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder position(Double... position) {
            position(Arrays.asList(position));
            return this;
        }

        public final PositionalAccuracy.Builder getAccuracy() {
            return accuracy != null ? accuracy.toBuilder() : null;
        }

        public final void setAccuracy(PositionalAccuracy.BuilderImpl accuracy) {
            this.accuracy = accuracy != null ? accuracy.build() : null;
        }

        @Override
        public final Builder accuracy(PositionalAccuracy accuracy) {
            this.accuracy = accuracy;
            return this;
        }

        public final Double getDeviationDistance() {
            return deviationDistance;
        }

        public final void setDeviationDistance(Double deviationDistance) {
            this.deviationDistance = deviationDistance;
        }

        @Override
        public final Builder deviationDistance(Double deviationDistance) {
            this.deviationDistance = deviationDistance;
            return this;
        }

        public final Boolean getProxyDetected() {
            return proxyDetected;
        }

        public final void setProxyDetected(Boolean proxyDetected) {
            this.proxyDetected = proxyDetected;
        }

        @Override
        public final Builder proxyDetected(Boolean proxyDetected) {
            this.proxyDetected = proxyDetected;
            return this;
        }

        @Override
        public InferredState build() {
            return new InferredState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
