/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an element of a leg within a route. A step contains instructions for how to move to the next step in the
 * leg.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Step implements SdkPojo, Serializable, ToCopyableBuilder<Step.Builder, Step> {
    private static final SdkField<List<Double>> START_POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("StartPosition")
            .getter(getter(Step::startPosition))
            .setter(setter(Builder::startPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartPosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> END_POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("EndPosition")
            .getter(getter(Step::endPosition))
            .setter(setter(Builder::endPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndPosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> DISTANCE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Distance").getter(getter(Step::distance)).setter(setter(Builder::distance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Distance").build()).build();

    private static final SdkField<Double> DURATION_SECONDS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("DurationSeconds").getter(getter(Step::durationSeconds)).setter(setter(Builder::durationSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DurationSeconds").build()).build();

    private static final SdkField<Integer> GEOMETRY_OFFSET_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("GeometryOffset").getter(getter(Step::geometryOffset)).setter(setter(Builder::geometryOffset))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeometryOffset").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(START_POSITION_FIELD,
            END_POSITION_FIELD, DISTANCE_FIELD, DURATION_SECONDS_FIELD, GEOMETRY_OFFSET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<Double> startPosition;

    private final List<Double> endPosition;

    private final Double distance;

    private final Double durationSeconds;

    private final Integer geometryOffset;

    private Step(BuilderImpl builder) {
        this.startPosition = builder.startPosition;
        this.endPosition = builder.endPosition;
        this.distance = builder.distance;
        this.durationSeconds = builder.durationSeconds;
        this.geometryOffset = builder.geometryOffset;
    }

    /**
     * For responses, this returns true if the service returned a value for the StartPosition property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStartPosition() {
        return startPosition != null && !(startPosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The starting position of a step. If the position is the first step in the leg, this position is the same as the
     * start position of the leg.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStartPosition} method.
     * </p>
     * 
     * @return The starting position of a step. If the position is the first step in the leg, this position is the same
     *         as the start position of the leg.
     */
    public final List<Double> startPosition() {
        return startPosition;
    }

    /**
     * For responses, this returns true if the service returned a value for the EndPosition property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEndPosition() {
        return endPosition != null && !(endPosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The end position of a step. If the position the last step in the leg, this position is the same as the end
     * position of the leg.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndPosition} method.
     * </p>
     * 
     * @return The end position of a step. If the position the last step in the leg, this position is the same as the
     *         end position of the leg.
     */
    public final List<Double> endPosition() {
        return endPosition;
    }

    /**
     * <p>
     * The travel distance between the step's <code>StartPosition</code> and <code>EndPosition</code>.
     * </p>
     * 
     * @return The travel distance between the step's <code>StartPosition</code> and <code>EndPosition</code>.
     */
    public final Double distance() {
        return distance;
    }

    /**
     * <p>
     * The estimated travel time, in seconds, from the step's <code>StartPosition</code> to the <code>EndPosition</code>
     * . . The travel mode and departure time that you specify in the request determines the calculated time.
     * </p>
     * 
     * @return The estimated travel time, in seconds, from the step's <code>StartPosition</code> to the
     *         <code>EndPosition</code>. . The travel mode and departure time that you specify in the request determines
     *         the calculated time.
     */
    public final Double durationSeconds() {
        return durationSeconds;
    }

    /**
     * <p>
     * Represents the start position, or index, in a sequence of steps within the leg's line string geometry. For
     * example, the index of the first step in a leg geometry is <code>0</code>.
     * </p>
     * <p>
     * Included in the response for queries that set <code>IncludeLegGeometry</code> to <code>True</code>.
     * </p>
     * 
     * @return Represents the start position, or index, in a sequence of steps within the leg's line string geometry.
     *         For example, the index of the first step in a leg geometry is <code>0</code>. </p>
     *         <p>
     *         Included in the response for queries that set <code>IncludeLegGeometry</code> to <code>True</code>.
     */
    public final Integer geometryOffset() {
        return geometryOffset;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasStartPosition() ? startPosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEndPosition() ? endPosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(distance());
        hashCode = 31 * hashCode + Objects.hashCode(durationSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(geometryOffset());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Step)) {
            return false;
        }
        Step other = (Step) obj;
        return hasStartPosition() == other.hasStartPosition() && Objects.equals(startPosition(), other.startPosition())
                && hasEndPosition() == other.hasEndPosition() && Objects.equals(endPosition(), other.endPosition())
                && Objects.equals(distance(), other.distance()) && Objects.equals(durationSeconds(), other.durationSeconds())
                && Objects.equals(geometryOffset(), other.geometryOffset());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Step").add("StartPosition", startPosition() == null ? null : "*** Sensitive Data Redacted ***")
                .add("EndPosition", endPosition() == null ? null : "*** Sensitive Data Redacted ***").add("Distance", distance())
                .add("DurationSeconds", durationSeconds()).add("GeometryOffset", geometryOffset()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StartPosition":
            return Optional.ofNullable(clazz.cast(startPosition()));
        case "EndPosition":
            return Optional.ofNullable(clazz.cast(endPosition()));
        case "Distance":
            return Optional.ofNullable(clazz.cast(distance()));
        case "DurationSeconds":
            return Optional.ofNullable(clazz.cast(durationSeconds()));
        case "GeometryOffset":
            return Optional.ofNullable(clazz.cast(geometryOffset()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StartPosition", START_POSITION_FIELD);
        map.put("EndPosition", END_POSITION_FIELD);
        map.put("Distance", DISTANCE_FIELD);
        map.put("DurationSeconds", DURATION_SECONDS_FIELD);
        map.put("GeometryOffset", GEOMETRY_OFFSET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Step, T> g) {
        return obj -> g.apply((Step) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Step> {
        /**
         * <p>
         * The starting position of a step. If the position is the first step in the leg, this position is the same as
         * the start position of the leg.
         * </p>
         * 
         * @param startPosition
         *        The starting position of a step. If the position is the first step in the leg, this position is the
         *        same as the start position of the leg.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startPosition(Collection<Double> startPosition);

        /**
         * <p>
         * The starting position of a step. If the position is the first step in the leg, this position is the same as
         * the start position of the leg.
         * </p>
         * 
         * @param startPosition
         *        The starting position of a step. If the position is the first step in the leg, this position is the
         *        same as the start position of the leg.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startPosition(Double... startPosition);

        /**
         * <p>
         * The end position of a step. If the position the last step in the leg, this position is the same as the end
         * position of the leg.
         * </p>
         * 
         * @param endPosition
         *        The end position of a step. If the position the last step in the leg, this position is the same as the
         *        end position of the leg.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endPosition(Collection<Double> endPosition);

        /**
         * <p>
         * The end position of a step. If the position the last step in the leg, this position is the same as the end
         * position of the leg.
         * </p>
         * 
         * @param endPosition
         *        The end position of a step. If the position the last step in the leg, this position is the same as the
         *        end position of the leg.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endPosition(Double... endPosition);

        /**
         * <p>
         * The travel distance between the step's <code>StartPosition</code> and <code>EndPosition</code>.
         * </p>
         * 
         * @param distance
         *        The travel distance between the step's <code>StartPosition</code> and <code>EndPosition</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distance(Double distance);

        /**
         * <p>
         * The estimated travel time, in seconds, from the step's <code>StartPosition</code> to the
         * <code>EndPosition</code>. . The travel mode and departure time that you specify in the request determines the
         * calculated time.
         * </p>
         * 
         * @param durationSeconds
         *        The estimated travel time, in seconds, from the step's <code>StartPosition</code> to the
         *        <code>EndPosition</code>. . The travel mode and departure time that you specify in the request
         *        determines the calculated time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationSeconds(Double durationSeconds);

        /**
         * <p>
         * Represents the start position, or index, in a sequence of steps within the leg's line string geometry. For
         * example, the index of the first step in a leg geometry is <code>0</code>.
         * </p>
         * <p>
         * Included in the response for queries that set <code>IncludeLegGeometry</code> to <code>True</code>.
         * </p>
         * 
         * @param geometryOffset
         *        Represents the start position, or index, in a sequence of steps within the leg's line string geometry.
         *        For example, the index of the first step in a leg geometry is <code>0</code>. </p>
         *        <p>
         *        Included in the response for queries that set <code>IncludeLegGeometry</code> to <code>True</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometryOffset(Integer geometryOffset);
    }

    static final class BuilderImpl implements Builder {
        private List<Double> startPosition = DefaultSdkAutoConstructList.getInstance();

        private List<Double> endPosition = DefaultSdkAutoConstructList.getInstance();

        private Double distance;

        private Double durationSeconds;

        private Integer geometryOffset;

        private BuilderImpl() {
        }

        private BuilderImpl(Step model) {
            startPosition(model.startPosition);
            endPosition(model.endPosition);
            distance(model.distance);
            durationSeconds(model.durationSeconds);
            geometryOffset(model.geometryOffset);
        }

        public final Collection<Double> getStartPosition() {
            if (startPosition instanceof SdkAutoConstructList) {
                return null;
            }
            return startPosition;
        }

        public final void setStartPosition(Collection<Double> startPosition) {
            this.startPosition = PositionCopier.copy(startPosition);
        }

        @Override
        public final Builder startPosition(Collection<Double> startPosition) {
            this.startPosition = PositionCopier.copy(startPosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder startPosition(Double... startPosition) {
            startPosition(Arrays.asList(startPosition));
            return this;
        }

        public final Collection<Double> getEndPosition() {
            if (endPosition instanceof SdkAutoConstructList) {
                return null;
            }
            return endPosition;
        }

        public final void setEndPosition(Collection<Double> endPosition) {
            this.endPosition = PositionCopier.copy(endPosition);
        }

        @Override
        public final Builder endPosition(Collection<Double> endPosition) {
            this.endPosition = PositionCopier.copy(endPosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endPosition(Double... endPosition) {
            endPosition(Arrays.asList(endPosition));
            return this;
        }

        public final Double getDistance() {
            return distance;
        }

        public final void setDistance(Double distance) {
            this.distance = distance;
        }

        @Override
        public final Builder distance(Double distance) {
            this.distance = distance;
            return this;
        }

        public final Double getDurationSeconds() {
            return durationSeconds;
        }

        public final void setDurationSeconds(Double durationSeconds) {
            this.durationSeconds = durationSeconds;
        }

        @Override
        public final Builder durationSeconds(Double durationSeconds) {
            this.durationSeconds = durationSeconds;
            return this;
        }

        public final Integer getGeometryOffset() {
            return geometryOffset;
        }

        public final void setGeometryOffset(Integer geometryOffset) {
            this.geometryOffset = geometryOffset;
        }

        @Override
        public final Builder geometryOffset(Integer geometryOffset) {
            this.geometryOffset = geometryOffset;
            return this;
        }

        @Override
        public Step build() {
            return new Step(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
