/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.m2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A batch job identifier in which the batch jobs to run are identified by an Amazon S3 location.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3BatchJobIdentifier implements SdkPojo, Serializable,
        ToCopyableBuilder<S3BatchJobIdentifier.Builder, S3BatchJobIdentifier> {
    private static final SdkField<String> BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("bucket")
            .getter(getter(S3BatchJobIdentifier::bucket)).setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucket").build()).build();

    private static final SdkField<JobIdentifier> IDENTIFIER_FIELD = SdkField.<JobIdentifier> builder(MarshallingType.SDK_POJO)
            .memberName("identifier").getter(getter(S3BatchJobIdentifier::identifier)).setter(setter(Builder::identifier))
            .constructor(JobIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identifier").build()).build();

    private static final SdkField<String> KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("keyPrefix").getter(getter(S3BatchJobIdentifier::keyPrefix)).setter(setter(Builder::keyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("keyPrefix").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD,
            IDENTIFIER_FIELD, KEY_PREFIX_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final JobIdentifier identifier;

    private final String keyPrefix;

    private S3BatchJobIdentifier(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.identifier = builder.identifier;
        this.keyPrefix = builder.keyPrefix;
    }

    /**
     * <p>
     * The Amazon S3 bucket that contains the batch job definitions.
     * </p>
     * 
     * @return The Amazon S3 bucket that contains the batch job definitions.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * Identifies the batch job definition. This identifier can also point to any batch job definition that already
     * exists in the application or to one of the batch job definitions within the directory that is specified in
     * <code>keyPrefix</code>.
     * </p>
     * 
     * @return Identifies the batch job definition. This identifier can also point to any batch job definition that
     *         already exists in the application or to one of the batch job definitions within the directory that is
     *         specified in <code>keyPrefix</code>.
     */
    public final JobIdentifier identifier() {
        return identifier;
    }

    /**
     * <p>
     * The key prefix that specifies the path to the folder in the S3 bucket that has the batch job definitions.
     * </p>
     * 
     * @return The key prefix that specifies the path to the folder in the S3 bucket that has the batch job definitions.
     */
    public final String keyPrefix() {
        return keyPrefix;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(identifier());
        hashCode = 31 * hashCode + Objects.hashCode(keyPrefix());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3BatchJobIdentifier)) {
            return false;
        }
        S3BatchJobIdentifier other = (S3BatchJobIdentifier) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(identifier(), other.identifier())
                && Objects.equals(keyPrefix(), other.keyPrefix());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3BatchJobIdentifier").add("Bucket", bucket()).add("Identifier", identifier())
                .add("KeyPrefix", keyPrefix()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "identifier":
            return Optional.ofNullable(clazz.cast(identifier()));
        case "keyPrefix":
            return Optional.ofNullable(clazz.cast(keyPrefix()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("bucket", BUCKET_FIELD);
        map.put("identifier", IDENTIFIER_FIELD);
        map.put("keyPrefix", KEY_PREFIX_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3BatchJobIdentifier, T> g) {
        return obj -> g.apply((S3BatchJobIdentifier) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3BatchJobIdentifier> {
        /**
         * <p>
         * The Amazon S3 bucket that contains the batch job definitions.
         * </p>
         * 
         * @param bucket
         *        The Amazon S3 bucket that contains the batch job definitions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * Identifies the batch job definition. This identifier can also point to any batch job definition that already
         * exists in the application or to one of the batch job definitions within the directory that is specified in
         * <code>keyPrefix</code>.
         * </p>
         * 
         * @param identifier
         *        Identifies the batch job definition. This identifier can also point to any batch job definition that
         *        already exists in the application or to one of the batch job definitions within the directory that is
         *        specified in <code>keyPrefix</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifier(JobIdentifier identifier);

        /**
         * <p>
         * Identifies the batch job definition. This identifier can also point to any batch job definition that already
         * exists in the application or to one of the batch job definitions within the directory that is specified in
         * <code>keyPrefix</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobIdentifier.Builder} avoiding the need
         * to create one manually via {@link JobIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #identifier(JobIdentifier)}.
         * 
         * @param identifier
         *        a consumer that will call methods on {@link JobIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identifier(JobIdentifier)
         */
        default Builder identifier(Consumer<JobIdentifier.Builder> identifier) {
            return identifier(JobIdentifier.builder().applyMutation(identifier).build());
        }

        /**
         * <p>
         * The key prefix that specifies the path to the folder in the S3 bucket that has the batch job definitions.
         * </p>
         * 
         * @param keyPrefix
         *        The key prefix that specifies the path to the folder in the S3 bucket that has the batch job
         *        definitions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyPrefix(String keyPrefix);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private JobIdentifier identifier;

        private String keyPrefix;

        private BuilderImpl() {
        }

        private BuilderImpl(S3BatchJobIdentifier model) {
            bucket(model.bucket);
            identifier(model.identifier);
            keyPrefix(model.keyPrefix);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final JobIdentifier.Builder getIdentifier() {
            return identifier != null ? identifier.toBuilder() : null;
        }

        public final void setIdentifier(JobIdentifier.BuilderImpl identifier) {
            this.identifier = identifier != null ? identifier.build() : null;
        }

        @Override
        public final Builder identifier(JobIdentifier identifier) {
            this.identifier = identifier;
            return this;
        }

        public final String getKeyPrefix() {
            return keyPrefix;
        }

        public final void setKeyPrefix(String keyPrefix) {
            this.keyPrefix = keyPrefix;
        }

        @Override
        public final Builder keyPrefix(String keyPrefix) {
            this.keyPrefix = keyPrefix;
            return this;
        }

        @Override
        public S3BatchJobIdentifier build() {
            return new S3BatchJobIdentifier(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
