/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medicalimaging;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.medicalimaging.model.CopyImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.CopyImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetDatastoreRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetDatastoreResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageFrameResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataResponse;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetRequest;
import software.amazon.awssdk.services.medicalimaging.model.GetImageSetResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse;
import software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest;
import software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse;
import software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobRequest;
import software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobResponse;
import software.amazon.awssdk.services.medicalimaging.model.TagResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.TagResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.UntagResourceRequest;
import software.amazon.awssdk.services.medicalimaging.model.UntagResourceResponse;
import software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataRequest;
import software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataResponse;
import software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsPublisher;
import software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresPublisher;
import software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsPublisher;
import software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsPublisher;

/**
 * Service client for accessing AWS Health Imaging asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * This is the <i>AWS HealthImaging API Reference</i>. AWS HealthImaging is a HIPAA eligible service that empowers
 * healthcare providers, life science organizations, and their software partners to store, analyze, and share medical
 * images in the cloud at petabyte scale. For an introduction to the service, see the <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/what-is.html"> <i>AWS HealthImaging Developer
 * Guide</i> </a>.
 * </p>
 * <note>
 * <p>
 * We recommend using one of the AWS Software Development Kits (SDKs) for your programming language, as they take care
 * of request authentication, serialization, and connection management. For more information, see <a
 * href="http://aws.amazon.com/developer/tools">Tools to build on AWS</a>.
 * </p>
 * </note>
 * <p>
 * The following sections list AWS HealthImaging API actions categorized according to functionality. Links are provided
 * to actions within this Reference, along with links back to corresponding sections in the <i>AWS HealthImaging
 * Developer Guide</i> where you can view tested code examples.
 * </p>
 * <p class="title">
 * <b>Data store actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_CreateDatastore.html">CreateDatastore</a>
 * – See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/create-data-store.html">Creating a data
 * store</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_GetDatastore.html">GetDatastore</a> – See
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/get-data-store.html">Getting data store
 * properties</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_ListDatastores.html">ListDatastores</a> –
 * See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/list-data-stores.html">Listing data
 * stores</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_DeleteDatastore.html">DeleteDatastore</a>
 * – See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/delete-data-store.html">Deleting a data
 * store</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Import job actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_StartDICOMImportJob.html">
 * StartDICOMImportJob</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/start-dicom-import-job.html">Starting an import
 * job</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_GetDICOMImportJob.html">GetDICOMImportJob</a>
 * – See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/get-dicom-import-job.html">Getting import
 * job properties</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_ListDICOMImportJobs.html">
 * ListDICOMImportJobs</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/list-dicom-import-jobs.html">Listing import jobs</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Image set access actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_SearchImageSets.html">SearchImageSets</a>
 * – See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/search-image-sets.html">Searching image
 * sets</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_GetImageSet.html">GetImageSet</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/get-image-set-properties.html">Getting image set
 * properties</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_GetImageSetMetadata.html">
 * GetImageSetMetadata</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/get-image-set-metadata.html">Getting image set
 * metadata</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_GetImageFrame.html">GetImageFrame</a> –
 * See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/get-image-frame.html">Getting image set pixel
 * data</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Image set modification actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_ListImageSetVersions.html">
 * ListImageSetVersions</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/list-image-set-versions.html">Listing image set
 * versions</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_UpdateImageSetMetadata.html">
 * UpdateImageSetMetadata</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/update-image-set-metadata.html">Updating image set
 * metadata</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_CopyImageSet.html">CopyImageSet</a> – See
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/copy-image-set.html">Copying an image set</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_DeleteImageSet.html">DeleteImageSet</a> –
 * See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/delete-image-set.html">Deleting an image
 * set</a>.
 * </p>
 * </li>
 * </ul>
 * <p class="title">
 * <b>Tagging actions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_TagResource.html">TagResource</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/tag-resource.html">Tagging a resource</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_ListTagsForResource.html">
 * ListTagsForResource</a> – See <a
 * href="https://docs.aws.amazon.com/healthimaging/latest/devguide/list-tag-resource.html">Listing tags for a
 * resource</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/healthimaging/latest/APIReference/API_UntagResource.html">UntagResource</a> –
 * See <a href="https://docs.aws.amazon.com/healthimaging/latest/devguide/untag-resource.html">Untagging a resource</a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface MedicalImagingAsyncClient extends AwsClient {
    String SERVICE_NAME = "medical-imaging";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "medical-imaging";

    /**
     * <p>
     * Copy an image set.
     * </p>
     *
     * @param copyImageSetRequest
     * @return A Java Future containing the result of the CopyImageSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.CopyImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CopyImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CopyImageSetResponse> copyImageSet(CopyImageSetRequest copyImageSetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Copy an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CopyImageSetRequest.Builder} avoiding the need to
     * create one manually via {@link CopyImageSetRequest#builder()}
     * </p>
     *
     * @param copyImageSetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.CopyImageSetRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CopyImageSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.CopyImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CopyImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CopyImageSetResponse> copyImageSet(Consumer<CopyImageSetRequest.Builder> copyImageSetRequest) {
        return copyImageSet(CopyImageSetRequest.builder().applyMutation(copyImageSetRequest).build());
    }

    /**
     * <p>
     * Create a data store.
     * </p>
     *
     * @param createDatastoreRequest
     * @return A Java Future containing the result of the CreateDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.CreateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CreateDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateDatastoreResponse> createDatastore(CreateDatastoreRequest createDatastoreRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Create a data store.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateDatastoreRequest.Builder} avoiding the need
     * to create one manually via {@link CreateDatastoreRequest#builder()}
     * </p>
     *
     * @param createDatastoreRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.CreateDatastoreRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.CreateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/CreateDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateDatastoreResponse> createDatastore(
            Consumer<CreateDatastoreRequest.Builder> createDatastoreRequest) {
        return createDatastore(CreateDatastoreRequest.builder().applyMutation(createDatastoreRequest).build());
    }

    /**
     * <p>
     * Delete a data store.
     * </p>
     * <note>
     * <p>
     * Before a data store can be deleted, you must first delete all image sets within it.
     * </p>
     * </note>
     *
     * @param deleteDatastoreRequest
     * @return A Java Future containing the result of the DeleteDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.DeleteDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteDatastoreResponse> deleteDatastore(DeleteDatastoreRequest deleteDatastoreRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Delete a data store.
     * </p>
     * <note>
     * <p>
     * Before a data store can be deleted, you must first delete all image sets within it.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteDatastoreRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteDatastoreRequest#builder()}
     * </p>
     *
     * @param deleteDatastoreRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.DeleteDatastoreRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.DeleteDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteDatastoreResponse> deleteDatastore(
            Consumer<DeleteDatastoreRequest.Builder> deleteDatastoreRequest) {
        return deleteDatastore(DeleteDatastoreRequest.builder().applyMutation(deleteDatastoreRequest).build());
    }

    /**
     * <p>
     * Delete an image set.
     * </p>
     *
     * @param deleteImageSetRequest
     * @return A Java Future containing the result of the DeleteImageSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.DeleteImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteImageSet"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteImageSetResponse> deleteImageSet(DeleteImageSetRequest deleteImageSetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Delete an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteImageSetRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteImageSetRequest#builder()}
     * </p>
     *
     * @param deleteImageSetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.DeleteImageSetRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteImageSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.DeleteImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/DeleteImageSet"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteImageSetResponse> deleteImageSet(Consumer<DeleteImageSetRequest.Builder> deleteImageSetRequest) {
        return deleteImageSet(DeleteImageSetRequest.builder().applyMutation(deleteImageSetRequest).build());
    }

    /**
     * <p>
     * Get the import job properties to learn more about the job or job progress.
     * </p>
     * <note>
     * <p>
     * The <code>jobStatus</code> refers to the execution of the import job. Therefore, an import job can return a
     * <code>jobStatus</code> as <code>COMPLETED</code> even if validation issues are discovered during the import
     * process. If a <code>jobStatus</code> returns as <code>COMPLETED</code>, we still recommend you review the output
     * manifests written to S3, as they provide details on the success or failure of individual P10 object imports.
     * </p>
     * </note>
     *
     * @param getDicomImportJobRequest
     * @return A Java Future containing the result of the GetDICOMImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetDicomImportJobResponse> getDICOMImportJob(GetDicomImportJobRequest getDicomImportJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the import job properties to learn more about the job or job progress.
     * </p>
     * <note>
     * <p>
     * The <code>jobStatus</code> refers to the execution of the import job. Therefore, an import job can return a
     * <code>jobStatus</code> as <code>COMPLETED</code> even if validation issues are discovered during the import
     * process. If a <code>jobStatus</code> returns as <code>COMPLETED</code>, we still recommend you review the output
     * manifests written to S3, as they provide details on the success or failure of individual P10 object imports.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDicomImportJobRequest.Builder} avoiding the need
     * to create one manually via {@link GetDicomImportJobRequest#builder()}
     * </p>
     *
     * @param getDicomImportJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetDicomImportJobRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetDICOMImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetDicomImportJobResponse> getDICOMImportJob(
            Consumer<GetDicomImportJobRequest.Builder> getDicomImportJobRequest) {
        return getDICOMImportJob(GetDicomImportJobRequest.builder().applyMutation(getDicomImportJobRequest).build());
    }

    /**
     * <p>
     * Get data store properties.
     * </p>
     *
     * @param getDatastoreRequest
     * @return A Java Future containing the result of the GetDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetDatastoreResponse> getDatastore(GetDatastoreRequest getDatastoreRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get data store properties.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDatastoreRequest.Builder} avoiding the need to
     * create one manually via {@link GetDatastoreRequest#builder()}
     * </p>
     *
     * @param getDatastoreRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetDatastoreRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetDatastoreResponse> getDatastore(Consumer<GetDatastoreRequest.Builder> getDatastoreRequest) {
        return getDatastore(GetDatastoreRequest.builder().applyMutation(getDatastoreRequest).build());
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getImageFrame(GetImageFrameRequest getImageFrameRequest,
            AsyncResponseTransformer<GetImageFrameResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageFrameRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageFrameRequest#builder()}
     * </p>
     *
     * @param getImageFrameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest.Builder} to create a
     *        request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getImageFrame(Consumer<GetImageFrameRequest.Builder> getImageFrameRequest,
            AsyncResponseTransformer<GetImageFrameResponse, ReturnT> asyncResponseTransformer) {
        return getImageFrame(GetImageFrameRequest.builder().applyMutation(getImageFrameRequest).build(), asyncResponseTransformer);
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     *
     * @param getImageFrameRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetImageFrameResponse> getImageFrame(GetImageFrameRequest getImageFrameRequest, Path destinationPath) {
        return getImageFrame(getImageFrameRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Get an image frame (pixel data) for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageFrameRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageFrameRequest#builder()}
     * </p>
     *
     * @param getImageFrameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageFrameRequest.Builder} to create a
     *        request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated image frame information.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageFrame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageFrame" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetImageFrameResponse> getImageFrame(Consumer<GetImageFrameRequest.Builder> getImageFrameRequest,
            Path destinationPath) {
        return getImageFrame(GetImageFrameRequest.builder().applyMutation(getImageFrameRequest).build(), destinationPath);
    }

    /**
     * <p>
     * Get image set properties.
     * </p>
     *
     * @param getImageSetRequest
     * @return A Java Future containing the result of the GetImageSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetImageSetResponse> getImageSet(GetImageSetRequest getImageSetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get image set properties.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetRequest.Builder} avoiding the need to
     * create one manually via {@link GetImageSetRequest#builder()}
     * </p>
     *
     * @param getImageSetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetImageSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSet" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetImageSetResponse> getImageSet(Consumer<GetImageSetRequest.Builder> getImageSetRequest) {
        return getImageSet(GetImageSetRequest.builder().applyMutation(getImageSetRequest).build());
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getImageSetMetadata(GetImageSetMetadataRequest getImageSetMetadataRequest,
            AsyncResponseTransformer<GetImageSetMetadataResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param getImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest.Builder} to create
     *        a request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getImageSetMetadata(
            Consumer<GetImageSetMetadataRequest.Builder> getImageSetMetadataRequest,
            AsyncResponseTransformer<GetImageSetMetadataResponse, ReturnT> asyncResponseTransformer) {
        return getImageSetMetadata(GetImageSetMetadataRequest.builder().applyMutation(getImageSetMetadataRequest).build(),
                asyncResponseTransformer);
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     *
     * @param getImageSetMetadataRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetImageSetMetadataResponse> getImageSetMetadata(
            GetImageSetMetadataRequest getImageSetMetadataRequest, Path destinationPath) {
        return getImageSetMetadata(getImageSetMetadataRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Get metadata attributes for an image set.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param getImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.GetImageSetMetadataRequest.Builder} to create
     *        a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The blob containing the aggregated metadata information for the image set.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.GetImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/GetImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetImageSetMetadataResponse> getImageSetMetadata(
            Consumer<GetImageSetMetadataRequest.Builder> getImageSetMetadataRequest, Path destinationPath) {
        return getImageSetMetadata(GetImageSetMetadataRequest.builder().applyMutation(getImageSetMetadataRequest).build(),
                destinationPath);
    }

    /**
     * <p>
     * List import jobs created for a specific data store.
     * </p>
     *
     * @param listDicomImportJobsRequest
     * @return A Java Future containing the result of the ListDICOMImportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListDicomImportJobsResponse> listDICOMImportJobs(
            ListDicomImportJobsRequest listDicomImportJobsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List import jobs created for a specific data store.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDicomImportJobsRequest.Builder} avoiding the
     * need to create one manually via {@link ListDicomImportJobsRequest#builder()}
     * </p>
     *
     * @param listDicomImportJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the ListDICOMImportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListDicomImportJobsResponse> listDICOMImportJobs(
            Consumer<ListDicomImportJobsRequest.Builder> listDicomImportJobsRequest) {
        return listDICOMImportJobs(ListDicomImportJobsRequest.builder().applyMutation(listDicomImportJobsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsPublisher publisher = client.listDICOMImportJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsPublisher publisher = client.listDICOMImportJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDicomImportJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDICOMImportJobsPublisher listDICOMImportJobsPaginator(ListDicomImportJobsRequest listDicomImportJobsRequest) {
        return new ListDICOMImportJobsPublisher(this, listDicomImportJobsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsPublisher publisher = client.listDICOMImportJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDICOMImportJobsPublisher publisher = client.listDICOMImportJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDICOMImportJobs(software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDicomImportJobsRequest.Builder} avoiding the
     * need to create one manually via {@link ListDicomImportJobsRequest#builder()}
     * </p>
     *
     * @param listDicomImportJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDicomImportJobsRequest.Builder} to create
     *        a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDICOMImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDICOMImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDICOMImportJobsPublisher listDICOMImportJobsPaginator(
            Consumer<ListDicomImportJobsRequest.Builder> listDicomImportJobsRequest) {
        return listDICOMImportJobsPaginator(ListDicomImportJobsRequest.builder().applyMutation(listDicomImportJobsRequest)
                .build());
    }

    /**
     * <p>
     * List data stores.
     * </p>
     *
     * @param listDatastoresRequest
     * @return A Java Future containing the result of the ListDatastores operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListDatastoresResponse> listDatastores(ListDatastoresRequest listDatastoresRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List data stores.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDatastoresRequest.Builder} avoiding the need to
     * create one manually via {@link ListDatastoresRequest#builder()}
     * </p>
     *
     * @param listDatastoresRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListDatastores operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListDatastoresResponse> listDatastores(Consumer<ListDatastoresRequest.Builder> listDatastoresRequest) {
        return listDatastores(ListDatastoresRequest.builder().applyMutation(listDatastoresRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresPublisher publisher = client.listDatastoresPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresPublisher publisher = client.listDatastoresPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatastoresRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDatastoresPublisher listDatastoresPaginator(ListDatastoresRequest listDatastoresRequest) {
        return new ListDatastoresPublisher(this, listDatastoresRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresPublisher publisher = client.listDatastoresPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListDatastoresPublisher publisher = client.listDatastoresPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatastores(software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDatastoresRequest.Builder} avoiding the need to
     * create one manually via {@link ListDatastoresRequest#builder()}
     * </p>
     *
     * @param listDatastoresRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListDatastoresRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListDatastores"
     *      target="_top">AWS API Documentation</a>
     */
    default ListDatastoresPublisher listDatastoresPaginator(Consumer<ListDatastoresRequest.Builder> listDatastoresRequest) {
        return listDatastoresPaginator(ListDatastoresRequest.builder().applyMutation(listDatastoresRequest).build());
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     *
     * @param listImageSetVersionsRequest
     * @return A Java Future containing the result of the ListImageSetVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListImageSetVersionsResponse> listImageSetVersions(
            ListImageSetVersionsRequest listImageSetVersionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List image set versions.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListImageSetVersionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListImageSetVersionsRequest#builder()}
     * </p>
     *
     * @param listImageSetVersionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the ListImageSetVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListImageSetVersionsResponse> listImageSetVersions(
            Consumer<ListImageSetVersionsRequest.Builder> listImageSetVersionsRequest) {
        return listImageSetVersions(ListImageSetVersionsRequest.builder().applyMutation(listImageSetVersionsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsPublisher publisher = client.listImageSetVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsPublisher publisher = client.listImageSetVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listImageSetVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListImageSetVersionsPublisher listImageSetVersionsPaginator(ListImageSetVersionsRequest listImageSetVersionsRequest) {
        return new ListImageSetVersionsPublisher(this, listImageSetVersionsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsPublisher publisher = client.listImageSetVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.ListImageSetVersionsPublisher publisher = client.listImageSetVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImageSetVersions(software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListImageSetVersionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListImageSetVersionsRequest#builder()}
     * </p>
     *
     * @param listImageSetVersionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListImageSetVersionsRequest.Builder} to create
     *        a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListImageSetVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListImageSetVersions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListImageSetVersionsPublisher listImageSetVersionsPaginator(
            Consumer<ListImageSetVersionsRequest.Builder> listImageSetVersionsRequest) {
        return listImageSetVersionsPaginator(ListImageSetVersionsRequest.builder().applyMutation(listImageSetVersionsRequest)
                .build());
    }

    /**
     * <p>
     * Lists all tags associated with a medical imaging resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all tags associated with a medical imaging resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.ListTagsForResourceRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     * <note>
     * <p>
     * <code>SearchImageSets</code> accepts a single search query parameter and returns a paginated response of all
     * image sets that have the matching criteria. All date range queries must be input as
     * <code>(lowerBound, upperBound)</code>.
     * </p>
     * <p>
     * By default, <code>SearchImageSets</code> uses the <code>updatedAt</code> field for sorting in descending order
     * from newest to oldest.
     * </p>
     * </note>
     *
     * @param searchImageSetsRequest
     * @return A Java Future containing the result of the SearchImageSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<SearchImageSetsResponse> searchImageSets(SearchImageSetsRequest searchImageSetsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Search image sets based on defined input attributes.
     * </p>
     * <note>
     * <p>
     * <code>SearchImageSets</code> accepts a single search query parameter and returns a paginated response of all
     * image sets that have the matching criteria. All date range queries must be input as
     * <code>(lowerBound, upperBound)</code>.
     * </p>
     * <p>
     * By default, <code>SearchImageSets</code> uses the <code>updatedAt</code> field for sorting in descending order
     * from newest to oldest.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchImageSetsRequest.Builder} avoiding the need
     * to create one manually via {@link SearchImageSetsRequest#builder()}
     * </p>
     *
     * @param searchImageSetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the SearchImageSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<SearchImageSetsResponse> searchImageSets(
            Consumer<SearchImageSetsRequest.Builder> searchImageSetsRequest) {
        return searchImageSets(SearchImageSetsRequest.builder().applyMutation(searchImageSetsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsPublisher publisher = client.searchImageSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsPublisher publisher = client.searchImageSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchImageSetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchImageSetsPublisher searchImageSetsPaginator(SearchImageSetsRequest searchImageSetsRequest) {
        return new SearchImageSetsPublisher(this, searchImageSetsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsPublisher publisher = client.searchImageSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.medicalimaging.paginators.SearchImageSetsPublisher publisher = client.searchImageSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchImageSets(software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchImageSetsRequest.Builder} avoiding the need
     * to create one manually via {@link SearchImageSetsRequest#builder()}
     * </p>
     *
     * @param searchImageSetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.SearchImageSetsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.SearchImageSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/SearchImageSets"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchImageSetsPublisher searchImageSetsPaginator(Consumer<SearchImageSetsRequest.Builder> searchImageSetsRequest) {
        return searchImageSetsPaginator(SearchImageSetsRequest.builder().applyMutation(searchImageSetsRequest).build());
    }

    /**
     * <p>
     * Start importing bulk data into an <code>ACTIVE</code> data store. The import job imports DICOM P10 files found in
     * the S3 prefix specified by the <code>inputS3Uri</code> parameter. The import job stores processing results in the
     * file specified by the <code>outputS3Uri</code> parameter.
     * </p>
     *
     * @param startDicomImportJobRequest
     * @return A Java Future containing the result of the StartDICOMImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.StartDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/StartDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDicomImportJobResponse> startDICOMImportJob(
            StartDicomImportJobRequest startDicomImportJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Start importing bulk data into an <code>ACTIVE</code> data store. The import job imports DICOM P10 files found in
     * the S3 prefix specified by the <code>inputS3Uri</code> parameter. The import job stores processing results in the
     * file specified by the <code>outputS3Uri</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartDicomImportJobRequest.Builder} avoiding the
     * need to create one manually via {@link StartDicomImportJobRequest#builder()}
     * </p>
     *
     * @param startDicomImportJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.StartDicomImportJobRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the StartDICOMImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.StartDICOMImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/StartDICOMImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartDicomImportJobResponse> startDICOMImportJob(
            Consumer<StartDicomImportJobRequest.Builder> startDicomImportJobRequest) {
        return startDICOMImportJob(StartDicomImportJobRequest.builder().applyMutation(startDicomImportJobRequest).build());
    }

    /**
     * <p>
     * Adds a user-specifed key and value tag to a medical imaging resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds a user-specifed key and value tag to a medical imaging resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from a medical imaging resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from a medical imaging resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Update image set metadata attributes.
     * </p>
     *
     * @param updateImageSetMetadataRequest
     * @return A Java Future containing the result of the UpdateImageSetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.UpdateImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UpdateImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateImageSetMetadataResponse> updateImageSetMetadata(
            UpdateImageSetMetadataRequest updateImageSetMetadataRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Update image set metadata attributes.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateImageSetMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateImageSetMetadataRequest#builder()}
     * </p>
     *
     * @param updateImageSetMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.medicalimaging.model.UpdateImageSetMetadataRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the UpdateImageSetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to throttling.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the service.</li>
     *         <li>InternalServerException An unexpected error occurred during processing of the request.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ServiceQuotaExceededException The request caused a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MedicalImagingException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MedicalImagingAsyncClient.UpdateImageSetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/medical-imaging-2023-07-19/UpdateImageSetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateImageSetMetadataResponse> updateImageSetMetadata(
            Consumer<UpdateImageSetMetadataRequest.Builder> updateImageSetMetadataRequest) {
        return updateImageSetMetadata(UpdateImageSetMetadataRequest.builder().applyMutation(updateImageSetMetadataRequest)
                .build());
    }

    @Override
    default MedicalImagingServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link MedicalImagingAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MedicalImagingAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MedicalImagingAsyncClient}.
     */
    static MedicalImagingAsyncClientBuilder builder() {
        return new DefaultMedicalImagingAsyncClientBuilder();
    }
}
