/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.memorydb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a collection of nodes in a cluster. One node in the node group is the read/write primary node. All the
 * other nodes are read-only Replica nodes.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Shard implements SdkPojo, Serializable, ToCopyableBuilder<Shard.Builder, Shard> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Shard::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Shard::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> SLOTS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Slots")
            .getter(getter(Shard::slots)).setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Slots").build()).build();

    private static final SdkField<List<Node>> NODES_FIELD = SdkField
            .<List<Node>> builder(MarshallingType.LIST)
            .memberName("Nodes")
            .getter(getter(Shard::nodes))
            .setter(setter(Builder::nodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Nodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Node> builder(MarshallingType.SDK_POJO)
                                            .constructor(Node::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> NUMBER_OF_NODES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfNodes").getter(getter(Shard::numberOfNodes)).setter(setter(Builder::numberOfNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfNodes").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STATUS_FIELD,
            SLOTS_FIELD, NODES_FIELD, NUMBER_OF_NODES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String status;

    private final String slots;

    private final List<Node> nodes;

    private final Integer numberOfNodes;

    private Shard(BuilderImpl builder) {
        this.name = builder.name;
        this.status = builder.status;
        this.slots = builder.slots;
        this.nodes = builder.nodes;
        this.numberOfNodes = builder.numberOfNodes;
    }

    /**
     * <p>
     * The name of the shard
     * </p>
     * 
     * @return The name of the shard
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The current state of this replication group - creating, available, modifying, deleting.
     * </p>
     * 
     * @return The current state of this replication group - creating, available, modifying, deleting.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The keyspace for this shard.
     * </p>
     * 
     * @return The keyspace for this shard.
     */
    public final String slots() {
        return slots;
    }

    /**
     * For responses, this returns true if the service returned a value for the Nodes property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasNodes() {
        return nodes != null && !(nodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list containing information about individual nodes within the shard
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodes} method.
     * </p>
     * 
     * @return A list containing information about individual nodes within the shard
     */
    public final List<Node> nodes() {
        return nodes;
    }

    /**
     * <p>
     * The number of nodes in the shard
     * </p>
     * 
     * @return The number of nodes in the shard
     */
    public final Integer numberOfNodes() {
        return numberOfNodes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodes() ? nodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(numberOfNodes());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Shard)) {
            return false;
        }
        Shard other = (Shard) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(status(), other.status())
                && Objects.equals(slots(), other.slots()) && hasNodes() == other.hasNodes()
                && Objects.equals(nodes(), other.nodes()) && Objects.equals(numberOfNodes(), other.numberOfNodes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Shard").add("Name", name()).add("Status", status()).add("Slots", slots())
                .add("Nodes", hasNodes() ? nodes() : null).add("NumberOfNodes", numberOfNodes()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "Nodes":
            return Optional.ofNullable(clazz.cast(nodes()));
        case "NumberOfNodes":
            return Optional.ofNullable(clazz.cast(numberOfNodes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Slots", SLOTS_FIELD);
        map.put("Nodes", NODES_FIELD);
        map.put("NumberOfNodes", NUMBER_OF_NODES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Shard, T> g) {
        return obj -> g.apply((Shard) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Shard> {
        /**
         * <p>
         * The name of the shard
         * </p>
         * 
         * @param name
         *        The name of the shard
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The current state of this replication group - creating, available, modifying, deleting.
         * </p>
         * 
         * @param status
         *        The current state of this replication group - creating, available, modifying, deleting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The keyspace for this shard.
         * </p>
         * 
         * @param slots
         *        The keyspace for this shard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(String slots);

        /**
         * <p>
         * A list containing information about individual nodes within the shard
         * </p>
         * 
         * @param nodes
         *        A list containing information about individual nodes within the shard
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodes(Collection<Node> nodes);

        /**
         * <p>
         * A list containing information about individual nodes within the shard
         * </p>
         * 
         * @param nodes
         *        A list containing information about individual nodes within the shard
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodes(Node... nodes);

        /**
         * <p>
         * A list containing information about individual nodes within the shard
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.memorydb.model.Node.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.memorydb.model.Node#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.memorydb.model.Node.Builder#build()} is called immediately and its
         * result is passed to {@link #nodes(List<Node>)}.
         * 
         * @param nodes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.memorydb.model.Node.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodes(java.util.Collection<Node>)
         */
        Builder nodes(Consumer<Node.Builder>... nodes);

        /**
         * <p>
         * The number of nodes in the shard
         * </p>
         * 
         * @param numberOfNodes
         *        The number of nodes in the shard
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfNodes(Integer numberOfNodes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String status;

        private String slots;

        private List<Node> nodes = DefaultSdkAutoConstructList.getInstance();

        private Integer numberOfNodes;

        private BuilderImpl() {
        }

        private BuilderImpl(Shard model) {
            name(model.name);
            status(model.status);
            slots(model.slots);
            nodes(model.nodes);
            numberOfNodes(model.numberOfNodes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final String getSlots() {
            return slots;
        }

        public final void setSlots(String slots) {
            this.slots = slots;
        }

        @Override
        public final Builder slots(String slots) {
            this.slots = slots;
            return this;
        }

        public final List<Node.Builder> getNodes() {
            List<Node.Builder> result = NodeListCopier.copyToBuilder(this.nodes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodes(Collection<Node.BuilderImpl> nodes) {
            this.nodes = NodeListCopier.copyFromBuilder(nodes);
        }

        @Override
        public final Builder nodes(Collection<Node> nodes) {
            this.nodes = NodeListCopier.copy(nodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodes(Node... nodes) {
            nodes(Arrays.asList(nodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodes(Consumer<Node.Builder>... nodes) {
            nodes(Stream.of(nodes).map(c -> Node.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getNumberOfNodes() {
            return numberOfNodes;
        }

        public final void setNumberOfNodes(Integer numberOfNodes) {
            this.numberOfNodes = numberOfNodes;
        }

        @Override
        public final Builder numberOfNodes(Integer numberOfNodes) {
            this.numberOfNodes = numberOfNodes;
            return this;
        }

        @Override
        public Shard build() {
            return new Shard(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
