/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworks.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.opsworks.OpsWorksClient;
import software.amazon.awssdk.services.opsworks.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.opsworks.model.DescribeAppsRequest;
import software.amazon.awssdk.services.opsworks.model.DescribeAppsResponse;
import software.amazon.awssdk.services.opsworks.model.DescribeDeploymentsRequest;
import software.amazon.awssdk.services.opsworks.model.DescribeDeploymentsResponse;
import software.amazon.awssdk.services.opsworks.model.DescribeInstancesRequest;
import software.amazon.awssdk.services.opsworks.model.DescribeInstancesResponse;
import software.amazon.awssdk.services.opsworks.model.OpsWorksRequest;
import software.amazon.awssdk.services.opsworks.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultOpsWorksWaiter implements OpsWorksWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final OpsWorksClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeAppsResponse> appExistsWaiter;

    private final Waiter<DescribeDeploymentsResponse> deploymentSuccessfulWaiter;

    private final Waiter<DescribeInstancesResponse> instanceOnlineWaiter;

    private final Waiter<DescribeInstancesResponse> instanceRegisteredWaiter;

    private final Waiter<DescribeInstancesResponse> instanceStoppedWaiter;

    private final Waiter<DescribeInstancesResponse> instanceTerminatedWaiter;

    private DefaultOpsWorksWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = OpsWorksClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.appExistsWaiter = Waiter.builder(DescribeAppsResponse.class).acceptors(appExistsWaiterAcceptors())
                .overrideConfiguration(appExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.deploymentSuccessfulWaiter = Waiter.builder(DescribeDeploymentsResponse.class)
                .acceptors(deploymentSuccessfulWaiterAcceptors())
                .overrideConfiguration(deploymentSuccessfulWaiterConfig(builder.overrideConfiguration)).build();
        this.instanceOnlineWaiter = Waiter.builder(DescribeInstancesResponse.class).acceptors(instanceOnlineWaiterAcceptors())
                .overrideConfiguration(instanceOnlineWaiterConfig(builder.overrideConfiguration)).build();
        this.instanceRegisteredWaiter = Waiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceRegisteredWaiterAcceptors())
                .overrideConfiguration(instanceRegisteredWaiterConfig(builder.overrideConfiguration)).build();
        this.instanceStoppedWaiter = Waiter.builder(DescribeInstancesResponse.class).acceptors(instanceStoppedWaiterAcceptors())
                .overrideConfiguration(instanceStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.instanceTerminatedWaiter = Waiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceTerminatedWaiterAcceptors())
                .overrideConfiguration(instanceTerminatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeAppsResponse> waitUntilAppExists(DescribeAppsRequest describeAppsRequest) {
        return appExistsWaiter.run(() -> client.describeApps(applyWaitersUserAgent(describeAppsRequest)));
    }

    @Override
    public WaiterResponse<DescribeAppsResponse> waitUntilAppExists(DescribeAppsRequest describeAppsRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return appExistsWaiter.run(() -> client.describeApps(applyWaitersUserAgent(describeAppsRequest)),
                appExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDeploymentsResponse> waitUntilDeploymentSuccessful(
            DescribeDeploymentsRequest describeDeploymentsRequest) {
        return deploymentSuccessfulWaiter
                .run(() -> client.describeDeployments(applyWaitersUserAgent(describeDeploymentsRequest)));
    }

    @Override
    public WaiterResponse<DescribeDeploymentsResponse> waitUntilDeploymentSuccessful(
            DescribeDeploymentsRequest describeDeploymentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return deploymentSuccessfulWaiter.run(
                () -> client.describeDeployments(applyWaitersUserAgent(describeDeploymentsRequest)),
                deploymentSuccessfulWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceOnline(DescribeInstancesRequest describeInstancesRequest) {
        return instanceOnlineWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceOnline(DescribeInstancesRequest describeInstancesRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return instanceOnlineWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceOnlineWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceRegistered(DescribeInstancesRequest describeInstancesRequest) {
        return instanceRegisteredWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceRegistered(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceRegisteredWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceStopped(DescribeInstancesRequest describeInstancesRequest) {
        return instanceStoppedWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceStopped(DescribeInstancesRequest describeInstancesRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return instanceStoppedWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceTerminated(DescribeInstancesRequest describeInstancesRequest) {
        return instanceTerminatedWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstancesResponse> waitUntilInstanceTerminated(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceTerminatedWaiter.run(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceTerminatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAppsResponse>> appExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAppsResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(400, WaiterState.FAILURE));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDeploymentsResponse>> deploymentSuccessfulWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDeploymentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Deployments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "successful"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Deployments").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Deployments[].Status=failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceOnlineWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "online"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=setup_failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "shutting_down"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=shutting_down) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "start_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=start_failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stopped) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stopping) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=terminating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminated"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=terminated) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stop_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stop_failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "registered"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=setup_failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "shutting_down"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=shutting_down) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stopped) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stopping) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=terminating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminated"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=terminated) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stop_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stop_failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "stopped"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "booting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=booting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=pending) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "requested"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=requested) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "running_setup"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=running_setup) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=setup_failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "start_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=start_failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stop_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=stop_failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceTerminatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "booting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=booting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "online"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=online) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=pending) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "requested"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=requested) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "running_setup"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=running_setup) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=setup_failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "start_failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Instances[].Status=start_failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration appExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration deploymentSuccessfulWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceOnlineWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceRegisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceTerminatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static OpsWorksWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends OpsWorksRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements OpsWorksWaiter.Builder {
        private OpsWorksClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public OpsWorksWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public OpsWorksWaiter.Builder client(OpsWorksClient client) {
            this.client = client;
            return this;
        }

        public OpsWorksWaiter build() {
            return new DefaultOpsWorksWaiter(this);
        }
    }
}
